/**
 * Plugin Name: CRS Booker
 * Plugin URI: http://www.car-rental-software.it
 * Description: CRS Booker
 * Author: My Appy
 * Author URI: http://www.myappy.it/
 * Version: 1.2.8
 * Text Domain: crs-booker
 * License: GPLv2 or later
 *
 * Copyright: (c) 2024-2025 My Appy (info@myappy.net)
 *
 * @author    My Appy
 * @copyright Copyright (c) 2024-2025, My Appy
 *
 */
var CRSBKR_BUTTONS_OK = 1;
var CRSBKR_BUTTONS_SUBMIT_CANCEL = 2;
var CRSBKR_BUTTONS_SUBMIT_CANCEL_DELETE = 3;

var CRSBKR_ALERT_COUNTER = 0;
var CRSBKR_COUNTRIES = [{"uid":"",code:"","name":""}, {"uid":"1","code":"AF","name":"AFGHANISTAN"},{"uid":"2","code":"AL","name":"ALBANIA"},{"uid":"3","code":"DZ","name":"ALGERIA"},{"uid":"4","code":"AD","name":"ANDORRA"},{"uid":"5","code":"AO","name":"ANGOLA"},{"uid":"6","code":"AI","name":"ANGUILLA"},{"uid":"7","code":"AG","name":"ANTIGUA E BARBUDA"},{"uid":"8","code":"AN","name":"ANTILLE OLANDESI"},{"uid":"9","code":"SA","name":"ARABIA SAUDITA"},{"uid":"10","code":"AR","name":"ARGENTINA"},{"uid":"11","code":"AM","name":"ARMENIA"},{"uid":"12","code":"AW","name":"ARUBA"},{"uid":"13","code":"AU","name":"AUSTRALIA"},{"uid":"14","code":"AT","name":"AUSTRIA"},{"uid":"15","code":"AZ","name":"AZERBAIGIAN"},{"uid":"16","code":"BS","name":"BAHAMAS"},{"uid":"17","code":"BH","name":"BAHRAIN"},{"uid":"18","code":"BD","name":"BANGLADESH"},{"uid":"19","code":"BB","name":"BARBADOS"},{"uid":"20","code":"BE","name":"BELGIO"},{"uid":"21","code":"BZ","name":"BELIZE"},{"uid":"22","code":"BJ","name":"BENIN"},{"uid":"23","code":"BM","name":"BERMUDA"},{"uid":"24","code":"BT","name":"BHUTAN"},{"uid":"25","code":"BY","name":"BIELORUSSIA"},{"uid":"26","code":"MM","name":"BIRMANIA"},{"uid":"27","code":"BO","name":"BOLIVIA"},{"uid":"28","code":"BQ","name":"BONAIRE"},{"uid":"29","code":"BA","name":"BOSNIA ED ERZEGOVINA"},{"uid":"30","code":"BW","name":"BOTSWANA"},{"uid":"31","code":"BR","name":"BRASILE"},{"uid":"32","code":"BN","name":"BRUNEI"},{"uid":"33","code":"BG","name":"BULGARIA"},{"uid":"34","code":"BF","name":"BURKINA FASO"},{"uid":"35","code":"BI","name":"BURUNDI"},{"uid":"36","code":"KH","name":"CAMBOGIA"},{"uid":"37","code":"CM","name":"CAMERUN"},{"uid":"38","code":"CA","name":"CANADA"},{"uid":"39","code":"CV","name":"CAPO VERDE"},{"uid":"40","code":"TD","name":"CIAD"},{"uid":"41","code":"CL","name":"CILE"},{"uid":"42","code":"CN","name":"CINA"},{"uid":"43","code":"CY","name":"CIPRO"},{"uid":"44","code":"VA","name":"CITTÀ DEL VATICANO"},{"uid":"45","code":"CO","name":"COLOMBIA"},{"uid":"46","code":"KM","name":"COMORE"},{"uid":"47","code":"KP","name":"COREA DEL NORD"},{"uid":"48","code":"KR","name":"COREA DEL SUD"},{"uid":"49","code":"CI","name":"COSTA D'AVORIO"},{"uid":"50","code":"CR","name":"COSTA RICA"},{"uid":"51","code":"HR","name":"CROAZIA"},{"uid":"52","code":"CU","name":"CUBA"},{"uid":"53","code":"CW","name":"CURAÇAO"},{"uid":"54","code":"DK","name":"DANIMARCA"},{"uid":"55","code":"DM","name":"DOMINICA"},{"uid":"56","code":"EC","name":"ECUADOR"},{"uid":"57","code":"EG","name":"EGITTO"},{"uid":"58","code":"SV","name":"EL SALVADOR"},{"uid":"59","code":"AE","name":"EMIRATI ARABI UNITI"},{"uid":"60","code":"ER","name":"ERITREA"},{"uid":"61","code":"EE","name":"ESTONIA"},{"uid":"62","code":"ET","name":"ETIOPIA"},{"uid":"63","code":"FJ","name":"FIGI"},{"uid":"64","code":"PH","name":"FILIPPINE"},{"uid":"65","code":"FI","name":"FINLANDIA"},{"uid":"66","code":"FR","name":"FRANCIA"},{"uid":"67","code":"GA","name":"GABON"},{"uid":"68","code":"GM","name":"GAMBIA"},{"uid":"69","code":"GE","name":"GEORGIA"},{"uid":"70","code":"GS","name":"GEORGIA DEL SUD E ISOLE SANDWICH MERIDIONALI"},{"uid":"71","code":"DE","name":"GERMANIA"},{"uid":"72","code":"GH","name":"GHANA"},{"uid":"73","code":"JM","name":"GIAMAICA"},{"uid":"74","code":"JP","name":"GIAPPONE"},{"uid":"75","code":"DJ","name":"GIBUTI"},{"uid":"76","code":"JO","name":"GIORDANIA"},{"uid":"77","code":"GR","name":"GRECIA"},{"uid":"78","code":"GD","name":"GRENADA"},{"uid":"79","code":"GL","name":"GROENLANDIA"},{"uid":"80","code":"GP","name":"GUADALUPA"},{"uid":"81","code":"GU","name":"GUAM"},{"uid":"82","code":"GT","name":"GUATEMALA"},{"uid":"83","code":"GN","name":"GUINEA"},{"uid":"84","code":"GQ","name":"GUINEA EQUATORIALE"},{"uid":"85","code":"GW","name":"GUINEA-BISSAU"},{"uid":"86","code":"GY","name":"GUYANA"},{"uid":"87","code":"GF","name":"GUYANA FRANCESE"},{"uid":"88","code":"HT","name":"HAITI"},{"uid":"89","code":"HN","name":"HONDURAS"},{"uid":"90","code":"HK","name":"HONG KONG"},{"uid":"91","code":"IN","name":"INDIA"},{"uid":"92","code":"ID","name":"INDONESIA"},{"uid":"93","code":"IR","name":"IRAN"},{"uid":"94","code":"IQ","name":"IRAQ"},{"uid":"95","code":"IE","name":"IRLANDA"},{"uid":"96","code":"IS","name":"ISLANDA"},{"uid":"97","code":"CX","name":"ISOLA DEL NATALE"},{"uid":"98","code":"NF","name":"ISOLA NORFOLK"},{"uid":"99","code":"KY","name":"ISOLE CAYMAN"},{"uid":"100","code":"CC","name":"ISOLE COCOS E KEELING"},{"uid":"101","code":"CK","name":"ISOLE COOK"},{"uid":"102","code":"FK","name":"ISOLE FALKLAND"},{"uid":"103","code":"MP","name":"ISOLE MARIANNE SETTENTRIONALI"},{"uid":"104","code":"MH","name":"ISOLE MARSHALL"},{"uid":"105","code":"PN","name":"ISOLE PITCAIRN"},{"uid":"106","code":"SB","name":"ISOLE SALOMONE"},{"uid":"107","code":"TF","name":"ISOLE SPARSE NELL'OCEANO INDIANO"},{"uid":"108","code":"VI","name":"ISOLE VERGINI AMERICANE"},{"uid":"109","code":"VG","name":"ISOLE VERGINI BRITANNICHE"},{"uid":"110","code":"IL","name":"ISRAELE"},{"uid":"111","code":"IT","name":"ITALIA"},{"uid":"112","code":"KZ","name":"KAZAKISTAN"},{"uid":"113","code":"KE","name":"KENYA"},{"uid":"114","code":"KG","name":"KIRGHIZISTAN"},{"uid":"115","code":"KI","name":"KIRIBATI"},{"uid":"116","code":"KW","name":"KUWAIT"},{"uid":"117","code":"LA","name":"LAOS"},{"uid":"118","code":"LS","name":"LESOTHO"},{"uid":"119","code":"LV","name":"LETTONIA"},{"uid":"120","code":"LB","name":"LIBANO"},{"uid":"121","code":"LR","name":"LIBERIA"},{"uid":"122","code":"LY","name":"LIBIA"},{"uid":"123","code":"LI","name":"LIECHTENSTEIN"},{"uid":"124","code":"LT","name":"LITUANIA"},{"uid":"125","code":"LU","name":"LUSSEMBURGO"},{"uid":"126","code":"MO","name":"MACAO"},{"uid":"127","code":"MG","name":"MADAGASCAR"},{"uid":"128","code":"MW","name":"MALAWI"},{"uid":"129","code":"MV","name":"MALDIVE"},{"uid":"130","code":"MY","name":"MALESIA"},{"uid":"131","code":"ML","name":"MALI"},{"uid":"132","code":"MT","name":"MALTA"},{"uid":"133","code":"MA","name":"MAROCCO"},{"uid":"134","code":"MQ","name":"MARTINICA"},{"uid":"135","code":"MR","name":"MAURITANIA"},{"uid":"136","code":"MU","name":"MAURITIUS"},{"uid":"137","code":"YT","name":"MAYOTTE"},{"uid":"138","code":"MX","name":"MESSICO"},{"uid":"139","code":"MD","name":"MOLDAVIA"},{"uid":"140","code":"MN","name":"MONGOLIA"},{"uid":"141","code":"ME","name":"MONTENEGRO"},{"uid":"142","code":"MS","name":"MONTSERRAT"},{"uid":"143","code":"MZ","name":"MOZAMBICO"},{"uid":"144","code":"NA","name":"NAMIBIA"},{"uid":"145","code":"NR","name":"NAURU"},{"uid":"146","code":"NP","name":"NEPAL"},{"uid":"147","code":"NI","name":"NICARAGUA"},{"uid":"148","code":"NE","name":"NIGER"},{"uid":"149","code":"NG","name":"NIGERIA"},{"uid":"150","code":"NU","name":"NIUE"},{"uid":"151","code":"NO","name":"NORVEGIA"},{"uid":"152","code":"NC","name":"NUOVA CALEDONIA"},{"uid":"153","code":"NZ","name":"NUOVA ZELANDA"},{"uid":"154","code":"OM","name":"OMAN"},{"uid":"155","code":"NL","name":"PAESI BASSI"},{"uid":"156","code":"PK","name":"PAKISTAN"},{"uid":"157","code":"PW","name":"PALAU"},{"uid":"158","code":"PS","name":"PALESTINA"},{"uid":"159","code":"PA","name":"PANAMA"},{"uid":"160","code":"PG","name":"PAPUA NUOVA GUINEA"},{"uid":"161","code":"PY","name":"PARAGUAY"},{"uid":"162","code":"PE","name":"PERÙ"},{"uid":"163","code":"PF","name":"POLINESIA FRANCESE"},{"uid":"164","code":"PL","name":"POLONIA"},{"uid":"165","code":"PR","name":"PORTO RICO"},{"uid":"166","code":"PT","name":"PORTOGALLO"},{"uid":"167","code":"MC","name":"MONACO"},{"uid":"168","code":"QA","name":"QATAR"},{"uid":"169","code":"GB","name":"REGNO UNITO"},{"uid":"170","code":"CZ","name":"REPUBBLICA CECA"},{"uid":"171","code":"CF","name":"REPUBBLICA CENTRAFRICANA"},{"uid":"172","code":"CG","name":"REPUBBLICA DEL CONGO"},{"uid":"173","code":"CD","name":"REPUBBLICA DEMOCRATICA DEL CONGO"},{"uid":"174","code":"MK","name":"MACEDONIA"},{"uid":"175","code":"DO","name":"REPUBBLICA DOMINICANA"},{"uid":"176","code":"RE","name":"RIUNIONE"},{"uid":"177","code":"RO","name":"ROMANIA"},{"uid":"178","code":"RW","name":"RUANDA"},{"uid":"179","code":"RU","name":"RUSSIA"},{"uid":"180","code":"BQ","name":"SABA"},{"uid":"181","code":"KN","name":"SAINT KITTS E NEVIS"},{"uid":"182","code":"VC","name":"SAINT VINCENT E GRENADINE"},{"uid":"183","code":"BL","name":"SAINT-BARTHÃ‰LEMY"},{"uid":"184","code":"MF","name":"SAINT-MARTIN"},{"uid":"185","code":"PM","name":"SAINT-PIERRE E MIQUELON"},{"uid":"186","code":"WS","name":"SAMOA"},{"uid":"187","code":"AS","name":"SAMOA AMERICANE"},{"uid":"188","code":"SM","name":"SAN MARINO"},{"uid":"189","code":"LC","name":"SANTA LUCIA"},{"uid":"190","code":"SH","name":"SANT'ELENA, ASCENSIONE E TRISTAN DA CUNHA"},{"uid":"191","code":"ST","name":"SÃO TOMÉ E PRÍNCIPE"},{"uid":"192","code":"SN","name":"SENEGAL"},{"uid":"193","code":"RS","name":"SERBIA"},{"uid":"194","code":"SC","name":"SEYCHELLES"},{"uid":"195","code":"SL","name":"SIERRA LEONE"},{"uid":"196","code":"SG","name":"SINGAPORE"},{"uid":"197","code":"BQ","name":"SINT EUSTATIUS"},{"uid":"198","code":"SX","name":"SINT MAARTEN"},{"uid":"199","code":"SY","name":"SIRIA"},{"uid":"200","code":"SK","name":"SLOVACCHIA"},{"uid":"201","code":"SI","name":"SLOVENIA"},{"uid":"202","code":"SO","name":"SOMALIA"},{"uid":"203","code":"ES","name":"SPAGNA"},{"uid":"204","code":"LK","name":"SRI LANKA"},{"uid":"205","code":"FM","name":"STATI FEDERATI DI MICRONESIA"},{"uid":"206","code":"US","name":"STATI UNITI D'AMERICA"},{"uid":"207","code":"ZA","name":"SUDAFRICA"},{"uid":"208","code":"SD","name":"SUDAN"},{"uid":"209","code":"SS","name":"SUDAN DEL SUD"},{"uid":"210","code":"SR","name":"SURINAME"},{"uid":"211","code":"SE","name":"SVEZIA"},{"uid":"212","code":"CH","name":"SVIZZERA"},{"uid":"213","code":"SZ","name":"SWAZILAND"},{"uid":"214","code":"TJ","name":"TAGIKISTAN"},{"uid":"215","code":"TW","name":"TAIWAN"},{"uid":"216","code":"TZ","name":"TANZANIA"},{"uid":"217","code":"TH","name":"THAILANDIA"},{"uid":"218","code":"TL","name":"TIMOR EST"},{"uid":"219","code":"TG","name":"TOGO"},{"uid":"220","code":"TK","name":"TOKELAU"},{"uid":"221","code":"TO","name":"TONGA"},{"uid":"222","code":"TT","name":"TRINIDAD E TOBAGO"},{"uid":"223","code":"TN","name":"TUNISIA"},{"uid":"224","code":"TR","name":"TURCHIA"},{"uid":"225","code":"TM","name":"TURKMENISTAN"},{"uid":"226","code":"TC","name":"TURKS E CAICOS"},{"uid":"227","code":"TV","name":"TUVALU"},{"uid":"228","code":"UA","name":"UCRAINA"},{"uid":"229","code":"UG","name":"UGANDA"},{"uid":"230","code":"HU","name":"UNGHERIA"},{"uid":"231","code":"UY","name":"URUGUAY"},{"uid":"232","code":"UZ","name":"UZBEKISTAN"},{"uid":"233","code":"VU","name":"VANUATU"},{"uid":"234","code":"VE","name":"VENEZUELA"},{"uid":"235","code":"VN","name":"VIETNAM"},{"uid":"236","code":"WF","name":"WALLIS E FUTUNA"},{"uid":"237","code":"YE","name":"YEMEN"},{"uid":"238","code":"ZM","name":"ZAMBIA"},{"uid":"239","code":"ZW","name":"ZIMBABWE"},{"uid":"240","code":"XK","name":"KOSOVO"}];

var CRSBKR_ID_TYPES = [ 'id', 'id_diplomatic', 'id_electronic', 'license', 'passport', 'passport_diplomatic', 'passport_service' ];

var CRSBKR_LANG = document.documentElement.lang;
if (CRSBKR_LANG && CRSBKR_LANG.indexOf('-') != -1)
{
	CRSBKR_LANG = CRSBKR_LANG.substr(0, CRSBKR_LANG.indexOf('-'));
}

if (CRSBKR_LANG && CRSBKR_LANG.indexOf('_') != -1)
{
	CRSBKR_LANG = CRSBKR_LANG.substr(0, CRSBKR_LANG.indexOf('_'));
}

var CRSBKR_RESULT_DETAILS =
[
	{ id : 'interior_dimensions', values : ['weight_limit', 'volume', 'height', 'width', 'length', 'wheel_arc_width'] },
	{ id : 'exterior_dimensions', values : ['wheelbase', 'height', 'width', 'length'] },
	{ id : 'vehicle_details', values : ['engine_size', 'fuel_type', 'emission_class', 'pax', 'lifting_platform', 'doors', 'radio', 'air', 'tank_size'] },
	{ id : 'chm_equipment', values : ['gps', 'mainsail', 'teak', 'bow_thruster'] },
	{ id : 'chm_characteristics', values : ['length_overall', 'year', 'type', 'engine', 'tank_size', 'water_tank_size'] },
	{ id : 'description', values : ['text'] }
];

function crsbkr_alert(message, buttons, element, positiveCallback, negativeCallback, deleteCallback)
{
	CRSBKR_ALERT_COUNTER++;

	if (!buttons)
	{
		buttons = CRSBKR_BUTTONS_OK;
	}

	// Disable the body.
	var body = document.getElementById('body');
	if (body == null)
	{
		var bodies = document.getElementsByTagName('body');
		body = bodies[0];
	}

	var holder = document.getElementById('alert-holder');
	if (holder == null)
	{
		holder = document.getElementById('body');
	}

	if (holder == null)
	{
		holder = body;
	}

	// Show the overlay.
	var overlay = document.getElementById('crsbkr_overlay');
	if (!overlay)
	{
		overlay = crsbkr_en('div');
		body.appendChild(overlay);
		overlay.id = 'crsbkr_overlay';
	}

	overlay.className = 'crsbkr_overlay-shown';

	// Create the element.
	var div = crsbkr_en('div');
	div.className = 'crsbkr_alert';
	div.onclick = overlay.onclick;

	message = message ? message.replace(/\n/g, '<br />') : '';
	if (message && message.length > 0)
	{
		var span = crsbkr_en('span');
		span.innerHTML = message;
		div.appendChild(span);

		div.message = span;
	}

	// Is there an input element given?
	if (element)
	{
		div.appendChild(element);
	}

	var buttonDiv = crsbkr_en('div');
	buttonDiv.className = 'buttons';
	if (buttons === CRSBKR_BUTTONS_OK || buttons.constructor === Array && buttons.length == 1)
	{
		var okButton = crsbkr_en('button');
		div.okButton = okButton;
		okButton.innerHTML = buttons == CRSBKR_BUTTONS_OK ? CRSBKR_STRINGS['label.ok'] : buttons[0];
		okButton.onclick = (e) =>
		{
			var leaveOpen = false;
			if (positiveCallback)
			{
				leaveOpen = positiveCallback();
			}

			if (!leaveOpen)
			{
				crsbkr_alert_dismiss(div);
			}
		};

		buttonDiv.appendChild(okButton);
	}
	else if (buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL || buttons.constructor === Array && buttons.length == 2)
	{
		var submitButton = crsbkr_en('button');
		div.submitButton = submitButton;
		submitButton.innerHTML = buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL ? CRSBKR_STRINGS['label.submit'] :
				(buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL ? CRSBKR_STRINGS['label.ok'] : buttons[0]);
		submitButton.onclick = (e) =>
		{
			var leaveOpen = false;
			if (positiveCallback)
			{
				leaveOpen = positiveCallback();
			}

			if (!leaveOpen)
			{
				crsbkr_alert_dismiss(div);
			}
		};

		var cancelButton = crsbkr_en('button');
		div.cancelButton = cancelButton;
		cancelButton.innerHTML = buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL ? CRSBKR_STRINGS['label.cancel'] : buttons[1];
		cancelButton.onclick = (e) =>
		{
			var leaveOpen = false;
			if (negativeCallback)
			{
				leaveOpen = negativeCallback();
			}

			if (!leaveOpen)
			{
				crsbkr_alert_dismiss(div);
			}
		};

		buttonDiv.appendChild(submitButton);
		buttonDiv.appendChild(cancelButton);
	}
	else if (buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL_DELETE || buttons.constructor === Array && buttons.length == 3)
	{
		var submitButton = crsbkr_en('button');
		div.submitButton = submitButton;
		submitButton.innerHTML = buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL_DELETE ? CRSBKR_STRINGS['label.submit'] : buttons[0];
		submitButton.onclick = (e) =>
		{
			var leaveOpen = false;
			if (positiveCallback)
			{
				leaveOpen = positiveCallback();
			}

			if (!leaveOpen)
			{
				crsbkr_alert_dismiss(div);
			}
		};

		var cancelButton = crsbkr_en('button');
		div.cancelButton = cancelButton;
		cancelButton.innerHTML = buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL_DELETE ? CRSBKR_STRINGS['label.cancel'] : buttons[1];
		cancelButton.onclick = (e) =>
		{
			var leaveOpen = false;
			if (negativeCallback)
			{
				leaveOpen = negativeCallback();
			}

			if (!leaveOpen)
			{
				crsbkr_alert_dismiss(div);
			}
		};

		var deleteButton = crsbkr_en('button');
		div.deleteButton = deleteButton;
		deleteButton.innerHTML = buttons === CRSBKR_BUTTONS_SUBMIT_CANCEL_DELETE ? CRSBKR_STRINGS['label.delete'] : buttons[2];
		deleteButton.onclick = (e) =>
		{
			var leaveOpen = false;
			if (deleteCallback)
			{
				leaveOpen = deleteCallback();
			}

			if (!leaveOpen)
			{
				crsbkr_alert_dismiss(div);
			}
		};

		buttonDiv.appendChild(submitButton);
		buttonDiv.appendChild(cancelButton);
		buttonDiv.appendChild(deleteButton);
	}

	div.appendChild(buttonDiv);
	holder.appendChild(div);

	return div;
}

function crsbkr_alert_dismiss(div)
{
	// Get the document.
	var body = document.getElementById('body');
	if (body == null)
	{
		var bodies = document.getElementsByTagName('body');
		body = bodies[0];
	}

	var holder = document.getElementById('alert-holder');
	if (holder == null)
	{
		holder = document.getElementById('body');
	}

	if (holder == null)
	{
		holder = body;
	}

	if (div.parentNode)
	{
		if (--CRSBKR_ALERT_COUNTER == 0)
		{
			var overlay = document.getElementById('crsbkr_overlay');
			if (overlay)
			{
				overlay.className = 'crsbkr_overlay';
			}
		}

		div.parentNode.removeChild(div);
	}

	if (holder != body)
	{
		holder.style.display = 'none';
	}
}

function crsbkr_array_get(a, k, noe, d)
{
	if (!a || !k && typeof k != 'string')
	{
		if (noe !== undefined)
		{
			return noe;
		}
		else
		{
			return d;
		}
	}

	var p = ('' + k).indexOf('.');
	var v = null;

	if (a && Object.prototype.toString.call(a) == '[object Object]' && a.hasOwnProperty(k) || Array.isArray(a) && !isNaN(k) && a.length > k || p == -1)
	{
		v = a && Object.prototype.toString.call(a) == '[object Object]' && a.hasOwnProperty(k) || Array.isArray(a) && !isNaN(k) && a.length > k ? a[k] : d;
		if (noe !== undefined)
		{
			if (v === undefined || v === null || v === '')
			{
				v = noe;
			}
			else
			{
			}
		}
	}
	else
	{
		var kk = k.substr(0, p);
		k = k.substr(p + 1);

		v = a && Object.prototype.toString.call(a) == '[object Object]' && a.hasOwnProperty(kk) || Array.isArray(a) && !isNaN(kk) && a.length > kk ? crsbkr_array_get(a[kk], k, noe, d) : d;
		if (noe !== undefined)
		{
			if (v === undefined || v === null || v === '')
			{
				v = noe;
			}
			else
			{
			}
		}
	}

	return v;
}

function crsbkr_array_set(a, k, v)
{
	if (!a || Object.prototype.toString.call(a) != '[object Object]' && Object.prototype.toString.call(a) != '[object Array]')
	{
		return;
	}

	var p = k.indexOf('.');
	if (p == -1)
	{
		a[k] = v;
	}
	else
	{
		var kk = k.substr(0, p);
		k = k.substr(p + 1);

		if (!a.hasOwnProperty(kk))
		{
			a[kk] = {};
		}
		else if (Array.isArray(a[kk]) && a[kk].length == 0)
		{
			a[kk] = {};
		}

		crsbkr_array_set(a[kk], k, v);
	}
}

function crsbkr_base64_encode(input)
{
	var keyStr = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
	var bytes = (input.length / 4) * 3;
	var arrayBuffer = new ArrayBuffer(bytes);

	var removePaddingChars = function(input)
	{
		var lkey = keyStr.indexOf(input.charAt(input.length - 1));
		if (lkey == 64)
		{
			return input.substring(0, input.length - 1);
		}

		return input;
	}

	// Get last chars to see if are valid
	input = removePaddingChars(input);
	input = removePaddingChars(input);

	var bytes = parseInt((input.length / 4) * 3, 10);

	var uarray;
	var chr1, chr2, chr3;
	var enc1, enc2, enc3, enc4;
	var i = 0;
	var j = 0;

	if (arrayBuffer)
	{
		uarray = new Uint8Array(arrayBuffer);
	}
	else
	{
		uarray = new Uint8Array(bytes);
	}

	input = input.replace(/[^A-Za-z0-9\+\/\=]/g, "");

	for (i = 0; i < bytes; i += 3)
	{
		// Get the 3 octects in 4 ascii chars
		enc1 = keyStr.indexOf(input.charAt(j++));
		enc2 = keyStr.indexOf(input.charAt(j++));
		enc3 = keyStr.indexOf(input.charAt(j++));
		enc4 = keyStr.indexOf(input.charAt(j++));

		chr1 = (enc1 << 2) | (enc2 >> 4);
		chr2 = ((enc2 & 15) << 4) | (enc3 >> 2);
		chr3 = ((enc3 & 3) << 6) | enc4;

		uarray[i] = chr1;

		if (enc3 != 64) uarray[i+1] = chr2;
		if (enc4 != 64) uarray[i+2] = chr3;
	}

	return uarray;
}

function crsbkr_currency(value, forServer)
{
	if (value != null && ('' + value).length > 0)
	{
		value = Math.round(('' + value).replace(',', '.') * 100);
		var euros = value >= 0 ? Math.floor(value / 100) : Math.ceil(value / 100);
		var cents = '' + Math.abs(value % 100);
		while (cents.length < 2)
		{
			cents = '0' + cents;
		}

		value = (value < 0 ? (forServer ? '-' : '- ') : '') + (forServer ? '' : '€ ')
				+ Math.abs(euros) + (forServer ? '.' : ',') + cents;
	}
	else
	{
		value = '';
	}

	return value;
}

function crsbkr_date(value, forServer, type)
{
	if (value != null && Object.prototype.toString.call(value) != '[object Date]')
	{
		if (forServer)
		{
			value = value.substring(6, 10) + '-' + value.substring(3, 5) + '-' + value.substring(0, 2) + value.substring(10);
			if (value.length == 10)
			{
				value += ' 00:00:00';
			}
		}
		else
		{
		}
	}

	var date = value == null || Object.prototype.toString.call(value) == '[object Date]' ? value : new Date(value.replace(' ', 'T') + (forServer ? '' : '+00:00'));
	if (date != null && date != undefined && Object.prototype.toString.call(date) == '[object Date]'
			&& !isNaN(date.getTime()))
	{
		var year = forServer ? date.getUTCFullYear() : date.getFullYear();
		var month = forServer ? date.getUTCMonth() + 1 : date.getMonth() + 1;
		var day = forServer ? date.getUTCDate() : date.getDate();
		var hours = forServer ? date.getUTCHours() : date.getHours();
		var min = forServer ? date.getUTCMinutes() : date.getMinutes();
		var sec = forServer ? date.getUTCSeconds() : date.getSeconds();

		year = (year < 10 ? '000' : (year < 100 ? '00' : (year < 1000 ? '0' : ''))) + year;
		month = (month < 10 ? '0' : '') + month;
		day = (day < 10 ? '0' : '') + day;
		hours = (hours < 10 ? '0' : '') + hours;
		min = (min < 10 ? '0' : '') + min;
		sec = (sec < 10 ? '0' : '') + sec;

		if (forServer)
		{
			value = year + '-' + month + '-' + day + ' ' + hours + ':' + min + ':' + sec;
		}
		else
		{
			value = type == undefined || type == 'date' || type == 'datetime' ? day + '-' + month + '-' + year : '';
			value += (type == undefined || type == 'time' || type == 'datetime') ? (value == '' ? '' : ' ') + hours + ':' + min : '';
		}
	}
	else
	{
		value = '';
	}

	return value;
}

async function crsbkr_digest(data)
{
	const utf8 = new TextEncoder().encode(document.cookie);
	const hashBuffer = await window.crypto.subtle.digest('SHA-256', utf8);
	const hashArray = Array.from(new Uint8Array(hashBuffer));

	return hashArray.map((bytes) => bytes.toString(16).padStart(2, '0')).join('');
}

function crsbkr_e(id)
{
	return document.getElementById(id);
}


function crsbkr_en(e)
{
	return document.createElement(e);
}

function crsbkr_error(error, message, callback)
{
	if (error && typeof error === 'object')
	{
		if (!message)
		{
			message = error && error.message ? error.message : null;
		}

		error = error.error ? error.error : null;
	}

	var thisMessage = CRSBKR_STRINGS.error + "\n" + error
			+ (message && typeof message === 'object' ? ' - ' + JSON.stringify(message) : (message && message.length > 0 ? ' - ' + message : ''));
	crsbkr_alert(thisMessage, CRSBKR_BUTTONS_OK, null, () =>
	{
		if (callback)
		{
			callback();
		}

		return false;

	}, null);

	return thisMessage;
}

function crsbkr_fbq(a, event, params)
{
	if (typeof fbq !== 'undefined')
	{
		fbq(a, event, params);

		var fbp = document.cookie.split(";").filter(function(c) { return c.includes('_fbp=') }).map(function (c) { return c.split('_fbp=')[1]; });
		fbp = fbp ? fbp[0] : null;

		var fbc = document.cookie.split(";").filter(function(c) { return c.includes('_fbc=') }).map(function (c) { return c.split('_fbc=')[1]; });
		fbc = fbc ? fbc[0] : null;

		crsbkr_request('POST', 'fbq',
		{
			action_source : 'website',
			event_name : event,
			event_source_url : window.location.href,
			event_time : Math.floor(Date.now() / 1000),
			custom_data : params,
			user_data :
			{
				client_user_agent : navigator.userAgent,
				fbc : fbc,
				fbp : fbp,
			},
		});
	}
}

function crsbkr_format_date(value, forServer, type, dateHolder)
{
	var result = '';
	var timezone = 'Europe/Rome';

	if (value != null)
	{
		if (Object.prototype.toString.call(value) == '[object Date]')
		{
			var year = value.getFullYear();
			var month = value.getMonth() + 1;
			var day = value.getDate();
			var hours = value.getHours();
			var min = value.getMinutes();
			var sec = value.getSeconds();

			month = (month < 10 ? '0' : '') + month;
			day = (day < 10 ? '0' : '') + day;
			hours = (hours < 10 ? '0' : '') + hours;
			min = (min < 10 ? '0' : '') + min;
			sec = (sec < 10 ? '0' : '') + sec;

			if (forServer)
			{
				value = day + '-' + month + '-' + year + ' ' + hours + ':' + min + ':' + sec;
			}
			else
			{
				value = year + '-' + month + '-' + day + ' ' + hours + ':' + min + ':' + sec;
			}
		}

		if (forServer)
		{
			value = value.substring(6, 10) + '-' + value.substring(3, 5) + '-' + value.substring(0, 2) + value.substring(10);
			if (value.length == 10)
			{
				value += 'T00:00:00';
			}
			else
			{
				value = value.replace(' ', 'T');
			}

			try
			{
				var offset = new Date(value).toLocaleString('en-US', { timeZone : timezone, timeZoneName : 'longOffset' });
				offset = offset.substr(-6);

				value += offset;

				var thisDate = new Date(value);
				result = thisDate.toISOString().replace('T', ' ').substr(0, 19);

				if (dateHolder)
				{
					dateHolder.date = thisDate;
				}
			}
			catch (e)
			{
				result = '';
			}
		}
		else
		{
			try
			{
				result = new Date(value.replace(' ', 'T') + '+00:00');
				if (!result || result == 'Invalid Date')
				{
					result = '';
				}
				else
				{
					if (dateHolder)
					{
						dateHolder.date = result;
					}

					result = result.toLocaleString('it-IT', { timeZone : timezone, timeZoneName : 'longOffset' }).replaceAll('/', '-').replace(', ', ' ').substr(0, 19);
					if (type == undefined || type == 'date')
					{
						result = result.substr(0, 10);
					}
					else if (type == 'time')
					{
						result = result.substr(11, 5);
					}
					else
					{
						result = result.substr(0, 16);
					}
				}
			}
			catch (e)
			{
				result = '';
			}
		}
	}

	return result;
}

function crsbkr_loading()
{
	var div = crsbkr_en('div');
	div.className = 'crsbkr_loading';
	div.innerHTML = '<div><div></div><div></div><div></div><div></div></div>';

	return div;
}

function crsbkr_module(data)
{
	var sessionHex = crsbkr_digest(document.cookie);

	crsbkr_fbq('trackCustom', 'crsbkr_module', { controller_form : crsbkr_array_get(data, 'controller_form.code', null), session : sessionHex });

	var holder = document.getElementById('crsbkr_module');
	while (holder.lastChild)
	{
		holder.removeChild(holder.lastChild);
	}

	var days = [ 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday' ];

	if (crsbkr_array_get(data, 'controller_form.type') == 'search')
	{
		var form = crsbkr_en('div');
		holder.appendChild(form);
		form.classList.add('search');
		form.id = 'crsbkr_search_form';

		var sync = null;

		var params = localStorage.getItem('crsbkr_module_' + crsbkr_array_get(data, 'controller_form.uid'));
		if (params && params.length > 0)
		{
			params = JSON.parse(params);
		}

		if (!params)
		{
			params = {};
		}

		///
		//
		// GROUPS
		//
		///
		var groups = crsbkr_array_get(data, 'controller_form.params.groups', []);
		if (crsbkr_array_get(data, 'controller_form.params.select_group', false))
		{
			var groupsDiv = crsbkr_en('div');
			form.appendChild(groupsDiv);
			groupsDiv.classList.add('groups');

			var ul = crsbkr_en('ul');
			groupsDiv.appendChild(ul);

			for (var i = 0; i < groups.length; i++)
			{
				var li = crsbkr_en('li');
				ul.appendChild(li);

				if (crsbkr_array_get(groups[i], 'icon_url', '') != '')
				{
					var img = crsbkr_en('img');
					li.appendChild(img);
					img.src = groups[i].icon_url;
				}

				var span = crsbkr_en('span');
				li.appendChild(span);
				span.innerHTML = crsbkr_array_get(groups[i], 'translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(groups[i], 'translations.en.name', groups[i].name));

				li.onclick = ((ul, li, i, group) =>
				{
					return () =>
					{
						params.group = i;
						for (var ii = 0; ii < ul.children.length; ii++)
						{
							ul.children[ii].classList.remove('selected');
						}

						li.classList.add('selected');

						if (li.offsetLeft < ul.scrollLeft)
						{
							ul.scrollLeft = Math.max(0, li.offsetLeft - 5);
							ul.onscroll();
						}
						else if (li.offsetLeft + li.offsetWidth > ul.scrollLeft + ul.offsetWidth)
						{
							ul.scrollLeft = li.offsetLeft + li.offsetWidth - ul.offsetWidth;
							ul.onscroll();
						}

						if (sync)
						{
							sync();
						}
					}
				})(ul, li, i, groups[i]);

				if (crsbkr_array_get(params, 'group', 0) == i)
				{
					li.classList.add('selected');
				}
			}

			var leftScroller = crsbkr_en('span');
			leftScroller.className = 'groups_lscroller';
			groupsDiv.appendChild(leftScroller);
			leftScroller.onclick = ((ul) =>
			{
				return () =>
				{
					var width = parseInt(ul.children[0].offsetWidth) + parseInt(window.getComputedStyle(ul.children[0]).marginRight.replace('px', ''));
					var index = Math.floor(ul.scrollLeft / width);

					var selectedIndex = crsbkr_array_get(params, 'group', 0);
					ul.children[Math.max(0, selectedIndex - 1)].onclick();
				}
			})(ul);

			var rightScroller = crsbkr_en('span');
			rightScroller.className = 'groups_rscroller';
			groupsDiv.appendChild(rightScroller);
			rightScroller.onclick = ((ul) =>
			{
				return () =>
				{
					var width = parseInt(ul.children[0].offsetWidth) + parseInt(window.getComputedStyle(ul.children[0]).marginRight.replace('px', ''));
					var index = Math.floor(ul.scrollLeft / width);

					var selectedIndex = crsbkr_array_get(params, 'group', 0);

					ul.children[Math.min(ul.children.length - 1, selectedIndex + 1)].onclick();
				}
			})(ul);

			ul.onscroll = ((ul, leftScroller, rightScroller) =>
			{
				return () =>
				{
					leftScroller.style.display = ul.scrollLeft > 0 ? 'block' : 'none';
					rightScroller.style.display = ul.scrollLeft < ul.scrollWidth - ul.offsetWidth - 12 ? 'block' : 'none';
				}
			})(ul, leftScroller, rightScroller);

			leftScroller.style.display = ul.scrollLeft > 0 ? 'block' : 'none';
			rightScroller.style.display = ul.scrollLeft < ul.scrollWidth - ul.offsetWidth - 12 ? 'block' : 'none';

			for (var i = 0; i < ul.children.length; i++)
			{
				if (ul.children[i].classList.contains('selected'))
				{
					ul.children[i].onclick();
				}
			}
		}

		///
		// BASES
		// Only show the bases with the given groups.
		///
		var bases = [];
		for (var i = 0; i < crsbkr_array_get(data, 'controller_form.params.bases', []).length; i++)
		{
			var base = data.controller_form.params.bases[i];
			bases.push(base);
		}

		var fieldset = crsbkr_en('fieldset');
		form.appendChild(fieldset);

		var checkoutBaseInput = crsbkr_en('span');
		var checkoutBaseList = crsbkr_en('ul');
		var checkoutDateError = crsbkr_en('div');
		var checkoutDay = crsbkr_en('span');
		var checkoutDow = crsbkr_en('span');
		var checkoutMonth = crsbkr_en('span');
		var checkoutTime = crsbkr_en('span');
		var checkoutTimeError = crsbkr_en('span');
		var checkoutTimeHolder = crsbkr_en('span');
		var checkoutTimeIndicator = crsbkr_en('span');
		var checkoutTimeIndicatorBubble = crsbkr_en('span');
		var checkoutTimeIndicatorLabel = crsbkr_en('span');
		var checkoutTimeIndicatorLeft = crsbkr_en('span');
		var checkoutTimeIndicatorRight = crsbkr_en('span');
		var checkoutTimeLabel = crsbkr_en('span');
		var checkoutTimeSet = {};

		var checkinBaseInput = crsbkr_en('span');
		var checkinBaseLabel = crsbkr_en('p');
		var checkinBaseList = crsbkr_en('ul');
		var checkinBaseSelect = crsbkr_en('input');
		var checkinDateError = crsbkr_en('div');
		var checkinDay = crsbkr_en('span');
		var checkinDow = crsbkr_en('span');
		var checkinMonth = crsbkr_en('span');
		var checkinTime = crsbkr_en('span');
		var checkinTimeError = crsbkr_en('span');
		var checkinTimeHolder = crsbkr_en('span');
		var checkinTimeIndicator = crsbkr_en('span');
		var checkinTimeIndicatorBubble = crsbkr_en('span');
		var checkinTimeIndicatorLabel = crsbkr_en('span');
		var checkinTimeIndicatorLeft = crsbkr_en('span');
		var checkinTimeIndicatorRight = crsbkr_en('span');
		var checkinTimeLabel = crsbkr_en('span');
		var checkinTimeSet = {};

		for (var h = 0; h < 24; h++)
		{
			for (var m = 0; m < 60; m += 15)
			{
				var key = (h < 10 ? '0' : '') + h + ':' + (m < 10 ? '0' : '') + m;
				checkoutTimeSet[key] = true;
				checkinTimeSet[key] = true;
			}
		}

		var calendar = crsbkr_en('div');

		var searchButton = crsbkr_en('button');

		checkoutDateError.classList.add('error');
		checkinDateError.classList.add('error');

		///
		//
		// CHECKOUT BASE
		//
		///
		var p = crsbkr_en('p');
		fieldset.appendChild(p);
		p.innerHTML = CRSBKR_STRINGS['search.checkout_base'];

		var div = crsbkr_en('div');
		fieldset.appendChild(div);
		div.classList.add('input');

		div.appendChild(checkoutBaseInput);

		var innerHolder = crsbkr_en('div');
		div.appendChild(innerHolder);

		var blur = crsbkr_en('div');
		div.appendChild(blur);
		blur.classList.add('blur');

		var p = crsbkr_en('p');
		innerHolder.appendChild(p);
		p.innerHTML = CRSBKR_STRINGS['search.selectBase'];

		innerHolder.appendChild(checkoutBaseList);

		checkoutBaseInput.base = null;
		div.onclick = () =>
		{
			checkoutBaseInput.parentNode.classList.add('open');
			checkoutBaseInput.parentNode.classList.remove('error');

			if (checkinBaseList)
			{
				checkinBaseList.classList.remove('open');
			}

			calendar.classList.remove('open');
			calendarOperation = 'checkout';

			calendarTimeLabel.classList.add('time_label_hidden');
			calendarTimeLabel.classList.remove('time_label_shown');

			for (var i = 0; i < bases.length; i++)
			{
				checkoutBaseList.children[i].style.display = bases[i].available ? 'block' : 'none';
				if (bases[i].available && i === params.checkout_base)
				{
					checkoutBaseList.children[i].classList.add('selected');
					var scrollTop = Math.min(checkoutBaseList.scrollHeight - checkoutBaseList.getBoundingClientRect().height, checkoutBaseList.children[i].offsetTop - checkoutBaseList.children[0].offsetTop);

					checkoutBaseList.scrollTop = scrollTop;
				}
				else
				{
					checkoutBaseList.children[i].classList.remove('selected');
				}
			}
		};

		for (var i = 0; i < bases.length; i++)
		{
			var base = bases[i];

			var li = crsbkr_en('li');
			checkoutBaseList.appendChild(li);

			var icon = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.icon', crsbkr_array_get(base, 'translations.en.icon', crsbkr_array_get(base, 'icon', '')));
			if (icon != '')
			{
				var img = crsbkr_en('img');
				li.appendChild(img);
				img.src = icon;
			}

			var title = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.title', crsbkr_array_get(base, 'translations.en.title', crsbkr_array_get(base, 'title', '')));
			var span = crsbkr_en('span');
			li.appendChild(span);
			span.innerHTML = title;

			var subtitle = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.subtitle', crsbkr_array_get(base, 'translations.en.subtitle', crsbkr_array_get(base, 'subtitle', '')))
			if (subtitle != '')
			{
				var span2 = crsbkr_en('span');
				li.appendChild(span2);
				span2.innerHTML = subtitle;
			}

			li.onclick = ((i, base, title) =>
			{
				return (e) =>
				{
					checkoutBaseInput.base = i;
					checkoutBaseInput.innerHTML = title;

					params.checkout_base = checkoutBaseInput.base;
					if (!checkinBaseInput || checkinBaseSelect && !checkinBaseSelect.checked)
					{
						params.checkin_base = checkoutBaseInput.base;
						if (checkinBaseInput)
						{
							checkinBaseInput.base = checkoutBaseInput.base;
							checkinBaseInput.innerHTML = title;
						}
					}

					e && e.preventDefault();
					e && e.stopPropagation();

					checkoutBaseInput.parentNode.classList.remove('open');

					sync();
				}
			})(i, base, title);

			if (crsbkr_array_get(params, 'checkout_base', false) === i)
			{
				checkoutBaseInput.innerHTML = title;
				checkoutBaseInput.base = i;

				checkoutBaseInput.parentNode.classList.remove('error');
			}
		}

		///
		//
		// CHECKIN BASE
		//
		///
		if (crsbkr_array_get(data, 'controller_form.params.select_checkin_base', 0) == 1)
		{
			var p = crsbkr_en('p');
			fieldset.appendChild(p);

			checkinBaseSelect.type = 'checkbox';
			p.appendChild(checkinBaseSelect);

			checkinBaseSelect.onchange = () =>
			{
				if (checkinBaseSelect.checked)
				{
					checkinBaseLabel.style.display = 'block';
					checkinBaseInput.parentNode.style.display = 'block';
				}
				else
				{
					params.checkin_base = crsbkr_array_get(params, 'checkout_base');
					sync();

					checkinBaseLabel.style.display = 'none';
					checkinBaseInput.parentNode.style.display = 'none';
					checkinBaseInput.innerHTML = checkoutBaseInput.textContent;
				}
			}

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = CRSBKR_STRINGS['search.checkin_base.tick'];

			fieldset.appendChild(checkinBaseLabel);
			checkinBaseLabel.innerHTML = CRSBKR_STRINGS['search.checkin_base'];

			var div = crsbkr_en('div');
			fieldset.appendChild(div);
			div.classList.add('input');

			div.appendChild(checkinBaseInput);

			var innerHolder = crsbkr_en('div');
			div.appendChild(innerHolder);

			var blur = crsbkr_en('div');
			div.appendChild(blur);
			blur.classList.add('blur');

			var p = crsbkr_en('p');
			innerHolder.appendChild(p);
			p.innerHTML = CRSBKR_STRINGS['search.selectBase'];

			innerHolder.appendChild(checkinBaseList);

			checkinBaseSelect.checked = crsbkr_array_get(params, 'checkin_base', '') != crsbkr_array_get(params, 'checkout_base', '') ? true : false;
			checkinBaseLabel.style.display = checkinBaseSelect.checked ? 'block' : 'none';
			div.style.display = checkinBaseSelect.checked ? 'block' : 'none';

			checkinBaseInput.base = null;
			div.onclick = () =>
			{
				checkinBaseInput.parentNode.classList.remove('error');
				checkinBaseInput.parentNode.classList.add('open');

				checkoutBaseInput.parentNode.classList.remove('open');
				calendar.classList.remove('open');
				calendarOperation = 'checkout';

				calendarTimeLabel.classList.add('time_label_hidden');
				calendarTimeLabel.classList.remove('time_label_shown');

				var checkoutBase = crsbkr_array_get(params, 'checkout_base', false) !== false ? bases[params.checkout_base] : null;
				for (var i = 0; i < bases.length; i++)
				{
					var available = bases[i].available && (!checkoutBase || checkoutBase.userIds.filter(value => bases[i].userIds.includes(value)).length > 0);

					if (params.checkout_base != i && available)
					{
						// Be sure one way is enabled.
						fromBaseIds = crsbkr_array_get(checkoutBase, 'bases', '').split(',').filter(n => ('' + n).length > 0);
						toBaseIds = crsbkr_array_get(bases[i], 'bases', '').split(',').filter(n => ('' + n).length > 0);

						var found = false;
						for (var ii = 0; ii < data.base_transfer_time.length; ii++)
						{
							if (crsbkr_array_get(data.base_transfer_time[ii], 'one_way', 0) != 0 && fromBaseIds.indexOf(data.base_transfer_time[ii].from_base_id) != -1 && toBaseIds.indexOf(data.base_transfer_time[ii].to_base_id) != -1)
							{
								found = true;
								break;
							}
						}

						if (!found)
						{
							available = false;
						}
					}

					checkinBaseList.children[i].style.display = available ? 'block' : 'none';

					if (available && params.checkin_base === i)
					{
						checkinBaseList.children[i].classList.add('selected');
						checkinBaseList.scrollTop = Math.min(checkinBaseList.scrollHeight - checkinBaseList.offsetHeight, checkinBaseList.children[i].offsetTop - checkinBaseList.children[0].offsetTop);
					}
					else
					{
						checkinBaseList.children[i].classList.remove('selected');
					}
				}
			}

			for (var i = 0; i < bases.length; i++)
			{
				var base = bases[i];

				var li = crsbkr_en('li');
				checkinBaseList.appendChild(li);

				var icon = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.icon', crsbkr_array_get(base, 'translations.en.icon', crsbkr_array_get(base, 'icon', '')));
				if (icon != '')
				{
					var img = crsbkr_en('img');
					li.appendChild(img);
					img.src = icon;
				}

				var title = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.title', crsbkr_array_get(base, 'translations.en.title', crsbkr_array_get(base, 'title', '')));
				var span = crsbkr_en('span');
				li.appendChild(span);
				span.innerHTML = title;

				var subtitle = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.subtitle', crsbkr_array_get(base, 'translations.en.subtitle', crsbkr_array_get(base, 'subtitle', '')));
				if (subtitle != '')
				{
					var span2 = crsbkr_en('span');
					li.appendChild(span2);
					span2.innerHTML = subtitle;
				}

				li.onclick = ((i, base, title) =>
				{
					return (e) =>
					{
						checkinBaseInput.base = i;
						checkinBaseInput.innerHTML = title;

						params.checkin_base = checkinBaseInput.base;

						e && e.preventDefault();
						e && e.stopPropagation();

						checkinBaseInput.parentNode.classList.remove('open');

						sync();
					}
				})(i, base, title);

				if (crsbkr_array_get(params, 'checkin_base', false) === i)
				{
					checkinBaseInput.innerHTML = title;
					checkinBaseInput.base = i;

					checkinBaseInput.parentNode.classList.remove('error');
				}
			}
		}
		else
		{
			checkinBaseInput = null;
			checkinBaseList = null;
			checkinBaseSelect.checked = false;
		}

		///
		//
		// PROMO CODE
		//
		///
		if (crsbkr_array_get(data, 'controller_form.params.promo_code', 0) == 1)
		{
			var p = crsbkr_en('p');
			fieldset.appendChild(p);

			var promoCodeSelect = crsbkr_en('input');
			promoCodeSelect.type = 'checkbox';
			p.appendChild(promoCodeSelect);

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = CRSBKR_STRINGS['search.promo_code.select'];

			var div = crsbkr_en('div');
			fieldset.appendChild(div);
			div.className = 'input';

			var promoCodeInput = crsbkr_en('input');
			div.appendChild(promoCodeInput);
			promoCodeInput.placeholder = CRSBKR_STRINGS['search.promo_code.placeholder'];
			promoCodeInput.type = 'text';
			promoCodeInput.value = crsbkr_array_get(params, 'promo_code', '').trim();

			if (promoCodeInput.value != '')
			{
				promoCodeSelect.checked = true;
				promoCodeInput.parentNode.style.display = 'block';
			}
			else
			{
				promoCodeSelect.checked = false;
				promoCodeInput.parentNode.style.display = 'none';
			}

			promoCodeSelect.onchange = () =>
			{
				if (promoCodeSelect.checked)
				{
					promoCodeInput.parentNode.style.display = 'block';
				}
				else
				{
					promoCodeInput.parentNode.style.display = 'none';
				}
			}

			promoCodeInput.onchange = () =>
			{
				params.promo_code = promoCodeInput.value;
				sync();
			}
		}

		///
		//
		// ADDITIONAL FIELDS?
		//
		///
		var thisAccs = crsbkr_array_get(params, 'accessories', '').split(',').filter(n => ('' + n).length > 0);
		for (var i = 0; i < crsbkr_array_get(data, 'controller_form.params.accessories', []).length; i++)
		{
			var acc = data.controller_form.params.accessories[i];
			var name = crsbkr_array_get(acc, 'translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(acc, 'translations.en.name', crsbkr_array_get(acc, 'name', '')))
			if (name == '')
			{
				continue;
			}

			var p = crsbkr_en('p');
			fieldset.appendChild(p);

			var select = crsbkr_en('input');
			select.type = 'checkbox';
			p.appendChild(select);

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = name;

			select.checked = thisAccs.indexOf('' + i) != -1;
			select.onchange = ((select, i, acc) =>
			{
				return () =>
				{
					var thisAccs = crsbkr_array_get(params, 'accessories', '').split(',').filter(n => ('' + n).length > 0);
					if (thisAccs.indexOf('' + i) == -1 && select.checked)
					{
						thisAccs.push('' + i);
						params.accessories = thisAccs.filter(n => ('' + n).length > 0).join(',');

						sync();
					}
					else if (thisAccs.indexOf('' + i) != -1 && !select.checked)
					{
						thisAccs.splice(thisAccs.indexOf('' + i), 1);
						params.accessories = thisAccs.filter(n => ('' + n).length > 0).join(',');

						sync();
					}
				}
			})(select, i, acc);
		}

		///
		//
		// CREATE THE DATE AND TIME FORM ELEMENTS
		//
		///
		var div = crsbkr_en('div');
		fieldset.appendChild(div);
		div.className = 'date';

		for (var ci = 0; ci < 2; ci++)
		{
			var type = ci == 0 ? 'checkout' : 'checkin';

			var innerDiv = crsbkr_en('div');
			div.appendChild(innerDiv);

			var day = ci == 0 ? checkoutDay : checkinDay;
			var dow = ci == 0 ? checkoutDow : checkinDow;
			var month = ci == 0 ? checkoutMonth : checkinMonth;
			var time = ci == 0 ? checkoutTime : checkinTime;

			innerDiv.appendChild(day);
			day.className = 'day';

			var span = crsbkr_en('span');
			innerDiv.appendChild(span);
			span.className = 'date_holder';

			span.appendChild(dow);
			dow.className = 'dow';

			span.appendChild(month);
			month.className = 'month';

			innerDiv.appendChild(time);
			time.className = 'time';
		}

		var arrow = crsbkr_en('div');
		arrow.className = 'arrow';
		div.appendChild(arrow);
		arrow.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_arrow.png\')';

		///
		//
		// Calendar popup
		//
		///
		div.appendChild(calendar);
		calendar.classList.add('calendar_holder');

		var blur = crsbkr_en('div');
		div.appendChild(blur);
		blur.classList.add('blur');

		var checkoutDateId = null;
		var checkinDateId = null;

		var p = crsbkr_en('p');
		calendar.appendChild(p);
		p.innerHTML = CRSBKR_STRINGS['search.selectDate'];

		var calendarCloseButton = crsbkr_en('button');
		p.appendChild(calendarCloseButton);
		calendarCloseButton.classList.add('close');
		calendarCloseButton.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_close.png\')';
		calendarCloseButton.onclick = () =>
		{
			calendar.classList.remove('open');
			calendarOperation = 'checkout';

			calendarTimeLabel.classList.add('time_label_hidden');
			calendarTimeLabel.classList.remove('time_label_shown');

			sync();
		}

		var calendarTimeLabel = crsbkr_en('p');

		var calendarOperation = 'checkout';
		calendar.classList.add('checkout');

		calendarTimeLabel.classList.add('time_label_hidden');
		calendarTimeLabel.classList.remove('time_label_shown');

		var calendarButtons = crsbkr_en('div');
		calendar.appendChild(calendarButtons);
		calendarButtons.className = 'buttons';

		var calendarLeft = crsbkr_en('button');
		calendarButtons.appendChild(calendarLeft);
		calendarLeft.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_arrow.png\')';
		calendarLeft.onclick = () =>
		{
			var ii = 0;
			for (var i = 0; i < calendarMonths.children.length; i++)
			{
				if (calendarMonths.children[i].classList.contains('selected'))
				{
					ii = i;
					break;
				}
			}

			if (ii > 0)
			{
				calendarMonths.children[ii].classList.remove('selected');
				calendarMonths.children[ii - 1].classList.add('selected');

				sync();
			}
		}

		var calendarRight = crsbkr_en('button');
		calendarButtons.appendChild(calendarRight);
		calendarRight.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_arrow.png\')';
		calendarRight.onclick = () =>
		{
			var ii = 0;
			for (var i = 0; i < calendarMonths.children.length; i++)
			{
				if (calendarMonths.children[i].classList.contains('selected'))
				{
					ii = i;
					break;
				}
			}

			if (ii < calendarMonths.children.length - 2)
			{
				calendarMonths.children[ii].classList.remove('selected');
				calendarMonths.children[ii + 1].classList.add('selected');

				sync();
			}
		}

		var calendarMonths = crsbkr_en('div');
		calendar.appendChild(calendarMonths);
		calendarMonths.classList.add('months');

		// Load the months
		var today = new Date();
		today.setHours(0, 0, 0, 0);
		todayId = today.getFullYear() * 10000 + (today.getMonth() + 1) * 100 + today.getDate();

		var calendarFirstDay = null;
		var calendarLastDay = null;

		for (var i = 0; i < 12; i++)
		{
			var date = new Date(today.getTime());
			date.setDate(1);
			date.setMonth(date.getMonth() + i);
			date.setHours(0, 0, 0, 0);

			var month = crsbkr_en('div');
			calendarMonths.appendChild(month);

			if (i == 0)
			{
				month.classList.add('selected');
			}

			var title = crsbkr_en('h1');
			month.appendChild(title);

			title.innerHTML = CRSBKR_STRINGS['month.' + date.getMonth()] + ' ' + date.getFullYear();

			var table = crsbkr_en('table');
			month.appendChild(table);

			var tbody = crsbkr_en('tbody');
			table.appendChild(tbody);

			var tr = crsbkr_en('tr');
			tbody.appendChild(tr);

			for (var j = 0; j < 7; j++)
			{
				var td = crsbkr_en('td');
				tr.appendChild(td);
				td.innerHTML = CRSBKR_STRINGS['weekday.' + j].substr(0, 3);
			}

			var m = date.getMonth();
			var tr = crsbkr_en('tr');
			tbody.appendChild(tr);

			// Empty cells before first day of month.
			var day = date.getDay() == 0 ? 7 : date.getDay();
			while (tr.children.length < day - 1)
			{
				var td = crsbkr_en('td');
				tr.appendChild(td);
				td.classList.add('none');
				td.onmousedown = (e) =>
				{
					e.preventDefault();
					e.stopPropagation();
				}
			}

			// Days.
			while (m == date.getMonth())
			{
				day = date.getDay() == 0 ? 7 : date.getDay();
				if (day == 1 && date.getDate() != 1)
				{
					tr = crsbkr_en('tr');
					tbody.appendChild(tr);
				}

				calendarLastDay = date.getFullYear() * 10000 + (date.getMonth() + 1) * 100 + date.getDate();
				if (!calendarFirstDay)
				{
					calendarFirstDay = calendarLastDay;
				}

				var td = crsbkr_en('td');
				tr.appendChild(td);
				td.id = 'crsbkr_calendar_' + calendarLastDay;

				var span = crsbkr_en('span');
				td.appendChild(span);
				span.innerHTML = date.getDate();

				td.onmouseover = ((td, span) =>
				{
					return () =>
					{
						var thisId = td.id.replace('crsbkr_calendar_', '');
						if (calendarOperation == 'checkout')
						{
						}
						else if (thisId >= checkoutDateId)
						{
							checkinDateId = thisId;
							sync();
						}
						else
						{
							checkinDateId = checkoutDateId;
							sync();
						}
					}
				})(td, span);

				td.onclick = ((td, span) =>
				{
					return () =>
					{
						if (td.classList.contains('disabled'))
						{
							return;
						}

						if (calendarOperation == 'checkout' || td.id.replace('crsbkr_calendar_', '') < checkoutDateId)
						{
							checkoutDateId = td.id.replace('crsbkr_calendar_', '');
							checkinDateId = checkoutDateId;

							var thisTime = checkoutDateId.substr(6, 2) + '-' + checkoutDateId.substr(4, 2) + '-' + checkoutDateId.substr(0, 4) + ' ' + checkoutTime.textContent + ':00';
							thisTime = crsbkr_format_date(thisTime, true, 'datetime');

							params.checkout_time = thisTime;

							thisTime = checkoutDateId.substr(6, 2) + '-' + checkoutDateId.substr(4, 2) + '-' + checkoutDateId.substr(0, 4) + ' ' + checkinTime.textContent + ':00';
							thisTime = crsbkr_format_date(thisTime, true, 'datetime');
							params.checkin_time = thisTime;

							calendarOperation = 'checkin';

							calendar.classList.add('checkin');
							calendar.classList.remove('checkout');

							sync();
						}
						else
						{
							checkinDateId = td.id.replace('crsbkr_calendar_', '');

							var thisTime = checkinDateId.substr(6, 2) + '-' + checkinDateId.substr(4, 2) + '-' + checkinDateId.substr(0, 4) + ' ' + checkinTime.textContent + ':00';
							thisTime = crsbkr_format_date(thisTime, true, 'datetime');

							params.checkin_time = thisTime;

							calendarOperation = 'checkout';
							calendar.classList.add('checkout');
							calendar.classList.remove('checkin');

							calendarTimeLabel.classList.remove('time_label_hidden');
							calendarTimeLabel.classList.add('time_label_shown');

							sync();
						}
					}
				})(td, span);

				date.setDate(date.getDate() + 1);
			}

			// Empty cells at the end.
			day = tr.children.length;
			for (; day < 7; day++)
			{
				var td = crsbkr_en('td');
				tr.appendChild(td);
				td.classList.add('none');
				td.onmousedown = (e) =>
				{
					e.preventDefault();
					e.stopPropagating();
				}
			}
		}

		///
		//
		// Date errors
		//
		///
		var div = crsbkr_en('div');
		fieldset.appendChild(div);
		div.className = 'date';

		div.appendChild(checkoutDateError);
		div.appendChild(checkinDateError);

		checkoutDay.parentNode.onclick = () =>
		{
			calendarOperation = 'checkout';
			calendar.classList.add('checkout');
			calendar.classList.remove('checkin');

			calendar.classList.add('open');

			calendarTimeLabel.classList.add('time_label_hidden');
			calendarTimeLabel.classList.remove('time_label_shown');

			checkoutBaseInput.parentNode.classList.remove('open');
			if (checkinBaseInput)
			{
				checkinBaseInput.parentNode.classList.remove('open');
			}

			var dateHolder = {};
			var dateStr = crsbkr_format_date(params.checkout_time, false, 'date', dateHolder);
			var dateObj = dateHolder.date;

			if (dateObj)
			{
				var n = dateObj.getMonth() - today.getMonth();
				if (n >= 0 && n < calendarMonths.children.length)
				{
					var scrollTop = Math.min(calendarMonths.scrollHeight - calendarMonths.getBoundingClientRect().height, calendarMonths.children[n].offsetTop);
					calendarMonths.scrollTop = scrollTop;
				}
			}
		};

		checkinDay.parentNode.onclick = checkoutDay.parentNode.onclick;

		var p = calendarTimeLabel;
		calendar.appendChild(p);
		p.className = 'time_label time_label_hidden';

		var span = crsbkr_en('span');
		p.appendChild(span);
		span.innerHTML = CRSBKR_STRINGS['search.selectTime'];

		var a = crsbkr_en('a');
		p.appendChild(a);
		a.className = 'show';
		a.innerHTML = CRSBKR_STRINGS['search.selectTime.show'];
		a.onclick = ((p) =>
		{
			return () =>
			{
				p.classList.remove('time_label_hidden');
				p.classList.add('time_label_shown');
			}
		})(p);

		var a = crsbkr_en('a');
		p.appendChild(a);
		a.className = 'hide';
		a.innerHTML = CRSBKR_STRINGS['search.selectTime.hide'];
		a.onclick = ((p) =>
		{
			return () =>
			{
				p.classList.remove('time_label_shown');
				p.classList.add('time_label_hidden');
			}
		})(p);

		p.appendChild(crsbkr_en('br'));

		var span = crsbkr_en('span');
		span.innerHTML = CRSBKR_STRINGS['search.selectTime.hint'];

		p.appendChild(span);

		for (var i = 0; i < 2; i++)
		{
			var timeHolder = i == 0 ? checkoutTimeHolder : checkinTimeHolder;
			var timeLabel = i == 0 ? checkoutTimeLabel : checkinTimeLabel;
			var indicator = i == 0 ? checkoutTimeIndicator : checkinTimeIndicator;
			var indicatorBubble = i == 0 ? checkoutTimeIndicatorBubble : checkinTimeIndicatorBubble;
			var indicatorLabel = i == 0 ? checkoutTimeIndicatorLabel : checkinTimeIndicatorLabel;
			var indicatorLeft = i == 0 ? checkoutTimeIndicatorLeft : checkinTimeIndicatorLeft;
			var indicatorRight = i == 0 ? checkoutTimeIndicatorRight : checkinTimeIndicatorRight;

			var span = crsbkr_en('span');
			calendar.appendChild(span);
			span.className = 'time';
			span.onclick = ((p) =>
			{
				return () =>
				{
					p.classList.add('time_label_shown');
					p.classList.remove('time_label_hidden');
				}
			})(p);

			var label = crsbkr_en('span');
			span.appendChild(label);
			label.innerHTML = CRSBKR_STRINGS['search.' + (i == 0 ? 'checkout' : 'checkin') + '_time'];

			label.appendChild(timeLabel);
			span.appendChild(timeHolder);

			timeHolder.appendChild(indicatorLeft);
			timeHolder.appendChild(indicatorRight);
			timeHolder.appendChild(indicator);
			indicator.classList.add('indicator');

			indicator.appendChild(indicatorLabel);
			indicator.appendChild(indicatorBubble);

			var draggingOffset = 0;
			indicator.onmousedown = ((indicator) =>
			{
				return (e) =>
				{
					var rect = indicator.getBoundingClientRect();
					draggingOffset = e.pageX - (rect.x + rect.width / 2);

					indicator.classList.add('dragging');
				}
			})(indicator);

			indicator.ontouchstart = indicator.onmousedown;

			var fn = ((indicator, label) =>
			{
				return (e) =>
				{
					e.preventDefault();
					e.stopPropagation();

					var dateId = indicator == checkoutTimeIndicator ? checkoutDateId : checkinDateId;
					var thisTime = dateId.substr(6, 2) + '-' + dateId.substr(4, 2) + '-' + dateId.substr(0, 4) + ' ' + label.textContent + ':00';
					thisTime = crsbkr_format_date(thisTime, true, 'datetime');

					params[indicator == checkoutTimeIndicator ? 'checkout_time' : 'checkin_time'] = thisTime;
					indicator.classList.remove('dragging');

					sync();
				}
			})(indicator, indicatorLabel);

			indicator.onmouseout = fn;
			indicator.onmouseup = fn;
			indicator.ontouchend = fn;
			indicator.ontouchcancel = fn;

			timeHolder.onmousedown = ((timeHolder, indicator, indicatorLeft, label, bubble, timeLabel) =>
			{
				return (e) =>
				{
					if (indicator.classList.contains('dragging'))
					{
						return;
					}

					var x = e.clientX;
					if (e.type.includes(`touch`))
					{
						const { touches, changedTouches } = e.originalEvent ?? e;
						const touch = touches[0] ?? changedTouches[0];

						x = touch.pageX;
					}

					var rect = timeHolder.getBoundingClientRect();

					if (x >= rect.x && x <= rect.x + rect.width + indicator.offsetWidth / 2)
					{
						x -= rect.x;

						var percent = Math.min(100, (x * 100 / timeHolder.offsetWidth));

						indicator.style.left = percent + '%';
						indicatorLeft.style.width = percent + '%';

						var minutes = (24 * 60 - 1) * percent / 100;
						var factor = Math.min(24 * 60 / 15 - 1, minutes / 15.0);
						minutes = Math.round(factor) * 15;

						var hours = Math.floor(minutes / 60);
						minutes = minutes % 60;

						var time = (hours < 10 ? '0' : '') + hours + ':' + (minutes < 10 ? '0' : '') + minutes;
						label.innerHTML = time;
						timeLabel.innerHTML = time;
						bubble.innerHTML = time;

						var dateId = indicator == checkoutTimeIndicator ? checkoutDateId : checkinDateId;
						var thisTime = dateId.substr(6, 2) + '-' + dateId.substr(4, 2) + '-' + dateId.substr(0, 4) + ' ' + label.textContent + ':00';
						thisTime = crsbkr_format_date(thisTime, true, 'datetime');

						params[indicator == checkoutTimeIndicator ? 'checkout_time' : 'checkin_time'] = thisTime;
						sync();
					}
				}
			})(timeHolder, indicator, indicatorLeft, indicatorLabel, indicatorBubble, timeLabel);

			timeHolder.onmousemove = ((timeHolder, indicator, indicatorLeft, label, bubble, timeLabel, timeSet, timeError) =>
			{
				return (e) =>
				{
					var x = e.clientX;
					if (e.type.includes(`touch`))
					{
						const { touches, changedTouches } = e.originalEvent ?? e;
						const touch = touches[0] ?? changedTouches[0];

						x = touch.pageX;
					}

					var rect = timeHolder.getBoundingClientRect();
					if (indicator.classList.contains('dragging'))
					{
						if (x >= rect.x && x <= rect.x + rect.width + indicator.offsetWidth / 2)
						{
							x -= rect.x;

							var percent = Math.min(100, (x * 100 / timeHolder.offsetWidth));

							indicator.style.left = percent + '%';
							indicatorLeft.style.width = percent + '%';

							var minutes = (24 * 60 - 1) * percent / 100;
							var factor = Math.min(24 * 60 / 15 - 1, minutes / 15.0);
							minutes = Math.round(factor) * 15;

							var hours = Math.floor(minutes / 60);
							minutes = minutes % 60;

							var time = (hours < 10 ? '0' : '') + hours + ':' + (minutes < 10 ? '0' : '') + minutes;
							label.innerHTML = time;
							bubble.innerHTML = time;
							timeLabel.innerHTML = time;

							var hours = crsbkr_array_get(timeSet, time, 0);
							if (hours == 0)
							{
								timeHolder.className = 'error';
								timeError.className = 'error';

								timeError.innerHTML = CRSBKR_STRINGS['search.error.baseClosedHours.' + (timeError == checkoutTimeError ? 'checkout' : 'checkin')];

							}
							else if (hours == 1)
							{
								timeHolder.className = '';
								timeError.className = 'error';

								timeError.innerHTML = '&nbsp;';
							}
							else
							{
								timeHolder.className = 'warn';
								timeError.className = 'warn';

								timeError.innerHTML = CRSBKR_STRINGS['search.error.baseOutOfHours.' + (timeError == checkoutTimeError ? 'checkout' : 'checkin')];
							}
						}
					}
				}
			})(timeHolder, indicator, indicatorLeft, indicatorLabel, indicatorBubble, timeLabel, i == 0 ? checkoutTimeSet : checkinTimeSet, i == 0 ? checkoutTimeError : checkinTimeError);

			timeHolder.ontouchmove = timeHolder.onmousemove;

			calendar.appendChild(i == 0 ? checkoutTimeError : checkinTimeError);
		}

		checkoutTimeError.classList.add('error');
		checkinTimeError.classList.add('error');

		calendar.appendChild(crsbkr_en('span'));

		var calendarSearchButton = crsbkr_en('button');
		calendar.appendChild(calendarSearchButton);
		calendarSearchButton.classList.add('search');
		calendarSearchButton.innerHTML = CRSBKR_STRINGS['label.search'];
		calendarSearchButton.onclick = () =>
		{
			calendarTimeLabel.classList.add('time_label_hidden');
			calendarTimeLabel.classList.remove('time_label_shown');

			sync();

			if (searchButton.disabled)
			{
				calendarCloseButton.click();
			}
			else
			{
				searchButton.click();
			}
		}

		///
		//
		// FORM VALIDATION
		//
		///
		sync = () =>
		{
			localStorage.setItem('crsbkr_module_' + crsbkr_array_get(data, 'controller_form.uid'), JSON.stringify(params));

			var group = crsbkr_array_get(groups, '' + crsbkr_array_get(params, 'group', '0'), null);

			///
			// BASES
			// Only show the bases with the given groups.
			///
			var groupAcrissCodes = [];
			var thisGroups = crsbkr_array_get(group, 'groups', '').split(',').filter(n => ('' + n).length > 0);

			for (var i = 0; i < data.acriss_code_web.length; i++)
			{
				if (thisGroups.indexOf('' + data.acriss_code_web[i].uid) != -1)
				{
					groupAcrissCodes.push(...(data.acriss_code_web[i].acriss_code_ids.split(',').filter(n => ('' + n).length > 0)));
				}
			}

			for (var i = 0; i < bases.length; i++)
			{
				var base = bases[i];
				base.userIds = [];

				var baseIds = crsbkr_array_get(base, 'bases', '').split(',').filter(n => ('' + n).length > 0);

				var found = false;
				for (var ii = 0; ii < data.base.length; ii++)
				{
					if (baseIds.indexOf('' + data.base[ii].uid) == -1)
					{
						continue;
					}

					var baseAcrissCodes = crsbkr_array_get(data.base[ii], 'settings.web.acriss_codes', '').split(',').filter(n => ('' + n).length > 0);
					var intersect = groupAcrissCodes.filter(value => baseAcrissCodes.includes(value));

					if (intersect.length > 0)
					{
						found = true;

						if (base.userIds.indexOf('' + data.base[ii].user_uid) == -1)
						{
							base.userIds.push(data.base[ii].user_uid);
						}
					}
				}

				base.available = found;

				if (checkoutBaseInput.base == i)
				{
					if (!found)
					{
						checkoutBaseInput.parentNode.classList.add('error');
					}
					else
					{
						checkoutBaseInput.parentNode.classList.remove('error');
					}
				}

				if (checkinBaseInput && checkinBaseInput.base == i)
				{
					if (!found)
					{
						checkinBaseInput.parentNode.classList.add('error');
					}
					else
					{
						checkinBaseInput.parentNode.classList.remove('error');
					}
				}
			}

			if (checkinBaseInput && !checkoutBaseInput.parentNode.classList.contains('error') && !checkinBaseInput.parentNode.classList.contains('error') && checkoutBaseInput.base != checkinBaseInput.base && data.base_transfer_time)
			{
				// Be sure one way is enabled.
				fromBaseIds = crsbkr_array_get(bases[checkoutBaseInput.base], 'bases', '').split(',').filter(n => ('' + n).length > 0);
				toBaseIds = crsbkr_array_get(bases[checkinBaseInput.base], 'bases', '').split(',').filter(n => ('' + n).length > 0);

				var found = false;
				for (var i = 0; i < data.base_transfer_time.length; i++)
				{
					if (crsbkr_array_get(data.base_transfer_time[i], 'one_way', 0) != 0 && fromBaseIds.indexOf(data.base_transfer_time[i].from_base_id) != -1 && toBaseIds.indexOf(data.base_transfer_time[i].to_base_id) != -1)
					{
						console.log(data.base_transfer_time);
						found = true;
						break;
					}
				}

				if (found)
				{
					checkinBaseInput.parentNode.classList.remove('error');
				}
				else
				{
					checkinBaseInput.parentNode.classList.add('error');
				}
			}

			///
			// OPENING HOURS
			///
			for (var d = 0; d < 2; d++)
			{
				var type = d == 0 ? 'checkout' : 'checkin';

				var dayField = d == 0 ? checkoutDay : checkinDay;
				var dowField = d == 0 ? checkoutDow : checkinDow;
				var monthField = d == 0 ? checkoutMonth : checkinMonth;
				var timeField = d == 0 ? checkoutTime : checkinTime;
				var timeLabel = d == 0 ? checkoutTimeLabel : checkinTimeLabel;
				var otherTimeField = d == 0 ? checkoutTimeIndicator : checkinTimeIndicator;
				var otherTimeFieldBubble = d == 0 ? checkoutTimeIndicatorBubble : checkinTimeIndicatorBubble;
				var otherTimeFieldLabel = d == 0 ? checkoutTimeIndicatorLabel : checkinTimeIndicatorLabel;
				var otherTimeFieldLeft = d == 0 ? checkoutTimeIndicatorLeft : checkinTimeIndicatorLeft;
				var timeSet = d == 0 ? checkoutTimeSet : checkinTimeSet;

				var error = d == 0 ? checkoutDateError : checkinDateError;
				var timeError = d == 0 ? checkoutTimeError : checkinTimeError;

				dayField.parentNode.classList.remove('error');
				error.innerHTML = '';
				timeError.innerHTML = '&nbsp;';

				var dateHolder = {};
				var dateStr = crsbkr_format_date(params[type + '_time'], false, 'date', dateHolder);
				var dateObj = dateHolder.date;

				if (!dateStr || dateStr == '' || !dateObj)
				{
					location.reload();
					return;
				}

				var day = dateObj ? (dateObj.getDay() == 0 ? 7 : dateObj.getDay()) : 0;
				var timeStr = crsbkr_format_date(params[type + '_time'], false, 'time');

				dayField.innerHTML = dateStr.substr(0, 2);
				dowField.innerHTML = ('' + CRSBKR_STRINGS['weekday.' + (day - 1)]).substr(0, 3).toLowerCase();

				monthField.innerHTML = ('' + CRSBKR_STRINGS['month.' + (parseInt(dateStr.substr(3, 2)) - 1)]).substr(0, 3).toLowerCase();
				timeField.innerHTML = timeStr;
				timeLabel.innerHTML = timeStr;

				otherTimeField.style.left = ((parseInt(timeStr.substr(0, 2)) * 60 + parseInt(timeStr.substr(3, 2))) * 100 / (24 * 60)) + '%';
				otherTimeFieldBubble.innerHTML = timeStr;
				otherTimeFieldLabel.innerHTML = timeStr;
				otherTimeFieldLeft.style.width = ((parseInt(timeStr.substr(0, 2)) * 60 + parseInt(timeStr.substr(3, 2))) * 100 / (24 * 60)) + '%';

				var thisTime = parseInt(dateStr.substr(6, 4) + dateStr.substr(3, 2) + dateStr.substr(0, 2));
				dateStr = crsbkr_format_date(dateStr + ' 00:00:00', true, 'date');

				var base = crsbkr_array_get(bases, '' + crsbkr_array_get(params, type + '_base'));
				if (!base || !base.available)
				{
					base = null;
				}

				Object.keys(timeSet).forEach(key => timeSet[key] = 1);

				if (base)
				{
					var baseIds = crsbkr_array_get(base, 'bases', '').split(',').filter(n => ('' + n).length > 0);

					var allHours = [];
					var open = false;
					var timeFound = false;

					for (var bi = 0; bi < baseIds.length; bi++)
					{
						open = false;
						timeFound = false;

						for (var i = 0; i < data.base.length; i++)
						{
							var thisBase = data.base[i];
							if (baseIds[bi] != thisBase.uid)
							{
								continue;
							}

							open = crsbkr_array_get(thisBase, 'closed_days', []).length > 0 ? false : true;
							for (dd in thisBase.closed_days)
							{
								open = true;

								closedDay = thisBase.closed_days[dd];
								if (crsbkr_array_get(closedDay, 'date_from', '') <= dateStr && dateStr <= crsbkr_array_get(closedDay, 'date_to', ''))
								{
									open = false;
									break;
								}
							}

							if (open)
							{
								var thisAllHours = [];
								open = false;

								var priority = null;
								for (var ii = 0; ii < crsbkr_array_get(data, 'base_opening_hours', []).length; ii++)
								{
									// Is it for this base and with at least the same priority as the last valid block?
									var hours = data.base_opening_hours[ii];
									if (crsbkr_array_get(hours, 'base_ids', []).indexOf('' + thisBase.uid) == -1 || priority !== null && parseInt(crsbkr_array_get(hours, 'priority', 0)) < parseInt(priority))
									{
										continue;
									}

									// Is it for the right period?
									var found = false;
									for (var iii = 0; iii < crsbkr_array_get(hours, 'validity', []).length; iii++)
									{
										if (crsbkr_array_get(hours.validity[iii], 'date_from', '') <= dateStr && dateStr <= crsbkr_array_get(hours.validity[iii], 'date_to', ''))
										{
											found = true;
										}
									}

									if (!found)
									{
										continue;
									}

									var thisPriority = crsbkr_array_get(hours, 'priority', 0);
									if (priority == null || thisPriority > priority)
									{
										thisAllHours = [];
									}

									priority = thisPriority;

									// Check whether we have opening hours for the correct day of the week and operation.
									if (crsbkr_array_get(hours, 'valid_' + crsbkr_array_get(days, '' + (day - 1), '') + 's', 0) == 1 && crsbkr_array_get(hours, 'for_' + type, 0) == 1)
									{
										thisAllHours.push({ from : crsbkr_array_get(hours, 'hours_from', '00:00'), to : crsbkr_array_get(hours, 'hours_to', '23:59'), ooh : crsbkr_array_get(hours, 'price_' + type, 0) > 0 });
									}
								}

								allHours.push(...thisAllHours);
							}
						}

						for (var i in timeSet)
						{
							if (!timeSet.hasOwnProperty(i))
							{
								continue;
							}

							var time = i;

							var found = false;
							var ooh = false;

							for (var ii = 0; ii < allHours.length; ii++)
							{
								if (allHours[ii].from <= time && time <= allHours[ii].to)
								{
									found = true;
									ooh = ooh || allHours[ii].ooh;
									open = true;

									break;
								}
							}

							timeSet[i] = found ? 1 + (ooh ? 1 : 0) : 0;
							if (timeStr.localeCompare(time) == 0 && found)
							{
								timeFound = true;
							}
						}

						if (open && timeFound)
						{
							break;
						}
					}

					if (dateObj < new Date())
					{
						error.innerHTML = CRSBKR_STRINGS['search.error.pastDate'];

						timeError.className = 'error';
						timeError.innerHTML = error.innerHTML;
					}
					else if (!open)
					{
						if (dayField.textContent.length > 0)
						{
							error.innerHTML = CRSBKR_STRINGS['search.error.baseClosed'];

							timeError.className = 'error';
							timeError.innerHTML = error.innerHTML;
						}
					}
					else
					{
						error.innerHTML = '&nbsp;';
						var hours = crsbkr_array_get(timeSet, timeStr, 0);
						if (hours == 0)
						{
							error.innerHTML = CRSBKR_STRINGS['search.error.baseClosedHours.' + type];

							otherTimeFieldLabel.parentNode.parentNode.classList.add('error');
							otherTimeFieldLabel.parentNode.parentNode.classList.remove('warn');

							timeError.className = 'error';
							timeError.innerHTML = CRSBKR_STRINGS['search.error.baseClosedHours.' + type];
						}
						else if (hours == 1)
						{
							otherTimeFieldLabel.parentNode.parentNode.classList.remove('error');
							otherTimeFieldLabel.parentNode.parentNode.classList.remove('warn');

							timeError.className = 'error';
						}
						else
						{
							otherTimeFieldLabel.parentNode.parentNode.classList.remove('error');
							otherTimeFieldLabel.parentNode.parentNode.classList.add('warn');

							timeError.className = 'warn';
							timeError.innerHTML = CRSBKR_STRINGS['search.error.baseOutOfHours.' + type];
						}
					}
				}
			}

			// Mark the selected days in the calendar.
			var baseIds = crsbkr_array_get(base, 'bases', '').split(',').filter(n => ('' + n).length > 0);
			for (var dd = parseInt(calendarFirstDay); dd <= calendarLastDay; dd += dd % 100 == 31 ? 70 : 1)
			{
				var dds = '' + dd;
				var e = crsbkr_e('crsbkr_calendar_' + dd);
				if (!e)
				{
					continue;
				}

				if (dd == checkoutDateId && dd == checkinDateId)
				{
					e.className = 'checkout checkin';
				}
				else if (dd == checkoutDateId)
				{
					e.className = 'checkout';
				}
				else if (dd == checkinDateId)
				{
					e.className = 'checkin';
				}
				else if (checkoutDateId < dd && dd < checkinDateId)
				{
					e.className = 'between';
				}
				else
				{
					e.className = '';

					var open = true;
					if (base)
					{
						var dateStr = crsbkr_format_date(dds.substr(6, 2) + '-' + dds.substr(4, 2) + '-' + dds.substr(0, 4) + ' 00:00:00', true, 'datetime');

						open = false;
						for (var bi = 0; bi < baseIds.length; bi++)
						{
							for (var i = 0; i < data.base.length; i++)
							{
								var thisBase = data.base[i];
								if (baseIds[bi] != thisBase.uid)
								{
									continue;
								}

								open = crsbkr_array_get(thisBase, 'closed_days', []).length > 0 ? false : true;
								for (ddc in thisBase.closed_days)
								{
									open = true;

									closedDay = thisBase.closed_days[ddc];
									if (crsbkr_array_get(closedDay, 'date_from', '') <= dateStr && dateStr <= crsbkr_array_get(closedDay, 'date_to', ''))
									{
										open = false;
										break;
									}
								}

								if (open)
								{
									break;
								}
							}

							if (open)
							{
								break;
							}
						}
					}

					if (open && (calendarOperation == 'checkout' && dd >= todayId || calendarOperation == 'checkin' && dd >= checkoutDateId))
					{
						e.classList.remove('disabled');
					}
					else
					{
						e.classList.add('disabled');
					}
				}
			}

			if (checkoutDateError.textContent.trim() == '' && checkinDateError.textContent.trim() == '' && params.checkout_time >= params.checkin_time && !checkoutTimeIndicator.classList.contains('dragging') && !checkinTimeIndicator.classList.contains('dragging')
					&& (calendarOperation != 'checkin' || !calendar.classList.contains('open')))
			{
				checkinDateError.innerHTML = CRSBKR_STRINGS['search.error.checkinBeforeCheckout'];

				checkinTimeError.innerHTML = CRSBKR_STRINGS['search.error.checkinBeforeCheckout'];
				checkinTimeError.className = 'error';
			}

			if (searchButton.parentNode)
			{
				searchButton.parentNode.href = CRSBKR_RESULTS_PAGE + '&f=' + data.controller_form.code + '&q=' + btoa(JSON.stringify(params));
				searchButton.disabled = checkoutBaseInput.parentNode.classList.contains('error') || checkinBaseInput && checkinBaseInput.parentNode.classList.contains('error') || checkoutDay.parentNode.classList.contains('error') || checkinDay.parentNode.classList.contains('error') || checkoutDateError.textContent.trim() != '' || checkinDateError.textContent.trim() != '' || crsbkr_array_get(params, 'checkout_base', null) === null || crsbkr_array_get(params, 'checkout_time', null) === null || crsbkr_array_get(params, 'checkin_base', null) === null || crsbkr_array_get(params, 'checkin_time', null) === null;
			}
		}

		///
		//
		// CHECKOUT & CHECKIN DATE
		//
		///
		var defaultCheckoutDayOffset = parseInt(crsbkr_array_get(data, 'controller_form.params.default_checkout_time_day_offset', 1));
		var defaultCheckoutTimeOffset = parseInt(crsbkr_array_get(data, 'controller_form.params.default_checkout_time_offset', 120));
		var defaultCheckinDayOffset = parseInt(crsbkr_array_get(data, 'controller_form.params.default_checkin_time_day_offset', 2));

		var dateHolder = {};
		var thisTime = crsbkr_format_date(params.checkout_time, false, 'datetime', dateHolder);
		var thisCheckoutDateObj = dateHolder.date;

		if (crsbkr_array_get(params, 'checkout_time', '') == '' || !thisCheckoutDateObj)
		{
			do
			{
				thisCheckoutDateObj = new Date();
				thisCheckoutDateObj.setDate(thisCheckoutDateObj.getDate() + defaultCheckoutDayOffset);
				thisCheckoutDateObj.setHours(defaultCheckoutDayOffset == 0 ? thisCheckoutDateObj.getHours() + 1 : 10, defaultCheckoutDayOffset == 0 ? defaultCheckoutTimeOffset : 0, 0, 0);

				params.checkout_time = crsbkr_format_date(thisCheckoutDateObj, true);

				sync();

				defaultCheckoutDayOffset++;
			} while (checkoutDateError.textContent != '' && defaultCheckoutDayOffset < 10)
		}

		var thisTime = crsbkr_format_date(params.checkin_time, false, 'datetime', dateHolder);
		var thisCheckinDateObj = dateHolder.date;

		if (crsbkr_array_get(params, 'checkin_time', '') == '' || !thisCheckinDateObj)
		{
			do
			{
				thisCheckinDateObj = new Date(thisCheckoutDateObj.getTime());
				thisCheckinDateObj.setDate(thisCheckinDateObj.getDate() + defaultCheckinDayOffset);
				thisCheckinDateObj.setHours(thisCheckinDateObj.getHours(), 0, 0, 0);

				params.checkin_time = crsbkr_format_date(thisCheckinDateObj, true);

				sync();

				defaultCheckinDayOffset++;
			} while (checkinDateError.textContent != '' && defaultCheckinDayOffset < 10)
		}

		checkoutDateId = '' + (thisCheckoutDateObj.getFullYear() * 10000 + (thisCheckoutDateObj.getMonth() + 1) * 100 + thisCheckoutDateObj.getDate());
		checkinDateId = '' + (thisCheckinDateObj.getFullYear() * 10000 + (thisCheckinDateObj.getMonth() + 1) * 100 + thisCheckinDateObj.getDate());

		var dateObj = null;
		var dateHolder = {};

		crsbkr_format_date(params.checkout_time, false, 'datetime', dateHolder);
		dateObj = dateHolder.date;

		if (dateObj)
		{
			var n = dateObj.getMonth() - today.getMonth();
			if (n >= 0 && n < calendarMonths.children.length)
			{
				for (var i = 0; i < calendarMonths.children.length; i++)
				{
					if (i == n)
					{
						calendarMonths.children[i].classList.add('selected');
					}
					else
					{
						calendarMonths.children[i].classList.remove('selected');
					}
				}
			}
		}

		///
		//
		// SEARCH BUTTON
		//
		///
		var a = crsbkr_en('a');
		fieldset.appendChild(a);
		a.appendChild(searchButton);

		searchButton.innerHTML = CRSBKR_STRINGS['search.search'];

		sync();
	}
	else if (!data || crsbkr_array_get(data, 'error', '') != '')
	{
		var form = crsbkr_en('span');
		holder.appendChild(form);
		form.classList.add('error');
		form.innerHTML = '<b>' + CRSBKR_STRINGS['error'] + '</b><br /><pre>' + crsbkr_array_get(data, 'error', '') + '</pre>';
	}
}

function crsbkr_request(method, request, params, callback, progressCallback)
{
	var url = CRSBKR_GW_URL;

	var xmlhttp = null;
	if (window.XMLHttpRequest)
	{
		xmlhttp = new XMLHttpRequest();
	}
	else
	{
		xmlhttp = new ActiveXObject('Microsoft.XMLHTTP');
	}

	// Callback function.
	if (progressCallback)
	{
		xmlhttp.onprogress = progressCallback;

		if (xmlhttp.upload)
		{
			xmlhttp.upload.onprogress = progressCallback;
			xmlhttp.upload.addEventListener('progress', progressCallback);
		}
	}

	xmlhttp.onreadystatechange = () =>
	{
		if (xmlhttp.readyState == 4)
		{
			if (xmlhttp.status == 200)
			{
				var json = null;

				try
				{
					json = JSON.parse(xmlhttp.responseText);
				}
				catch (e)
				{
					json = null;
				}

				if (callback)
				{
					callback(json && json.success, !json || json.error || !json.success ? (!json ? xmlhttp.responseText : json) : json.result, xmlhttp);
				}
			}
			else
			{
				var json = null;
				try
				{
					json = JSON.parse(xmlhttp.responseText);
				}
				catch (e)
				{
					json = null;
				}

				if (callback)
				{
					callback(false, json ? json : xmlhttp.responseText, xmlhttp);
				}
			}
		}
	};

	// Get the parameters.
	var getp = (url.indexOf('?') != -1 ? '&' : '?') + 'action=crsbkr_ajax';
	var postp = 'method=' + encodeURIComponent(method) + '&request=' + encodeURIComponent(request)
			+ '&params=' + btoa(encodeURIComponent(JSON.stringify(params)));

	xmlhttp.open('POST', url + getp, true);
	xmlhttp.setRequestHeader('Content-type', 'application/x-www-form-urlencoded');

	xmlhttp.send(postp);

	return xmlhttp;
}

function crsbkr_search_results(params)
{
	if (!crsbkr_array_get(params, 'data.controller_form.code', null))
	{
		return;
	}

	var sessionHex = crsbkr_digest(document.cookie);

	var data = params.data;
	var input = params.input;
	var reservation = params.reservation;
	var results = params.results;
	var selectedQuote = params.selected_quote;

	var fbParams =
	{
		checkout_time : input.checkout_time,
		checkin_time : input.checkin_time,
		checkout_base : crsbkr_array_get(data, 'controller_form.params.bases.' + crsbkr_array_get(input, 'checkout_base') + '.title'),
		checkin_base : crsbkr_array_get(data, 'controller_form.params.bases.' + crsbkr_array_get(input, 'checkin_base') + '.title'),
		controller_form : crsbkr_array_get(data, 'controller_form.code', null),
		session : sessionHex
	}

	if (crsbkr_array_get(input, 'group', '') != '')
	{
		fbParams.group = crsbkr_array_get(data, 'controller_form.params.groups.' + crsbkr_array_get(input, 'group') + '.name');
	}

	var holder = document.getElementById('crsbkr_search_results');
	while (holder.lastChild)
	{
		holder.removeChild(holder.lastChild);
	}

	var loading = crsbkr_loading();
	holder.appendChild(loading);

	var p = crsbkr_en('p');
	loading.appendChild(p);
	p.innerHTML = CRSBKR_STRINGS['results.loading'];

	// Send a request for each checkout/checkin base.
	var requests = results || reservation ? null : [];
	if (requests !== null)
	{
		if (!results)
		{
			results = [];
		}

		var checkoutBases = [...new Set(crsbkr_array_get(data, 'controller_form.params.bases.' + crsbkr_array_get(input, 'checkout_base') + '.bases', '').split(',').filter(n => ('' + n).length > 0))];
		var checkinBases = [...new Set(crsbkr_array_get(data, 'controller_form.params.bases.' + crsbkr_array_get(input, 'checkin_base') + '.bases', '').split(',').filter(n => ('' + n).length > 0))];

		crsbkr_fbq('trackCustom', 'crsbkr_search_results', fbParams);

		for (var i = 0; i < checkoutBases.length; i++)
		{
			// Get the checkout base.
			var checkoutBase = null;
			for (var ii = 0; ii < data.base.length; ii++)
			{
				if (data.base[ii].uid == checkoutBases[i])
				{
					checkoutBase = data.base[ii];
					break;
				}
			}

			if (!checkoutBase)
			{
				continue;
			}

			for (var j = 0; j < checkinBases.length; j++)
			{
				// Get the checkin base.
				var checkinBase = null;
				for (var jj = 0; jj < data.base.length; jj++)
				{
					if (data.base[jj].uid == checkinBases[j])
					{
						checkinBase = data.base[jj];
						break;
					}
				}

				if (!checkinBase || checkoutBase.user_uid != checkinBase.user_uid)
				{
					continue;
				}

				// Get the acriss codes.
				var acrissCodes = [];
				var groups = [...new Set(crsbkr_array_get(data, 'controller_form.params.groups.' + crsbkr_array_get(input, 'group') + '.groups', '').split(',').filter(n => ('' + n).length > 0))];

				for (var k = 0; k < groups.length; k++)
				{
					// Get the group.
					for (var kk = 0; kk < data.acriss_code_web.length; kk++)
					{
						if (data.acriss_code_web[kk].user_uid != checkoutBase.user_uid)
						{
							continue;
						}

						if (data.acriss_code_web[kk].uid == groups[k])
						{
							acrissCodes.push(...new Set(crsbkr_array_get(data.acriss_code_web[kk], 'acriss_code_ids', '').split(',').filter(n => ('' + n).length > 0)));
						}
					}
				}

				var baseAcrissCodes = [...new Set(crsbkr_array_get(checkoutBase, 'settings.web.acriss_codes', '').split(',').filter(n => ('' + n).length > 0))];
				acrissCodes = acrissCodes.filter(value => baseAcrissCodes.includes(value));

				// Send the request.
				var params =
				{
					accessories : 'all',
					acriss_code_ids : acrissCodes.join(','),
					checkout_base_id : checkoutBase.uid,
					checkin_base_id : checkinBase.uid,
					checkout_time : input.checkout_time,
					checkin_time : input.checkin_time,
					tariff_type : 'web',
					vat_incl : crsbkr_array_get(data, 'controller_form.params.vat_incl', 0),
				};

				// Get the accessories.
				var selectedAccs = [];
				var thisAccessories = crsbkr_array_get(input, 'accessories', '').split(',').filter(n => ('' + n).length > 0);
				for (var k = 0; k < thisAccessories.length; k++)
				{
					var accessories = [...new Set(crsbkr_array_get(data, 'controller_form.params.accessories.' + thisAccessories[k] + '.accessories', '').split(',').filter(n => ('' + n).length > 0))];
					for (var kk = 0; kk < crsbkr_array_get(data, 'accessory', []).length; kk++)
					{
						if (data.accessory[kk].user_uid != checkoutBase.user_uid || accessories.indexOf(data.accessory[kk].uid) == -1)
						{
							continue;
						}

						selectedAccs.push(data.accessory[kk].uid);
					}
				}

				if (crsbkr_array_get(input, 'promo_code', '') != '')
				{
					params.promo_code = input.promo_code;
				}

				var user = null;
				for (var k = 0; !user && k < crsbkr_array_get(data, 'user', []).length; k++)
				{
					if (crsbkr_array_get(data.user[k], 'uid') == checkoutBase.user_uid)
					{
						user = data.user[k];
					}
				}

				var broker = null;
				var apiEntity = null;

				for (var k = 0; !apiEntity && k < crsbkr_array_get(data, 'broker', []).length; k++)
				{
					if (crsbkr_array_get(data.broker[k], 'user_uid') == checkoutBase.user_uid)
					{
						broker = data.broker[k];
						apiEntity = 'broker/' + data.broker[k].uid;
					}
				}

				var agent = null;
				for (var k = 0; !apiEntity && k < crsbkr_array_get(data, 'agent', []).length; k++)
				{
					if (crsbkr_array_get(data.agent[k], 'user_uid') == checkoutBase.user_uid)
					{
						agent = data.agent[k];
						apiEntity = 'agent/' + data.agent[k].uid;
					}
				}

				if (!apiEntity)
				{
					continue;
				}

				var request = { error : null, request : null, result : null };
				requests.push(request);
				request.id = requests.length - 1;

				((request) =>
				{
					request.request = crsbkr_request('GET', 'user/' + checkoutBase.user_uid + '/' + apiEntity + '/quote', params, (success, response) =>
					{
						if (success && response)
						{
							for (var i = crsbkr_array_get(response, 'quotes', []).length - 1; i >= 0; i--)
							{
								if (!response.quotes[i].accessories)
								{
									var thisSelectedAccs = JSON.parse(JSON.stringify(selectedAccs));

									// Be sure we have all the accessories.
									response.quotes[i].accessories = JSON.parse(JSON.stringify(response.accessory_quotes));
									for (var ii = response.quotes[i].accessories.length - 1; ii >= 0; ii--)
									{
										var acc = response.quotes[i].accessories[ii];

										var available = false;
										var excluded = false;
										var required = false;

										for (var j in crsbkr_array_get(acc, 'accessory.availability', []))
										{
											if (crsbkr_array_get(acc.accessory.availability[j], 'acriss_code_id', '') == '' || crsbkr_array_get(acc.accessory.availability[j], 'acriss_code_id', '') == crsbkr_array_get(response.quotes[i], 'acriss_code.uid', ''))
											{
												available = true;
												break;
											}
										}

										for (var j in crsbkr_array_get(acc, 'accessory.exclusion', []))
										{
											if (crsbkr_array_get(acc.accessory.exclusion[j], 'acriss_code_id', '') == '' || crsbkr_array_get(acc.accessory.exclusion[j], 'acriss_code_id', '') == crsbkr_array_get(response.quotes[i], 'acriss_code.uid', ''))
											{
												excluded = true;
												break;
											}
										}

										for (var j in crsbkr_array_get(acc, 'accessory.requirement', []))
										{
											if (crsbkr_array_get(acc.accessory.requirement[j], 'acriss_code_id', '') == '' || crsbkr_array_get(acc.accessory.requirement[j], 'acriss_code_id', '') == crsbkr_array_get(response.quotes[i], 'acriss_code.uid', ''))
											{
												required = true;
												break;
											}
										}

										acc.available = available;
										acc.required = required;
										acc.excluded = excluded;

										if (!available || excluded)
										{
											response.quotes[i].accessories.splice(ii, 1);
										}
										else if (thisSelectedAccs.indexOf(crsbkr_array_get(acc, 'accessory.uid')) != -1)
										{
											acc.required = true;
											thisSelectedAccs.splice(thisSelectedAccs.indexOf(crsbkr_array_get(acc, 'accessory.uid')), 1);
										}
									}

									// If we haven't got all necessary accessories, ignore this quote.
									if (thisSelectedAccs.length > 0)
									{
										response.quotes.splice(i, 1);
									}
									else
									{
										response.quotes[i].agent = agent;
										response.quotes[i].broker = broker;
										response.quotes[i].user = user;
									}
								}
							}

							request.result = response;
						}
						else
						{
							request.error = response.error;
							console.log('error for request: ' + request.id + ': ' + response.error);
						}
					});
				})(request);

				break;
			}
		}
	}

	var done = requests === null;
	var now = new Date().getTime();

	var callback = () =>
	{
		if (requests !== null)
		{
			done = true;
			for (var i = 0; i < requests.length; i++)
			{
				if (requests[i].result === null && requests[i].error === null)
				{
					done = false;
					break;
				}
			}

			if (!done && new Date().getTime() - now > 10000)
			{
				done = true;

				for (var i = 0; i < requests.length; i++)
				{
					if (requests[i].result === null && requests[i].error === null)
					{
						requests[i].request.abort();
					}
				}
			}

			if (!done)
			{
				setTimeout(callback, 500);
				return;
			}

			for (var i = 0; i < requests.length; i++)
			{
				if (requests[i].result)
				{
					results.push(...new Set(crsbkr_array_get(requests[i], 'result.quotes', [])));
				}
			}

			// Sort the results.
			results.sort((lhs, rhs) => { return crsbkr_array_get(lhs, 'rate', 0) - crsbkr_array_get(rhs, 'rate', 0); });
		}

		///
		//
		// GOT A RESERVATION?
		//
		///
		if (reservation && !selectedQuote)
		{
			selectedQuote = localStorage.getItem('crsbkr_module_' + crsbkr_array_get(data, 'controller_form.uid') + '_' + reservation);
			selectedQuote = selectedQuote && selectedQuote != '' ? JSON.parse(selectedQuote) : null;

			if (selectedQuote)
			{
				input = selectedQuote.input;
				selectedQuote = selectedQuote.quote;

				var resData =
				{
					uid : reservation,
					surname : crsbkr_array_get(input, 'client.business', crsbkr_array_get(input, 'client.name', crsbkr_array_get(input, 'driver.name'))),
					email : crsbkr_array_get(input, 'client.email', crsbkr_array_get(input, 'driver.email'))
				};

				crsbkr_request('GET', 'client_reservation/' + btoa(JSON.stringify(resData)) + "?expand=details", null, (success, response) =>
				{
					if (success && response)
					{
						params.input = input;
						params.reservation = response;
						params.selected_quote = selectedQuote;

						crsbkr_search_results(params);
					}
					else
					{
						var pre = crsbkr_en('pre');
						pre.innerHTML = JSON.stringify(response, '', 3);
						holder.appendChild(pre);
					}
				});
			}
		}
		///
		//
		// SELECTED QUOTE
		//
		///
		else if (selectedQuote)
		{
			var quote = selectedQuote;

			var summary = crsbkr_en('div');
			holder.appendChild(summary);
			summary.classList.add('summary');

			var a = crsbkr_en('a');
			summary.appendChild(a);
			a.name = 'crsbkr_summary';

			var img = crsbkr_en('img');
			summary.appendChild(img);
			img.src = crsbkr_array_get(quote, 'tariffs.0.tariff_group.image_url');

			var group = crsbkr_array_get(quote, 'tariffs.0.tariff_group.tariff_group_template');

			var p =  crsbkr_en('p');
			summary.appendChild(p);
			p.className = 'group';
			p.innerHTML = CRSBKR_STRINGS['quote.group'];

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = group;

			if (results)
			{
				var modify = crsbkr_en('a');
				p.appendChild(modify);
				modify.href = '';
				modify.innerHTML = CRSBKR_STRINGS['quote.modify'];
				modify.onclick = ((data, input, results) =>
				{
					return (e) =>
					{
						e.preventDefault();
						e.stopPropagation();

						crsbkr_search_results({ data : data, input : input, results : results });
					}
				})(data, input, results);
			}

			var checkoutBase = null;
			var checkinBase = null;

			for (var i = 0; i < data.base.length; i++)
			{
				if (data.base[i].uid == quote.checkout_base_id)
				{
					checkoutBase = data.base[i];
				}

				if (data.base[i].uid == quote.checkin_base_id)
				{
					checkinBase = data.base[i];
				}
			}

			crsbkr_fbq('trackCustom', 'crsbkr_quote', { ...fbParams, ...{
				quote : group
			} });

			var p = crsbkr_en('p');
			summary.appendChild(p);
			p.className = 'base';
			p.innerHTML = CRSBKR_STRINGS['quote.checkout_base'];

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = crsbkr_array_get(checkoutBase, 'settings.web.title');

			if (crsbkr_array_get(checkoutBase, 'settings.web.address', '') != '')
			{
				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = crsbkr_array_get(checkoutBase, 'settings.web.address');
			}

			var p = crsbkr_en('p');
			summary.appendChild(p);
			p.className = 'base';
			p.innerHTML = CRSBKR_STRINGS['quote.checkin_base'];

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = crsbkr_array_get(checkinBase, 'settings.web.title');

			if (crsbkr_array_get(checkoutBase, 'settings.web.address', '') != '')
			{
				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = crsbkr_array_get(checkinBase, 'settings.web.address');
			}

			var p = crsbkr_en('p');
			summary.appendChild(p);
			p.className = 'date';
			p.innerHTML = CRSBKR_STRINGS['quote.checkout_date'];

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = crsbkr_format_date(quote.checkout_time, false, 'datetime');

			var p = crsbkr_en('p');
			summary.appendChild(p);
			p.className = 'date';
			p.innerHTML = CRSBKR_STRINGS['quote.checkin_date'];

			var span = crsbkr_en('span');
			p.appendChild(span);
			span.innerHTML = crsbkr_format_date(quote.checkin_time, false, 'datetime');

			if (crsbkr_array_get(quote, 'user.platform', 'crs') == 'crs')
			{
				var p = crsbkr_en('p');
				summary.appendChild(p);
				p.className = 'km_limit';
				p.innerHTML = CRSBKR_STRINGS['quote.km_limit'];

				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = crsbkr_array_get(quote, 'km_max', 0) == 0 ? CRSBKR_STRINGS['quote.km_limit.unlimited'] : quote.km_max + '/' + CRSBKR_STRINGS['quote.km_limit.day'];
			}

			var costsDiv = crsbkr_en('div');
			summary.appendChild(costsDiv);

			var p = crsbkr_en('p');
			costsDiv.appendChild(p);
			p.innerHTML = CRSBKR_STRINGS['quote.rate'];

			var submitButton = crsbkr_en('button');
			var submitLoading = crsbkr_loading();
			var submitTotals = 0;

			var printCosts = () =>
			{
				while (costsDiv.lastChild != costsDiv.firstChild)
				{
					costsDiv.removeChild(costsDiv.lastChild);
				}

				var totals = {};

				var p = crsbkr_en('p');
				costsDiv.appendChild(p);

				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = crsbkr_array_get(quote, 'tariffs.0.tariff_group.tariff_group_template');

				var rate = crsbkr_array_get(quote, 'rate', 0);
				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = crsbkr_currency(rate);

				var vatInt = crsbkr_vat_int(crsbkr_array_get(quote, 'tax_system', 'it_std_22'));
				totals[vatInt] = crsbkr_array_get(totals, vatInt, 0) + parseFloat(crsbkr_currency(rate, true));

				for (var i = 0; i < crsbkr_array_get(quote, 'discounts', []).length; i++)
				{
					var p = crsbkr_en('p');
					costsDiv.appendChild(p);

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = crsbkr_array_get(quote, 'discounts.' + i + '.desc');

					var rate = -crsbkr_array_get(quote, 'discounts.' + i + '.value', 0);
					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = crsbkr_currency(rate);

					var vatInt = crsbkr_vat_int(crsbkr_array_get(quote, 'discounts.' + i + '.tax_system', 'it_std_22'));
					totals[vatInt] = crsbkr_array_get(totals, vatInt, 0) + parseFloat(crsbkr_currency(rate, true));
				}

				if (crsbkr_array_get(quote, 'insurances', []).length > 0)
				{
					var i = crsbkr_array_get(quote, 'insurance', 0);

					var p = crsbkr_en('p');
					costsDiv.appendChild(p);

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = crsbkr_array_get(quote, 'insurances.' + i + '.insurance_class.settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(quote, 'insurances.' + i + '.insurance_class.settings.translations.en.name', crsbkr_array_get(quote, 'insurances.' + i + '.insurance_class.name')));

					var rate = crsbkr_array_get(quote, 'insurances.' + i + '.rate', 0);
					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = crsbkr_currency(rate);

					var vatInt = crsbkr_vat_int(crsbkr_array_get(quote, 'insurances.' + i + '.tax_system', 'it_std_22'));
					totals[vatInt] = crsbkr_array_get(totals, vatInt, 0) + parseFloat(crsbkr_currency(rate, true));
				}

				for (var o = 0; o < 2; o++)
				{
					for (var i = 0; i < crsbkr_array_get(quote, 'accessories', []).length; i++)
					{
						var acc = quote.accessories[i];

						if (!acc.available || acc.excluded || o == 0 && !acc.required || o == 1 && acc.required || o == 1 && !crsbkr_array_get(acc, 'selected', false))
						{
							continue;
						}

						var p = crsbkr_en('p');
						costsDiv.appendChild(p);

						var span = crsbkr_en('span');
						p.appendChild(span);
						span.innerHTML = (crsbkr_array_get(acc, 'accessory.countable', 0) != 0 && crsbkr_array_get(acc, 'quantity', 1) != 1 ? crsbkr_array_get(acc, 'quantity') + '×' : '') + crsbkr_array_get(acc, 'accessory.settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(acc, 'accessory.settings.translations.en.name', crsbkr_array_get(acc, 'accessory.name')));

						var rate = crsbkr_array_get(acc, 'rate', 0);
						var span = crsbkr_en('span');
						p.appendChild(span);
						span.innerHTML = crsbkr_currency(rate);

						var vatInt = crsbkr_vat_int(crsbkr_array_get(acc, 'tax_system', 'it_std_22'));
						totals[vatInt] = crsbkr_array_get(totals, vatInt, 0) + parseFloat(crsbkr_currency(rate, true));
					}
				}

				var allTotals = 0;
				for (var k in totals)
				{
					if (totals.hasOwnProperty(k))
					{
						allTotals += totals[k];
					}
				}

				if (crsbkr_array_get(quote, 'vat_incl', 0) == 0)
				{
					var p = crsbkr_en('p');
					costsDiv.appendChild(p);
					p.className = 'totals';

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = CRSBKR_STRINGS['quote.totals.vatExcl'];

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = crsbkr_currency(allTotals);

					var allVat = 0;
					for (var k in totals)
					{
						if (totals.hasOwnProperty(k))
						{
							allVat += Math.round(totals[k] * k) / 100;
						}
					}

					var p = crsbkr_en('p');
					costsDiv.appendChild(p);

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = CRSBKR_STRINGS['quote.totals.vat'];

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = crsbkr_currency(allVat);

					allTotals += allVat;
				}

				var p = crsbkr_en('p');
				costsDiv.appendChild(p);
				p.className = 'totals';

				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = CRSBKR_STRINGS['quote.totals'];

				var span = crsbkr_en('span');
				p.appendChild(span);
				span.innerHTML = crsbkr_currency(allTotals);

				submitTotals = Math.round(allTotals * (quote.agent ? crsbkr_array_get(quote, 'agent.requested_down_payment', 1) : crsbkr_array_get(quote, 'broker.requested_down_payment', 1)) * 100) / 100;
				submitButton.innerHTML = CRSBKR_STRINGS[reservation ? 'quote.pay' : 'quote.submit'].replace('{totals}', crsbkr_currency(submitTotals));
			}

			printCosts();

			///
			//
			// COVERAGES
			//
			///
			var div = crsbkr_en('div');
			holder.appendChild(div);
			div.className = 'quote';

			if (crsbkr_array_get(quote, 'insurances', []).length > 0)
			{
				var h1 = crsbkr_en('h1');
				div.appendChild(h1);
				h1.innerHTML = CRSBKR_STRINGS['quote.insurances'];

				var p = crsbkr_en('p');
				div.appendChild(p);
				p.innerHTML = CRSBKR_STRINGS['quote.insurances.desc'];

				var ul = crsbkr_en('ul');
				div.appendChild(ul);

				if (crsbkr_array_get(quote, 'insurance', false) === false)
				{
					quote.insurance = 0;
				}

				if (reservation)
				{
					ul.classList.add('disabled');
				}

				for (var i = 0; i < quote.insurances.length; i++)
				{
					var li = crsbkr_en('li');
					ul.appendChild(li);
					li.className = crsbkr_array_get(quote, 'insurance', 0) == i ? 'selected' : '';

					if (!reservation)
					{
						li.onclick = ((i, li) =>
						{
							return () =>
							{
								if (crsbkr_array_get(quote, 'insurance', 0) != i)
								{
									li.parentNode.children[crsbkr_array_get(quote, 'insurance', 0)].classList.remove('selected');
								}

								quote.insurance = i;
								li.classList.add('selected');

								crsbkr_fbq('trackCustom', 'crsbkr_quote_insurance', { ...fbParams, ...{
									quote : group,
									insurance : crsbkr_array_get(quote.insurances[i], 'insurance_class.name', '')
								} });

								printCosts();
							}
						})(i, li);
					}

					var p = crsbkr_en('p');
					li.appendChild(p);
					p.className = 'title';
					p.innerHTML = crsbkr_array_get(quote.insurances[i], 'insurance_class.settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(quote.insurances[i], 'insurance_class.settings.translations.en.name', crsbkr_array_get(quote.insurances[i], 'insurance_class.name', '')));

					var info = crsbkr_en('div');
					p.appendChild(info);
					info.className = 'info';
					info.innerHTML = 'ⓘ';

					var p = crsbkr_en('p');
					li.appendChild(p);
					p.className = 'rate';
					p.innerHTML = crsbkr_array_get(quote.insurances[i], 'rate', 0) == 0 ? CRSBKR_STRINGS['quote.insurances.included'] : crsbkr_currency(quote.insurances[i].rate);

					var p = crsbkr_en('p');
					li.appendChild(p);
					p.className = 'tick';
					p.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_tick.png\')';

					var infoHolder = crsbkr_en('div');
					info.appendChild(infoHolder);

					var infoArrow = crsbkr_en('div');
					infoHolder.appendChild(infoArrow);

					var infoBox = crsbkr_en('div');
					infoHolder.appendChild(infoBox);

					var p = crsbkr_en('p');
					infoBox.appendChild(p);

					var span = crsbkr_en('span');
					p.appendChild(span);
					span.innerHTML = CRSBKR_STRINGS['quote.insurances.excess'];

					var span2 = crsbkr_en('span');
					p.appendChild(span2);
					span2.innerHTML = crsbkr_currency(crsbkr_array_get(quote.insurances[i], 'insurance_class.excess', 0));

					if (crsbkr_array_get(quote.insurances[i], 'insurance_class.excess_info', '') != '')
					{
						var p = crsbkr_en('p');
						infoBox.appendChild(p);

						var span = crsbkr_en('span');
						p.appendChild(span);
						span.innerHTML = CRSBKR_STRINGS['quote.insurances.excess.info'];

						var span2 = crsbkr_en('span');
						p.appendChild(span2);
						span2.innerHTML = crsbkr_array_get(quote.insurances[i], 'insurance_class.excess_info', '');

						if (crsbkr_array_get(quote, 'user.terms_file_url', '') != '')
						{
							var p = crsbkr_en('p');
							infoBox.appendChild(p);
							p.appendChild(crsbkr_en('span'));

							var a = crsbkr_en('a');
							p.appendChild(a);
							a.href = quote.user.terms_file_url;
							a.innerHTML = CRSBKR_STRINGS['quote.insurances.excess.info.link'];
							a.target = 'crsbkr_terms';
						}
					}
					else
					{
						var p = crsbkr_en('p');
						infoBox.appendChild(p);

						var span = crsbkr_en('span');
						p.appendChild(span);
						span.innerHTML = CRSBKR_STRINGS['quote.insurances.excess.damage'];

						var span2 = crsbkr_en('span');
						p.appendChild(span2);
						span2.innerHTML = crsbkr_currency(crsbkr_array_get(quote.insurances[i], 'insurance_class.excess_damage', 0));

						var p = crsbkr_en('p');
						infoBox.appendChild(p);

						var span = crsbkr_en('span');
						p.appendChild(span);
						span.innerHTML = CRSBKR_STRINGS['quote.insurances.excess.theft'];

						var span2 = crsbkr_en('span');
						p.appendChild(span2);
						span2.innerHTML = crsbkr_currency(crsbkr_array_get(quote.insurances[i], 'insurance_class.excess_theft', 0));
					}
				}
			}

			///
			//
			// EXTRAS
			//
			///
			var hasExtras = false;
			var withQuantity = false;
			for (var i = 0; i < crsbkr_array_get(quote, 'accessories', []).length; i++)
			{
				var acc = quote.accessories[i];
				if (acc.available && !acc.excluded && !acc.required)
				{
					hasExtras = true;
					withQuantity = withQuantity || crsbkr_array_get(acc, 'accessory.countable', 0) != 0;
				}
			}

			if (hasExtras)
			{
				var h1 = crsbkr_en('h1');
				div.appendChild(h1);
				h1.innerHTML = CRSBKR_STRINGS['quote.accessories'];

				var p = crsbkr_en('p');
				div.appendChild(p);
				p.innerHTML = CRSBKR_STRINGS['quote.accessories.desc'];

				var ul = crsbkr_en('ul');
				div.appendChild(ul);

				if (reservation)
				{
					ul.classList.add('disabled');
				}

				for (var i = 0; i < quote.accessories.length; i++)
				{
					var acc = quote.accessories[i];
					if (!acc.available || acc.excluded || acc.required)
					{
						continue;
					}

					var li = crsbkr_en('li');
					ul.appendChild(li);
					li.className = crsbkr_array_get(acc, 'selected', false) ? 'selected' : '';

					var p = crsbkr_en('p');
					li.appendChild(p);
					p.className = 'title';
					p.innerHTML = crsbkr_array_get(acc, 'accessory.settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(acc, 'accessory.settings.translations.en.name', crsbkr_array_get(acc, 'accessory.name', '')));

					var p = crsbkr_en('p');
					li.appendChild(p);
					p.className = 'rate';
					p.innerHTML = crsbkr_currency(acc.rate);

					var tickP = crsbkr_en('p');
					if (crsbkr_array_get(acc, 'accessory.countable', 0) != 0)
					{
						var p = crsbkr_en('p');
						li.appendChild(p);
						p.className = 'quantity_decrease';
						p.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_minus.png\')';

						if (!reservation)
						{
							p.onclick = ((acc, li, p) =>
							{
								return (e) =>
								{
									!e || e.stopPropagation();
									!e || e.preventDefault();

									acc.quantity = Math.max(0, crsbkr_array_get(acc, 'quantity', 1) - 1);
									acc.rate = crsbkr_array_get(acc, 'unit_rate', 0) * acc.quantity;

									if (acc.quantity == 0)
									{
										delete acc.quantity;
										li.onclick();
									}

									p.innerHTML = crsbkr_array_get(acc, 'selected', 0) != 0 ? crsbkr_array_get(acc, 'quantity', 1) : '';

									crsbkr_fbq('trackCustom', 'crsbkr_quote_extra', { ...fbParams, ...{
										quote : group,
										extra : crsbkr_array_get(acc, 'accessory.name', ''),
										quantity : acc.quantity ? acc.quantity : 0
									} });

									printCosts();
								}
							})(acc, li, tickP);
						}

						li.appendChild(tickP);
						tickP.className = 'tick' + (withQuantity ? ' wide' : '');
						tickP.innerHTML = crsbkr_array_get(acc, 'selected', 0) != 0 ? crsbkr_array_get(acc, 'quantity', 1) : '';

						var p = crsbkr_en('p');
						li.appendChild(p);
						p.className = 'quantity_increase';
						p.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_plus.png\')';

						if (!reservation)
						{
							p.onclick = ((acc, li, p) =>
							{
								return (e) =>
								{
									!e || e.stopPropagation();
									!e || e.preventDefault();

									acc.quantity = crsbkr_array_get(acc, 'quantity', 1) + 1;
									acc.rate = crsbkr_array_get(acc, 'unit_rate', 0) * acc.quantity;

									p.innerHTML = crsbkr_array_get(acc, 'selected', 0) != 0 ? crsbkr_array_get(acc, 'quantity', 1) : '';

									crsbkr_fbq('trackCustom', 'crsbkr_quote_extra', { ...fbParams, ...{
										quote : group,
										extra : crsbkr_array_get(acc, 'accessory.name', ''),
										quantity : acc.quantity
									} });

									printCosts();
								}
							})(acc, li, tickP);
						}
					}
					else
					{
						li.appendChild(tickP);
						tickP.className = 'tick' + (withQuantity ? ' wide' : '');
						tickP.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_tick.png\')';
					}

					if (!reservation)
					{
						li.onclick = ((acc, li, p) =>
						{
							return () =>
							{
								if (crsbkr_array_get(acc, 'selected', false))
								{
									acc.selected = false;
									li.classList.remove('selected');
								}
								else
								{
									acc.selected = true;
									li.classList.add('selected');
								}

								if (crsbkr_array_get(acc, 'accessory.countable', 0) != 0)
								{
									p.innerHTML = crsbkr_array_get(acc, 'selected', 0) != 0 ? crsbkr_array_get(acc, 'quantity', 1) : '';
								}

								crsbkr_fbq('trackCustom', 'crsbkr_quote_extra', { ...fbParams, ...{
									quote : group,
									extra : crsbkr_array_get(acc, 'accessory.name', ''),
									selected : acc.selected
								} });

								printCosts();
							}
						})(acc, li, tickP);
					}
				}
			}

			///
			//
			// BOOKING FIELDS
			//
			///
			var allFields = [ 'first_name', 'name', 'phone', 'tax_number', 'email', 'email_confirm', 'address', 'city', 'zip_code', 'province', 'country_id', 'nationality', 'date_of_birth', 'place_of_birth' ];
			var fields = crsbkr_array_get(data, 'controller_form.params.driver_fields', '').split(',').filter(n => ('' + n).length > 0);
			var requiredDriverFields = crsbkr_array_get(data, 'controller_form.params.required_driver_fields', '').split(',').filter(n => ('' + n).length > 0);

			if (fields.indexOf('email') != -1)
			{
				fields.push('email_confirm');
			}

			if (requiredDriverFields.indexOf('email') != -1)
			{
				requiredDriverFields.push('email_confirm');
			}

			if (fields.indexOf('notes') != -1 && (!reservation || crsbkr_array_get(quote, 'notes', '') != ''))
			{
				var h1 = crsbkr_en('h1');
				div.appendChild(h1);
				h1.innerHTML = CRSBKR_STRINGS['booking.driver.notes'];

				var textarea = crsbkr_en('textarea');
				div.appendChild(textarea);
				textarea.innerHTML = crsbkr_array_get(quote, 'notes', '');

				textarea.onchange = ((textarea) =>
				{
					return () =>
					{
						quote.notes = textarea.value;
					}
				})(textarea);

				if (reservation)
				{
					textarea.classList.add('disabled');
				}
			}

			var h1 = crsbkr_en('h1');
			div.appendChild(h1);
			h1.innerHTML = CRSBKR_STRINGS['booking.driver'];

			var p = crsbkr_en('p');
			div.appendChild(p);
			p.innerHTML = CRSBKR_STRINGS['booking.driver.desc'];

			var fieldset = crsbkr_en('div');
			div.appendChild(fieldset);
			fieldset.className = 'fieldset';

			if (reservation)
			{
				fieldset.classList.add('disabled');
			}

			for (var i = 0; i < allFields.length; i++)
			{
				if (fields.indexOf(allFields[i]) == -1)
				{
					continue;
				}

				var d = crsbkr_en('div');
				fieldset.appendChild(d);

				var label = crsbkr_en('label');
				d.appendChild(label);
				label.innerHTML = CRSBKR_STRINGS['booking.driver.' + allFields[i]] + (requiredDriverFields.indexOf(allFields[i]) != -1 ? '*' : '');

				if (allFields[i] == 'country_id')
				{
					var select = crsbkr_en('select');
					d.appendChild(select);
					select.id = 'crsbkr_driver_' + allFields[i];
					select.value = crsbkr_array_get(input, 'driver.' + allFields[i], '');

					for (var ii = 0; ii < CRSBKR_COUNTRIES.length; ii++)
					{
						var option = crsbkr_en('option');
						select.appendChild(option);
						option.value = CRSBKR_COUNTRIES[ii].uid;
						option.innerHTML = CRSBKR_COUNTRIES[ii].name;
					}

					select.onchange = ((field, thisInput) =>
					{
						return () =>
						{
							thisInput.classList.remove('error');
							if (!input.driver)
							{
								input.driver = {};
							}

							input.driver[field] = thisInput.value;
						}
					})(allFields[i], select);

					thisInput.onfocus = ((field, thisInput) =>
					{
						return () =>
						{
							thisInput.classList.remove('error');
						}
					})(allFields[i], select);
				}
				else if (allFields[i] == 'nationality')
				{
					var select = crsbkr_en('select');
					d.appendChild(select);
					select.id = 'crsbkr_driver_' + allFields[i];
					select.value = crsbkr_array_get(input, 'driver.' + allFields[i], '');

					for (var ii = 0; ii < CRSBKR_COUNTRIES.length; ii++)
					{
						var option = crsbkr_en('option');
						select.appendChild(option);
						option.value = CRSBKR_COUNTRIES[ii].code;
						option.innerHTML = CRSBKR_COUNTRIES[ii].name;
					}

					select.onchange = ((field, thisInput) =>
					{
						return () =>
						{
							thisInput.classList.remove('error');
							if (!input.driver)
							{
								input.driver = {};
							}

							input.driver[field] = thisInput.value;
						}
					})(allFields[i], select);

					thisInput.onfocus = ((field, thisInput) =>
					{
						return () =>
						{
							thisInput.classList.remove('error');
						}
					})(allFields[i], select);
				}
				else
				{
					var thisInput = crsbkr_en('input');
					d.appendChild(thisInput);
					thisInput.id = 'crsbkr_driver_' + allFields[i];
					thisInput.value = crsbkr_array_get(input, 'driver.' + allFields[i], '');
					thisInput.onchange = ((field, thisInput) =>
					{
						return () =>
						{
							thisInput.classList.remove('error');
							if (!input.driver)
							{
								input.driver = {};
							}

							input.driver[field] = thisInput.value;
						}
					})(allFields[i], thisInput);

					thisInput.onfocus = ((field, thisInput) =>
					{
						return () =>
						{
							thisInput.classList.remove('error');
						}
					})(allFields[i], thisInput);
				}
			}

			///
			//
			// INVOICING FIELDS
			//
			///
			var allFields = [ 'business', 'vat_number', 'first_name', 'name', 'phone', 'email', 'tax_number', 'address', 'city', 'zip_code', 'province', 'country_id', 'date_of_birth', 'place_of_birth', 'pec', 'sdi_code' ];
			var fields = crsbkr_array_get(data, 'controller_form.params.invoicing_fields', '').split(',').filter(n => ('' + n).length > 0);
			var requiredInvoicingFields = crsbkr_array_get(data, 'controller_form.params.required_invoicing_fields', '').split(',').filter(n => ('' + n).length > 0);

			if (fields.length > 0 && (!reservation || input.client))
			{
				var h1 = crsbkr_en('h1');
				div.appendChild(h1);
				h1.innerHTML = CRSBKR_STRINGS['booking.invoice'];

				var p = crsbkr_en('p');
				div.appendChild(p);
				p.innerHTML = CRSBKR_STRINGS['booking.invoice.desc'];

				var fieldset = crsbkr_en('div');
				div.appendChild(fieldset);
				fieldset.className = 'fieldset';

				if (reservation)
				{
					fieldset.classList.add('disabled');
				}

				for (var i = 0; i < allFields.length; i++)
				{
					if (fields.indexOf(allFields[i]) == -1)
					{
						continue;
					}

					var d = crsbkr_en('div');
					fieldset.appendChild(d);

					var label = crsbkr_en('label');
					d.appendChild(label);
					label.innerHTML = CRSBKR_STRINGS['booking.invoice.' + allFields[i]] + (requiredInvoicingFields.indexOf(allFields[i]) != -1 ? '*' : '');

					if (allFields[i] == 'country_id')
					{
						var select = crsbkr_en('select');
						d.appendChild(select);
						select.id = 'crsbkr_invoicing_' + allFields[i];
						select.value = crsbkr_array_get(input, 'invoice.' + allFields[i], '');

						for (var ii = 0; ii < CRSBKR_COUNTRIES.length; ii++)
						{
							var option = crsbkr_en('option');
							select.appendChild(option);
							option.value = CRSBKR_COUNTRIES[ii].uid;
							option.innerHTML = CRSBKR_COUNTRIES[ii].name;
						}

						select.onchange = ((field, thisInput) =>
						{
							return () =>
							{
								if (!input.client)
								{
									input.client = {};
								}

								input.client[field] = thisInput.value;
							}
						})(allFields[i], select);

						select.onfocus = ((field, thisInput) =>
						{
							return () =>
							{
								thisInput.classList.remove('error');
							}
						})(allFields[i], select);
					}
					else
					{
						var thisInput = crsbkr_en('input');
						d.appendChild(thisInput);
						thisInput.id = 'crsbkr_invoicing_' + allFields[i];
						thisInput.value = crsbkr_array_get(input, 'client.' + allFields[i], '');
						thisInput.onchange = ((field, thisInput) =>
						{
							return () =>
							{
								thisInput.classList.remove('error');
								if (!input.client)
								{
									input.client = {};
								}

								input.client[field] = thisInput.value;
							}
						})(allFields[i], thisInput);

						thisInput.onfocus = ((field, thisInput) =>
						{
							return () =>
							{
								thisInput.classList.remove('error');
								if (field == 'pec' || field == 'sdi_code')
								{
									var e = crsbkr_e('crsbkr_invoicing_' + (field == 'pec' ? 'sdi_code' : 'pec'));
									if (e)
									{
										e.classList.remove('error');
									}
								}
							}
						})(allFields[i], thisInput);
					}
				}
			}

			///
			//
			// TERMS AND CONDITIONS
			//
			///
			var privacyLink = crsbkr_array_get(data, 'user', []).length > 1 ? crsbkr_array_get(data, 'controller_form.params.privacy_file_url', crsbkr_array_get(quote, 'user.privacy_file_url')) : crsbkr_array_get(data, 'controller_form.params.privacy_file_url');
			var termsLink = crsbkr_array_get(data, 'user', []).length > 1 ? crsbkr_array_get(data, 'controller_form.params.terms_file_url', crsbkr_array_get(quote, 'user.terms_file_url')) : crsbkr_array_get(data, 'controller_form.params.terms_file_url');

			var ul = crsbkr_en('ul');
			div.appendChild(ul);
			ul.className = 'terms';

			var terms = CRSBKR_STRINGS['quote.terms'];
			if (termsLink)
			{
				terms = terms.replace(CRSBKR_STRINGS['quote.terms.link'], '<a href="' + termsLink + '" target="crsbkr_terms">' + CRSBKR_STRINGS['quote.terms.link'] + '</a>');
			}

			var termsLi = crsbkr_en('li');
			ul.appendChild(termsLi);

			if (!reservation)
			{
				termsLi.onclick = () =>
				{
					termsLi.classList.remove('error');
					if (termsLi.classList.contains('selected'))
					{
						termsLi.classList.remove('selected')
					}
					else
					{
						termsLi.classList.add('selected');
					}
				};
			}
			else
			{
				ul.classList.add('disabled');
			}

			var p = crsbkr_en('p');
			termsLi.appendChild(p);
			p.classList.add('tick');
			p.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_tick.png\')';

			var p = crsbkr_en('p');
			termsLi.appendChild(p);
			p.innerHTML = terms;
			if (termsLink)
			{
				p.getElementsByTagName('a')[0].onclick = (e) => { e.stopPropagation(); };
			}

			var privacy = CRSBKR_STRINGS['quote.privacy'];
			if (privacyLink)
			{
				privacy = privacy.replace(CRSBKR_STRINGS['quote.privacy.link'], '<a href="' + privacyLink + '" target="crsbkr_terms">' + CRSBKR_STRINGS['quote.privacy.link'] + '</a>');
			}

			var privacyLi = crsbkr_en('li');
			ul.appendChild(privacyLi);

			if (!reservation)
			{
				privacyLi.onclick = () =>
				{
					privacyLi.classList.remove('error');
					if (privacyLi.classList.contains('selected'))
					{
						privacyLi.classList.remove('selected')
					}
					else
					{
						privacyLi.classList.add('selected');
					}
				};
			}

			var p = crsbkr_en('p');
			privacyLi.appendChild(p);
			p.classList.add('tick');
			p.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_tick.png\')';

			var p = crsbkr_en('p');
			privacyLi.appendChild(p);
			p.innerHTML = privacy;
			if (privacyLink)
			{
				p.getElementsByTagName('a')[0].onclick = (e) => { e.stopPropagation(); };
			}

			if (reservation)
			{
				termsLi.classList.add('selected');
				privacyLi.classList.add('selected');
			}

			///
			//
			// GOT A RESERVATION?
			//
			///
			if (reservation)
			{
				if (crsbkr_array_get(reservation, 'client_reservation.status') == 'booked')
				{
					var p = crsbkr_en('p');
					div.appendChild(p);
					p.className = 'success';
					p.innerHTML = CRSBKR_STRINGS['quote.confirmation'].replace('{reservation}', crsbkr_array_get(reservation, 'client_reservation.stn_id_reservation', crsbkr_array_get(reservation, 'client_reservation.ref_id_reservation')));
				}
				else if (crsbkr_array_get(reservation, 'client_reservation.status') == 'canceled')
				{
					var p = crsbkr_en('p');
					div.appendChild(p);
					p.className = 'error';
					p.innerHTML = CRSBKR_STRINGS['quote.canceled'].replace('{reservation}', crsbkr_array_get(reservation, 'client_reservation.stn_id_reservation', crsbkr_array_get(reservation, 'client_reservation.ref_id_reservation')));
				}
				else
				{
					var error = null;
					if (crsbkr_array_get(params, 'mpe', '') != '')
					{
						var error = crsbkr_en('p');
						div.appendChild(error);
						error.className = 'error';
						error.innerHTML = params.mpe;
					}

					var cancelButton = crsbkr_en('button');
					div.appendChild(cancelButton);
					cancelButton.className = 'submit cancel';
					cancelButton.innerHTML = CRSBKR_STRINGS['quote.cancel'];
					cancelButton.onclick = () =>
					{
						cancelButton.style.display = 'none';
						submitButton.style.display = 'none';
						submitLoading.style.display = 'block';

						crsbkr_fbq('trackCustom', 'crsbkr_reservation_cancel', { ...fbParams, ...{
							quote : group,
							client_reservation : reservation.client_reservation.uid
						} });

						crsbkr_request('POST', 'client_reservation/' + reservation.client_reservation.uid + '/cancel', null, (success, response) =>
						{
							submitLoading.style.display = 'none';
							if (success && response)
							{
								div.removeChild(error);

								var p = crsbkr_en('p');
								div.appendChild(p);
								p.className = 'error';
								p.innerHTML = CRSBKR_STRINGS['quote.canceled'].replace('{reservation}', crsbkr_array_get(reservation, 'client_reservation.stn_id_reservation', crsbkr_array_get(reservation, 'client_reservation.ref_id_reservation')));
							}
							else
							{
								submitLoading.style.display = 'none';
								cancelButton.style.display = 'inline-block';
								submitButton.style.display = 'inline-block';

								crsbkr_alert(CRSBKR_STRINGS['quote.error'] + (response.error ? '\n' + response.error + (response.message ? ' - ' + response.message : '') : ''));
							}
						});
					};

					if (crsbkr_array_get(reservation, 'paylink', '') != '')
					{
						div.appendChild(submitButton);
						submitButton.className = 'submit';
						submitButton.onclick = () =>
						{
							window.location = reservation.paylink;
						}
					}

					submitLoading.style.display = 'none';
					div.appendChild(submitLoading);
				}
			}
			else
			{
				///
				//
				// SUBMIT BUTTON
				//
				///
				div.appendChild(submitButton);
				div.appendChild(submitLoading);
				submitLoading.style.display = 'none';

				submitButton.className = 'submit';
				submitButton.onclick = () =>
				{
					// Do we have all mandatory fields given?
					var error = null;
					var errorFields = [];

					for (var i = 0; i < requiredDriverFields.length; i++)
					{
						var field = requiredDriverFields[i];
						if (crsbkr_array_get(input, 'driver.' + field, '') == '')
						{
							error = CRSBKR_STRINGS['quote.error.required'];
							errorFields.push(field);

							var e = crsbkr_e('crsbkr_driver_' + field);
							if (e)
							{
								e.classList.add('error');
							}
						}
					}

					for (var i = 0; i < requiredInvoicingFields.length; i++)
					{
						var field = requiredInvoicingFields[i];
						if (crsbkr_array_get(input, 'client.' + field, '') == '')
						{
							if (field == 'pec' && crsbkr_array_get(input, 'client.sdi_code', '') != '' || field == 'sdi_code' && crsbkr_array_get(input, 'client.pec', '') != '')
							{
								continue;
							}

							error = CRSBKR_STRINGS['quote.error.required'];
							errorFields.push(field);

							var e = crsbkr_e('crsbkr_invoicing_' + field);
							if (e)
							{
								e.classList.add('error');
							}
						}
					}

					var emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;
					if (!error && crsbkr_array_get(input, 'driver.email', '') != '' && !emailRegex.test(input.driver.email))
					{
						var e = crsbkr_e('crsbkr_driver_email');
						if (e)
						{
							e.classList.add('error');
						}

						error = CRSBKR_STRINGS['quote.error.emailNotValid'].replace('{email}', input.driver.email);
					}

					if (!error && crsbkr_array_get(input, 'driver.email', '') != '' && crsbkr_array_get(input, 'driver.email_confirm', '') != input.driver.email)
					{
						var e = crsbkr_e('crsbkr_driver_email_confirm');
						if (e)
						{
							e.classList.add('error');
						}

						error = CRSBKR_STRINGS['quote.error.emailsNotMatching'];
					}

					if (!error && crsbkr_array_get(input, 'client.email', '') != '' && !emailRegex.test(input.client.email))
					{
						var e = crsbkr_e('crsbkr_client_email');
						if (e)
						{
							e.classList.add('error');
						}

						error = CRSBKR_STRINGS['quote.error.emailNotValid'].replace('{email}', input.client.email);
					}

					if (!error && !termsLi.classList.contains('selected'))
					{
						error = CRSBKR_STRINGS['quote.error.terms'];
						termsLi.classList.add('error');
					}

					if (!error && !privacyLi.classList.contains('selected'))
					{
						error = CRSBKR_STRINGS['quote.error.privacy'];
						privacyLi.classList.add('error');
					}

					if (error)
					{
						crsbkr_fbq('trackCustom', 'crsbkr_reservation_form_error', { ...fbParams, ...{
							error : error,
							error_fields : errorFields,
							quote : group
						} });

						crsbkr_alert(error);
					}
					else
					{
						submitButton.style.display = 'none';
						submitLoading.style.display = 'block';

						if (input.client)
						{
							var keys = Object.keys(input.client);
							for (var kk = 0; kk < keys.length; kk++)
							{
								if (crsbkr_array_get(input.client, keys[kk], '') == '')
								{
									delete input.client[keys[kk]];
								}
							}

							var keys = Object.keys(input.client);
							if (keys.length == 0 || keys.length == 1 && keys[0] == 'type')
							{
								input.client = null;
							}
						}

						var submitParams =
						{
							json_reservation :
							{
								quote : JSON.parse(JSON.stringify(quote)),
							},
							client_driver : input.driver ? JSON.parse(JSON.stringify(input.driver)) : null,
							client : input.client ? JSON.parse(JSON.stringify(input.client)) : JSON.parse(JSON.stringify(input.driver))
						}

						if (crsbkr_array_get(submitParams.client, 'nationality', '') != '')
						{
							delete submitParams.client.nationality;
						}

						if (crsbkr_array_get(submitParams.client, 'business', '') != '')
						{
							submitParams.client.contact_person = (crsbkr_array_get(submitParams.client, 'first_name', '') + ' ' + crsbkr_array_get(submitParams.client, 'name', '')).trim();
							submitParams.client.name = submitParams.client.business;
							submitParams.client.type = 'business';

							delete submitParams.client.business;
						}
						else
						{
							submitParams.client.type = 'private';
						}

						if (crsbkr_array_get(submitParams.client, 'date_of_birth', '') != '')
						{
							submitParams.client.date_of_birth = crsbkr_format_date(submitParams.client.date_of_birth, true, 'date')
						}

						if (crsbkr_array_get(submitParams.client_driver, 'date_of_birth', '') != '')
						{
							submitParams.client_driver.date_of_birth = crsbkr_format_date(submitParams.client_driver.date_of_birth, true, 'date')
						}

						var keys = [ 'checkout_time', 'checkout_base_id', 'checkin_time', 'checkin_base_id' ];
						for (var i = 0; i < keys.length; i++)
						{
							submitParams[keys[i]] = quote[keys[i]];
						}

						submitParams.initiate_only = true;
						submitParams.tariff_type = 'web';
						submitParams.type = 'reservation';
						submitParams.vat_incl = crsbkr_array_get(data, 'controller_form.params.vat_incl', 0);
						submitParams.acriss_code_id = crsbkr_array_get(quote, 'acriss_code.uid');

						submitParams.payment =
						{
							amount : submitTotals,
							referer : window.location.href.indexOf('#') != -1 ? window.location.href.substr(0, window.location.href.indexOf('#')) : window.location.href
						};

						if (crsbkr_array_get(submitParams, 'json_reservation.quote.notes', '') != '')
						{
							submitParams.notes = submitParams.json_reservation.quote.notes;
							delete submitParams.json_reservation.quote.notes;
						}

						if (crsbkr_array_get(input, 'promo_code', '') != '')
						{
							submitParams.promo_code = input.promo_code;
						}

						if (crsbkr_array_get(quote, 'tariff_category.uid', '') != '')
						{
							submitParams.tariff_category_id = quote.tariff_category.uid;
						}

						if (submitParams.client_driver.email_confirm)
						{
							delete submitParams.client_driver.email_confirm;
						}

						if (submitParams.client.email_confirm)
						{
							delete submitParams.client.email_confirm;
						}

						submitParams.json_reservation.accessory_quotes = crsbkr_array_get(submitParams.json_reservation.quote, 'accessories', []);
						delete submitParams.json_reservation.quote.accessories;

						// Get the accessories.
						submitParams.accessories = '';

						var thisAccessories = crsbkr_array_get(input, 'accessories', '').split(',').filter(n => ('' + n).length > 0);
						for (var k = 0; k < thisAccessories.length; k++)
						{
							var accessories = [...new Set(crsbkr_array_get(data, 'controller_form.params.accessories.' + thisAccessories[k] + '.accessories', '').split(',').filter(n => ('' + n).length > 0))];
							for (var kk = 0; kk < crsbkr_array_get(data, 'accessory', []).length; kk++)
							{
								if (data.accessory[kk].user_uid != checkoutBase.user_uid || accessories.indexOf(data.accessory[kk].uid) == -1)
								{
									continue;
								}

								submitParams.accessories += (submitParams.accessories == '' ? '' : ',') + data.accessory[kk].uid;
							}
						}

						for (var i = submitParams.json_reservation.accessory_quotes.length - 1; i >= 0; i--)
						{
							if (!crsbkr_array_get(submitParams.json_reservation.accessory_quotes[i], 'selected', false)
									&& crsbkr_array_get(submitParams.json_reservation.accessory_quotes[i], 'required', 0) == 0)
							{
								submitParams.json_reservation.accessory_quotes.splice(i, 1);
							}
							else if (crsbkr_array_get(submitParams.json_reservation.accessory_quotes[i], 'required', 0) == 0)
							{
								submitParams.accessories += (submitParams.accessories == '' ? '' : ',') + submitParams.json_reservation.accessory_quotes[i].accessory.uid;
							}
						}

						if (crsbkr_array_get(submitParams.json_reservation.quote, 'insurance', false) !== false)
						{
							submitParams.json_reservation.quote.insurance = crsbkr_array_get(submitParams.json_reservation.quote, 'insurances.' + submitParams.json_reservation.quote.insurance);
						}

						var agent = submitParams.json_reservation.quote.agent;
						var broker = submitParams.json_reservation.quote.broker;
						var user = submitParams.json_reservation.quote.user;

						delete submitParams.json_reservation.quote.agent;
						delete submitParams.json_reservation.quote.broker;
						delete submitParams.json_reservation.quote.user;

						crsbkr_fbq('trackCustom', 'crsbkr_reservation', { ...fbParams, ...{
							quote : group
						} });

						crsbkr_request('POST', 'user/' + crsbkr_array_get(user, 'uid') + '/' + (agent ? 'agent/' + agent.uid : 'broker/' + broker.uid) + '/client_reservation', submitParams, (success, response) =>
						{
							if (success && response)
							{
								localStorage.setItem('crsbkr_module_' + crsbkr_array_get(data, 'controller_form.uid') + '_' + crsbkr_array_get(response, 'client_reservation.uid'), JSON.stringify({ input : input, quote : quote, reservation : response }));

								if (response.paylink)
								{
									crsbkr_fbq('trackCustom', 'crsbkr_reservation_payment', { ...fbParams, ...{
										quote : group,
										client_reservation : response.client_reservation.uid,
										paylink : response.paylink
									} });

									window.location = response.paylink;
								}
							}
							else
							{
								submitLoading.style.display = 'none';
								submitButton.style.display = 'block';

								crsbkr_alert(CRSBKR_STRINGS['quote.error'] + (response.error ? '\n' + response.error + (response.message ? ' - ' + response.message : '') : ''));
							}
						});
					}
				};
			}
		}
		///
		//
		// RESULTS LIST
		//
		///
		else
		{
			var multipleBases = false;
			if (crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) != 0)
			{
				holder.classList.add('full-width');

				// Group the results if we're in full width.
				for (var i = 0; i < results.length; i++)
				{
					var thisCheckoutBase = crsbkr_array_get(results[i], 'checkout_base_id');
					var thisCheckinBase = crsbkr_array_get(results[i], 'checkin_base_id');

					var aid = crsbkr_array_get(results[i], 'acriss_code.uid');

					if (!results[i].options)
					{
						results[i].options = [];
						for (var ii = i + 1; ii < results.length; ii++)
						{
							if (crsbkr_array_get(results[ii], 'acriss_code.uid') == aid && crsbkr_array_get(results[ii], 'checkout_base_id') == thisCheckoutBase && crsbkr_array_get(results[ii], 'checkin_base_id') == thisCheckinBase)
							{
								results[i].options.push(results[ii]);
								results.splice(ii, 1);

								ii--;
							}
						}
					}
				}
			}

			// Check whether we have to deal with different bases.
			var checkoutBase = null;
			var checkinBase = null;

			for (var i = 0; i < results.length; i++)
			{
				var thisCheckoutBase = crsbkr_array_get(results[i], 'checkout_base_id');
				var thisCheckinBase = crsbkr_array_get(results[i], 'checkin_base_id');

				if (checkoutBase == null)
				{
					checkoutBase = thisCheckoutBase;
					checkinBase = thisCheckinBase;
				}
				else if (thisCheckoutBase != checkoutBase || thisCheckinBase != checkinBase)
				{
					multipleBases = true;
				}
			}

			///
			//
			// HEADER
			//
			//
			var div = crsbkr_en('div');
			holder.appendChild(div);
			div.id = 'crsbkr_search_results_header';

			var ul = crsbkr_en('ul');
			div.appendChild(ul);

			var li = crsbkr_en('li');
			ul.appendChild(li);

			var span = crsbkr_en('span');
			li.appendChild(span);
			span.className = 'input';
			span.innerHTML = CRSBKR_STRINGS['results.checkout'];

			var base = crsbkr_array_get(data, 'controller_form.params.bases.' + crsbkr_array_get(input, 'checkout_base'), {});
			var span = crsbkr_en('span');
			li.appendChild(span);
			span.className = 'base';
			span.innerHTML = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.title', crsbkr_array_get(base, 'translations.en.title', crsbkr_array_get(base, 'title', '')));

			var span = crsbkr_en('span');
			li.appendChild(span);
			span.className = 'date';
			span.innerHTML = crsbkr_format_date(crsbkr_array_get(input, 'checkout_time'), false, 'datetime');

			var li = crsbkr_en('li');
			ul.appendChild(li);

			var span = crsbkr_en('span');
			li.appendChild(span);
			span.className = 'input';
			span.innerHTML = CRSBKR_STRINGS['results.checkin'];

			var base = crsbkr_array_get(data, 'controller_form.params.bases.' + crsbkr_array_get(input, 'checkin_base'), {});
			var span = crsbkr_en('span');
			li.appendChild(span);
			span.className = 'base';
			span.innerHTML = crsbkr_array_get(base, 'translations.' + CRSBKR_LANG + '.title', crsbkr_array_get(base, 'translations.en.title', crsbkr_array_get(base, 'title', '')));

			var span = crsbkr_en('span');
			li.appendChild(span);
			span.className = 'date';
			span.innerHTML = crsbkr_format_date(crsbkr_array_get(input, 'checkin_time'), false, 'datetime');

			var li = crsbkr_en('li');
			ul.appendChild(li);

			var a = crsbkr_en('a');
			li.appendChild(a);
			a.href = window.location.href;
			a.innerHTML = CRSBKR_STRINGS['results.change'];

			var ee = crsbkr_e('crsbkr_search_form');
			if (ee)
			{
				a.onclick = (e) =>
				{
					e.stopPropagation();
					e.preventDefault();

					crsbkr_e('crsbkr_search_form').parentNode.classList.add('visible');
				}
			}
			else
			{
				var i = a.href.indexOf('q=');
				if (i != -1)
				{
					var len = a.href.indexOf('&', i);
					len = len == -1 ? a.href.length - i : len - i;

					a.href = a.href.substr(0, i - 1) + a.href.substr(i + len);
				}
			}

			///
			//
			// NO RESULTS
			//
			///
			if (results.length == 0)
			{
				var div = crsbkr_en('div');
				holder.appendChild(div);
				div.className = 'error';

				var h2 = crsbkr_en('h2');
				div.appendChild(h2);
				h2.innerHTML = CRSBKR_STRINGS['results.none.title'];

				var p = crsbkr_en('p');
				div.appendChild(p);
				p.innerHTML = CRSBKR_STRINGS['results.none.text'];
			}
			else
			{
				///
				//
				// RESULT GROUPS
				//
				///
				var groups = null;
				var groupsDiv = null;
				if (crsbkr_array_get(data, 'controller_form.params.result_groups', false))
				{
					groups = JSON.parse(JSON.stringify(crsbkr_array_get(data, 'controller_form.params.groups', [])));
					for (var i = groups.length - 1; i >= 0; i--)
					{
						groups[i].uid = i;
						if (crsbkr_array_get(groups[i], 'icon_url', '') == '')
						{
							groups.splice(i, 1);
						}
						else
						{
							groups[i].acriss_code_ids = [];
							groups[i].min = null;
						}
					}

					groupsDiv = crsbkr_en('div');
					holder.appendChild(groupsDiv);
					groupsDiv.id = 'crsbkr_search_results_groups';
				}

				///
				//
				// RESULTS
				//
				///
				var ul = crsbkr_en('ul')
				holder.appendChild(ul);

				for (var ri = 0; ri < results.length; ri++)
				{
					var quote = results[ri];

					var li = crsbkr_en('li');
					ul.appendChild(li);

					var h1 = crsbkr_en('h1');
					li.appendChild(h1);
					h1.innerHTML = crsbkr_array_get(quote, 'tariffs.0.tariff_group.label');

					var h2 = crsbkr_en('h2');
					li.appendChild(h2);
					h2.innerHTML = crsbkr_array_get(quote, 'tariffs.0.tariff_group.tariff_group_template');

					if (multipleBases)
					{
						var checkoutBase = null;
						var checkinBase = null;

						for (var i = 0; (!checkoutBase || !checkinBase) && i < data.base.length; i++)
						{
							if (crsbkr_array_get(data.base[i], 'uid') == crsbkr_array_get(quote, 'checkout_base_id'))
							{
								checkoutBase = data.base[i];
							}

							if (crsbkr_array_get(data.base[i], 'uid') == crsbkr_array_get(quote, 'checkin_base_id'))
							{
								checkinBase = data.base[i];
							}
						}

						var div = crsbkr_en('div');
						li.appendChild(div);
						div.classList.add('base');
						div.style.backgroundImage = 'url(\'' + CRSBKR_DIR + 'images/icon_marker.png' + '\')';

						var span = crsbkr_en('span');
						div.appendChild(span);
						span.innerHTML = crsbkr_array_get(checkoutBase, 'settings.web.translations.' + CRSBKR_LANG + '.title', crsbkr_array_get(checkoutBase, 'settings.web.translations.en.title', crsbkr_array_get(checkoutBase, 'settings.web.title', crsbkr_array_get(quote, 'checkout_base_id'))));

						var span = crsbkr_en('span');
						div.appendChild(span);
						span.innerHTML = crsbkr_array_get(checkinBase, 'settings.web.translations.' + CRSBKR_LANG + '.title', crsbkr_array_get(checkinBase, 'settings.web.translations.en.title', crsbkr_array_get(checkinBase, 'settings.web.title', crsbkr_array_get(quote, 'checkin_base_id'))));
					}

					var box = crsbkr_en('div');
					li.appendChild(box);
					box.className = 'box';

					var info = crsbkr_en('div');
					box.appendChild(info);
					info.className = 'info';

					var img = crsbkr_en('img');
					img.style.backgroundImage = 'url(\'' + crsbkr_array_get(quote, 'tariffs.0.tariff_group.image_url') + '\')';
					info.appendChild(img);

					var table = crsbkr_en('table');
					info.appendChild(table);

					var tr = crsbkr_en('tr');
					table.appendChild(tr);

					var td = crsbkr_en('td');
					td.className = 'params';
					tr.appendChild(td);

					var div = crsbkr_en('div');
					td.appendChild(div);

					if (crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters', '') != '')
					{
						var keys = ['seats', 'doors', 'small_bags', 'big_bags', 'cabins', 'seats_no', 'berths', 'toilets', 'type'];
						for (var k in keys)
						{
							var key = keys[k];
							if (crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.' + key, '') != '')
							{
								var p1 = null;
								var p2 = null;
								var s = '';

								var is = Object.keys(crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.' + key, '')).sort();
								for (var i in is)
								{
									i = is[i];
									if (crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.' + key + '.' + i, ''))
									{
										if (p1 == null)
										{
											s += (s == '' ? '' : ', ') + i.substr(1);
											p1 = i.substr(1);
											p2 = p1;
										}
										else if (p2 == parseInt(i.substr(1)) - 1)
										{
											p2 = i.substr(1);
										}
										else
										{
											if (p2 != p1)
											{
												s += '-' + p2;
											}

											s += (s == '' ? '' : ', ') + i.substr(1);

											p1 = i.substr(1);
											p2 = p1;
										}
									}
								}

								if (p2 != p1)
								{
									s += '-' + p2;
								}

								if (s != '')
								{
									if (key == 'cabins' && crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.cabin_skipper.v1', false))
									{
										s += '+1';
									}

									var ss = crsbkr_en('span');
									ss.className = key.toLowerCase();
									ss.innerHTML = s + (CRSBKR_STRINGS['results.params.vehicle_filters.' + key] ? ' ' + CRSBKR_STRINGS['results.params.vehicle_filters.' + key] : '');
									ss.style.backgroundImage = 'url(\'' + CRSBKR_DIR + '/images/icon_param_' + key.toLowerCase().replaceAll('_', '') + '.png' + '\')';
									div.appendChild(ss);
								}
							}
						}

						if (crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.gearshift', ''))
						{
							var s = '';
							for (var i in crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.gearshift', ''))
							{
								s += (s == '' ? '' : ', ') + CRSBKR_STRINGS['results.params.vehicle_filters.gearshift.' + i];
							}

							if (s != '')
							{
								var ss = crsbkr_en('span');
								ss.className = 'gearshift';
								ss.innerHTML = s;
								ss.style.backgroundImage = 'url(\'' + CRSBKR_DIR + '/images/icon_param_gearshift.png' + '\')';
								div.appendChild(ss);
							}
						}

						if (crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.fuel_type', ''))
						{
							var s = '';
							for (var i in crsbkr_array_get(quote, 'tariffs.0.tariff_group.params.vehicle_filters.fuel_type', ''))
							{
								s += (s == '' ? '' : ', ') + CRSBKR_STRINGS['results.params.vehicle_filters.fuel_type.' + i];
							}

							if (s != '')
							{
								var ss = crsbkr_en('span');
								ss.className = 'fueltype';
								ss.innerHTML = s;
								ss.style.backgroundImage = 'url(\'' + CRSBKR_DIR + '/images/icon_param_fueltype.png' + '\')';
								div.appendChild(ss);
							}
						}
					}

					// Print the km limit only if it's the same for all options.
					if (crsbkr_array_get(quote, 'user.platform', 'crs') == 'crs' && quote.hasOwnProperty('km_max'))
					{
						var limit = quote.km_max;
						var noLimit = false;
						if (crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) != 0)
						{
							for (var i in quote.options)
							{
								if (quote.options[i].km_max != limit)
								{
									noLimit = true;
									break;
								}
							}
						}

						if (!noLimit && crsbkr_array_get(quote, 'km_max', '') != '')
						{
							var ss = crsbkr_en('span');
							ss.className = 'kmlimit';
							ss.innerHTML = quote.km_max + ' km/' +  CRSBKR_STRINGS['results.km_limit.day'];
							ss.style.backgroundImage = 'url(\'' + CRSBKR_DIR + '/images/icon_param_kmlimit.png' + '\')';
							div.appendChild(ss);
						}
						else if (!noLimit)
						{
							var ss = crsbkr_en('span');
							ss.className = 'kmlimit';
							ss.innerHTML = CRSBKR_STRINGS['results.params.vehicle_filters.km_limit.none'];
							ss.style.backgroundImage = 'url(\'' + CRSBKR_DIR + '/images/icon_param_kmlimit.png' + '\')';
							div.appendChild(ss);
						}
					}

					if (div.children.length == 0)
					{
						div.innerHTML = '&nbsp;';
					}

					var rate = 0;
					if (crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) == 0)
					{
						var td = crsbkr_en('td');
						td.className = 'rate';
						tr.appendChild(td);

						var div = crsbkr_en('div');
						td.appendChild(div);

						var span = crsbkr_en('span');
						span.innerHTML = CRSBKR_STRINGS['results.rate'];
						div.appendChild(span);

						var span = crsbkr_en('span');
						span.appendChild(document.createTextNode(crsbkr_currency(crsbkr_array_get(quote, 'rate', 0))));
						div.appendChild(span);

						rate = parseFloat(crsbkr_array_get(quote, 'rate', 0)) - parseFloat(crsbkr_array_get(quote, 'discount', 0));
						if (crsbkr_array_get(quote, 'discount', 0) != 0)
						{
							var span2 = crsbkr_en('span');
							span.classList.add('discount');
							span2.innerHTML = span.textContent;
							span.innerHTML = crsbkr_currency(rate);
							span.appendChild(span2);
						}

						var a = crsbkr_en('a');
						div.appendChild(a);
						a.href = '#crsbkr_summary';

						var button = crsbkr_en('button');
						button.innerHTML = CRSBKR_STRINGS['results.select'];
						a.appendChild(button);
						button.onclick = ((quote) =>
						{
							return () =>
							{
								crsbkr_search_results({ data : data, input : input, results : results, selected_quote : quote });
							}
						})(quote);

						// Any required extras?
						var extras = '';
						if (quote.accessories)
						{
							for (var i in quote.accessories)
							{
								var acc = quote.accessories[i];

								var available = false;
								var excluded = false;
								var required = false;

								for (var j in crsbkr_array_get(acc, 'accessory.availability', []))
								{
									if (crsbkr_array_get(acc.accessory.availability[j], 'acriss_code_id', '') == '' || crsbkr_array_get(acc.accessory.availability[j], 'acriss_code_id', '') == crsbkr_array_get(quote, 'acriss_code.uid', ''))
									{
										available = true;
										break;
									}
								}

								for (var j in crsbkr_array_get(acc, 'accessory.exclusion', []))
								{
									if (crsbkr_array_get(acc.accessory.exclusion[j], 'acriss_code_id', '') == '' || crsbkr_array_get(acc.accessory.exclusion[j], 'acriss_code_id', '') == crsbkr_array_get(quote, 'acriss_code.uid', ''))
									{
										excluded = true;
										break;
									}
								}

								for (var j in crsbkr_array_get(acc, 'accessory.requirement', []))
								{
									if (crsbkr_array_get(acc.accessory.requirement[j], 'acriss_code_id', '') == '' || crsbkr_array_get(acc.accessory.requirement[j], 'acriss_code_id', '') == crsbkr_array_get(quote, 'acriss_code.uid', ''))
									{
										required = true;
										break;
									}
								}

								if (available && !excluded && required && crsbkr_array_get(acc, 'rate', 0) > 0)
								{
									extras += (extras == '' ? '' : '<br />') + CRSBKR_STRINGS['results.extra']
											.replace('{name}', crsbkr_array_get(acc, 'accessory.settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(acc, 'accessory.settings.translations.en.name', crsbkr_array_get(acc, 'accessory.name', ''))))
											.replace('{price}', crsbkr_currency(crsbkr_array_get(acc, 'rate', 0)));
								}
							}
						}

						if (extras != '')
						{
							var span = crsbkr_en('span');
							info.appendChild(span);
							span.className = 'hint';
							span.innerHTML = extras;
						}
					}

					// Any other details?
					var detailsBox = null;
					if (crsbkr_array_get(data, 'controller_form.params.result_details', '') != '' && crsbkr_array_get(quote, 'tariffs.0.tariff_group.params', '') != ''
							|| crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) != 0)
					{
						var groupParams = crsbkr_array_get(quote, 'tariffs.0.tariff_group.params', {});

						detailsBox = crsbkr_en('div');
						detailsBox.className = 'details';

						var div = crsbkr_en('div');
						div.className = 'params';

						var allDetails = crsbkr_array_get(data, 'controller_form.params.result_details', '').split(',').filter(n => ('' + n).length > 0);
						for (var jj in CRSBKR_RESULT_DETAILS)
						{
							var details = null;
							var id = CRSBKR_RESULT_DETAILS[jj].id;

							for (var j in allDetails)
							{
								if (allDetails[j] == id)
								{
									details = CRSBKR_RESULT_DETAILS[jj].values;
								}
							}

							if (details && crsbkr_array_get(groupParams, id, false))
							{
								var uul = crsbkr_en('ul');
								uul.className = 'param_' + id;

								for (var jj in details)
								{
									var value = crsbkr_array_get(groupParams, id + '.' + details[jj], '');
									if (value != '')
									{
										var lli = crsbkr_en('li');
										uul.appendChild(lli);

										if (crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj], '') != '')
										{
											var span = crsbkr_en('span');
											lli.appendChild(span);
											span.innerHTML = crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj], '') + ':';
										}

										var thisValue = '';
										if (typeof value === 'object')
										{
											for (var kk in value)
											{
												if (!value[kk])
												{
													continue;
												}

												var stringValue = crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj] + '.' + kk, '');
												thisValue += (thisValue == '' ? '' : ', ') + (stringValue === '' ? kk + (crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj] + '.unit', '') != '' ? crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj] + '.unit', '') : '') : stringValue);
											}
										}
										else
										{
											var stringValue = crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj] + '.' + value, '');
											thisValue += stringValue === '' ? value + (crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj] + '.unit', '') != '' ? crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id + '.' + details[jj] + '.unit', '') : '') : stringValue;
										}

										var span = crsbkr_en('span');
										lli.appendChild(span);
										span.innerHTML = thisValue;
									}
								}

								if (uul.children.length > 0)
								{
									var innerDiv = crsbkr_en('div');
									var hh = crsbkr_en('h3');
									hh.innerHTML = crsbkr_array_get(CRSBKR_STRINGS, 'results.params.' + id, '');
									innerDiv.appendChild(hh);
									innerDiv.appendChild(uul);
									div.appendChild(innerDiv);
								}
							}
						}

						if (div.children.length > 0)
						{
							detailsBox.appendChild(div);
							box.appendChild(detailsBox);

							if (crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) == 0)
							{
								var topScroller = crsbkr_en('span');
								topScroller.className = 'crsbkr_result_details_tscroller';
								detailsBox.appendChild(topScroller);
								topScroller.onclick = ((div, topScroller, bottomScroller) =>
								{
									return () =>
									{
										div.scrollTop = Math.max(0, div.scrollTop - 50);
									}
								})(div, topScroller, bottomScroller);


								var bottomScroller = crsbkr_en('span');
								bottomScroller.className = 'crsbkr_result_details_bscroller';
								detailsBox.appendChild(bottomScroller);
								bottomScroller.onclick = ((div, topScroller, bottomScroller) =>
								{
									return () =>
									{
										div.scrollTop = Math.min(div.scrollHeight - div.offsetHeight, div.scrollTop + 50);
									}
								})(div, topScroller, bottomScroller);

								div.onscroll = ((div, topScroller, bottomScroller) =>
								{
									return () =>
									{
										topScroller.style.display = div.scrollTop > 0 ? 'block' : 'none';
										bottomScroller.style.display = div.scrollTop < div.scrollHeight - div.offsetHeight ? 'block' : 'none';
									}
								})(div, topScroller, bottomScroller);

								topScroller.style.display = div.scrollTop > 0 ? 'block' : 'none';
								bottomScroller.style.display = div.scrollTop < div.scrollHeight - div.offsetHeight ? 'block' : 'none';
							}

							var button = crsbkr_en('button');
							li.appendChild(button);
							button.className = 'details_toggle';
							button.innerHTML = CRSBKR_STRINGS['results.details'];
							button.onclick = ((button, detailsBox) =>
							{
								return () =>
								{
									if (!detailsBox.classList.contains('shown'))
									{
										detailsBox.classList.add('shown');
										button.innerHTML = CRSBKR_STRINGS['results.details.back'];
									}
									else
									{
										detailsBox.classList.remove('shown');
										button.innerHTML = CRSBKR_STRINGS['results.details'];
									}
								}
							})(button, detailsBox);
						}
						else if (crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) != 0)
						{
							box.appendChild(detailsBox);
						}
					}

					if (crsbkr_array_get(data, 'controller_form.params.full_width_results', 0) != 0)
					{
						var div = crsbkr_en('div');
						detailsBox.appendChild(div);
						div.className = 'options';

						var noDays = Math.ceil((new Date(input.checkin_time.replace(' ', 'T'))
								- new Date(input.checkout_time.replace(' ', 'T'))) / (24.0 * 60 * 60 * 1000));

						var uul = crsbkr_en('ul');
						div.appendChild(uul);

						var li = crsbkr_en('li');
						uul.appendChild(li);
						li.className = 'days';
						li.innerHTML = noDays == 1 ? CRSBKR_STRINGS['results.options.day'] : CRSBKR_STRINGS['results.options.days'].replace('{days}', noDays);

						var options = [quote];
						if (quote.options)
						{
							for (var i in quote.options)
							{
								if (quote.options.hasOwnProperty(i))
								{
									options.push(quote.options[i]);
								}
							}
						}

						for (var i in options)
						{
							var quote = options[i];

							var li = crsbkr_en('li');
							uul.appendChild(li);

							var a = crsbkr_en('a');
							li.appendChild(a);
							a.href = '#crsbkr_summary';
							a.onclick = ((quote) =>
							{
								return () =>
								{
									crsbkr_search_results({ data : data, input : input, results : results, selected_quote : quote });
								}
							})(options[i]);

							var thisRate = parseFloat(crsbkr_array_get(quote, 'rate', 0)) - parseFloat(crsbkr_array_get(quote, 'discount', 0));
							rate = rate == 0 ? thisRate : Math.min(rate, thisRate);

							var category = crsbkr_array_get(quote, 'tariff_category', {});
							category = crsbkr_array_get(category, 'settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(category, 'settings.translations.en.name', crsbkr_array_get(category, 'name', '')));
							if (category != '')
							{
								var span = crsbkr_en('span');
								a.appendChild(span);
								span.className = 'tariff';
								span.innerHTML = category;
							}

							var span = crsbkr_en('span');
							a.appendChild(span);
							span.className = 'rate';

							var rate = parseFloat(crsbkr_array_get(quote, 'rate', 0)) - parseFloat(crsbkr_array_get(quote, 'discount', 0));
							if (options[i].discount && options[i].discount != 0)
							{
								span.classList.add('discount');

								var span2 = crsbkr_en('span');
								span.appendChild(span2);
								span2.innerHTML = crsbkr_currency(crsbkr_array_get(quote, 'rate', 0));
							}

							span.appendChild(document.createTextNode(crsbkr_currency(rate)));
						}

						// Any required optionals?
						var extras = '';
						for (var i in quote.accessories)
						{
							if (quote.accessories[i].required && quote.accessories[i].rate > 0)
							{
								extras += (extras == '' ? '' : '<br />') + CRSBKR_STRINGS['results.extra']
										.replace('{name}', crsbkr_array_get(quote.accessories[i], 'accessory.settings.translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(quote.accessories[i], 'accessory.settings.translations.en.name', crsbkr_array_get(quote.accessories[i], 'accessory.name', ''))))
										.replace('{price}', crsbkr_currency(quote.accessories[i].rate));
							}
						}

						var hint = CRSBKR_STRINGS['results.options.hint'];
						if (extras != '')
						{
							hint += (hint == '' ? '' : '<br /><br />') + extras;
						}

						var li = crsbkr_en('li');
						uul.appendChild(li);
						li.innerHTML = hint;
					}

					if (groups)
					{
						var acrissCodeId = crsbkr_array_get(quote, 'acriss_code.uid');
						for (var i = 0; i < groups.length; i++)
						{
							if (groups[i].acriss_code_ids.indexOf('' + acrissCodeId) == -1)
							{
								var found = false;

								var groupIds = crsbkr_array_get(groups[i], 'groups', '').split(',').filter(n => ('' + n).length > 0);
								for (var j = 0; j < crsbkr_array_get(data, 'acriss_code_web', []).length; j++)
								{
									if (groupIds.indexOf('' + crsbkr_array_get(data.acriss_code_web[j], 'uid', '')) != -1 && crsbkr_array_get(data.acriss_code_web[j], 'acriss_code_ids', '').split(',').filter(n => ('' + n).length > 0).indexOf('' + acrissCodeId) != -1)
									{
										found = true;

										groups[i].acriss_code_ids.push(acrissCodeId);
										break;
									}
								}
							}

							if (!found)
							{
								continue;
							}

							groups[i].min = groups[i].min === null ? rate : Math.min(groups[i].min, rate);
						}
					}
				}

				///
				//
				// PRINT GROUPS
				//
				///
				if (groups)
				{
					var groupsUl = crsbkr_en('ul');
					groupsDiv.appendChild(groupsUl);

					var selectedGroups = [];

					for (var i in groups)
					{
						if (groups[i].acriss_code_ids.length == 0)
						{
							continue;
						}

						var group = groups[i];
						var li = crsbkr_en('li');
						li.style.backgroundImage = 'url(\'' + group.icon_url + '\')';

						var span = crsbkr_en('span');
						span.className = 'name';
						span.innerHTML = crsbkr_array_get(group, 'translations.' + CRSBKR_LANG + '.name', crsbkr_array_get(group, 'translations.en.name', crsbkr_array_get(group, 'name', '')));
						li.appendChild(span);

						var span = document.createElement('span');
						span.className = 'rate';
						span.innerHTML = CRSBKR_STRINGS['results.group.rate'].replace('{rate}', crsbkr_currency(group.min));
						li.appendChild(span);

						groupsUl.appendChild(li);

						li.onclick = ((li, group) =>
						{
							return () =>
							{
								if (selectedGroups.indexOf('' + group.uid) != -1)
								{
									selectedGroups.splice(selectedGroups.indexOf('' + group.uid), 1);
									li.classList.remove('selected');
								}
								else
								{
									selectedGroups.push('' + group.uid);
									li.classList.add('selected');
								}

								for (var ii = 0; ii < results.length; ii++)
								{
									var quote = results[ii];

									var found = selectedGroups.length == 0;
									for (var iii = 0; !found && iii < groups.length; iii++)
									{
										if (selectedGroups.indexOf('' + groups[iii].uid) != -1 && groups[iii].acriss_code_ids.indexOf('' + quote.acriss_code.uid) != -1)
										{
											found = true;
										}
									}

									ul.children[ii].style.display = found ? 'block' : 'none';
								}
							}
						})(li, group);
					}

					var leftScroller = crsbkr_en('span');
					leftScroller.id = 'crsbkr_search_results_groups_lscroller';
					groupsDiv.appendChild(leftScroller);
					leftScroller.onclick = () =>
					{
						for (var i = groupsUl.children.length - 1; i >= 0; i--)
						{
							if (groupsUl.children[i].offsetLeft - 5 < groupsUl.scrollLeft + 5)
							{
								groupsUl.scrollLeft = Math.min(groupsUl.scrollWidth - groupsUl.offsetWidth, groupsUl.children[i].offsetLeft - 10);
								break;
							}
						}
					};

					var rightScroller = crsbkr_en('span');
					rightScroller.id = 'crsbkr_search_results_groups_rscroller';
					groupsDiv.appendChild(rightScroller);
					rightScroller.onclick = () =>
					{
						for (var i = 0; i < groupsUl.children.length; i++)
						{
							if (groupsUl.children[i].offsetLeft - 5 > groupsUl.scrollLeft + 5)
							{
								groupsUl.scrollLeft = Math.min(groupsUl.scrollWidth - groupsUl.offsetWidth, groupsUl.children[i].offsetLeft - 10);
								break;
							}
						}
					};

					groupsUl.onscroll = () =>
					{
						leftScroller.style.display = groupsUl.scrollLeft > 0 ? 'block' : 'none';
						rightScroller.style.display = groupsUl.scrollLeft < groupsUl.scrollWidth - groupsUl.offsetWidth ? 'block' : 'none';
					}

					leftScroller.style.display = groupsUl.scrollLeft > 0 ? 'block' : 'none';
					rightScroller.style.display = groupsUl.scrollLeft < groupsUl.scrollWidth - groupsUl.offsetWidth ? 'block' : 'none';
				}
			}
		}

		loading.style.display = 'none';
	}

	if (requests === null)
	{
		callback();
	}
	else
	{
		setTimeout(callback, 500);
	}
}

async function crsbkr_sleep(ms)
{
	await new Promise(r => setTimeout(r, ms));
}

function crsbkr_upload_file(element, field, url, callback)
{
	var type = null;
	var value = null;

	// Get the file url.
	var dummy = crsbkr_en('img');
	dummy.src = CRSBKR_DIR + 'images/no_image.png';

	var alertObj = null;

	var form = crsbkr_en('div');
	form.className = 'file';

	var p = crsbkr_en('p');
	form.appendChild(p);
	p.innerHTML = CRSBKR_STRINGS['alert.file.intro'];

	var header = crsbkr_en('ul');
	form.appendChild(header);
	header.className = 'header';

	var camera = null;
	var existing = null;
	var upload = null;
	var urlins = null;

	var headerExisting = null;
	var headerUpload = null;
	var headerCamera = null;
	var headerUrlIns = null;

	if (value)
	{
		headerExisting = crsbkr_en('li');
		header.appendChild(headerExisting);
		headerExisting.innerHTML = CRSBKR_STRINGS['alert.file.existing'];

		existing = crsbkr_en('div');
		form.appendChild(existing);
		existing.className = 'content existing';
	}

	if ('mediaDevices' in navigator && 'getUserMedia' in navigator.mediaDevices)
	{
		headerCamera = crsbkr_en('li');
		header.appendChild(headerCamera);
		headerCamera.innerHTML = CRSBKR_STRINGS['alert.file.camera'];

		camera = crsbkr_en('div');
		form.appendChild(camera);
		camera.className = 'content camera';

		camera.none = crsbkr_en('p');
		camera.appendChild(camera.none);
		camera.none.innerHTML = CRSBKR_STRINGS['alert.file.camera.none'];

		camera.crop = crsbkr_en('div');
		camera.appendChild(camera.crop);
		camera.crop.className = 'crop';

		var inner = crsbkr_en('div');
		camera.crop.appendChild(inner);

		camera.shutter = crsbkr_en('span');
		camera.appendChild(camera.shutter);
		camera.shutter.className = 'shutter';

		var innerSpan = crsbkr_en('span');
		camera.shutter.appendChild(innerSpan);

		camera.confirm = crsbkr_en('span');
		camera.appendChild(camera.confirm);
		camera.confirm.className = 'shutter';

		var innerSpan = crsbkr_en('span');
		camera.confirm.appendChild(innerSpan);
		innerSpan.innerHTML = '✓';

		camera.cancel = crsbkr_en('span');
		camera.appendChild(camera.cancel);
		camera.cancel.className = 'cancel';

		var innerSpan = crsbkr_en('span');
		camera.cancel.appendChild(innerSpan);
		innerSpan.innerHTML = '×';
	}

	headerUpload = crsbkr_en('li');
	header.appendChild(headerUpload);
	headerUpload.innerHTML = CRSBKR_STRINGS['alert.file.upload'];

	upload = crsbkr_en('div');
	form.appendChild(upload);
	upload.className = 'content upload';

	if (field.type == 'file' || field.type == 'image_url')
	{
		headerUrlIns = crsbkr_en('li');
		header.appendChild(headerUrlIns);
		headerUrlIns.innerHTML = CRSBKR_STRINGS['alert.file.url'];

		urlins = crsbkr_en('div');
		form.appendChild(urlins);
		urlins.className = 'content urlins';
	}

	var selected = null;
	var canGoBackToSelected = false;

	headerUpload.onclick = async function (e)
	{
		if (e)
		{
			canGoBackToSelected = false;
			selected = this;
		}
		else
		{
			canGoBackToSelected = true;
		}

		if (headerCamera)
		{
			if (this == headerCamera)
			{
				headerCamera.classList.add('selected');

				var constraints =
				{
					video : { width : 1280 }
				};

				try
				{
					await navigator.mediaDevices.getUserMedia(constraints)

					var devices = await navigator.mediaDevices.enumerateDevices();
					var videoDevices = devices.filter(device => device.kind === 'videoinput');

					var cameras = [];
					for (var i = 0; i < videoDevices.length; i++)
					{
						cameras.push({ value : videoDevices[i].deviceId, label : videoDevices[i].label });
					}

					var device = cameras.length > 0 ? cameras[0] : '';

					if (device)
					{
						camera.none.style.display = 'none';
						if (cameras.length > 1 && !camera.cameraSelect)
						{
							camera.cameraSelect = crsbkr_en('div');
							camera.appendChild(camera.cameraSelect);
							camera.cameraSelect.className = 'camera-select';
							camera.cameraSelect.onclick = () =>
							{
								for (var i = 0; i < cameras.length; i++)
								{
									if (cameras[i].value == device.value)
									{
										device = i == cameras.length - 1 ? cameras[0] : cameras[i + 1];
										break;
									}
								}

								camera.startStream(device);
							};

							var img = crsbkr_en('img');
							camera.cameraSelect.appendChild(img);
							img.src = CRSBKR_DIR + 'images/icon_rotate.png';
						}
						else if (cameras.length > 1)
						{
							camera.cameraSelect.style.display = 'block';
						}

						if (!camera.flip)
						{
							camera.flip = crsbkr_en('div');
							camera.appendChild(camera.flip);
							camera.flip.className = 'flip';
							camera.flip.onclick = () =>
							{
								if (camera.flip.classList.contains('selected'))
								{
									camera.flip.classList.remove('selected');
								}
								else
								{
									camera.flip.classList.add('selected');
								}

								camera.canvas.getContext('2d').scale(-1, 1);
							}

							var img = crsbkr_en('img');
							camera.flip.appendChild(img);
							img.src = CRSBKR_DIR + 'images/icon_flip.png';
						}

						camera.stop = () =>
						{
							if (camera.timer)
							{
								window.clearInterval(camera.timer);
								camera.timer = null;
							}

							if (camera.stream)
							{
								camera.stream.getTracks().forEach((track) => { track.stop(); camera.stream.removeTrack(track); });
								camera.stream = null;
							}

							if (camera.video)
							{
								camera.video.pause();

								camera.video.srcObject = null;
								camera.video.removeAttribute('src');
								camera.video.load();
								camera.video = null;
							}

							if (camera.onkeydown)
							{
								document.body.removeEventListener('keydown', camera.onkeydown);
								camera.onkeydown = null;
							}
						};

						camera.startStream = async (thisDevice) =>
						{
							camera.stop();

							camera.onkeydown = (e) =>
							{
								if (e.keyCode == 13)
								{
									camera.shutter.onclick();
								}
							}

							document.body.addEventListener('keydown', camera.onkeydown);

							camera.cancel.style.display = 'none';
							camera.confirm.style.display = 'none';
							camera.crop.style.display = 'none';
							camera.flip.style.display = 'block';
							camera.shutter.style.display = 'block';
							alertObj.submitButton.style.display = 'none';

							var updatedConstraints = JSON.parse(JSON.stringify(constraints));
							updatedConstraints.video.deviceId = { exact : thisDevice.value };

							camera.stream = await navigator.mediaDevices.getUserMedia(updatedConstraints);
							camera.streaming = true;

							camera.video = crsbkr_en('video');
							camera.video.autoplay = 'autoplay';
							camera.video.srcObject = camera.stream;
							camera.video.onloadedmetadata = () =>
							{
								var flip = false;
								if (!camera.canvas)
								{
									camera.canvas = crsbkr_en('canvas');
									camera.appendChild(camera.canvas);
								}
								else if (camera.flip.classList.contains('selected'))
								{
									flip = true;

									camera.canvas.getContext('2d').scale(-1, 1);
									camera.flip.classList.remove('selected');
								}

								camera.canvas.width = camera.video.videoWidth;
								camera.canvas.height = camera.video.videoHeight;

								if (flip)
								{
									camera.flip.onclick();
								}

								camera.timer = window.setInterval(() =>
								{
									if (!camera.streaming)
									{
										return;
									}

									camera.canvas.getContext('2d').drawImage(camera.video, 0, 0, (camera.streaming && camera.flip.classList.contains('selected') ? -1 : 1) * camera.canvas.width, camera.canvas.height);
								}, 30);

								camera.shutter.onclick = () =>
								{
									camera.streaming = false;

									if (camera.flip.classList.contains('selected'))
									{
										camera.canvas.getContext('2d').scale(-1, 1);
									}

									camera.canvas.getContext('2d').drawImage(camera.video, 0, 0, camera.canvas.width, camera.canvas.height);

									camera.stop();

									camera.flip.style.display = 'none';
									camera.shutter.style.display = 'none';
									camera.confirm.style.display = 'block';
									camera.cancel.style.display = 'block';
									camera.cancel.onclick = () =>
									{
										camera.startStream(device);
									}

									if (camera.cameraSelect)
									{
										camera.cameraSelect.style.display = 'none';
									}

									while (camera.crop.firstChild.lastChild)
									{
										camera.crop.firstChild.removeChild(camera.crop.firstChild.lastChild);
									}

									var corners = [];
									var dragging = null;
									var screenX = 0;
									var screenY = 0;

									for (var i = 0; i < 4; i++)
									{
										var arrow = crsbkr_en('div');
										inner.appendChild(arrow);
										corners.push(arrow);

										arrow.onpointerdown = ((i) =>
										{
											return function(e)
											{
												dragging = i;

												screenX = e.screenX;
												screenY = e.screenY;

												e.preventDefault();
												e.stopPropagation();
											}
										})(i);
									}

									// Fetch the 4 border width values
									var styling = window.getComputedStyle(camera.crop, null);
									var topBorder = parseInt(styling.getPropertyValue('border-top-width').replace('px', ''));
									var rightBorder = parseInt(styling.getPropertyValue('border-right-width').replace('px', ''));
									var bottomBorder = parseInt(styling.getPropertyValue('border-bottom-width').replace('px', ''));
									var leftBorder = parseInt(styling.getPropertyValue('border-left-width').replace('px', ''));

									camera.crop.onpointerdown = (e) =>
									{
										dragging = 5;

										screenX = e.screenX;
										screenY = e.screenY;

										e.preventDefault();
										e.stopPropagation();
									}

									camera.crop.onpointermove = (e) =>
									{
										if (dragging !== null)
										{
											var dx = e.screenX - screenX;
											var dy = e.screenY - screenY;

											screenX = e.screenX;
											screenY = e.screenY;

											var rect = camera.crop.getBoundingClientRect();

											var styling = getComputedStyle(camera.crop, null);
											var top = parseInt(styling.getPropertyValue('top').replace('px', ''));
											var right = parseInt(styling.getPropertyValue('right').replace('px', ''));
											var bottom = parseInt(styling.getPropertyValue('bottom').replace('px', ''));
											var left = parseInt(styling.getPropertyValue('left').replace('px', ''));

											if (dragging == 5)
											{
												dx = -leftBorder > left + dx ? -leftBorder - left : (-rightBorder > right - dx ? right + rightBorder : dx);
												dy = -topBorder > top + dy ? -topBorder - top : (-bottomBorder > bottom - dy ? bottom + bottomBorder : dy);
											}

											if (dragging == 0 || dragging == 2 || dragging == 5)
											{
												camera.crop.style.left = Math.max(-leftBorder, left + dx) + 'px';
											}

											if (dragging == 1 || dragging == 3 || dragging == 5)
											{
												camera.crop.style.right = Math.max(-rightBorder, right - dx) + 'px';
											}

											if (dragging == 0 || dragging == 1 || dragging == 5)
											{
												camera.crop.style.top = Math.max(-topBorder, top + dy) + 'px';
											}

											if (dragging == 2 || dragging == 3 || dragging == 5)
											{
												camera.crop.style.bottom = Math.max(-bottomBorder, bottom - dy) + 'px';
											}
										}
									}

									alertObj.parentNode.onpointerup = () =>
									{
										dragging = null;
									}

									camera.crop.style.display = 'block';
								}

								camera.confirm.onclick = () =>
								{
									// Fetch the 4 border width values
									var styling = window.getComputedStyle(camera.crop, null);
									var topBorder = parseInt(styling.getPropertyValue('border-top-width').replace('px', ''));
									var rightBorder = parseInt(styling.getPropertyValue('border-right-width').replace('px', ''));
									var bottomBorder = parseInt(styling.getPropertyValue('border-bottom-width').replace('px', ''));
									var leftBorder = parseInt(styling.getPropertyValue('border-left-width').replace('px', ''));

									var rect = camera.crop.getBoundingClientRect();

									var canvasRect = camera.canvas.getBoundingClientRect();
									var scaleX = camera.canvas.width / camera.canvas.offsetWidth;
									var scaleY = camera.canvas.height / camera.canvas.offsetHeight;

									var thisCanvas = crsbkr_en('canvas');
									thisCanvas.width = (rect.width - leftBorder - rightBorder) * scaleX;
									thisCanvas.height = (rect.height - topBorder - bottomBorder) * scaleY;

									thisCanvas.getContext('2d').drawImage(camera.canvas, (rect.left + leftBorder - canvasRect.left) * scaleX, (rect.top + topBorder - canvasRect.top) * scaleY, thisCanvas.width, thisCanvas.height, 0, 0, thisCanvas.width, thisCanvas.height);

									headerUpload.onclick(null);

									camera.crop.style.display = 'none';
									camera.shutter.style.display = 'block';
									camera.confirm.style.display = 'none';

									var data = thisCanvas.toDataURL('image/png');
									alertObj.parentNode.ondrop({ dataTransfer : { files : [ { size : 0, name : 'image.png', data : data } ] } });
								}
							};
						};

						camera.startStream(device);
					}
					else
					{
						alertObj.submitButton.style.display = 'inline-block';
						for (var i = 0; i < camera.children.length; i++)
						{
							camera.children[i].style.display = camera.children[i] == camera.none ? 'block' : 'none';
						}
					}
				}
				catch (e)
				{
					alertObj.submitButton.style.display = 'inline-block';
					for (var i = 0; i < camera.children.length; i++)
					{
						camera.children[i].style.display = camera.children[i] == camera.none ? 'block' : 'none';
					}
				}
			}
			else
			{
				if (camera && camera.stop)
				{
					camera.stop();
				}

				headerCamera.classList.remove('selected');

				if (alertObj)
				{
					alertObj.submitButton.style.display = 'inline-block';
				}
			}

			camera.style.display = this == headerCamera ? 'block' : 'none';
		}

		if (headerExisting)
		{
			existing.style.display = this == headerExisting ? 'block' : 'none';
			if (this == headerExisting)
			{
				headerExisting.classList.add('selected');
			}
			else
			{
				headerExisting.classList.remove('selected');
			}
		}

		if (headerUpload)
		{
			upload.style.display = this == headerUpload ? 'block' : 'none';
			if (this == headerUpload)
			{
				headerUpload.classList.add('selected');
			}
			else
			{
				headerUpload.classList.remove('selected');
			}
		}

		if (headerUrlIns)
		{
			urlins.style.display = this == headerUrlIns ? 'block' : 'none';
			if (this == headerUrlIns)
			{
				headerUrlIns.classList.add('selected');
			}
			else
			{
				headerUrlIns.classList.remove('selected');
			}
		}
	}

	if (headerExisting)
	{
		headerExisting.onclick = headerUpload.onclick;
	}

	if (headerCamera)
	{
		headerCamera.onclick = headerUpload.onclick;
	}

	if (headerUrlIns)
	{
		headerUrlIns.onclick = headerUpload.onclick;
	}

	if (headerCamera && /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent))
	{
		headerCamera.onclick();
	}
	else
	{
		headerUpload.onclick();
	}

	var div = crsbkr_en('div');
	upload.appendChild(div);

	var span = crsbkr_en('span');
	div.appendChild(span);
	span.innerHTML = CRSBKR_STRINGS['alert.file.drag'];

	var span = crsbkr_en('span');
	div.appendChild(span);
	span.innerHTML = CRSBKR_STRINGS['alert.file.drop'];

	var selectButton = crsbkr_en('input');
	selectButton.id = 'file';
	selectButton.type = 'file';
	div.appendChild(selectButton);

	var selectButtonLabel = crsbkr_en('label');
	selectButtonLabel.htmlFor = 'file';
	div.appendChild(selectButtonLabel);

	var divInfo = crsbkr_en('div');
	upload.appendChild(divInfo);
	divInfo.className = 'info';

	var progressBarHolder = crsbkr_en('div');
	divInfo.appendChild(progressBarHolder);
	progressBarHolder.className = 'progressbar';

	var progressBar = crsbkr_en('div');
	progressBarHolder.appendChild(progressBar);
	progressBar.style.width = '0%';

	var uploadImg = crsbkr_en('img');
	divInfo.appendChild(uploadImg);
	uploadImg.src = CRSBKR_DIR + 'images/file.png';

	var uploadLine = crsbkr_en('span');
	divInfo.appendChild(uploadLine);

	var deleteButton = null;
	if (existing)
	{
// 		if (field.type == 'image_url')
// 		{
// 			var infoImg = crsbkr_en('img');
// 			existing.appendChild(infoImg);
// 			infoImg.src = value && value.startsWith('data:') ? value : CRSBKR_STRINGS['url.api') + 'file/' + btoa(value) + '?output=inline';
// 			infoImg.className = 'preview';
// 		}
// 		else
// 		{
// 			var infoImg = crsbkr_en('img');
// 			existing.appendChild(infoImg);
// 			infoImg.src = value && value.startsWith('data:') ? value : CRSBKR_STRINGS['url.api') + 'file/' + btoa(value) + '?output=inline';
//
// 			var infoLine = crsbkr_en('span');
// 			existing.appendChild(infoLine);
//
// 			infoLine.innerHTML = '<b>' + L('label.fileMime') + ':</b> ' + (metadata ? metadata.type : '-') + '<br />'
// 					+ '<b>' + L('label.fileSize') + ':</b> ' + (metadata ? metadata.size : '-') + '<br />';
// 		}
//
// 		var a = crsbkr_en('a');
// 		a.href = value && value.startsWith('data:') ? value : L('url.api') + 'file/' + btoa(value) + '?output=attachment' + (field.filename ? '&filename=' + encodeURI(field.filename) : '');
// 		a.target = '_blank';
//
// 		var button = crsbkr_en('button');
// 		button.className = 'add upload';
// 		button.appendChild(document.createTextNode(L('label.show')));
// 		a.appendChild(button);
// 		existing.appendChild(a);
//
// 		deleteButton = crsbkr_en('button');
// 		deleteButton.className = 'add delete';
// 		deleteButton.innerHTML = L('alert.file.delete');
// 		existing.appendChild(deleteButton);
//
// 		headerExisting.click();
	}

	var urlInput = null;
	if (urlins)
	{
		var labelDesc = crsbkr_en('span');
		labelDesc.innerHTML = CRSBKR_STRINGS['alert.file.url.intro'];

		urlInput = crsbkr_en('input');
		urlInput.style.placeholder = CRSBKR_STRINGS['label.url'];
		urlInput.value = !value || value.startsWith('attachment.php') ? '' : value;
		urlInput.type = 'text';

		urlins.appendChild(labelDesc);
		urlins.appendChild(urlInput);
	}

	var canceled = false;
	var cancelRequest = (callback) =>
	{
		if (camera && camera.stop)
		{
			camera.stop();
		}

		if (url)
		{
			crsbkr_request('POST', url,
			{
				url : value,
				cancel : true
			}, (success, response) =>
			{
				if (success && response)
				{
					value = null;
					if (callback)
					{
						callback();
					}
				}
				else
				{
					crsbkr_error(response);
				}
			});
		}
		else if (callback)
		{
			callback();
		}
	};

	var droppedFile = false;
	var dropping = false;

	var filename = null;

	alertObj = crsbkr_alert('', CRSBKR_BUTTONS_SUBMIT_CANCEL, form, () =>
	{
		if (camera && camera.stop)
		{
			camera.stop();
		}

		var leaveOpen = true;
		if (dropping)
		{
			crsbkr_error(CRSBKR_STRINGS['alert.file.dropping']);
		}
		else if (droppedFile && value)
		{
			if (url)
			{
				// Send confirmation to the server.
				crsbkr_request('POST', url,
				{
					url : value,
					confirm : true,
					filename : filename
				}, (success, response) =>
				{
					if (success && response)
					{
						if (element)
						{
							element.innerHTML = this.formatField(field.type, response.url, null, element, field);
						}

						if (callback)
						{
							callback(response);
						}

						crsbkr_alert_dismiss(alertObj);
					}
					else
					{
						crsbkr_error(response);
					}
				});
			}
			else if (callback)
			{
				leaveOpen = false;
				callback({ url : null, data : value, type : type });
			}
		}
		else if (headerUrlIns && headerUrlIns.classList.contains('selected')
				&& urlInput && urlInput.value !== '' && urlInput.value !== value)
		{
			if (url)
			{
				// If we have a URL given send it to the server.
				crsbkr_request('POST', url,
				{
					url : urlInput.value
				}, (success, response) =>
				{
					if (success && response)
					{
						if (element)
						{
							element.innerHTML = this.formatField(field.type, response.url, null, element, field);
						}

						if (callback)
						{
							callback(response.url);
						}

						crsbkr_alert_dismiss(alertObj);
					}
					else
					{
						crsbkr_error(response);
					}
				});
			}
			else if (callback)
			{
				leaveOpen = false;
				callback({ url : value });
			}
		}
		else if (!url)
		{
			callback({});
			leaveOpen = false;
		}
		else
		{
			leaveOpen = false;
		}

		return leaveOpen;
	},
	() =>
	{
		if (camera && camera.stop)
		{
			camera.stop();
		}

		var leaveOpen = false;
		if (droppedFile && value)
		{
			leaveOpen = true;
			cancelRequest(() => { crsbkr_alert_dismiss(alertObj); });
		}

		return leaveOpen;
	});

	alertObj.classList.add('no-padding')
	alertObj.dragCounter = 0;
	alertObj.parentNode.ondragenter = (e) =>
	{
		if (selected != headerUpload)
		{
			headerUpload.onclick(null);
		}

		alertObj.classList.add('dropping');
		alertObj.dragCounter++;
	};

	alertObj.parentNode.ondragleave = (e) =>
	{
		alertObj.dragCounter--;
		if (alertObj.dragCounter == 0)
		{
			alertObj.classList.remove('dropping');
			if (canGoBackToSelected)
			{
				selected.click();
			}
		}
	};

	alertObj.parentNode.ondragover = (e) =>
	{
		e.stopPropagation();
		e.preventDefault();
		e.dataTransfer.dropEffect = 'copy';
	}

	alertObj.parentNode.ondrop = (e) =>
	{
		alertObj.classList.remove('dropping');
		divInfo.style.display = 'block';

		var f = null;
		if (Object.prototype.toString.call(e) == '[object FileList]')
		{
			f = e[0];
		}
		else
		{
			!e.stopPropagation || e.stopPropagation();
			!e.preventDefault || e.preventDefault();

			f = e.dataTransfer.files[0];
		}

		if (!f)
		{
			return;
		}

		var readFile = (f) =>
		{
			dropping = true;

			// Get the size in a readable value.
			var sizeUnits = ['B', 'KB', 'MB', 'GB', 'TB'];
			var unitIndex = 0;

			var size = f.size;
			while (size / 1024.0 > 1)
			{
				size = size / 1024.0;
				unitIndex++;
			}

			size = Math.round(size);

			filename = f.name;
			uploadLine.innerHTML = '<b>' + CRSBKR_STRINGS['label.fileName'] + ':</b> ' + f.name + (size > 0 ? '<br /><b>' + CRSBKR_STRINGS['label.fileSize'] + ':</b> ' + size + sizeUnits[unitIndex] : '');

			var dataCallback = url ? (data) =>
			{
				// Callback that is used to send the file in chunks.
				uploadImg.src = data;

				data = data.substring(5);

				var base64 = 0;
				var base64Pos = data.indexOf('base64,');
				if (base64Pos != -1)
				{
					base64 = 1;
					data = data.substring(base64Pos + 7);
				}

				if (size == 0)
				{
					size = base64 ? atob(data).length : data.length;
					while (size / 1024.0 > 1)
					{
						size = size / 1024.0;
						unitIndex++;
					}

					size = Math.round(size);
				}

				var offset = 0;

				var fn = (success, response) =>
				{
					// Have we canceled the request?
					if (canceled)
					{
						cancelRequest(() =>
						{
							canceled = false;
							droppedFile = false;
						});
					}
					else if (success && response)
					{
						var progress = Math.min(100, Math.floor(offset * 100.0 / data.length));
						progressBar.style.width = '' + progress + '%';
						uploadLine.innerHTML = '<b>' + CRSBKR_STRINGS['label.fileName'] + ':</b> ' + f.name + '<br /><b>' + CRSBKR_STRINGS['label.fileSize'] + ':</b> ' + size + sizeUnits[unitIndex] + '<br />'
								+ '<b>' + CRSBKR_STRINGS['label.uploading'] + ':</b> ' + progress + '%';

						// Have we finished?
						if (offset == data.length)
						{
							dropping = false;
							value = response.url;
							if (urlInput)
							{
								urlInput.value = value;
							}
						}
						else
						{
							// The server offset might differ due to base64 decoding.
							var serverOffset = response.offset ? response.offset : 0;

							// Send the next chunk to the server.
							var chunkSize = 150 * 1024;
							if (offset + chunkSize > data.length)
							{
								chunkSize = data.length - offset;
							}

							var thisOffset = offset;
							offset += chunkSize;

							crsbkr_request('POST', url,
							{
								url : response.url ? response.url : null,
								filename : f.name,
								data : data.substring(thisOffset, thisOffset + chunkSize),
								offset : serverOffset,
								base64 : base64
							}, fn);
						}
					}
					else
					{
						crsbkr_alert(response.error);
					}
				};

				// Send the meta data to the server.
				fn(true, {});
			} : (data) =>
			{
				dropping = false;

				uploadImg.src = data;

				var tmpData = data.substring(5);

				var base64 = 0;
				var base64Pos = tmpData.indexOf('base64,');
				if (base64Pos != -1)
				{
					base64 = 1;
					tmpData = tmpData.substring(base64Pos + 7);
				}

				if (size == 0)
				{
					size = base64 ? atob(tmpData).length : data.length;
					while (size / 1024.0 > 1)
					{
						size = size / 1024.0;
						unitIndex++;
					}

					size = Math.round(size);
					uploadLine.innerHTML = '<b>' + CRSBKR_STRINGS['label.fileName'] + ':</b> ' + f.name + (size > 0 ? '<br /><b>' + CRSBKR_STRINGS['label.fileSize'] + ':</b> ' + size + sizeUnits[unitIndex] : '');
				}

				type = file.type;
				value = data;
			};

			if (f.data)
			{
				droppedFile = true;
				dataCallback(f.data);
			}
			else
			{
				// Read the file content.
				var reader = new FileReader();
				reader.onprogress = (e) =>
				{
					var progress = Math.min(100, Math.floor(e.loaded * 100.0 / e.total));
					progressBar.style.width = '' + progress + '%';
				};

				reader.onload = (e) =>
				{
					dataCallback(e.target.result);
				}

				// Dropped a file...
				droppedFile = true;

				// Read in the div file as a data URL.
				reader.readAsDataURL(f);
			}
		};

		// If we already have an image set, cancel this one.
		if (droppedFile && value)
		{
			cancelRequest(((f) =>
			{
				return function()
				{
					readFile(f);
				}
			})(f));
		}
		else
		{
			readFile(f);
		}
	};

	selectButton.addEventListener('change', (e) =>
	{
		alertObj.parentNode.ondrop(selectButton.files);
	});

	if (deleteButton)
	{
		deleteButton.onclick = (e) =>
		{
			crsbkr_alert(CRSBKR_STRINGS['alert.file.deleteConfirm'], CRSBKR_BUTTONS_SUBMIT_CANCEL, null, () =>
			{
				if (url)
				{
					// Send the request to the server.
					crsbkr_request('POST', url, { remove : true }, (success, response) =>
					{
						if (success && response)
						{
							if (element)
							{
								element.innerHTML = this.formatField(field.type, null, false, element, field);
							}

							if (callback)
							{
								callback(response);
							}

							crsbkr_alert_dismiss(alertObj);
						}
						else
						{
							crsbkr_error(response);
						}
					});
				}
				else
				{
					type = null;
					value = null;

					alertObj.submitButton.onclick();
				}
			});
		};
	}
}

function crsbkr_vat_int(taxSystem)
{
	var res = 0;
	if (taxSystem)
	{
		var pos = taxSystem.lastIndexOf('_');
		if (pos != -1)
		{
			res = parseInt(taxSystem.substring(pos + 1));
		}
	}

	return res;
}

function crsbkr_webcheck(data)
{
	var holder = document.getElementById('crsbkr_webcheck');
	while (holder.lastChild)
	{
		holder.removeChild(holder.lastChild);
	}

	if (!data || data == '')
	{
		var form = crsbkr_webcheck_form();
		holder.appendChild(form);
	}
	else
	{
		var loading = crsbkr_loading();
		holder.appendChild(loading);
		loading.style.display = 'block';

		var p = crsbkr_en('p');
		loading.appendChild(p);
		p.innerHTML = CRSBKR_STRINGS['webcheck.loading'];

		var param = data;

		data = JSON.parse(atob(data));
		crsbkr_request('GET', 'client_reservation/' + param + '?expand=client,client_driver,details', null, (success, response) =>
		{
			loading.parentNode.removeChild(loading);

			if (success && response && response.client_reservation)
			{
				data = {};

				var refId = crsbkr_array_get(response.client_reservation, 'stn_id_reservation', crsbkr_array_get(response.client_reservation, 'ref_id_reservation'));

				var p = crsbkr_en('p');
				holder.appendChild(p);
				p.innerHTML = CRSBKR_STRINGS['webcheck.intro'].replace('{{ref_id}}', refId).replace('{ref_id}', refId);

				var fields =
				{
					'client_driver.0' :
					[
						'.first_name', '.name', '.email', '.phone', '.address', '.city', '.zip_code', '.province', '.country_id', '', '.tax_number', '', '.date_of_birth', '.place_of_birth'
					],

					'client_driver.0.id' :
					[
						'_type', '_number', '_issue_date', '_expiry_date', '_issuer', '', '_file_id', '_file_id_2',
					],

					'client_driver.0.license' :
					[
						'_number', '_issuer', '_issue_date', '_expiry_date', '_file_license', '_file_license_2'
					],

					'client_driver.1' :
					[
						'.first_name', '.name', '.email', '.phone', '.address', '.city', '.zip_code', '.province', '.country_id', '', '.tax_number', '', '.date_of_birth', '.place_of_birth'
					],

					'client_driver.1.id' :
					[
						'_type', '_number', '_issue_date', '_expiry_date', '_issuer', '', '_file_id', '_file_id_2',
					],

					'client_driver.1.license' :
					[
						'_number', '_issuer', '_issue_date', '_expiry_date', '_file_license', '_file_license_2'
					],

// 					'client' :
// 					[
// 						'.first_name', '.name', '.email', '.phone', '.address', '.city', '.zip_code', '.province', '.country_id', '', '.tax_number', '', '.date_of_birth', '.place_of_birth'
// 					]
				};

				var div = null;
				for (var s in fields)
				{
					if (!fields.hasOwnProperty(s))
					{
						continue;
					}

					if (s == 'client_driver.1' && crsbkr_array_get(response, 'client_driver.1.name', '') == '')
					{
						var p = crsbkr_en('p');
						holder.appendChild(p);

						var a = crsbkr_en('a');
						p.appendChild(a);
						a.innerHTML = CRSBKR_STRINGS['webcheck.client_driver.1.show'];
						a.onclick = ((p) =>
						{
							return () =>
							{
								p.classList.add('hidden');
							}
						})(p);
					}

					if (s.indexOf('.') == s.lastIndexOf('.'))
					{
						div = crsbkr_en('div');
						holder.appendChild(div);
						div.id = 'crsbkr_webcheck_' + s;

						var h2 = crsbkr_en('h2');
						div.appendChild(h2);
						h2.innerHTML = CRSBKR_STRINGS['webcheck.' + s];
					}
					else
					{
						var h2 = crsbkr_en('h3');
						div.appendChild(h2);
						h2.innerHTML = CRSBKR_STRINGS['webcheck.' + s];
					}

					var p = crsbkr_en('p');
					div.appendChild(p);
					p.innerHTML = CRSBKR_STRINGS['webcheck.' + s + '.desc'];

					var prev = null;
					for (var i = 0; i < fields[s].length; i++)
					{
						var field = fields[s][i];
						if (field == '' && prev)
						{
							prev.style.width = '100%';
							continue;
						}
						else if (field == '')
						{
							div.appendChild(crsbkr_en('div'));
							continue;
						}

						var div2 = crsbkr_en('div');
						div.appendChild(div2);

						var label = crsbkr_en('label');
						div2.appendChild(label);
						label.innerHTML = field.startsWith('_file') ? CRSBKR_STRINGS['webcheck.' + s.replace(/.\d/, '').replace('license', '').replace('id', '') + field.substr(1)] : CRSBKR_STRINGS['webcheck.' + s.replace(/.\d/, '') + field];

						prev = div2;

						if (field.startsWith('_file'))
						{
							if (crsbkr_array_get(response, s.replace('id', '').replace('license', '') + field.substr(1), '') == '')
							{
								var span = crsbkr_en('span');
								div2.appendChild(span);
								span.className = 'attachment';

								var img = crsbkr_en('img');
								span.appendChild(img);
								img.src = CRSBKR_DIR + 'images/no_image.png';

								var inner = crsbkr_en('span');
								span.appendChild(inner);
								inner.innerHTML = CRSBKR_STRINGS['webcheck.attachment.select'];

								span.onclick = ((span, img, s, field) =>
								{
									return () =>
									{
										field = s.replace('license', '').replace('id', '') + field.substr(1);
										crsbkr_upload_file(null, { key : field, type : 'file_upload', label : CRSBKR_STRINGS['webcheck.' + field.replace(/.\d/, '')] }, null, (e) =>
										{
											crsbkr_array_set(data, field, e.data);
											img.src = e.data;
										});
									}
								})(span, img, s, field);
							}
							else
							{
								var input = crsbkr_en('input');
								div2.appendChild(input);
								input.className = 'tick';
								input.id = 'crsbkr_webcheck_' + s + field;
								input.type = 'text';
								input.value = '✓';
							}
						}
						else if (field.startsWith('_type') && crsbkr_array_get(response, s  + field, '') == '')
						{
							var select = crsbkr_en('select');
							div2.appendChild(select);
							select.id = 'crsbkr_webcheck_' + s + field;

							var option = crsbkr_en('option');
							select.appendChild(option);
							option.value = '';
							option.innerHTML = '';

							for (var ii = 0; ii < CRSBKR_ID_TYPES.length; ii++)
							{
								var option = crsbkr_en('option');
								select.appendChild(option);
								option.value = CRSBKR_ID_TYPES[ii];
								option.innerHTML = CRSBKR_STRINGS['id_type.' + CRSBKR_ID_TYPES[ii]];
							}
						}
						else if (field == '.country_id' && crsbkr_array_get(response, s + field, '') == '')
						{
							var select = crsbkr_en('select');
							div2.appendChild(select);
							select.id = 'crsbkr_webcheck_' + s + field;

							for (var ii = 0; ii < CRSBKR_COUNTRIES.length; ii++)
							{
								var option = crsbkr_en('option');
								select.appendChild(option);
								option.value = CRSBKR_COUNTRIES[ii].uid;
								option.innerHTML = CRSBKR_COUNTRIES[ii].name;
							}
						}
						else
						{
							var input = crsbkr_en('input');
							div2.appendChild(input);
							input.id = 'crsbkr_webcheck_' + s + field;
							input.type = 'text';
							input.value = crsbkr_array_get(response, s + field, '');

							if (input.value != '')
							{
								input.className = 'tick';
								input.value = '✓ ' + input.value;
							}
						}
					}
				}

				var div = crsbkr_en('div');
				holder.appendChild(div);

				var h2 = crsbkr_en('h2');
				div.appendChild(h2);
				h2.innerHTML = CRSBKR_STRINGS['webcheck.outro.title'];

				var p = crsbkr_en('p');
				div.appendChild(p);
				p.innerHTML = CRSBKR_STRINGS['webcheck.outro'];

				var button = crsbkr_en('button');
				div.appendChild(button);
				button.innerHTML = CRSBKR_STRINGS['webcheck.submit'];
				button.onclick = () =>
				{
					var errors = [];
					var now = crsbkr_date(new Date(), true, 'date');

					var inputs = holder.getElementsByTagName('input');
					for (var i = 0; i < inputs.length; i++)
					{
						if (inputs[i].classList.contains('tick') || inputs[i].value.trim() == '')
						{
							continue;
						}

						var v = inputs[i].value;
						if (inputs[i].id.indexOf('date') != -1)
						{
							v = crsbkr_date(inputs[i].value, true, 'date');
							if (crsbkr_date(v, false, 'date') != inputs[i].value.replaceAll('/', '-').replaceAll('.', '-'))
							{
								var field = inputs[i].id.replace('crsbkr_webcheck_', '');
								var s = field.startsWith('client_driver.0') ? 'client_driver.0' : 'client_driver.1';
								field = field.replace(s + '.', '');

								errors.push(CRSBKR_STRINGS['webcheck.' + s] + ' - ' + CRSBKR_STRINGS['webcheck.client_driver.' + field] + ': ' + CRSBKR_STRINGS['webcheck.error.invalidFormat']);
								inputs[i].classList.add('error');
							}
							else if (inputs[i].id.indexOf('expiry_date') != -1 && v < now)
							{
								var field = inputs[i].id.replace('crsbkr_webcheck_', '');
								var s = field.startsWith('client_driver.0') ? 'client_driver.0' : 'client_driver.1';
								field = field.replace(s + '.', '');

								errors.push(CRSBKR_STRINGS['webcheck.' + s] + ' - ' + CRSBKR_STRINGS['webcheck.client_driver.' + field] + ': ' + CRSBKR_STRINGS['webcheck.error.expired']);
								inputs[i].classList.add('error');
							}
							else
							{
								inputs[i].classList.remove('error');
							}
						}

						crsbkr_array_set(data, inputs[i].id.replace('crsbkr_webcheck_', ''), v);
					}

					var inputs = holder.getElementsByTagName('select');
					for (var i = 0; i < inputs.length; i++)
					{
						if (inputs[i].classList.contains('tick') || inputs[i].value.trim() == '' || inputs[i].id.indexOf('country_id') != -1 && inputs[i].value == '0')
						{
							continue;
						}

						var v = inputs[i].value;
						crsbkr_array_set(data, inputs[i].id.replace('crsbkr_webcheck_', ''), v);
					}

					if (errors.length > 0)
					{
						crsbkr_alert(CRSBKR_STRINGS['webcheck.error'] + '<br />' + errors.join(', '));
					}
					else
					{
						var div = crsbkr_en('div');

						var progressBarHolder = crsbkr_en('div');
						div.appendChild(progressBarHolder);
						progressBarHolder.className = 'progressbar';

						var progressBar = crsbkr_en('div');
						progressBarHolder.appendChild(progressBar);
						progressBar.style.width = '0%';

						var span = crsbkr_en('span');
						div.appendChild(span);
						span.innerHTML = '0%';

						var canceled = false;
						var thisAlert = crsbkr_alert(CRSBKR_STRINGS['webcheck.submit.progress'], [ CRSBKR_STRINGS['label.cancel'] ], div, () =>
						{
							canceled = true;
						});

						var offset = 0;
						var record = null;
						var thisData = JSON.stringify(data);

						var fn = (success, response2) =>
						{
							if (success && response2)
							{
								if (canceled)
								{
									if (record)
									{
										crsbkr_request('POST', 'client_reservation_update/' + record.uid + '/file_url', { filename : 'data.json', cancel : true });
									}

									crsbkr_alert_dismiss(thisAlert);

									return;
								}
								else if (record == null && crsbkr_array_get(response2, 'client_reservation_update.uid', '') != '')
								{
									record = response2.client_reservation_update;
								}
								else if (record == null)
								{
									crsbkr_request('POST', 'client_reservation/' + response.client_reservation.uid + '/client_reservation_update', null, fn);
									return;
								}

								var progress = Math.min(100, Math.floor(offset * 100.0 / thisData.length));
								progressBar.style.width = '' + progress + '%';
								span.innerHTML = '' + progress + '%';

								// Have we finished?
								if (offset == thisData.length)
								{
									crsbkr_request('POST', 'client_reservation_update/' + record.uid + '/file_url', { filename : 'data.json', confirm : true }, (success, response3) =>
									{
										if (success && response3)
										{
											var p = crsbkr_en('p');
											button.parentNode.appendChild(p);
											p.innerHTML = CRSBKR_STRINGS['webcheck.submit.success'];

											button.parentNode.removeChild(button);
											holder.classList.add('disabled');

											crsbkr_alert_dismiss(thisAlert);
											crsbkr_alert(CRSBKR_STRINGS['webcheck.submit.success']);
										}
										else
										{
											crsbkr_alert(CRSBKR_STRINGS['error'] + '<br />' + CRSBKR_STRINGS['error.error'] + ': ' + response3.error, [ CRSBKR_STRINGS['label.cancel'] ], null, () =>
											{
												crsbkr_alert_dismiss(thisAlert);
											});
										}
									});
								}
								else
								{
									// The server offset might differ due to base64 decoding.
									var serverOffset = response2.offset ? response2.offset : 0;

									// Send the next chunk to the server.
									var chunkSize = 150 * 1024;
									if (offset + chunkSize > thisData.length)
									{
										chunkSize = thisData.length - offset;
									}

									var thisOffset = offset;
									offset += chunkSize;
									crsbkr_request('POST', 'client_reservation_update/' + record.uid + '/file_url',
									{
										url : response2.url ? response2.url : null,
										filename : 'data.json',
										data : thisData.substring(thisOffset, thisOffset + chunkSize),
										offset : serverOffset
									}, (success, response3) =>
									{
										if (success && response3)
										{
											fn(success, response3);
										}
										else
										{
											crsbkr_alert(CRSBKR_STRINGS['error'] + '<br />' + CRSBKR_STRINGS['error.error'] + ': ' + response3.error, [ CRSBKR_STRINGS['label.retry'], CRSBKR_STRINGS['label.cancel'] ], null, () =>
											{
												offset = thisOffset;
												fn(true, response2);
											}, () =>
											{
												crsbkr_alert_dismiss(thisAlert);
											});
										}
									});
								}
							}
							else
							{
								crsbkr_alert(CRSBKR_STRINGS['error'] + '<br />' + CRSBKR_STRINGS['error.error'] + ': ' + response2.error, [ CRSBKR_STRINGS['label.retry'], CRSBKR_STRINGS['label.cancel'] ], null, () =>
								{
									fn(true, response2);
								}, () =>
								{
									canceled = true;
									fn(true, response2);
								});
							}
						};

						fn(true, {});
					}
				}
			}
			else
			{
				var error = crsbkr_en('span');
				holder.appendChild(error);
				error.className = 'crsbkr_error';
				error.innerHTML = CRSBKR_STRINGS['webcheck.form.error'].replace('{{error}}', response.error);

				var form = crsbkr_webcheck_form(data);
				holder.appendChild(form);
			}
		});
	}
}

function crsbkr_webcheck_form(data)
{
	var form = crsbkr_en('form');
	form.method = 'get';

	var label = crsbkr_en('label');
	form.appendChild(label);
	label.innerHTML = CRSBKR_STRINGS['webcheck.form.voucher'];

	var voucher = crsbkr_en('input');
	form.appendChild(voucher);
	voucher.type = 'text';
	voucher.value = data && data.voucher_no ? data.voucher_no : '';
	voucher.onchange = () =>
	{
		if (voucher.value.length > 0)
		{
			voucher.classList.remove('error');
		}
	}

	var label = crsbkr_en('label');
	form.appendChild(label);
	label.innerHTML = CRSBKR_STRINGS['webcheck.form.surname'];

	var surname = crsbkr_en('input');
	form.appendChild(surname);
	surname.type = 'text';
	surname.value = data && data.surname ? data.surname : '';
	surname.onchange = () =>
	{
		if (surname.value.length > 0)
		{
			surname.classList.remove('error');
		}
	}

	var hidden = crsbkr_en('input');
	form.appendChild(hidden);
	hidden.name = 'q';
	hidden.type = 'hidden';

	var label = crsbkr_en('label');
	form.appendChild(label);

	var submit = crsbkr_en('input');
	form.appendChild(submit);
	submit.type = 'submit';
	submit.value = 'Invia';

	form.onsubmit = () =>
	{
		if (voucher.value.length > 0 && surname.value.length > 0)
		{
			var s = btoa(JSON.stringify({ voucher_no : voucher.value, surname : surname.value }));
			hidden.value = s;

			return true;
		}
		else
		{
			if (voucher.value.length == 0)
			{
				voucher.classList.add('error');
			}

			if (surname.value.length == 0)
			{
				surname.classList.add('error');
			}

			return false;
		}
	}

	return form;
}