<?php
namespace PPCR;
if ( ! defined('ABSPATH') ) { exit; }

class Options_Page {

    public static function boot(){
        add_action('admin_menu', [__CLASS__, 'menu']);
        add_action('admin_init', [__CLASS__, 'maybe_handle_post']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'assets']);
    }

    public static function menu(){
        add_menu_page(
            __('Post-Purchase Coupon Rules', 'crosssell-mailer-post-purchase-coupon-rules-lite'),
            __('Coupon Rules', 'crosssell-mailer-post-purchase-coupon-rules-lite'),
            'manage_options',
            'ppcr-options',
            [__CLASS__, 'render'],
            'dashicons-email-alt2',
            56
        );
    }
    public static function assets($hook){
        if ($hook !== 'toplevel_page_ppcr-options') return;

        $main_file = dirname(__DIR__) . '/crosssell-mailer-post-purchase-coupon-rules-lite.php';
        $base_url  = plugins_url('', $main_file);
        $css_path  = dirname(__DIR__) . '/assets/ppcr-admin.css';
        $js_path   = dirname(__DIR__) . '/assets/ppcr-admin.js';

        wp_enqueue_style(
            'ppcr-admin',
            $base_url . '/assets/ppcr-admin.css',
            [],
            file_exists($css_path) ? filemtime($css_path) : time()
        );

        wp_enqueue_script(
            'ppcr-admin',
            $base_url . '/assets/ppcr-admin.js',
            ['jquery'],
            file_exists($js_path) ? filemtime($js_path) : time(),
            true
        );

        // Antes calculabas esto en admin_head:
        $rules   = get_option( defined('PPCR_OPT_RULES') ? PPCR_OPT_RULES : 'ppcr_rules', [] );
        $has_one = ( is_array($rules) && count($rules) >= 1 );

        // Pasar el flag a JS de forma correcta:
        wp_localize_script('ppcr-admin', 'PPCR_SETTINGS', [
            'has_one_rule' => $has_one ? true : false,
        ]);

            if ( class_exists('\WooCommerce') ) {
                wp_enqueue_style('woocommerce_admin_styles');
                wp_enqueue_script('selectWoo');
                wp_enqueue_script('wc-enhanced-select');
            }
    }

    public static function maybe_handle_post(){
        if ( ! is_admin() ) {
            return;
        }
        if ( ! current_user_can('manage_options') ) {
            return;
        }
        if ( ! isset($_POST['ppcr_save']) ) {
            return;
        }

        check_admin_referer('ppcr_save_rules');

        // Normalizar entrada de reglas
        $rules = ( isset($_POST['rules']) && is_array($_POST['rules']) )
            ? array_values($_POST['rules'])
            : [];

        $clean     = [];
        $has_error = false;

        foreach ( $rules as $r ) {
            $trigger_products = isset($r['trigger_products'])
                ? array_map('absint', (array) $r['trigger_products'])
                : [];
            $target_products  = isset($r['target_products'])
                ? array_map('absint', (array) $r['target_products'])
                : [];

            // Validación básica: cada regla necesita al menos un trigger y un target
            if ( empty($trigger_products) || empty($target_products) ) {
                $has_error = true;
            }

            $discount_type = isset($r['discount_type'])
                ? sanitize_text_field($r['discount_type'])
                : 'percent';

            if ( ! in_array($discount_type, ['percent', 'fixed_cart', 'fixed_product'], true) ) {
                $discount_type = 'percent';
            }

            $clean[] = [
                'enabled'                 => ! empty($r['enabled']),
                'name'                    => sanitize_text_field( $r['name'] ?? '' ),
                'trigger_products'        => array_values( array_filter( $trigger_products ) ),
                'trigger_status'          => sanitize_text_field( $r['trigger_status'] ?? 'processing' ),
                'min_order'               => isset($r['min_order']) ? (float) $r['min_order'] : 0,
                'discount_type'           => $discount_type,
                'discount_amount'         => isset($r['discount_amount']) ? (float) $r['discount_amount'] : 0,
                'expires_days'            => isset($r['expires_days']) ? max(0, (int) $r['expires_days']) : 0,
                'usage_limit'             => isset($r['usage_limit']) ? max(1, (int) $r['usage_limit']) : 1,
                'free_shipping'           => ! empty($r['free_shipping']),
                'skip_if_target_in_order' => ! empty($r['skip_if_target_in_order']),
                'target_products'         => array_values( array_filter( $target_products ) ),
                'email_subject'           => sanitize_text_field( $r['email_subject'] ?? '' ),
                'email_body'              => wp_kses_post( $r['email_body'] ?? '' ),
                'coupon_card'             => ! empty($r['coupon_card']),
                'coupon_prefix'           => sanitize_text_field( $r['coupon_prefix'] ?? '' ),
            ];
    }

    // Si hay errores de configuración en alguna regla, mostramos aviso y no guardamos
    if ( $has_error ) {
        add_settings_error(
            'ppcr_messages',
            'ppcr_error',
            self::tr(
                'Each rule must have at least one trigger product and one target product.',
                'Cada regla debe tener al menos un producto activador y uno incentivado.',
                'Jede Regel benötigt mindestens ein Auslöser-Produkt und ein Zielprodukt.',
                'Chaque règle doit avoir au moins un produit déclencheur et un produit cible.',
                'Ogni regola deve avere almeno un prodotto attivatore e uno target.'
            ),
            'error'
        );
        return;
    }

    // Guardar reglas (aunque el array esté vacío, refleja el estado actual)
    update_option(
        defined('PPCR_OPT_RULES') ? PPCR_OPT_RULES : 'ppcr_rules',
        $clean,
        false
    );

    // Guardar ajustes globales
    if ( isset($_POST['settings']) && is_array($_POST['settings']) ) {
        $settings = [
            'from_name'       => sanitize_text_field( $_POST['settings']['from_name'] ?? '' ),
            'from_email'      => sanitize_email(      $_POST['settings']['from_email'] ?? '' ),
            'test_mode'       => ! empty($_POST['settings']['test_mode']),
            'test_recipient'  => sanitize_email(      $_POST['settings']['test_recipient'] ?? '' ),
            'use_wc_template' => ! empty($_POST['settings']['use_wc_template']),
            'delay_minutes'   => max(0, (int) ( $_POST['settings']['delay_minutes'] ?? 5 )),
        ];

        update_option(
            defined('PPCR_OPT_SETTINGS') ? PPCR_OPT_SETTINGS : 'ppcr_settings',
            $settings,
            false
        );
    }

    add_settings_error(
        'ppcr_messages',
        'ppcr_saved',
        self::tr(
            'Rules saved successfully.',
            'Reglas guardadas correctamente.',
            'Regeln erfolgreich gespeichert.',
            'Règles enregistrées avec succès.',
            'Regole salvate con successo.'
        ),
        'updated'
    );
}


    private static function tr($en, $es=null, $de=null, $fr=null, $it=null){
        $loc = function_exists('get_user_locale') ? get_user_locale() : ( function_exists('get_locale') ? get_locale() : 'en_US' );
        $lang = strtolower(substr($loc,0,2));
        if ($lang === 'es' && $es !== null) return $es;
        if ($lang === 'de' && $de !== null) return $de;
        if ($lang === 'fr' && $fr !== null) return $fr;
        if ($lang === 'it' && $it !== null) return $it;
        return $en;
    }

    private static function defaults_email_subject(){
        return self::tr('Your exclusive discount','Tu descuento exclusivo','Ihr exklusiver Rabatt','Votre remise exclusive','Il tuo sconto esclusivo');
    }

    private static function defaults_email_body(){
        return self::tr(
            "Hi {{first_name}},\n\nShop now: {{target_link}}\nHere’s your code: {{coupon_code}} ({{discount_amount}} {{discount_type}}).\nDon’t wait — this offer ends soon.",
            "Hola {{first_name}},\n\nCompra ahora: {{target_link}}\nAquí tienes tu código: {{coupon_code}} ({{discount_amount}} {{discount_type}}).\n¡Aprovéchalo! Esta oferta termina pronto.",
            "Hallo {{first_name}},\n\nJetzt kaufen: {{target_link}}\nHier ist dein Code: {{coupon_code}} ({{discount_amount}} {{discount_type}}).\nNicht verpassen — das Angebot endet bald.",
            "Bonjour {{first_name}},\n\nAchetez maintenant : {{target_link}}\nVoici votre code : {{coupon_code}} ({{discount_amount}} {{discount_type}}).\nNe tardez pas — l’offre se termine bientôt.",
            "Ciao {{first_name}},\n\nAcquista ora: {{target_link}}\nEcco il tuo codice: {{coupon_code}} ({{discount_amount}} {{discount_type}}).\nAffrettati — l’offerta termina presto."
        );
    }

    private static function token_labels(){
        return [
            ['{{first_name}}', self::tr('First name','Nombre','Name','Prénom','Nome')],
            ['{{last_name}}', self::tr('Last name','Apellidos','Nachname','Nom','Cognome')],
            ['{{full_name}}', self::tr('Full name','Nombre completo','Vollständiger Name','Nom complet','Nome completo')],
            ['{{email}}', self::tr('Email','Email','E-Mail','Email','Email')],
            ['{{billing_phone}}', self::tr('Phone','Teléfono','Telefon','Téléphone','Telefono')],
            ['{{billing_city}}', self::tr('Billing city','Ciudad de facturación','Rechnungsstadt','Ville de facturation','Città di fatturazione')],
            ['{{billing_country}}', self::tr('Billing country','País de facturación','Rechnungsland','Pays de facturation','Paese di fatturazione')],
            ['{{shipping_city}}', self::tr('Shipping city','Ciudad de envío','Versandstadt','Ville de livraison','Città di spedizione')],
            ['{{shipping_country}}', self::tr('Shipping country','País de envío','Versandland','Pays de livraison','Paese di spedizione')],
            ['{{order_id}}', self::tr('Order ID','ID de pedido','Bestell-ID','ID de commande','ID ordine')],
            ['{{coupon_code}}', self::tr('Coupon code','Código de cupón','Gutscheincode','Code promo','Codice coupon')],
            ['({{discount_amount}} {{discount_type}})', self::tr('Discount (amount+type)','Descuento (importe+tipo)','Rabatt (Betrag+Art)','Remise (montant+type)','Sconto (importo+tipo)')],
            ['{{target_link}}', self::tr('Target product link(s)','Enlace(s) a producto(s) destino','Zielprodukt-Link(s)','Lien(s) produit(s) cible(s)','Link prodotto target')],
            ['{{target_list}}', self::tr('Target products as list','Productos destino en lista','Zielprodukte als Liste','Produits cibles en liste','Prodotti target in lista')],
        ];
    }

    private static function product_select($name, $ids = [], $placeholder = 'Search products or variations…', $required = false){
        $ids = array_map('absint', (array)$ids);
        echo '<select class="wc-product-search" style="width:100%" name="'.esc_attr($name).'[]" multiple="multiple" data-placeholder="'.esc_attr($placeholder).'" data-action="woocommerce_json_search_products_and_variations"'.($required?' required':'').'>';
        if (!empty($ids)){
            $posts = get_posts([
                'post_type' => ['product','product_variation'],
                'post__in' => $ids,
                'numberposts' => -1,
            ]);
            foreach ($posts as $p){
                $label = get_the_title($p->ID);
                if ($p->post_type === 'product_variation'){
                    $parent = wp_get_post_parent_id($p->ID);
                    $parent_title = $parent ? get_the_title($parent) : 'Variation';
                    $product = function_exists('wc_get_product') ? wc_get_product($p->ID) : null;
                    $var_attrs = $product ? wc_get_formatted_variation($product, true) : '';
                    $label = $parent_title . ' – ' . $var_attrs;
                }
                $label = wp_strip_all_tags($label);
                echo '<option value="'.esc_attr($p->ID).'" selected="selected">'.esc_html(wp_html_excerpt($label.' (#'.$p->ID.')', 100, '…')).'</option>';
            }
        }
        echo '</select>';
    }

    public static function render(){
        if (!current_user_can('manage_options')) return;

        $rules = get_option( defined('PPCR_OPT_RULES') ? PPCR_OPT_RULES : 'ppcr_rules', [] );
           // indicar si hay al menos una regla guardada
         $has_one = is_array($rules) && count($rules) >= 1;

        // solo mostramos y editamos la primera.
            add_settings_error(
                'ppcr_messages',
                'ppcr_lite_limit',
                self::tr(
                    'Lite version: only the first rule is active. Upgrade to Pro for unlimited rules.',
                    'Versión Lite: solo la primera regla está activa. Actualiza a Pro para reglas ilimitadas.',
                    'Lite-Version: Nur die erste Regel ist aktiv. Upgrade auf Pro für unbegrenzte Regeln.',
                    'Version Lite : seule la première règle est active. Passez à Pro pour des règles illimitées.',
                    'Versione Lite: è attiva solo la prima regola. Passa a Pro per regole illimitate.'
                ),
                'warning'
            );

            $rules = array_slice( $rules, 0, 1 );
        

        $settings = get_option( defined('PPCR_OPT_SETTINGS') ? PPCR_OPT_SETTINGS : 'ppcr_settings', [
            'from_name'       => get_bloginfo('name'),
            'from_email'      => get_bloginfo('admin_email'),
            'test_mode'       => false,
            'test_recipient'  => '',
            'use_wc_template' => true,
            'delay_minutes'   => 5,
        ]);


        settings_errors('ppcr_messages');
        ?>
        <div class="wrap ppcr-wrap">
            <h1 class="wp-heading-inline">
                <?php echo esc_html( self::tr(
                    'CrossSell Mailer — Rules',
                    'CrossSell Mailer — Reglas',
                    'CrossSell Mailer — Regeln',
                    'CrossSell Mailer — Règles',
                    'CrossSell Mailer — Regole'
                ) ); ?>
            </h1>

            <form method="post">
                <?php wp_nonce_field('ppcr_save_rules'); ?>

                <div id="ppcr-rules">
                    <?php if ( $has_one ) : ?>
                        <?php
                        // La versión actual trabaja con una sola regla.
                        // Tomamos solo la primera regla guardada y la normalizamos al índice 0.
                        $first_rule = reset($rules);
                        self::rule_block(0, $first_rule);
                        ?>
                    <?php else : ?>
                        <div class="ppcr-empty">
                            <?php echo esc_html( self::tr(
                                'No rules yet. Click “Add rule”.',
                                'Aún no hay reglas. Pulsa “Añadir regla”.',
                                'Noch keine Regeln. Klicke auf „Regel hinzufügen“.',
                                'Aucune règle pour le moment. Cliquez sur « Ajouter une règle ».',
                                'Nessuna regola ancora. Clicca “Aggiungi regola”.'
                            ) ); ?>
                        </div>
                    <?php endif; ?>
                </div>

                <?php if ( ! $has_one ) : ?>
                    <p>
                        <button type="button" class="button button-primary" id="ppcr-add-rule">
                            <?php echo esc_html( self::tr(
                                'Add rule',
                                'Añadir regla',
                                'Regel hinzufügen',
                                'Ajouter une règle',
                                'Aggiungi regola'
                            ) ); ?>
                        </button>
                    </p>
                <?php endif; ?>

                <?php if ( $has_one ) : ?>
                    <p style="color:#666; font-size:13px;">
                        <?php echo esc_html__(
                            'This version supports one automation rule.',
                            'crosssell-mailer-post-purchase-coupon-rules-lite'
                        ); ?>
                        <a href="https://sellmailer.com" target="_blank">
                            <?php echo esc_html__(
                                'See Pro version for advanced workflows.',
                                'crosssell-mailer-post-purchase-coupon-rules-lite'
                            ); ?>
                        </a>
                    </p>
                <?php endif; ?>

                <h2 class="title">
                    <?php echo esc_html( self::tr(
                        'Global settings',
                        'Ajustes globales',
                        'Globale Einstellungen',
                        'Paramètres globaux',
                        'Impostazioni globali'
                    ) ); ?>
                </h2>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <?php echo esc_html( self::tr(
                                'Delay before sending (minutes)',
                                'Retraso antes de enviar (minutos)',
                                'Verzögerung vor dem Senden (Minuten)',
                                'Délai avant envoi (minutes)',
                                'Ritardo prima dell’invio (minuti)'
                            ) ); ?>
                        </th>
                        <td>
                            <input name="settings[delay_minutes]" type="number" min="0" step="1"
                                value="<?php echo isset($settings['delay_minutes']) ? intval($settings['delay_minutes']) : 5; ?>"
                                class="small-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php echo esc_html( self::tr(
                                'Use WooCommerce email template (header, footer, styles)',
                                'Usar plantilla de WooCommerce (cabecera, pie, estilos)',
                                'WooCommerce-E-Mail-Vorlage verwenden (Header, Footer, Styles)',
                                'Utiliser le modèle d’email WooCommerce (en-tête, pied, styles)',
                                'Usa il template email di WooCommerce (header, footer, stili)'
                            ) ); ?>
                        </th>
                        <td>
                            <label>
                                <input name="settings[use_wc_template]" type="checkbox" value="1"
                                    <?php echo ! empty($settings['use_wc_template']) ? 'checked' : ''; ?> />
                                <?php echo esc_html( self::tr(
                                    'Enable',
                                    'Habilitar',
                                    'Aktivieren',
                                    'Activer',
                                    'Abilita'
                                ) ); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php echo esc_html( self::tr(
                                'Sender name',
                                'Remitente (nombre)',
                                'Absendername',
                                'Nom de l’expéditeur',
                                'Nome mittente'
                            ) ); ?>
                        </th>
                        <td>
                            <input name="settings[from_name]" type="text" class="regular-text"
                                value="<?php echo esc_attr($settings['from_name']); ?>" required />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php echo esc_html( self::tr(
                                'Sender email',
                                'Remitente (email)',
                                'Absender-E-Mail',
                                'E-mail de l’expéditeur',
                                'Email mittente'
                            ) ); ?>
                        </th>
                        <td>
                            <input name="settings[from_email]" type="email" class="regular-text"
                                value="<?php echo esc_attr($settings['from_email']); ?>" required />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php echo esc_html( self::tr(
                                'Test mode',
                                'Modo test',
                                'Testmodus',
                                'Mode test',
                                'Modalità test'
                            ) ); ?>
                        </th>
                        <td>
                            <label>
                                <input name="settings[test_mode]" type="checkbox"
                                    <?php checked( ! empty($settings['test_mode']) ); ?> />
                                <?php echo esc_html( self::tr(
                                    'Do not send to customer; send to:',
                                    'No enviar al cliente; envía a:',
                                    'Nicht an Kunden senden; senden an:',
                                    'Ne pas envoyer au client ; envoyer à :',
                                    'Non inviare al cliente; invia a:'
                                ) ); ?>
                            </label>
                            <br>
                            <input name="settings[test_recipient]" type="email" class="regular-text"
                                value="<?php echo esc_attr($settings['test_recipient']); ?>"
                                placeholder="you@example.com" />
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" name="ppcr_save" class="button button-primary">
                        <?php echo esc_html( self::tr(
                            'Save changes',
                            'Guardar cambios',
                            'Änderungen speichern',
                            'Enregistrer les modifications',
                            'Salva modifiche'
                        ) ); ?>
                    </button>
                </p>
            </form>
            </div>


        <?php self::rule_template(); ?>
        <?php
    }

    private static function rule_block($i, $r = []){
        $d = wp_parse_args($r, [
            'enabled' => true,
            'name' => '',
            'trigger_products' => [],
            'trigger_status' => 'processing',
            'min_order' => '',
            'discount_type' => 'percent',
            'discount_amount' => '',
            'expires_days' => 7,
            'usage_limit' => 1,
            'free_shipping' => false,
            'skip_if_target_in_order' => false,
            'target_products' => [],
            'email_subject' => self::defaults_email_subject(),
            'email_body' => self::defaults_email_body(),
            'coupon_card' => true, // por defecto activado
            'coupon_prefix' => 'PROMO',
        ]);
        ?>
        <div class="ppcr-rule" data-index="<?php echo esc_attr($i); ?>">
            <div class="ppcr-rule__header">
                <strong class="ppcr-rule__title"><?php echo esc_html( self::tr('Rule','Regla','Regel','Règle','Regola') ); ?> <span class="ppcr-required" aria-hidden="true">*</span>:</strong>
                <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][name]" type="text" class="regular-text" value="<?php echo esc_attr($d['name']); ?>" placeholder="<?php echo esc_attr( self::tr('Rule name','Nombre de la regla','Regelname','Nom de la règle','Nome regola') ); ?>" required />
                <label class="ppcr-toggle"><input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][enabled]" type="checkbox" <?php checked($d['enabled']); ?> /> <?php echo esc_html( self::tr('Enabled','Activada','Aktiviert','Activé','Abilitata') ); ?></label>
                <button type="button" class="button-link-delete ppcr-remove"><?php echo esc_html( self::tr('Delete','Eliminar','Löschen','Supprimer','Elimina') ); ?></button>
            </div>

            <div class="ppcr-grid">
                <p class="ppcr-col-12">
                    <label><?php echo esc_html( self::tr('Trigger products','Productos que activan la regla','Auslöser-Produkte','Produits déclencheurs','Prodotti attivatori') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <?php self::product_select("rules[{$i}][trigger_products]", $d['trigger_products'], esc_attr( self::tr('Search products or variations…','Buscar productos o variaciones…','Produkte oder Variationen suchen…','Rechercher des produits ou variations…','Cerca prodotti o variazioni…') ), true); ?>
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Order status that triggers this rule','Estado del pedido que activa la regla','Bestellstatus, der die Regel auslöst','Statut de commande qui déclenche la règle','Stato ordine che attiva la regola') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <select name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][trigger_status]" required>
                        <option value="processing" <?php selected($d['trigger_status'], 'processing'); ?>><?php echo esc_html( self::tr('Processing (paid)','Procesando (pagado)','In Bearbeitung (bezahlt)','En cours (payé)','In lavorazione (pagato)') ); ?></option>
                        <option value="completed" <?php selected($d['trigger_status'], 'completed'); ?>><?php echo esc_html( self::tr('Completed','Completado','Abgeschlossen','Terminé','Completato') ); ?></option>
                        <option value="pending" <?php selected($d['trigger_status'], 'pending'); ?>><?php echo esc_html( self::tr('Pending payment','Pendiente de pago','Ausstehende Zahlung','En attente de paiement','In attesa di pagamento') ); ?></option>
                        <option value="on-hold" <?php selected($d['trigger_status'], 'on-hold'); ?>><?php echo esc_html( self::tr('On hold','En espera','In Wartestellung','En attente','In sospeso') ); ?></option>
                        <option value="cancelled" <?php selected($d['trigger_status'], 'cancelled'); ?>><?php echo esc_html( self::tr('Cancelled','Cancelado','Storniert','Annulé','Annullato') ); ?></option>
                        <option value="refunded" <?php selected($d['trigger_status'], 'refunded'); ?>><?php echo esc_html( self::tr('Refunded','Reembolsado','Erstattet','Remboursé','Rimborsato') ); ?></option>
                    </select>
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Minimum order amount','Importe mínimo del pedido','Mindestbestellwert','Montant minimum de commande','Importo minimo ordine') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][min_order]" type="number" step="0.01" min="0" value="<?php echo esc_attr($d['min_order']); ?>" required />
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Discount type','Tipo de descuento','Rabattart','Type de remise','Tipo di sconto') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <select name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][discount_type]" required>
                        <option value="percent" <?php selected($d['discount_type'], 'percent'); ?>><?php echo esc_html( self::tr('Percentage','Porcentaje','Prozentsatz','Pourcentage','Percentuale') ); ?></option>
                        <option value="fixed_cart" <?php selected($d['discount_type'], 'fixed_cart'); ?>><?php echo esc_html( self::tr('Fixed cart','Importe en carrito','Fester Warenkorb','Montant panier','Importo carrello') ); ?></option>
                        <option value="fixed_product" <?php selected($d['discount_type'], 'fixed_product'); ?>><?php echo esc_html( self::tr('Fixed product','Importe por producto','Fester Produkt','Montant produit','Importo per prodotto') ); ?></option>
                    </select>
                </p>
                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr(
                        'Coupon code prefix',
                        'Prefijo del código de cupón',
                        'Gutscheincode-Präfix',
                        'Préfixe du code promo',
                        'Prefisso del codice coupon'
                    ) ); ?></label>
                    <input
                        name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][coupon_prefix]"
                        type="text"
                        value="<?php echo esc_attr($d['coupon_prefix']); ?>"
                        placeholder="PROMO"
                    />
                    <span class="description">
                        <?php echo esc_html( self::tr(
                            'Used to build the code: PREFIX_{order_id}',
                            'Se usará para construir el código: PREFIJO_{id_pedido}',
                            'Wird für den Code verwendet: PRAEFIX_{bestell_id}',
                            'Utilisé pour construire le code : PREFIXE_{id_commande}',
                            'Usato per costruire il codice: PREFISSO_{id_ordine}'
                        ) ); ?>
                    </span>
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Discount amount','Cantidad de descuento','Rabattbetrag','Montant de remise','Importo sconto') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][discount_amount]" type="number" step="0.01" min="0.01" value="<?php echo esc_attr($d['discount_amount']); ?>" required />
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Expires in (days)','Caduca en (días)','Läuft ab in (Tagen)','Expire dans (jours)','Scade in (giorni)') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][expires_days]" type="number" min="0" value="<?php echo esc_attr($d['expires_days']); ?>" required />
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Usage limit per coupon','Límite de uso por cupón','Nutzungslimit pro Gutschein','Limite d’utilisation par coupon','Limite di utilizzo per coupon') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][usage_limit]" type="number" min="1" value="<?php echo esc_attr($d['usage_limit']); ?>" required />
                </p>

                <p class="ppcr-col-4">
                    <label><?php echo esc_html( self::tr('Free shipping','Envío gratis','Kostenloser Versand','Livraison gratuite','Spedizione gratuita') ); ?></label>
                    <label class="ppcr-toggle"><input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][free_shipping]" type="checkbox" <?php checked($d['free_shipping']); ?> /> <?php echo esc_html( self::tr('Enable','Habilitar','Aktivieren','Activer','Abilita') ); ?></label>
                </p>

                <p class="ppcr-col-12">
                    <label><?php echo esc_html( self::tr('Target products','Productos incentivados','Zielprodukte','Produits cibles','Prodotti incentivati') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <?php self::product_select("rules[{$i}][target_products]", $d['target_products'], esc_attr( self::tr('Search products or variations…','Buscar productos o variaciones…','Produkte oder Variationen suchen…','Rechercher des produits ou variations…','Cerca prodotti o variazioni…') ), true); ?>
                </p>

                <p class="ppcr-col-12">
                    <label class="ppcr-toggle"><input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][skip_if_target_in_order]" type="checkbox" <?php checked(!empty($d['skip_if_target_in_order'])); ?> /> <?php echo esc_html( self::tr('Do not apply the rule if the order already includes any target products','No aplicar la regla si el pedido ya incluye alguno de los productos incentivados','Regel nicht anwenden, wenn die Bestellung bereits Zielprodukte enthält','Ne pas appliquer la règle si la commande inclut déjà des produits cibles','Non applicare la regola se l’ordine include già prodotti target') ); ?></label>
                </p>
            </div>

            <div class="ppcr-email ppcr-email-open">
                <div class="ppcr-email-row">
                    <label><?php echo esc_html( self::tr('Subject','Asunto','Betreff','Objet','Oggetto') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][email_subject]" type="text" value="<?php echo esc_attr($d['email_subject']); ?>" class="large-text" required />
                </div>
                <div class="ppcr-email-row">
                    <label><?php echo esc_html( self::tr('Body (HTML allowed)','Cuerpo (HTML permitido)','Text (HTML erlaubt)','Corps (HTML autorisé)','Corpo (HTML consentito)') ); ?> <span class="ppcr-required" aria-hidden="true">*</span></label>
                    <textarea name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][email_body]" rows="6" class="large-text code" required><?php echo esc_textarea($d['email_body']); ?></textarea>
                    <div class="ppcr-tokenbar" aria-label="<?php echo esc_attr( self::tr('Insert variables','Insertar variables','Variablen einfügen','Insérer des variables','Inserisci variabili') ); ?>">
                        <?php foreach (self::token_labels() as $t): ?>
                            <button type="button" class="button ppcr-insert-token" data-token="<?php echo esc_attr($t[0]); ?>"><?php echo esc_html($t[1]); ?></button>
                        <?php endforeach; ?>
                    </div>
                </div>
                <div class="ppcr-email-row">
                    <label class="ppcr-toggle">
                        <input name="rules[<?php echo esc_attr( (string) (int) $i ); ?>][coupon_card]" type="checkbox" <?php echo !empty($d['coupon_card']) ? 'checked' : ''; ?> />
                        <?php echo esc_html( self::tr('Insert coupon card (logo + code) in email','Insertar tarjeta de cupón (logo + código) en el email','Gutscheinkarte (Logo + Code) in E-Mail einfügen','Insérer une carte coupon (logo + code) dans l’e-mail','Inserisci card coupon (logo + codice) nell’email') ); ?>
                    </label>
                </div>
            </div>
        </div>
        <?php
    }

    private static function rule_template(){
        ob_start();
        self::rule_block('__INDEX__');
        $tpl = ob_get_clean();
        ?>
        <script id="ppcr-rule-template" type="application/json">
            <?php echo wp_json_encode($tpl); ?>
        </script>
        <?php
    }
}
Options_Page::boot();
