<?php
namespace PPCR;
if ( ! defined('ABSPATH') ) { exit; }

class Engine {

    public static function boot_hooks(){
        // Hook a transiciones de estado relevantes
        $statuses = ['pending','processing','completed','on-hold','cancelled','refunded'];
        foreach ($statuses as $st){
            add_action('woocommerce_order_status_' . $st, function($order_id) use ($st){
                $order = function_exists('wc_get_order') ? wc_get_order($order_id) : null;
                if ($order){
                    self::maybe_schedule($order, $st);
                }
            }, 20, 1);
        }

        // También en thankyou (algunos gateways no cambian estado inmediatamente)
        add_action('woocommerce_thankyou', function($order_id){
            $order = function_exists('wc_get_order') ? wc_get_order($order_id) : null;
            if ($order){
                self::maybe_schedule($order, 'thankyou');
            }
        }, 10, 1);

        // 🔧 Acción diferida: aceptar 1 argumento (array o int)
        add_action('ppcr_delayed_trigger', [__CLASS__, 'handle_scheduled'], 10, 1);
    }

    private static function get_settings(){
        $defaults = [
            'delay_minutes' => 5,
        ];
        $s = get_option( defined('PPCR_OPT_SETTINGS') ? PPCR_OPT_SETTINGS : 'ppcr_settings', [] );
        return wp_parse_args($s, $defaults);
    }

    private static function get_rules(){
        $rules = get_option( defined('PPCR_OPT_RULES') ? PPCR_OPT_RULES : 'ppcr_rules', [] );
        if ( ! is_array( $rules ) ) {
            return [];
        }
            if ( count( $rules ) > 1 ) {
                $rules = array_slice( $rules, 0, 1 );
            }

        return $rules;
    }


    /**
     * Decide si hay que programar la acción diferida para este pedido/estado
     */
    public static function maybe_schedule(\WC_Order $order, $status){
        $rules = self::get_rules();
        if (empty($rules)) return;

        // Normalizar thankyou como processing (si tus reglas apuntan a 'processing')
        $normalized_status = ($status === 'thankyou') ? 'processing' : $status;

        // Si ninguna regla habilitada apunta a este estado, no hacemos nada
        $targets_status = false;
        foreach ($rules as $r){
            if (!empty($r['enabled']) && (!empty($r['trigger_status']) ? $r['trigger_status'] : 'processing') === $normalized_status){
                $targets_status = true; break;
            }
        }
        if (!$targets_status) return;

        $settings = self::get_settings();
        $order_id = $order->get_id();
        $delay    = max(0, ((int)$settings['delay_minutes']) * 60)-60;
        $when     = time() + $delay;

        // Evitar duplicados: cancelar programaciones anteriores para este pedido
        if ( function_exists('as_unschedule_action') ) {
            // Cancelar cualquier acción pendiente en el grupo 'cross-selling-rules'
            as_unschedule_action('ppcr_delayed_trigger', [ 'order_id' => $order_id ], 'crosssell-mailer-post-purchase-coupon-rules-lite');
        } else {
            // Fallback para WP-Cron
            wp_clear_scheduled_hook('ppcr_delayed_trigger', [ $order_id ]);
        }

        // Programar usando Action Scheduler si está disponible
        if ( function_exists('as_schedule_single_action') ) {
            as_schedule_single_action(
                $when,
                'ppcr_delayed_trigger',
                [ 'order_id' => $order_id ],
                'cross-selling-rules' // grupo opcional para organizar en AS
            );
        } else {
            // Fallback: WP-Cron
            wp_schedule_single_event(
                $when,
                'ppcr_delayed_trigger',
                [ $order_id ]
            );
        }
    }

    /**
     * Handler de la acción diferida. Acepta:
     * - array ['order_id'=>123] desde Action Scheduler
     * - int 123 desde wp-cron (fallback)
     */
    public static function handle_scheduled( $arg ){
        // Normalizar $order_id desde distintos orígenes
        $order_id = is_array($arg) ? (int)($arg['order_id'] ?? 0) : (int)$arg;
        if ( ! $order_id ) return;

        $order = function_exists('wc_get_order') ? wc_get_order($order_id) : null;
        if ( ! $order ) return;

        $rules = self::get_rules();
        if ( empty($rules) ) return;

        // Leer estado actual del pedido (más robusto que pasar estado al programar)
        $current_status = $order->get_status(); // 'processing', 'completed', etc.

        // Construir IDs comprados (producto y variación)
        $purchased_ids = [];
        foreach ( $order->get_items() as $item ){
            $pid = (int) $item->get_product_id();
            $vid = (int) $item->get_variation_id();
            if ( $pid ) { $purchased_ids[] = $pid; }
            if ( $vid ) { $purchased_ids[] = $vid; } // ✅ sin append
        }
        $purchased_ids = array_map('absint', array_unique($purchased_ids));

        foreach ( $rules as $r ){
            if ( empty($r['enabled']) ) continue;

            $target_status = ! empty($r['trigger_status']) ? $r['trigger_status'] : 'processing';
            if ( $target_status !== $current_status ) continue;

            // Debe incluir al menos un producto disparador
            $triggers = array_map('absint', (array)($r['trigger_products'] ?? []));
            if ( count(array_intersect($purchased_ids, $triggers)) === 0 ) continue;

            // Pedido mínimo
            $min = isset($r['min_order']) ? (float)$r['min_order'] : 0.0;
            if ( $min > 0 && (float)$order->get_total() < $min ) continue;

            // Saltar si el pedido ya incluye algún producto destino
            if ( ! empty($r['skip_if_target_in_order']) ){
                $targets = array_map('absint', (array)($r['target_products'] ?? []));
                if ( count(array_intersect($purchased_ids, $targets)) > 0 ) continue;
            }

            // Crear cupón y enviar email
            $coupon_code = self::create_coupon($r, $order_id);
            if ( $coupon_code ){
                Mailer::send_coupon_email($order, $r, $coupon_code);
            }
        }
    }

     /**
     * Crea el cupón según la regla y lo devuelve
     */
    private static function create_coupon(array $r, $order_id = 0){
        if (!class_exists('\\WC_Coupon')) return '';

        // Prefijo configurable por regla, o PROMO como valor por defecto
        $prefix = ! empty( $r['coupon_prefix'] )
            ? sanitize_text_field( $r['coupon_prefix'] )
            : 'PROMO';

        // Código base: {PREFIJO}_{ORDERID}
        $code = $prefix . '_' . absint($order_id);

        // Evitar colisiones (duplicados por reintentos, etc.)
        // 🔹 Mantiene la misma lógica que ya tienes
        if ( function_exists('wc_get_coupon_id_by_code') && wc_get_coupon_id_by_code($code) ) {
            $code .= '_' . substr(md5(microtime()), 0, 4);
        }

        $coupon = new \WC_Coupon();
        $coupon->set_code($code);
        $coupon->set_discount_type($r['discount_type'] ?? 'percent');
        $coupon->set_amount(isset($r['discount_amount']) ? (float)$r['discount_amount'] : 0);
        $coupon->set_free_shipping(!empty($r['free_shipping']));
        $coupon->set_usage_limit(max(1, intval($r['usage_limit'] ?? 1)));

        $days = max(0, intval($r['expires_days'] ?? 0));
        if ($days > 0){
            try {
                // Fecha de expiración en la zona horaria de WP
                $tz  = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
                $exp = new \DateTime('now', $tz);
                $exp->modify('+' . $days . ' days');
                $coupon->set_date_expires( $exp );
            } catch ( \Exception $e ) {
                // En caso de fallo de TZ, usar timestamp relativo
                $coupon->set_date_expires( time() + ( $days * DAY_IN_SECONDS ) );
            }
        }

        $targets = array_map('absint', (array)($r['target_products'] ?? []));
        if (!empty($targets)){
            $coupon->set_product_ids($targets);
        }

        $coupon->save();
        return $coupon->get_code();
    }

    /**
     * (Opcional) Generador alternativo de códigos
     */
    private static function random_code(){
        $alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $code = '';
        for ($i=0; $i<10; $i++){
            $code .= $alphabet[random_int(0, strlen($alphabet)-1)];
        }
        return $code;
    }
}
