import { dispatch, select } from '@wordpress/data'
import { STORE_NAME as store } from './store/constants'
import { create as createRichTextValue } from '@wordpress/rich-text'
import { getBlockType } from '@wordpress/blocks'
import { __ } from '@wordpress/i18n'
import { store as noticesStore } from '@wordpress/notices'
import { getSettings } from '@wordpress/date';

const createHeaders = (token) => {
	let headers = {
		'Accept': 'application/json',
		'Content-Type': 'application/json',
	};

	if (token !== undefined) {
		headers['Authorization'] = `Bearer ${token}`;
	}

	return headers;
};

const buildUrl = (endpoint, path, params) => {
	const queryString = new URLSearchParams(params).toString();
	return `${endpoint}${path}?${queryString}`;
};

const parseResponse = (fetch, options = {}) => {
	const {setUsage, setDisplaySubscriptionModal} = dispatch(store);

	return fetch
		.then(response => {
			if (!response.ok) {
				throw response.status;
			}
			return response.json();
		})
		.then(data => {
			if (data.usage !== undefined) {
				setUsage(data.usage);
			}
			return data;
		})
		.catch(e => {
			const { getStatus } = select(store);
			const { getNotices } = select(noticesStore);
			const { createErrorNotice} = dispatch(noticesStore);

			const noticeId = 'creator-assistant--'+e.toString();

			const existingNotice = getNotices().find((notice) => notice.id === noticeId);

			if (!existingNotice) {
				let status = getStatus();
				if (!status) {
					status = {};
				}
				const { is_premium } = status;

				let message  = __('An error occurred while requesting the Creator Assistant API service.', 'creator-assistant');
				if (options.errorMessage) {
					message = options.errorMessage;
				}

				let actions = [];

				if (e === 401) {
					message = __(
						'There was an authentication problem with the Creator Assistant API service. Please deactivate and reactivate the plugin to resolve the issue.',
						'creator-assistant'
					);
				} else if (e === 429) {
					if (is_premium) {
						message = __('You have exhausted the requests available with the Creator Assistant Pro plan.', 'creator-assistant');
					} else {
						message = __('The Creator Assistant API service is receiving too many requests and a temporary limit has been activated on Lite plan accounts. Please try again later or upgrade to Pro plan.', 'creator-assistant');
						actions = [
							{
								label: __('Upgrade to Pro plan', 'creator-assistant'),
								onClick: () => {
									setDisplaySubscriptionModal(true)
								}
							},
						];
					}
				} else if (e === 402 && !is_premium) {
					message = __('You have exhausted the requests available with the Lite plan. If you want to continue using Creator Assistant, upgrade to the Pro plan.', 'creator-assistant');
					actions = [
						{
							label: __('Upgrade to Pro plan', 'creator-assistant'),
							onClick: () => {
								setDisplaySubscriptionModal(true)
							}
						},
					];
				} else if (e === 503) {
					message = __('The Creator Assistant API service is temporarily down. We apologize for the inconvenience!', 'creator-assistant');
				} else if (e === 410) {
					message = __('The Creator Assistant API service has been permanently disabled.', 'creator-assistant');
				}

				createErrorNotice(message, {
					id: noticeId,
					type: 'snackbar',
					isDismissible: true,
					actions
				});
			}

			// throw e;
		});
}

export const sendData = (auth, route, body, options = {}) => {
	const { getApiData } = select(store);
	const { access_token, endpoint, path, locale, plugin_version, wp_version } = getApiData();
	const defaultBody = { locale, plugin_version, wp_version };

	const url = buildUrl(endpoint, path + route);
	const headers = createHeaders(auth ? access_token : undefined);

	const responsePromise = fetch(url, {
		method: 'POST',
		headers,
		body: JSON.stringify({ ...defaultBody, ...body }),
	});
	return parseResponse(responsePromise, options);
};

export const fetchData = (auth, route, params, options = {}) => {
	const { getApiData } = select(store)
	const { access_token, endpoint, path, locale, plugin_version, wp_version } = getApiData()
	const defaultParams = {locale, plugin_version, wp_version}

	const url = buildUrl(endpoint, path+route, { ...defaultParams, ...params });
	const headers = createHeaders(auth ? access_token : undefined);
	const responsePromise = fetch(url, {headers});
	return parseResponse(responsePromise, options);
};

export const reduceInnerBlocks = (blocks) => {
	return blocks.reduce(
		(accumulator, block) => {
			accumulator.push(block);
			if (block.innerBlocks && block.innerBlocks.length > 0) {
				accumulator = accumulator.concat(reduceInnerBlocks(block.innerBlocks));
			}

			return accumulator;
		},
		[],
	);
}

export const getBlocksData = blocks => {
	const flatBlocks = reduceInnerBlocks(blocks);

	let blocksData = [];
	flatBlocks.forEach(block => {
		getAttributesNameByBlockName(block.name).forEach(attributeName => {
			const text = getPlainTextFromHtml(block.attributes[attributeName] ?? '').trim();
			if (text === '') {
				return;
			}

			blocksData.push({
				blockId:       block.clientId,
				attributeName: attributeName,
				text
			});
		});
	});

	return blocksData;
}

function getAttributesNameByBlockName(blockName) {
	const blockType = getBlockType(blockName);

	let attributesName = [];
	for (const attributeName in blockType.attributes ?? []) {
		const source = blockType.attributes[attributeName].source ?? '';
		if (['rich-text', 'html'].includes(source)) {
			attributesName.push(attributeName);
		}
	}

	return attributesName;
}

export function getPlainTextFromHtml(html) {
	let value = createRichTextValue({
		html
	});

	return value.text;
}

export function formatNumber(number) {
	const locale = getSettings().l10n.locale
	return new Intl.NumberFormat(locale.replaceAll('_', '-')).format(number);
}

/**
 * Ordina un array di oggetti in ordine alfabetico rispetto alla proprietà 'label'.
 *
 * @param {Array} array - L'array di oggetti da ordinare.
 * @return {Array} - L'array ordinato.
 */
export function sortByLabel (array) {
	return array.sort((a, b) => {
		if (a.label < b.label) {
			return -1
		}
		if (a.label > b.label) {
			return 1
		}
		return 0
	})
}
