import { createReduxStore, dispatch, register, select } from '@wordpress/data'
import { STORE_NAME } from './constants'
import apiFetch from '@wordpress/api-fetch'
import { fetchData } from '../utils'
import { __ } from '@wordpress/i18n'
import {store as noticesStore} from '@wordpress/notices';

const DEFAULT_STATE = {
	displaySubscriptionModal: false,
	displayBillingModal: false,
	displayTosModal: false,
	displayTutorialModal: false,
	apiData: {},
	billingData: {
		ready: false
	},
};

const actions = {
	async fetchApiData() {
		const { getNotices } = select(noticesStore);
		const {createErrorNotice} = dispatch(noticesStore);

		const noticeId = 'creator-assistant--init';

		const existingNotice = getNotices().find((notice) => notice.id === noticeId);

		let value;
		if (!existingNotice) {
			value = await apiFetch({ path: '/creator-assistant/v1/editor' }).catch(() => {
				createErrorNotice(__('An error occurred while initializing Creator Assistant.', 'creator-assistant'), {
					id: noticeId,
					type: 'snackbar',
					isDismissible: true,
				});
			});
		}

		/**
		 * The set must happen only after the fetch otherwise anyone can change important properties like is_premium or other
		 */
		return {
			type: 'SET_API_DATA',
			value
		};
	},
	async fetchStatus() {
		const status = await fetchData(true, '/status', {});
		if (status) {
			status.allowSubscription = (status.payment.enabled || status.subscription !== null) && status.payment.price !== null;
		}

		/**
		 * The set must happen only after the fetch otherwise anyone can change important properties like is_premium or other
		 */
		return {
			type: 'SET_STATUS',
			value: status,
		};
	},
	async fetchBillingData() {
		const billingData = await fetchData(true, '/billing', {}, {
			errorMessage: __(
				'An error occurred while requesting your billing information from the API service.',
				'creator-assistant'
			)
		});
		if (billingData) {
			billingData.ready = true;
		}


		/**
		 * The set must happen only after the fetch otherwise anyone can change important properties like is_premium or other
		 */
		return {
			type: 'SET_BILLING_DATA',
			value: billingData,
		};
	},
	setDisplaySubscriptionModal(value) {
		return {
			type: 'SET_DISPLAY_SUBSCRIPTION_MODAL',
			value,
		};
	},
	setDisplayBillingModal(value) {
		return {
			type: 'SET_DISPLAY_BILLING_MODAL',
			value,
		};
	},
	setDisplayTosModal(value) {
		return {
			type: 'SET_DISPLAY_TOS_MODAL',
			value,
		};
	},
	setDisplayTutorialModal(value) {
		return {
			type: 'SET_DISPLAY_TUTORIAL_MODAL',
			value,
		};
	},
	setUsage(value) {
		return {
			type: 'SET_USAGE',
			value,
		};
	},
};

const selectors = {
	getApiData(state) {
		return state.apiData;
	},
	getStatus(state) {
		return state.status;
	},
	getUsage(state) {
		return state.usage;
	},
	getDisplaySubscriptionModal(state) {
		return state.displaySubscriptionModal;
	},
	getDisplayBillingModal(state) {
		return state.displayBillingModal;
	},
	getDisplayTosModal(state) {
		return state.displayTosModal;
	},
	getDisplayTutorialModal(state) {
		return state.displayTutorialModal;
	},
	getBillingData(state) {
		return {
			geographical_area: 'italy',
			customer_type: 'company',
			...state.billingData
		};
	},
};

function reducer(state = DEFAULT_STATE, action) {
	switch (action.type) {
		case 'SET_API_DATA':
			if (!action.value) {
				return {
					...state
				};
			}

			return {
				...state,
				apiData: action.value,
			};
		case 'SET_STATUS':
			if (!action.value) {
				return {
					...state
				};
			}

			const status = action.value;
			const usage = status.usage;
			delete status.usage;

			return {
				...state,
				status,
				usage
			};
		case 'SET_DISPLAY_SUBSCRIPTION_MODAL':
			return {
				...state,
				displaySubscriptionModal: action.value,
			};
		case 'SET_DISPLAY_BILLING_MODAL':
			return {
				...state,
				displayBillingModal: action.value,
			};
		case 'SET_DISPLAY_TOS_MODAL':
			return {
				...state,
				displayTosModal: action.value,
			};
		case 'SET_DISPLAY_TUTORIAL_MODAL':
			return {
				...state,
				displayTutorialModal: action.value,
			};
		case 'SET_BILLING_DATA':
			if (!action.value) {
				return {
					...state
				};
			}

			return {
				...state,
				billingData: action.value,
			};
		case 'SET_USAGE':
			return {
				...state,
				usage: action.value,
			};
		default:
			return state;
	}
}

const store = createReduxStore( STORE_NAME, {
	reducer,
	selectors,
	actions,
} );

register( store );
