import { __, sprintf } from '@wordpress/i18n'
import {
	__experimentalSpacer as Spacer,
	Button, Flex, FlexItem,
	Modal,
} from '@wordpress/components'
import { paragraphs } from './Activation'
import { sendData } from '../utils'
import { useDispatch, useSelect } from '@wordpress/data'
import { STORE_NAME as store } from '../store/constants'
import {store as noticesStore} from '@wordpress/notices';
import { useState } from '@wordpress/element'
import { check } from '@wordpress/icons'

export default function TosModal () {

	const {createSuccessNotice} = useDispatch(noticesStore);
	const {fetchApiData, setDisplayTutorialModal, setDisplayTosModal} = useDispatch(store);
	const { apiData, display } = useSelect((select) => {
		const {getApiData, getDisplayTosModal} = select(store);
		return {
			apiData: getApiData(),
			display: getDisplayTosModal()
		};
	}, []);
	const {setup, is_active} = apiData;
	const buttonType = is_active ? 'close' : 'accept';
	const [isBusy, setIsBusy] = useState(false);

	if (!display || !setup) {
		return null
	}

	const {wp_admin_url, wp_setup_url, wp_setup_hash} = setup;

	const onClose = () => setDisplayTosModal( false );

	const accept = async () => {
		setIsBusy(true)
		const response = await sendData(false, '/register', {
			settings_url: wp_admin_url,
			callback_url: wp_setup_url,
			callback_hash: wp_setup_hash,
		}, {
			errorMessage: __('An error occurred while activating Creator Assistant.', 'creator-assistant')
		});

		if (response !== undefined) {
			let action = await fetchApiData();
			if (action.value !== undefined) {
				await createSuccessNotice(__('Creator Assistant has been activated.', 'creator-assistant'), {
					type: 'snackbar',
				});
				onClose();
				setDisplayTutorialModal(true);
			}
		}
		setIsBusy(false)
	}

	const translatedText = sprintf(
		__(
			'Review the %1$sTerms of Service%2$s and our %3$sPrivacy Policy%4$s.',
			'creator-assistant'
		),
		'<a href="https://www.creator-assistant.it/terms-of-service" target="_blank">',
		'</a>',
		'<a href="https://www.creator-assistant.it/privacy-policy" target="_blank">',
		'</a>'
	)

	return (
		<Modal title={__('Creator Assistant Terms of Service', 'creator-assistant')}
			   onRequestClose={onClose}
			   size={'medium'}
		>
			{paragraphs.map((paragraph) => (
				<p>{paragraph}</p>
			))}
			<Spacer marginBottom={8} marginTop={buttonType === 'accept' ? 8 : 0}>
				<p dangerouslySetInnerHTML={{ __html: translatedText }}/>
			</Spacer>
			<Flex justify={'end'}>
				<FlexItem>
					<Button isBusy={isBusy} disabled={isBusy} variant="primary"
							icon={buttonType === 'accept' ? check : null}
							onClick={async () => {
								if (buttonType === 'accept') {
									await accept();
								} else {
									onClose();
								}
							}}
					>
						{buttonType === 'accept' ? __('Accept', 'creator-assistant') : __('Close', 'creator-assistant')}
					</Button>
				</FlexItem>
			</Flex>
		</Modal>
	)
}
