import { __, sprintf } from '@wordpress/i18n'
import {
	__experimentalSpacer as Spacer,
	Button,
	Card,
	CardBody, CardFooter,
	CardHeader,
	Flex, FlexItem, Modal, Tooltip, Icon
} from '@wordpress/components'
import { useDispatch, useSelect } from '@wordpress/data'
import { STORE_NAME as store } from '../store/constants'
import { info } from '@wordpress/icons';
import { formatNumber } from '../utils'
import Billing from './Billing'
import { maxTextLength } from '../blocks/ai-writer/settings'

export const featureLabels = {
	'ideasForTheTitle': __('%s requests to generate ideas on topics (AI Writer)', 'creator-assistant'),
	'createIndex': __('%s content generation (AI Writer)', 'creator-assistant'),
	'editText': __('%s changes via prompt', 'creator-assistant'),
	'summarizeText': __('%s text summaries', 'creator-assistant'),
	'suggestImprovements': __('%s suggestions for improvement', 'creator-assistant'),
	'input_credits': __('%s input credits', 'creator-assistant'),
	'output_credits': __('%s output credits', 'creator-assistant'),
};

export const featureTooltips = {
	'input_credits': __('Each request uses credits based on text length and processing complexity. Input credits cover processing the submitted text and instructions, while output credits apply to the length of the generated response.', 'creator-assistant'),
	'output_credits': __('Each request uses credits based on text length and processing complexity. Input credits cover processing the submitted text and instructions, while output credits apply to the length of the generated response.', 'creator-assistant'),
};

function FeatureList (props) {
	const {plan, otherFeatures} = props;

	return (
		<ul className={'creator-assistant--features'}>
			{Object.keys(featureLabels).map(function (key) {
				const number = formatNumber(plan[key]);
				const asterisk = <sup>*</sup>;
				return (
					<li>
						{featureTooltips[key] !== undefined && (
							<Tooltip text={featureTooltips[key]}><span><span>{sprintf(featureLabels[key], number)}{asterisk}</span> <Icon
								icon={info}/></span></Tooltip>
						)}
						{featureTooltips[key] === undefined && (
							<>
								{sprintf(featureLabels[key], number)}{asterisk}
							</>
						)}
					</li>
				)
			})}
			{otherFeatures.map(item => <li>{item}</li>)}
		</ul>
	);
}

export default function Subscription () {
	const { setDisplaySubscriptionModal, setDisplayBillingModal } = useDispatch(store)
	const { status, displaySubscriptionModal } = useSelect((select) => {
		const { getStatus, getDisplaySubscriptionModal } = select(store)
		return {
			status: getStatus(),
			displaySubscriptionModal: getDisplaySubscriptionModal(),
		}
	}, [])

	if (!status) {
		return null
	}
	const { is_premium, have_billing_data, payment, settings, allowSubscription, portal_url } = status
	const { usage_limits, words_limit } = settings;
	const {lite: maxTextLengthForLitePlan} = words_limit;
	const {price, url} = payment;
	let buttonUrl;
	let buttonTarget;
	if (is_premium) {
		buttonUrl = portal_url;
		buttonTarget = '_blank';
	} else {
		buttonUrl = url;
		buttonTarget = false;
	}

	const buttonLabel = is_premium ? __('Manage Subscription', 'creator-assistant') : __('Subscribe Now', 'creator-assistant');

	const openBillingModal = () => {
		setDisplaySubscriptionModal(false);
		setDisplayBillingModal(true);
	}

	const close = () => setDisplaySubscriptionModal(false);
	const {lite, pro} = usage_limits;

	return (
		<>
			{displaySubscriptionModal && (
				<Modal title={__('Creator Assistant: Plans', 'creator-assistant')}
					   onRequestClose={close}
					   size={allowSubscription ? 'fill' : 'medium'}
				>
					<div className={'creator-assistant--modal-container'}>
						<Spacer marginBottom={8} marginTop={0}>
							<div className={'creator-assistant--price-cards'+' '+(allowSubscription ? 'allow-subscription' : '')}>
								<Card className="price-card">
									<CardHeader>
										<h3>{__('Lite', 'creator-assistant')}</h3>
									</CardHeader>
									<CardBody>
										<h2>
											{sprintf(__('%s € / month', 'creator-assistant'), formatNumber(0))}
											<span className={'secondary-text'}>{__('VAT not applicable', 'creator-assistant')}</span>
										</h2>

										<FeatureList plan={lite} otherFeatures={[sprintf(__('Content generation (AI Writer) limited to %s words', 'creator-assistant'), formatNumber(maxTextLengthForLitePlan))]}/>
									</CardBody>
									<CardFooter>
										{!is_premium && <Button variant={'primary'} disabled>{__('Your Plan', 'creator-assistant')}</Button>}
									</CardFooter>
								</Card>
								{allowSubscription && <Card className="price-card">
									<CardHeader>
										<h3>{__('Pro', 'creator-assistant')}</h3>
									</CardHeader>
									<CardBody>
										<h2>
											{sprintf(__('%s € / month', 'creator-assistant'), formatNumber(price))}
											<span className={'secondary-text'}>{__('VAT not applicable', 'creator-assistant')}</span>
										</h2>
										<FeatureList plan={pro} otherFeatures={[sprintf(__('Content generation (AI Writer) up to %s words', 'creator-assistant'), formatNumber(maxTextLength))]}/>
									</CardBody>
									<CardFooter>
										<div style={{textAlign: 'center'}}>
											<div>
												{!have_billing_data && <Button variant={'primary'} onClick={openBillingModal}>{__( 'Set your billing information', 'creator-assistant' )}</Button>}
												{have_billing_data && <Button variant={'primary'} href={buttonUrl} target={buttonTarget}>{buttonLabel}</Button>}
											</div>
											<div className={'secondary-text'} style={{marginTop: '1em'}}>{__('Active immediately after payment', 'creator-assistant')}</div>
											<div className={'secondary-text'}>{__('Automatic renewal', 'creator-assistant')}</div>
											<div className={'secondary-text'}>{__('Cancel anytime without penalty', 'creator-assistant')}</div>
											<div className={'secondary-text'}>
												<a href={'https://www.creator-assistant.it/terms-of-service#withdrawal'} target={'_blank'}>{__('Right of withdrawal', 'creator-assistant')}</a>
											</div>
										</div>
									</CardFooter>
								</Card>}
							</div>
							<div>
								<p><sup>*</sup> {__('Requests not cumulative with monthly renewal.', 'creator-assistant')}</p>
							</div>
						</Spacer>
						<Flex justify={'end'} gap={4}>
							{allowSubscription && have_billing_data && <FlexItem>
								<Button variant={'secondary'}
										onClick={openBillingModal}>{__('Update your billing information', 'creator-assistant' )}</Button>
							</FlexItem>}
							<FlexItem>
								<Button variant="secondary" onClick={() => {
									close();
								}}>
									{__('Close', 'creator-assistant')}
								</Button>
							</FlexItem>
						</Flex>
					</div>
				</Modal>
			)}
			<Billing/>
		</>
	)
}
