import { useState } from '@wordpress/element'
import { displayShortcut } from '@wordpress/keycodes'
import { __ } from '@wordpress/i18n'
import {
	__experimentalSpacer as Spacer,
	Button, DropdownMenu,
	Flex,
	FlexItem,
	KeyboardShortcuts, MenuItem,
	Popover, Spinner,
	TextareaControl,
	ToolbarButton,
	ToolbarGroup
} from '@wordpress/components'
import { create as createRichTextValue, registerFormatType } from '@wordpress/rich-text'
import { BlockControls } from '@wordpress/block-editor'
import icon, {commentText, loading, domain} from './Icon'
import { sendData, sortByLabel } from '../utils'
import { useSelect } from '@wordpress/data'
import { STORE_NAME as store } from '../store/constants'

const RichTextPrompt = ( props ) => {
	// console.log( 'displayShortcut', displayShortcut );
	const shortcutsChar = 'e';

	const {value, onChange} = props;
	const [ isVisible, setIsVisible ] = useState( false );
	const [ isBusy, setIsBusy ] = useState( false );
	const [ prompt, setPrompt ] = useState( '' );
	const [ promptType, setPromptType ] = useState( 'custom' );
	const toggleVisible = () => {
		setIsVisible( ( state ) => ! state );
	};

	const shortcutLabel = displayShortcut[ 'primary' ]( shortcutsChar );
	let shortcuts = {};
	shortcuts['mod+' + shortcutsChar] = toggleVisible;

	const useAllText = value.start === value.end;
	const promptLabel = useAllText ? __('Edit the text according to your instructions:', 'creator-assistant') : __('Edit the selected text according to your instructions:', 'creator-assistant');

	const textModificationOptions = [
		{ value: 'elaborate', label: __('Elaborate on current topic', 'creator-assistant'), description: __('Type a short sentence to express a concept in the text field, then click this button and let Creator Assistant do the rest.', 'creator-assistant') },
		{ value: 'rewrite', label: __('Rewrite more clearly', 'creator-assistant') },
		{ value: 'synonyms', label: __('Replace with a synonym', 'creator-assistant'), isDisabled: useAllText },
		{ value: 'formal', label: __('Make more formal', 'creator-assistant') },
		{ value: 'friendly', label: __('Make friendlier', 'creator-assistant') },
		{ value: 'reduce', label: __('Reduce while retaining meaning', 'creator-assistant') },
		{ value: 'expand', label: __('Expand with details', 'creator-assistant') },
		{ value: 'simplify', label: __('Simplify the language', 'creator-assistant') },
		{ value: 'specify', label: __('Use technical language', 'creator-assistant') },
		{ value: 'seo', label: __('Optimize for SEO', 'creator-assistant') }
	];

	const translateLanguages = [
		{ value: 'en', label: __('English', 'creator-assistant') },
		{ value: 'zh', label: __('Chinese', 'creator-assistant') },
		{ value: 'es', label: __('Spanish', 'creator-assistant') },
		{ value: 'hi', label: __('Hindi', 'creator-assistant') },
		{ value: 'ar', label: __('Arabic', 'creator-assistant') },
		{ value: 'fr', label: __('French', 'creator-assistant') },
		{ value: 'ru', label: __('Russian', 'creator-assistant') },
		{ value: 'pt', label: __('Portuguese', 'creator-assistant') },
		{ value: 'bn', label: __('Bengali', 'creator-assistant') },
		{ value: 'de', label: __('German', 'creator-assistant') },
		{ value: 'ja', label: __('Japanese', 'creator-assistant') },
		{ value: 'ko', label: __('Korean', 'creator-assistant') },
		{ value: 'it', label: __('Italian', 'creator-assistant') },
		{ value: 'tr', label: __('Turkish', 'creator-assistant') },
		{ value: 'fa', label: __('Persian', 'creator-assistant') },
		{ value: 'ur', label: __('Urdu', 'creator-assistant') },
		{ value: 'vi', label: __('Vietnamese', 'creator-assistant') },
		{ value: 'ta', label: __('Tamil', 'creator-assistant') },
		{ value: 'pl', label: __('Polish', 'creator-assistant') },
		{ value: 'uk', label: __('Ukrainian', 'creator-assistant') }
	];

	const apiData = useSelect((select) => {
		const {getApiData} = select(store);
		return getApiData();
	}, []);
	const { is_active } = apiData;
	if (!is_active) {
		return null;
	}

	const send = async (prompt, promptType) => {
		if (promptType === 'custom' && prompt === '') {
			return;
		}

		setIsBusy(true);
		const data = await sendData(true, '/edit-text', {
			prompt,
			promptType,
			text: value.text,
			start: value.start,
			end: value.end
		});

		if (data) {
			let newText = createRichTextValue({
				text: data.text,
			});
			onChange(newText);
		}
		setIsBusy(false);
	}

	const actionDropdownMenuIcon = !['custom', 'translate'].includes(promptType) && isBusy ? loading : icon;
	const translateDropdownMenuIcon = promptType === 'translate' && isBusy ? loading : domain;

	return (
		<>
			<KeyboardShortcuts
				shortcuts={ shortcuts }
				bindGlobal
			/>
			<BlockControls>
				<ToolbarGroup>
					<DropdownMenu
						onToggle={() => {setIsVisible(false)}}
						icon={ translateDropdownMenuIcon }
						label={__('Translate with Creator Assistant', 'creator-assistant')}
						controls={
							sortByLabel(translateLanguages).map(option => {
								return {
									title: option.label,
									onClick: async () => {
										setPromptType('translate');
										await send(option.value, 'translate');
									},
								}
							})
						}
					/>
					<DropdownMenu
						onToggle={() => {setIsVisible(false)}}
						icon={ actionDropdownMenuIcon }
						label={__('Edit with Creator Assistant', 'creator-assistant')}
					>
						{ ( { onClose } ) => (
							<>
								{
									textModificationOptions.map(option => {
										return (
											<MenuItem
												info={option.description ?? ''}
												disabled={ option.isDisabled ?? !useAllText }
												onClick={ async () => {
													onClose();
													setPromptType(option.value);
													await send('', option.value);
												}}
											>
												{option.label}
											</MenuItem>
										);
									})
								}
							</>
						) }
					</DropdownMenu>
					<ToolbarButton
						shortcut={shortcutLabel}
						isActive={ isVisible }
						icon={commentText}
						title={__('Creator Assistant Prompt', 'creator-assistant')}
						onClick={ () => {
							toggleVisible();
						} }
					/>
				</ToolbarGroup>
			</BlockControls>
			{ isVisible && (
				<Popover
					anchor={ props.contentRef.current }
					className={'creator-assistant--popover-prompt'}
				>
					<Spacer padding={4} style={{minWidth: '350px'}}>
						<TextareaControl label={promptLabel} rows={3} onChange={value => setPrompt( value ) } help={__("Want to edit the text, but the available options aren't enough? Write here what you'd like to change. Tell Creator Assistant how you would do it with a person, and it'll take care of everything!", 'creator-assistant')} />
						<Flex justify={'end'} style={{marginTop: '20px'}}>
							{promptType === 'custom' && isBusy && <FlexItem>
								<Spinner/>
							</FlexItem>}
							<FlexItem>
								<Button
									isBusy={promptType === 'custom' && isBusy}
									variant={'primary'}
										text={__('Send', 'creator-assistant')}
										onClick={async () => {
											setPromptType('custom');
											await send(prompt, 'custom');
										}}
								/>
							</FlexItem>
						</Flex>
					</Spacer>
				</Popover>
			) }
		</>
	);
};

export default () => {
	registerFormatType( 'creator-assistant/prompt', {
		title: __('Creator Assistant Prompt', 'av_languages'),
		tagName: 'creator-assistant',
		className: null,
		edit: RichTextPrompt,
	} );
};
