import { __ } from '@wordpress/i18n'
import {
	__experimentalSpacer as Spacer,
	Button,
	Flex, FlexItem, Modal, Notice, RadioControl, TextControl,
	__experimentalGrid as Grid,
	Spinner
} from '@wordpress/components'
import { useDispatch, useSelect } from '@wordpress/data'
import { STORE_NAME as store } from '../store/constants'
import { sendData } from '../utils'
import { useEffect, useState } from '@wordpress/element'
import {store as noticesStore} from '@wordpress/notices';

export default function Billing () {
	const [isBusy, setIsBusy] = useState(false);
	const {setDisplayBillingModal, setDisplaySubscriptionModal, fetchBillingData, fetchStatus} = useDispatch(store);
	const {createSuccessNotice} = useDispatch(noticesStore);
	const { apiData, status, displayBillingModal, remoteBillingData } = useSelect((select) => {
		const {getStatus, getDisplayBillingModal, getBillingData, getApiData} = select(store);
		return {
			apiData: getApiData(),
			status: getStatus(),
			displayBillingModal: getDisplayBillingModal(),
			remoteBillingData: getBillingData()
		};
	}, []);
	const { is_active, locale } = apiData;

	useEffect(() => {
		if (is_active) {
			fetchBillingData()
		}
	}, [is_active])

	const [billingData, setBillingData] = useState({});
	const {ready} = remoteBillingData;
	useEffect(() => {
		if (ready) {
			setBillingData({...remoteBillingData});
		}
	}, [ready])

	const {customer_type, geographical_area} = billingData;

	if (!status) {
		return null;
	}
	const {portal_url, have_billing_address} = status;

	const close = () => {
		setDisplayBillingModal(false);
		setDisplaySubscriptionModal(true);
	};
	const save = async () => {
		setIsBusy(true);
		const response = await sendData(true, '/billing', {
			locale,
			metadata: billingData,
			labels: {
				tax_code: __( 'Tax Code', 'creator-assistant' ),
				legal_business_name: __( 'Legal Business Name', 'creator-assistant' ),
				vat_number: __( 'VAT Number', 'creator-assistant' ),
				sdi: __( 'SDI or PEC', 'creator-assistant' )
			}
		}, {
			errorMessage: __('An error occurred while saving.', 'creator-assistant')
		});

		if (response !== undefined) {
			await fetchBillingData();
			await fetchStatus();
			await createSuccessNotice(__('Saved data.', 'creator-assistant'), {
				type: 'snackbar',
			});
			close();
		}
		setIsBusy(false);
	};

	return (
		<>
			{displayBillingModal && (
				<Modal title={__('Creator Assistant: Billing Information', 'creator-assistant')}
					   onRequestClose={close}
					   size={'medium'}
				>
					<Spacer marginBottom={8} marginTop={0}>
						{!ready && <>
							<Flex align="center" justify="center">
								<FlexItem>
									<Spacer marginTop={8}>
										<Spinner/>
									</Spacer>
								</FlexItem>
							</Flex>
						</>}
						{ready && <>
							{have_billing_address && <>
								<Notice status="info" isDismissible={false}>
									<p>{__(
										'By clicking the button below, you will be taken directly to the Stripe portal. There, you can easily manage your subscriptions, update payment methods, edit billing details, and view your invoices. Everything is just a click away!',
										'creator-assistant'
									)}</p>
									<Button variant={'secondary'} href={portal_url} target={'_blank'}>{__( 'Go to the Stripe portal', 'creator-assistant' )}</Button>
								</Notice>
							</>}
							<p>{__( 'To correctly generate your invoice we need some additional information.', 'creator-assistant' )}</p>
							<Grid columns={1} gap={5}>
								<RadioControl
									label={__( 'What geographical area are you in?', 'creator-assistant' )}
									selected={ billingData.geographical_area }
									options={ [
										{ label: __( 'Italy', 'creator-assistant' ), value: 'italy' },
										{ label: __( 'Other', 'creator-assistant' ), value: 'other' },
									] }
									onChange={ ( value ) => setBillingData( {
										...billingData,
										geographical_area: value
									} ) }
								/>
								<RadioControl
									label={__( 'What type of customer are you?', 'creator-assistant' )}
									selected={ billingData.customer_type }
									options={ [
										{ label: __( 'Personal', 'creator-assistant' ), value: 'individual' },
										{ label: __( 'Company', 'creator-assistant' ), value: 'company' },
									] }
									onChange={ ( value ) => setBillingData( {
										...billingData,
										customer_type: value
									} ) }
								/>
								{'individual' === customer_type && <TextControl
									label={__( 'Tax Code', 'creator-assistant' )}
									value={ billingData.tax_code ?? '' }
									onChange={ ( value ) => setBillingData( {
										...billingData,
										tax_code: value
									} ) }
								/>}
								{'company' === customer_type && <>
									<TextControl
										label={__( 'Legal Business Name', 'creator-assistant' )}
										value={ billingData.legal_business_name ?? '' }
										onChange={ ( value ) => setBillingData( {
											...billingData,
											legal_business_name: value
										} ) }
									/>
									<TextControl
										label={__( 'VAT Number', 'creator-assistant' )}
										value={ billingData.vat_number ?? '' }
										onChange={ ( value ) => setBillingData( {
											...billingData,
											vat_number: value
										} ) }
									/>
									{'italy' === geographical_area && <TextControl
										label={__( 'SDI or PEC', 'creator-assistant' )}
										value={ billingData.sdi ?? '' }
										onChange={ ( value ) => setBillingData( {
											...billingData,
											sdi: value
										} ) }
									/>}
								</>}
							</Grid>
						</>}
					</Spacer>
					<Flex justify={'end'} gap={4}>
						<FlexItem>
							<Button variant="secondary" onClick={() => {
								close();
							}}>
								{__('Close', 'creator-assistant')}
							</Button>
						</FlexItem>
						<FlexItem>
							<Button isBusy={isBusy} disabled={isBusy || !ready} variant={'primary'} onClick={save}>{__( 'Save', 'creator-assistant' )}</Button>
						</FlexItem>
					</Flex>
				</Modal>
			)}
		</>
	)
}
