import {registerBlockType, createBlock, getBlockDefaultClassName} from "@wordpress/blocks";
import {useBlockProps, useInnerBlocksProps} from '@wordpress/block-editor';
import {__} from "@wordpress/i18n";
import {
	SelectControl,
	TextControl,
	RadioControl,
	__experimentalGrid as Grid,
	Button,
	BaseControl,
	RangeControl,
	Flex,
	FlexItem,
	ProgressBar,
	__experimentalSpacer as Spacer, Spinner
} from '@wordpress/components'
import { renderToString, useEffect, useState } from '@wordpress/element'
import { useDispatch, useSelect } from '@wordpress/data'
import icon from '../../components/Icon'
import { cleanForSlug } from '@wordpress/url';
import {pencil} from '@wordpress/icons';

import cloneDeep from 'lodash/cloneDeep';
import metadata from "./block.json";
import "./index.scss";
import {
	writingTones,
	textLengthMarks,
	audienceAgeGroups,
	audienceGeographyGroups,
	minTextLength,
	maxTextLength, contentTypes, satisfactionTones, languageComplexityOptions, languageComplexityHelpOptions
} from "./settings";
import { formatNumber, sendData } from '../../utils'
import PhraseRotator from '../../components/PhraseRotator'
import { STORE_NAME as store } from '../../store/constants'
import { store as noticesStore } from '@wordpress/notices'

const {name} = metadata;

registerBlockType( name, {
	...metadata,
	title: __('AI Writer', 'creator-assistant'),
	description: __('Use the intelligence of Creator Assistant to create new content in minutes.', 'creator-assistant'),
	keywords: [
		__('AI', 'creator-assistant'),
		__('text', 'creator-assistant'),
		__('automation', 'creator-assistant'),
		__('content', 'creator-assistant'),
		__('writer', 'creator-assistant')
	],
	icon: icon,
	edit: ({attributes, setAttributes, clientId}) => {
		const blockProps = useBlockProps();
		const innerBlocksProps = useInnerBlocksProps();

		const [isBusy, setIsBusy] = useState('');
		const [titleIdeas, setTitleIdeas] = useState([]);
		const [maxTextLengthForPlan, setMaxTextLengthForPlan] = useState(maxTextLength);

		const {replaceInnerBlocks} = useDispatch('core/block-editor');
		const {editPost} = useDispatch('core/editor');
		const {createWarningNotice, createInfoNotice} = useDispatch(noticesStore);
		const {setDisplaySubscriptionModal} = useDispatch(store);

		const { status } = useSelect((select) => {
			const {getStatus} = select(store);
			return {
				status: getStatus(),
			};
		}, []);

		useEffect(() => {
			if (!status) {
				return;
			}

			const { is_premium, settings } = status
			const { words_limit, usage_limits } = settings
			const { enabled: usage_limits_enabled } = usage_limits
			const { lite: maxTextLengthForLitePlan } = words_limit
			const newValue = is_premium || !usage_limits_enabled ? maxTextLength : Math.min(maxTextLength, maxTextLengthForLitePlan)
			setMaxTextLengthForPlan(newValue)
			if (attributes.textLength > newValue) {
				setAttributes({ textLength: newValue })
			}
		}, [status])

		if (!status) {
			return (<Flex align="center" justify="center">
				<FlexItem>
					<Spacer marginY={8}>
						<Spinner/>
					</Spacer>
				</FlexItem>
			</Flex>);
		}

		const generateContent = async () => {
			if (attributes.argument === '') {
				await displayArgumentWarning();

				return;
			}

			if (attributes.textLength > maxTextLengthForPlan) {
				setAttributes({ textLength: maxTextLengthForPlan })

				const message = sprintf(__('The Lite plan allows up to %s words. Upgrade to Pro for articles up to %s words!', 'creator-assistant'), formatNumber(maxTextLengthForPlan), formatNumber(maxTextLength))
				await createInfoNotice(message, {
					type: 'snackbar',
					isDismissible: true,
					actions: [
						{
							label: __('Upgrade to Pro plan', 'creator-assistant'),
							onClick: () => {
								setDisplaySubscriptionModal(true)
							}
						},
					]
				})
			}

			setIsBusy('content')
			replaceInnerBlocks(clientId, [])
			const body = {
				...attributes,
				textLength: Math.min(attributes.textLength, maxTextLengthForPlan),
			}

			const data = await sendData(true, '/create-index', body);
			if (data) {
				const items = data.items;
				const blocks = await Promise.all(
					items.map(async (item) => {
						const sectionTitle = item.content;
						const attributes = { content: sectionTitle, level: item.level, anchor: cleanForSlug(sectionTitle) };
						const block = createBlock('core/' + item.type, attributes);

						const sectionBlocks = await generateSection(sectionTitle, item.sectionLength);

						return [block, ...sectionBlocks];
					})
				);

				const listItems = items.map((item) => {
					const sectionTitle = item.content;
					const attributes = { content: renderToString(<a href={'#'+cleanForSlug(sectionTitle)}>{sectionTitle}</a>)};
					return createBlock('core/list-item', attributes);
				});

				if (listItems.length > 0) {
					blocks.unshift(createBlock('core/list', {}, listItems));
				}

				const flatBlocks = blocks.flat();
				replaceInnerBlocks(clientId, flatBlocks);
			}
			setIsBusy('');
		}

		const generateSection = (sectionTitle, sectionLength) => {
			const params = { ...attributes, sectionTitle, sectionLength };
			return new Promise(async (resolve) => {
				let blocks = [];
				const data = await sendData(true, '/create-text', params);
				if (data) {
					const items = data.items;
					const contentBlocks = items.filter(item => item.type === 'paragraph');
					blocks = contentBlocks.map(item => {
						const attributes = { content: item.content };
						return createBlock('core/' + item.type, attributes);
					});
				}

				resolve(blocks);
			});
		};

		const displayArgumentWarning = async () => {
			const message = __('First fill in the field:', 'creator-assistant') + ' "' + __('What do you want to talk about in the article?', 'creator-assistant') + '"'
			await createWarningNotice(message, {
				type: 'snackbar',
			});
		}

		return (
			<div {...blockProps}>
				<Grid columns={1} gap={5}>
					<SelectControl
						label={__('What type of content do you need assistance with?', 'creator-assistant')}
						onChange={(contentType) => {
							setAttributes({
								contentType,
								satisfactionTone: attributes.contentType === 'product-review' ? attributes.satisfactionTone : 'neutral'
							})
						}}
						value={attributes.contentType}
						options={contentTypes}
					/>
					<Flex align={'center'} className={'argument-row'}>
						<FlexItem style={{ flex: '1' }}>
							<TextControl
								className={'argument-input'}
								label={__('What do you want to talk about in the article?', 'creator-assistant')}
								help={__('Enter the topic or a draft idea for your article', 'creator-assistant')}
								value={attributes.argument}
								onChange={(argument) => setAttributes({argument})}
							/>
						</FlexItem>
						<FlexItem>
							<Button variant={'secondary'} text={__('Set as title', 'creator-assistant')} onClick={() => {
								editPost({title: attributes.argument});
							}}/>
						</FlexItem>
					</Flex>
					{attributes.contentType !== 'faq-responses' && (
						<>
							<BaseControl
								label={__('Would you like suggestions on related topics to cover?', 'creator-assistant')}
								className={'flex-space-between'}
							>
								<Button isBusy={isBusy === 'title'} disabled={isBusy !== ''} variant={'secondary'}
										text={__('Give me some ideas', 'creator-assistant')}
										onClick={async () => {
											if (attributes.argument === '') {
												await displayArgumentWarning();

												return;
											}

											const params = {
												argument: attributes.argument,
												contentType: attributes.contentType,
											};

											setIsBusy('title');
											const data = await sendData(true, '/ideas-for-the-title', params);
											if (data) {
												let items = cloneDeep(data.items);
												items.unshift(attributes.argument);
												setTitleIdeas(items);
											}
											setIsBusy('');
										}}
								/>
							</BaseControl>
							{titleIdeas.length > 0 && <RadioControl
								label={__('Which suggestion interests you the most?', 'creator-assistant')}
								selected={attributes.argument}
								options={titleIdeas.map(titleIdea => {
									return {
										label: titleIdea,
										value: titleIdea
									};
								})}
								onChange={(argument) => setAttributes({argument})}
							/>}
						</>
					)}
					<SelectControl
						label={__('What tone do you want to give the article?', 'creator-assistant')}
						onChange={(writingTone) => setAttributes({writingTone})}
						value={attributes.writingTone}
						options={writingTones}
					/>
					{attributes.contentType === 'product-review' && <SelectControl
						label={__('What is the tone of satisfaction for your review?', 'creator-assistant')}
						onChange={(satisfactionTone) => setAttributes({satisfactionTone})}
						value={attributes.satisfactionTone}
						options={satisfactionTones}
					/>}
					<RangeControl
						label={__('How detailed do you want the article to be?', 'creator-assistant')}
						help={__('Text length in words', 'creator-assistant')}
						marks={textLengthMarks}
						value={attributes.textLength}
						onChange={(textLength) => setAttributes({textLength})}
						min={minTextLength}
						max={maxTextLength}
						step={100}
					/>
					<RadioControl
						label={__('What type of language do you prefer for your article?', 'creator-assistant')}
						selected={attributes.languageComplexity}
						options={languageComplexityOptions}
						help={languageComplexityHelpOptions[attributes.languageComplexity]}
						onChange={(languageComplexity) => setAttributes({languageComplexity})}
					/>
					<SelectControl
						label={__('What age group is your article targeting?', 'creator-assistant')}
						onChange={(audienceAge) => setAttributes({audienceAge})}
						value={attributes.audienceAge}
						options={audienceAgeGroups}
					/>
					<SelectControl
						label={__('What is the geographical or cultural context of your audience?', 'creator-assistant')}
						onChange={(audienceGeo) => setAttributes({audienceGeo})}
						value={attributes.audienceGeo}
						options={audienceGeographyGroups}
					/>
					{attributes.audienceGeo === 'custom' &&
						<TextControl
							label={__('Custom Audience Geography', 'creator-assistant')}
							value={attributes.customAudienceGeo}
							onChange={(customAudienceGeo) => setAttributes({customAudienceGeo})}
						/>
					}
					<Button isBusy={isBusy === 'content'} disabled={isBusy !== ''} variant={'primary'}
							icon={pencil}
							text={__('Generate content', 'creator-assistant')}
							onClick={() => generateContent()}
					/>
				</Grid>
				{isBusy === 'content' && (
					<Spacer marginBottom={5} marginTop={5}>
						<Flex
							direction="column"
							align="center"
							justify="center"
						>
							<PhraseRotator/>
							<ProgressBar />
						</Flex>
					</Spacer>
				)}
				<div {...blockProps}>
					<div {...innerBlocksProps} />
				</div>
			</div>
		);
	},
	save: () => {
		const blockProps = useBlockProps.save();
		const classNamePrefix = getBlockDefaultClassName(metadata.name);
		const innerBlocksProps = useInnerBlocksProps.save({className: classNamePrefix+'__inner-blocks'});

		return (
			<div {...blockProps}>
				<div {...innerBlocksProps} />
			</div>
		);
	},
});
