<?php
/**
 * Plugin.
 *
 * @package CreatorAssistant
 */

declare( strict_types=1 );

namespace CreatorAssistant;

use Closure;
use RuntimeException;
use function get_plugin_data;

/**
 * Class Plugin manages the plugin update mechanism by hooking into specific actions
 * and determining if the plugin should be automatically updated.
 */
class Plugin {
	const NOTICES_TRANSIENT_NAME = 'creator_assistant_notices';
	/**
	 * Main plugin file path.
	 *
	 * @var string $plugin_file_path
	 */
	private static string $plugin_file_path;

	/**
	 * Plugin data.
	 *
	 * @var array|null $plugin_data
	 */
	private static ?array $plugin_data;

	/**
	 * Retrieves the file path of the plugin.
	 *
	 * @return string Returns a string containing the plugin's file path.
	 */
	public static function get_plugin_file_path(): string {
		return self::$plugin_file_path;
	}

	/**
	 * Initializes the plugin by hooking the auto-update functionality into WordPress.
	 *
	 * Adds an action for automatically updating plugins using a callback method from this class.
	 *
	 * @return void
	 */
	public static function init(): void {
		self::$plugin_file_path = plugin_dir_path( __FILE__ ) . 'creator-assistant.php';

		add_action( 'init', Closure::fromCallable( array( __CLASS__, 'load_plugin_textdomain' ) ) );
		add_action( 'admin_init', Closure::fromCallable( array( __CLASS__, 'set_plugin_data' ) ) );
		add_action( 'admin_notices', Closure::fromCallable( array( __CLASS__, 'print_notices' ) ) );

		register_deactivation_hook(
			self::get_plugin_file_path(),
			Closure::fromCallable( array( __CLASS__, 'uninstall_plugin' ) )
		);
	}

	/**
	 * Uninstalls the plugin by deleting the associated transient data.
	 *
	 * @return void
	 */
	private static function uninstall_plugin(): void {
		delete_transient( self::NOTICES_TRANSIENT_NAME );
	}

	/**
	 * Prints various admin notices based on certain conditions.
	 *
	 * This includes notices for checkout actions, access token detection, and other pre-defined notices.
	 *
	 * @return void
	 */
	private static function print_notices(): void {
		if ( isset( $_GET['creator_assistant'] ) ) {
			$action = sanitize_text_field( wp_unslash( $_GET['creator_assistant'] ) );
			if ( in_array( $action, array( 'checkout_success', 'checkout_cancel' ), true ) ) {
				if ( 'checkout_success' === $action ) {
					$type    = 'success';
					$message = sprintf( '<strong>%s</strong><br>%s', __( 'Thank you for subscribing to Creator Assistant!', 'creator-assistant' ), __( 'Your payment has been completed and you can now enjoy all the features the plugin has to offer. Happy working!', 'creator-assistant' ) );
				} else {
					$type = 'info';
					// translators: Checkout abort.
					$message = __( 'The %1$sCreator Assistant%2$s subscription purchase has been interrupted as requested. No problem! You can resume payment at any time directly from the sidebar in the content editor. See you soon!', 'creator-assistant' );
					$message = sprintf( $message, '<strong>', '</strong>' );
				}

				if ( ! empty( $message ) ) {
					wp_admin_notice(
						$message,
						array(
							'type'           => $type,
							'paragraph_wrap' => true,
						)
					);
				}
			}
		}

		$current_screen = get_current_screen();
		if ( ! in_array( $current_screen->id, array( 'plugins', 'dashboard' ), true ) ) {
			return;
		}

		if ( ! Api_Handler::have_access_token() && current_user_can( 'edit_posts' ) ) {
			// translators: Welcome message.
			$message = __( 'To start using all the features of %1$sCreator Assistant%2$s, we kindly ask you to accept our terms and conditions. You can do this by visiting the editing page of any content, such as a post or a page. Thank you for your cooperation and happy creating!', 'creator-assistant' );
			wp_admin_notice(
				sprintf( $message, '<strong>', '</strong>' ),
				array(
					'type'           => 'info',
					'paragraph_wrap' => true,
				)
			);

			return;
		}

		$notices = self::get_notices();
		if ( empty( $notices ) ) {
			return;
		}

		foreach ( $notices as $notice_id => $notice ) {
			$lang_code = substr( get_user_locale(), 0, 2 );
			$message   = $notice['message'][ $lang_code ] ?: $notice['message']['en'] ?? null;

			if ( ! empty( $message ) ) {
				wp_admin_notice(
					$message,
					array(
						'type'           => $notice['type'],
						'paragraph_wrap' => true,
						'id'             => $notice_id,
					)
				);
			}
		}
	}

	/**
	 * Retrieves notices from a transient. If the transient is empty, it fetches
	 * the notices from an external API and stores it in the transient.
	 *
	 * @return array Returns an array of notices.
	 */
	public static function get_notices(): array {
		$data = get_transient( self::NOTICES_TRANSIENT_NAME );
		if ( empty( $data ) ) {
			$plugin_data = self::get_plugin_data();

			$response = Api_Handler::fetch_data(
				true,
				'/wp/v1/notices',
				array(
					'locale'         => get_user_locale(),
					'plugin_version' => $plugin_data['Version'] ?? null,
					'wp_version'     => get_bloginfo( 'version' ),
				)
			);

			if ( $response->ok ) {
				$data = $response->data;
				set_transient( self::NOTICES_TRANSIENT_NAME, $data, DAY_IN_SECONDS );
			}
		}

		return is_array( $data['notices'] ) ? $data['notices'] : array();
	}

	/**
	 * Sets the plugin data by retrieving it from the main plugin file.
	 *
	 * @return void
	 * @throws RuntimeException If the plugin main file path is not set.
	 */
	public static function set_plugin_data(): void {
		if ( ! self::$plugin_file_path ) {
			throw new RuntimeException( 'Plugin main file path is not set' );
		}

		self::$plugin_data = get_plugin_data( self::$plugin_file_path, false, false );
	}

	/**
	 * Retrieves the plugin data.
	 *
	 * @return array Returns an array containing the plugin data.
	 * @throws RuntimeException If the plugin data is not set.
	 */
	public static function get_plugin_data(): array {
		if ( ! self::$plugin_data ) {
			throw new RuntimeException( 'Plugin data is not set' );
		}

		return self::$plugin_data;
	}

	/**
	 * Loads the plugin's translated text domain for localization.
	 *
	 * @return void
	 */
	private static function load_plugin_textdomain(): void {
		$path = dirname( plugin_basename( __FILE__ ) ) . '/languages';
		load_plugin_textdomain( 'creator-assistant', false, $path );
	}
}
