<?php
/**
 * Editor.
 *
 * @package CreatorAssistant
 */

declare( strict_types=1 );

namespace CreatorAssistant;

use Closure;

/**
 * Class Editor
 *
 * Handles the initialization and management of block assets for the editor.
 */
class Editor {
	/**
	 * Initializes the setup by adding the necessary actions for
	 * enqueuing block assets and loading blocks.
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'enqueue_block_assets', Closure::fromCallable( array( __CLASS__, 'enqueue_assets' ) ) );
		add_action( 'init', Closure::fromCallable( array( __CLASS__, 'load_blocks' ) ) );

		if ( version_compare( get_bloginfo( 'version' ), '5.8', '>=' ) ) {
			add_filter( 'block_categories_all', Closure::fromCallable( array( __CLASS__, 'add_block_category' ) ) );
		} else {
			add_filter( 'block_categories', Closure::fromCallable( array( __CLASS__, 'add_block_category' ) ) );
		}
	}

	/**
	 * Adds a new block category to the existing categories.
	 *
	 * @param array $categories The existing block categories.
	 *
	 * @return array The updated list of block categories, including the new category.
	 */
	private static function add_block_category( array $categories ): array {
		return array_merge(
			array(
				array(
					'slug'  => 'creator-assistant',
					'title' => __( 'Creator Assistant', 'creator-assistant' ),
				),
			),
			$categories,
		);
	}

	/**
	 * Enqueues the necessary assets for the plugin when in the admin dashboard.
	 * This includes JavaScript and CSS files, along with script translations.
	 *
	 * @return void
	 */
	private static function enqueue_assets(): void {
		if ( ! is_admin() ) {
			return;
		}

		$asset = require plugin_dir_path( __FILE__ ) . 'build/index.asset.php';
		wp_enqueue_script(
			'creator-assistant',
			plugin_dir_url( __FILE__ ) . 'build/index.js',
			$asset['dependencies'],
			$asset['version'],
			array(
				'in_footer' => true,
				'strategy'  => 'defer',
			)
		);
		wp_set_script_translations(
			'creator-assistant',
			'creator-assistant',
			plugin_dir_path( __FILE__ ) . 'languages'
		);

		$css_path = 'build/index.css';
		if ( file_exists( plugin_dir_path( __FILE__ ) . $css_path ) ) {
			wp_enqueue_style( 'creator-assistant', plugin_dir_url( __FILE__ ) . $css_path, array(), $asset['version'] );
		}
	}

	/**
	 * Loads custom blocks from the plugin's 'build/blocks' directory.
	 * This method scans the directory for valid block configurations
	 * and registers them with WordPress, including their render callbacks and scripts.
	 *
	 * @return void
	 */
	private static function load_blocks(): void {
		$blocks_dir = plugin_dir_path( __FILE__ ) . 'build/blocks';
		if ( ! file_exists( $blocks_dir ) ) {
			return;
		}

		foreach ( scandir( $blocks_dir ) as $block_dir ) {
			if ( in_array( $block_dir, array( '.', '..' ), true ) ) {
				continue;
			}

			$block_json  = $blocks_dir . '/' . $block_dir . '/block.json';
			$render_file = plugin_dir_path( __FILE__ ) . 'src/blocks/' . $block_dir . '/render.php';
			if ( file_exists( $render_file ) && file_exists( $block_json ) ) {
				$metadata   = wp_json_file_decode( $block_json, array( 'associative' => true ) );
				$block_name = $metadata['name'];
				add_filter(
					'block_type_metadata_settings',
					function ( $settings, $metadata ) use ( $block_name, $render_file ) {
						if ( $metadata['name'] === $block_name ) {
							$settings['render_callback'] = function ( $attributes, $content, $block ) use ( $render_file ) {
								ob_start();
								require $render_file;
								return ob_get_clean();
							};
						}

						return $settings;
					},
					10,
					2
				);
			}

			$block   = register_block_type_from_metadata( $block_json );
			$scripts = array(
				'editor_script_handles' => 'index.js',
				'script_handles'        => 'script.js',
				'view_script_handles'   => 'view.js',
			);

			foreach ( array_keys( $scripts ) as $script ) {
				if ( isset( $block->{$script} ) ) {
					foreach ( $block->{$script} as $script_handle ) {
						wp_script_add_data( $script_handle, 'strategy', 'defer' );
					}
				}
			}

			if ( is_admin() ) {
				wp_set_script_translations(
					$block->editor_script,
					'creator-assistant',
					plugin_dir_path( __FILE__ ) . 'languages'
				);
			}
		}
	}
}
