<?php
/**
 * Api Handler.
 *
 * @package CreatorAssistant
 */

declare(strict_types=1);

namespace CreatorAssistant;

use Closure;
use stdClass;
use WP_Error;
use WP_REST_Request;
use WP_REST_Server;

/**
 * Handles API interactions and REST routes for the Creator Assistant plugin.
 */
class Api_Handler {

	public const API_ENDPOINT_DEV         = 'https://api.wp-plugins.dev';
	public const API_ENDPOINT_PROD        = 'https://api.creator-assistant.it';
	public const ACCESS_TOKEN_OPTION_NAME = 'creator_assistant_access_token';
	public const SETUP_HASH_OPTION_NAME   = 'creator_assistant_setup_hash';
	public const REST_NAMESPACE           = 'creator-assistant/v1';
	public const REST_ROUTE_SETUP         = '/setup';
	public const REST_ROUTE_EDITOR        = '/editor';

	/**
	 * Initializes the plugin's functionality by registering necessary hooks.
	 */
	public static function init(): void {
		add_action( 'rest_api_init', Closure::fromCallable( array( __CLASS__, 'register_routes' ) ) );

		register_deactivation_hook(
			Plugin::get_plugin_file_path(),
			Closure::fromCallable( array( __CLASS__, 'uninstall_plugin' ) )
		);

		register_activation_hook(
			Plugin::get_plugin_file_path(),
			Closure::fromCallable( array( __CLASS__, 'install_plugin' ) )
		);
	}

	/**
	 * Retrieves the setup hash stored in the options.
	 *
	 * @return string|null returns the setup hash if available, null otherwise
	 */
	public static function get_setup_hash(): ?string {
		return get_option( self::SETUP_HASH_OPTION_NAME, null );
	}

	/**
	 * Installs or initializes the plugin by generating and setting up an authentication hash.
	 *
	 * @return void
	 */
	private static function install_plugin(): void {
		$auth_hash = self::get_setup_hash();
		if ( ! $auth_hash ) {
			$auth_hash = wp_hash( wp_rand() );
			update_option( self::SETUP_HASH_OPTION_NAME, $auth_hash );
		}
	}

	/**
	 * Uninstalls the plugin by deleting specific options from the database.
	 *
	 * @return void
	 */
	private static function uninstall_plugin(): void {
		delete_option( self::ACCESS_TOKEN_OPTION_NAME );
		delete_option( self::SETUP_HASH_OPTION_NAME );
	}

	/**
	 * Checks if an access token is available.
	 *
	 * @return bool returns true if an access token is present, false otherwise
	 */
	public static function have_access_token(): bool {
		return ! empty( self::get_access_token() );
	}

	/**
	 * Retrieves the appropriate API endpoint based on the environment.
	 *
	 * @return string returns the development API endpoint if in developer mode, otherwise returns the production API endpoint
	 */
	private static function get_api_endpoint(): string {
		return CREATOR_ASSISTANT_IS_DEV ? self::API_ENDPOINT_DEV : self::API_ENDPOINT_PROD;
	}

	/**
	 * Sends a POST request to the specified API endpoint.
	 *
	 * @param bool   $need_auth indicates if authentication is required for the request.
	 * @param string $path      the API endpoint path to send the request to.
	 * @param array  $body      the body of the request, default is an empty array.
	 *
	 * @return stdClass the parsed response from the API
	 */
	public static function send_data( bool $need_auth, string $path, array $body = array() ): stdClass {
		$response = wp_remote_post(
			self::get_api_endpoint() . $path,
			array(
				'headers'   => self::get_headers( $need_auth ),
				'body'      => wp_json_encode( $body ),
				'sslverify' => self::is_ssl_verify(),
			)
		);

		return self::parse_response( $response );
	}

	/**
	 * Fetches data from a specified path with optional authentication and parameters.
	 *
	 * @param bool   $need_auth whether authentication is required.
	 * @param string $path      the API endpoint path to fetch data from.
	 * @param array  $params    optional parameters to include in the query string.
	 *
	 * @return stdClass the parsed response from the API request
	 */
	public static function fetch_data( bool $need_auth, string $path, array $params = array() ): stdClass {
		$query_string = empty( $params ) ? '' : '?' . http_build_query( $params );
		$response     = wp_remote_get(
			self::get_api_endpoint() . $path . $query_string,
			array(
				'headers'   => self::get_headers( $need_auth ),
				'sslverify' => self::is_ssl_verify(),
			)
		);

		return self::parse_response( $response );
	}

	/**
	 * Retrieves the stored access token from the database.
	 *
	 * @return null|string the access token if it exists, or null if it does not
	 */
	private static function get_access_token(): ?string {
		return get_option( self::ACCESS_TOKEN_OPTION_NAME, null );
	}

	/**
	 * Registers custom REST API routes for the application.
	 */
	private static function register_routes(): void {
		register_rest_route(
			self::REST_NAMESPACE,
			self::REST_ROUTE_SETUP,
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => Closure::fromCallable( array( __CLASS__, 'setup_route' ) ),
				'args'                => array(
					'access_token' => array(
						'description'       => __( 'Access token for api requests.', 'creator-assistant' ),
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'validate_callback' => function ( $value ) {
							return is_string( $value ) && preg_match( '/^[a-zA-Z0-9]+$/', $value );
						},
					),
				),
				'permission_callback' => Closure::fromCallable( array( __CLASS__, 'setup_permission' ) ),
			)
		);

		register_rest_route(
			self::REST_NAMESPACE,
			self::REST_ROUTE_EDITOR,
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => Closure::fromCallable( array( __CLASS__, 'editor_route' ) ),
				'permission_callback' => Closure::fromCallable( array( __CLASS__, 'editor_permission' ) ),
			)
		);
	}

	/**
	 * Checks if the current user has permission to edit posts.
	 *
	 * @return bool returns true if the current user can edit posts, false otherwise
	 */
	private static function editor_permission(): bool {
		return current_user_can( 'edit_posts' );
	}

	/**
	 * Sets up permissions based on the provided WP REST request.
	 *
	 * @param WP_REST_Request $request the incoming REST request.
	 *
	 * @return bool returns true if the permissions are successfully set up and the hashes match, false otherwise
	 */
	private static function setup_permission( WP_REST_Request $request ): bool {
		$auth_header = $request->get_header( 'Authorization' );

		$header_hash = substr( $auth_header, strlen( 'Bearer ' ) );
		if ( ! $header_hash ) {
			return false;
		}

		$auth_hash = get_option( self::SETUP_HASH_OPTION_NAME );
		if ( ! $auth_hash ) {
			return false;
		}

		return hash_equals( $auth_hash, $header_hash );
	}

	/**
	 * Sets up the route by validating and storing the access token.
	 *
	 * @param WP_REST_Request $request the REST request object containing parameters.
	 *
	 * @return null|WP_Error returns a WP_Error if the access token is empty or cannot be saved, otherwise null
	 */
	private static function setup_route( WP_REST_Request $request ): ?WP_Error {
		$access_token = $request->get_param( 'access_token' );

		if ( ! $access_token ) {
			return new WP_Error( 'empty_access_token', '', array( 'status' => 400 ) );
		}

		$updated = update_option( self::ACCESS_TOKEN_OPTION_NAME, $access_token );
		if ( ! $updated ) {
			return new WP_Error( 'not_saved_access_token', '', array( 'status' => 304 ) );
		}

		return null;
	}

	/**
	 * Constructs and returns the editor route information.
	 *
	 * Loads necessary plugin data if not available during REST calls and sets up
	 * required fields including access token, API endpoint, current locale, plugin
	 * version, and WordPress version.
	 *
	 * @return array returns an array containing editor route information
	 */
	private static function editor_route(): array {
		// get_plugin_data is not loaded in REST calls.
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		Plugin::set_plugin_data();
		$plugin_data = Plugin::get_plugin_data();

		$locale = get_locale();

		return array(
			'is_active'      => self::have_access_token(),
			'access_token'   => self::get_access_token(),
			'endpoint'       => self::get_api_endpoint(),
			'path'           => '/wp/v1',
			'locale'         => $locale,
			'plugin_version' => $plugin_data['Version'],
			'wp_version'     => get_bloginfo( 'version' ),
			'setup'          => array(
				'wp_admin_url'  => admin_url(),
				'wp_setup_url'  => get_rest_url( null, self::REST_NAMESPACE . self::REST_ROUTE_SETUP ),
				'wp_setup_hash' => self::get_setup_hash(),
			),
		);
	}

	/**
	 * Generates the headers for an HTTP request.
	 *
	 * @param bool $need_auth indicates whether authorization is required.
	 *
	 * @return array returns an array of headers
	 */
	private static function get_headers( bool $need_auth ): array {
		$headers = array(
			'Content-Type' => 'application/json; charset=utf-8',
			'Referer'      => admin_url(),
		);

		if ( ! $need_auth ) {
			return $headers;
		}

		$access_token             = self::get_access_token();
		$headers['Authorization'] = 'Bearer ' . $access_token;

		return $headers;
	}

	/**
	 * Determines whether SSL verification is required.
	 *
	 * @return bool returns true if SSL verification is required, false if in development mode
	 */
	private static function is_ssl_verify(): bool {
		return ! CREATOR_ASSISTANT_IS_DEV;
	}

	/**
	 * Parses the given response and returns a standardized object.
	 *
	 * @param WP_Error|array $response the response to parse, which can be an array or a WP_Error object.
	 *
	 * @return stdClass a standardized object with the response status, a success flag, and the parsed response data or error message
	 */
	private static function parse_response( $response ): stdClass {
		if ( is_wp_error( $response ) ) {
			return (object) array(
				'status' => 500,
				'ok'     => false,
				'data'   => $response->get_error_message(),
			);
		}

		$response_obj         = new stdClass();
		$response_obj->status = wp_remote_retrieve_response_code( $response );
		$response_obj->ok     = 200 === $response_obj->status;

		if ( $response_obj->ok ) {
			$response_obj->data = json_decode( wp_remote_retrieve_body( $response ), true );
		} else {
			$response_obj->data = wp_remote_retrieve_body( $response );
		}

		return $response_obj;
	}
}
