<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://www.creamailer.fi
 * @since      1.0.0
 *
 * @package    CreamailerForNinjaForms
 * @subpackage CreamailerForNinjaForms/admin
 * @author     Creamailer <info@creamailer.fi>
 */
class Creamailer_For_Ninja_Forms_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

    protected $settings;

    /**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;

        $this->settings = get_option( 'creamailer-for-ninja-forms' );

        add_action( 'admin_init', array( $this, 'admin_init_hook' ));
        add_action( 'admin_menu', array( $this, 'admin_menu_hook' ));

        add_filter( 'ninja_forms_register_actions', array( $this, 'register_creamailer_ninja_forms_actions' ) );
    }

    public function register_creamailer_ninja_forms_actions($actions)
    {
        $actions['creamailer_for_ninja_forms_action'] = new CreamailerForNinjaFormsAction();

        return $actions;
    }

    public function admin_menu_hook() {
        add_options_page(
            'Creamailer settings for Ninja Forms',
            'Creamailer for Ninja Forms',
            'manage_options',
            'creamailer_for_ninja_forms_settings',
            array( $this, 'render_settings_page' )
        );
    }

    public function sanitize_settings( $input ) {
        $output = array();
        $output['access_token'] = isset( $input['access_token'] ) ? sanitize_text_field( $input['access_token'] ) : '';
        $output['shared_secret'] = isset( $input['shared_secret'] ) ? sanitize_text_field( $input['shared_secret'] ) : '';
        $output['list_id'] = isset( $input['list_id'] ) ? sanitize_text_field( $input['list_id'] ) : '';
        $output['optin_text'] = isset( $input['optin_text'] ) ? sanitize_text_field( $input['optin_text'] ) : '';
        $output['hide_consent'] = isset( $input['hide_consent'] ) ? '1' : '0';
        $output['send_autoresponders'] = isset( $input['send_autoresponders'] ) ? '1' : '0';
        $output['send_autoresponders_if_exists'] = isset( $input['send_autoresponders_if_exists'] ) ? '1' : '0';
        $output['error_log'] = isset( $input['error_log'] ) ? '1' : '0';

        return $output;
    }

    public function admin_init_hook() {
        register_setting(
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms',
            array(
                'sanitize_callback' => array( $this, 'sanitize_settings' ),
            )
        );

        add_settings_section(
            'creamailer-for-ninja-forms',
            '',
            array( $this, 'settings_section_html' ),
            'creamailer_for_ninja_forms_settings'
        );

        add_settings_field(
            'access_token',
            '',
            array( $this, 'print_access_token_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );

        add_settings_field(
            'shared_secret',
            '',
            array( $this, 'print_shared_secret_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );

        add_settings_field(
            'list_id',
            '',
            array( $this, 'print_list_id_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );
        add_settings_field(
            'optin_text',
            '',
            array( $this, 'print_optin_text_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );
        add_settings_field(
            'hide_consent',
            '',
            array( $this, 'print_consent_message_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );
        add_settings_field(
            'send_autoresponders',
            '',
            array( $this, 'print_send_autoresponders_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );
        add_settings_field(
            'send_autoresponders_if_exists',
            '',
            array( $this, 'print_send_autoresponders_if_exists_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );
        add_settings_field(
            'error_log',
            '',
            array( $this, 'print_error_log_field' ),
            'creamailer_for_ninja_forms_settings',
            'creamailer-for-ninja-forms',
            'creamailer-for-ninja-forms'
        );
    }

    protected function update_lists() {
        $creamailer = new CreamailerApiForNinjaForms(
            $this->settings['access_token'],
            $this->settings['shared_secret'],
            0,
            $this->settings['error_log']
        );

        $res = $creamailer->get_lists();

        if ( ! is_array( $res ) ) {
            $this->lists = null;
            return;
        }
        $this->lists = array();
        foreach ( $res as $i ) {
            array_push( $this->lists, $i );
        }
        /* XXX  probably wrong place to set this */
        $this->connected = true;
    }

    public function settings_section_html() {
        printf( '<h1>' . esc_html__( 'Creamailer integration settings for Ninja Forms', 'creamailer-for-ninja-forms' ) . '</h1>' );
        printf( '<div id="cfw-settings-header">' );
        printf( '<img src="%s/admin/images/logo.png" alt="Creamailer" height="200" />', esc_url(  CREAMAILER_NINJA_FORMS_PLUGIN_URL ) );
        printf( '<sup>Version %s</sup>', esc_html( CREAMAILER_FOR_NINJA_FORMS_VERSION ) );
        printf( '</div>' );
        printf( '<hr>' );
        printf( '<div id="creamailer-for-ninja-forms-plugin-api-container">' );
        printf( '<h2>' . esc_html__( 'API-Settings', 'creamailer-for-ninja-forms') . '</h2>' );
        if ( $this->connected ) {
            printf( '<small id="creamailer-for-ninja-forms-plugin-connected" class="creamailer-for-ninja-forms-plugin-connect-indicator">' . esc_html__( 'connected', 'creamailer-for-ninja-forms' ) . '</small>' );
        } else {
            printf( '<small id="creamailer-for-ninja-forms-plugin-not-connected" class="creamailer-for-ninja-forms-plugin-connect-indicator">' . esc_html__( 'No connection', 'creamailer-for-ninja-forms' ) . '</small>' );
        }
        printf( ' </div>' );
    }

    public function print_access_token_field() {
        printf( '<p class="cfw-required">' . esc_html__( 'Access token', 'creamailer-for-ninja-forms' ) . '</p>' );
        printf( '<input id="creamailer-for-ninja-forms-plugin-access-token-field" type="text" name="creamailer-for-ninja-forms[access_token]" value="%s" size="40" />', esc_html( $this->settings['access_token'] ) );
        printf( '<p>' . esc_html__( 'Your access token (asiakastunnus)', 'creamailer-for-ninja-forms' ) . '</p>' );

    }

    public function print_shared_secret_field() {
        printf( '<p class="cfw-required">' . esc_html__( 'Shared secred', 'creamailer-for-ninja-forms' ) . '</p>' );
        printf( '<input id="creamailer-for-ninja-forms-plugin-shared-secret-field" type="password" name="creamailer-for-ninja-forms[shared_secret]" value="%s" size="40" />', esc_html( $this->settings['shared_secret'] ) );
        printf( '<p>' . esc_html__( 'Your shared secred (yhteinen tunniste)', 'creamailer-for-ninja-forms' ) . '</p>' );
    }

    public function print_optin_text_field() {
        printf( '<p class="cfw-required">' . esc_html__( 'Opt-in text', 'creamailer-for-ninja-forms' ) . '</p>' );
        printf(
            '<input id="creamailer-for-ninja-forms-plugin-optin-text-field" type="text" name="creamailer-for-ninja-forms[optin_text]" value="%s" size="40" />',
            $this->settings['optin_text'] ?
                esc_html( $this->settings['optin_text'] ) :
                esc_html__( 'Subscribe to our newsletter', 'creamailer-for-ninja-forms' )
        );
        printf( '<p>' . esc_html__( 'Opt-in text', 'creamailer-for-ninja-forms' ) . '</p>' );

    }

    public function print_list_id_field() {
        if ( ! $this->lists ) {
            return;
        }

        printf( '<p class="cfw-required">' . esc_html__( 'Mailinglist', 'creamailer-for-ninja-forms' ) . '</p>' );
        printf( '<select name="creamailer-for-ninja-forms[list_id]" />' );
        foreach ( $this->lists as $i ) {


            printf(
                '<option value="%s" %s>%s</option>',
                esc_html( $i['id'] ),
                esc_html( $this->settings['list_id'] === $i['id'] ? 'selected' : '' ),
                esc_html( $i['name'] )
            );
        }
        printf( '</select>' );
        printf( '<p>' . esc_html__( 'Choose mailinglist', 'creamailer-for-ninja-forms' ) . '</p>' );
    }

    public function print_error_log_field() {
        printf(
            '<input id="creamailer-for-ninja-forms-plugin-error-log-field" type="checkbox" value="1" name="creamailer-for-ninja-forms[error_log]" %s />',
            $this->settings['error_log'] ? 'checked="checked"' : ''
        );
        printf( '<span id="creamailer-for-ninja-forms-plugin-error-log-text">' . esc_html__( 'Enable error log', 'creamailer-for-ninja-forms' ) . '</span>' );
    }

    public function print_consent_message_field() {
        printf( '<hr id="cfw-send-newsletters" />' );
        printf(
            '<input id="cfw-show-consent-field" type="checkbox" value="1" name="creamailer-for-ninja-forms[hide_consent]" %s />',
            (isset($this->settings['hide_consent']) && $this->settings['hide_consent'] === '1') ? 'checked="checked"' : ''
        );
        printf( '<span id="cfw-show-consent-text">' . esc_html__( 'Hide the permission question and add to the list', 'creamailer-for-ninja-forms' ) . '</span>' );
    }

    public function print_send_autoresponders_field() {
        printf(
            '<input id="creamailer-for-ninja-forms-plugin-send-autoresponders-field" type="checkbox" value="1" name="creamailer-for-ninja-forms[send_autoresponders]" %s />',
            $this->settings['send_autoresponders'] ? 'checked="checked"' : ''
        );
        printf( '<span id="creamailer-for-ninja-forms-plugin-send-autoresponders-text">' . esc_html__( 'Send mailinglists automatic messages', 'creamailer-for-ninja-forms' ) . '</span>' );
    }

    public function print_send_autoresponders_if_exists_field() {
        printf(
            '<input id="creamailer-for-ninja-forms-plugin-send-autoresponders-if-exists-field" type="checkbox" value="1" name="creamailer-for-ninja-forms[send_autoresponders_if_exists]" %s />',
            $this->settings['send_autoresponders_if_exists'] ? 'checked="checked"' : ''
        );
        printf( '<span id="creamailer-for-ninja-forms-plugin-send-autoresponders-if-exists-text">' . esc_html__( 'Send mailinglists automatic messages even if subscriber exists', 'creamailer-for-ninja-forms' ) . '</span>' );
    }

    public function render_settings_page() {
        $this->update_lists();
        printf( '<div id="creamailer-for-ninja-forms-plugin-settings">' );
        printf( '<form action="options.php" method="post">' );
        settings_fields( 'creamailer-for-ninja-forms' );
        do_settings_sections( 'creamailer_for_ninja_forms_settings' );
        submit_button();
        printf( '</form>' );
        printf( '<hr />' );
        printf( '<h1>' . esc_html__( 'Need help?', 'creamailer-for-ninja-forms' ) . '</h1>' );
        printf( '<a href="https://tuki.creamailer.fi/hc/fi/articles/360021562680">' . esc_html__( 'See the official help article', 'creamailer-for-ninja-forms' ) . '</a>' );
        printf(
            '<p><a href="%s/log/%s.log">' . esc_html__( 'View error log', 'creamailer-for-ninja-forms' ) . '</a></p>',
            esc_html( CREAMAILER_FOR_NINJA_FORMS_VERSION ),
            esc_html( md5( $this->settings['access_token'] ) )
        );
        printf( '</div>' );
    }

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url(__FILE__) . 'css/creamailer-for-ninja-forms-admin.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, plugin_dir_url(__FILE__) . 'js/creamailer-for-ninja-forms-admin.js', array( 'jquery' ), $this->version, false );
	}
}
