<?php
/**
 * Plugin Name: Craftify AI Content Publisher
 * Description: Publish blog posts directly from Craftify AI to your WordPress site.
 * Version: 1.0.2
 * Author: Craftify AI
 * Author URI: https://www.craftify.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: craftify-ai-content-publisher
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CRAFTIFY_AI_VERSION', '1.0.2');
define('CRAFTIFY_AI_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CRAFTIFY_AI_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CRAFTIFY_AI_API_URL', 'https://api.craftify.ai');

/**
 * Main Craftify AI Plugin Class
 */
class Craftify_AI {
    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Plugin options
     */
    private $options;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->options = get_option('craftify_ai_options', array());

        // Initialize hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

        // Register activation hook
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        if (!get_option('craftify_ai_options')) {
            update_option('craftify_ai_options', array(
                'api_key' => '',
                'site_name' => get_bloginfo('name'),
                'default_status' => 'draft',
                'default_author' => get_current_user_id(),
            ));
        }

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Optionally unregister from Craftify AI
        $this->unregister_site();
        flush_rewrite_rules();
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            __('Craftify AI', 'craftify-ai-content-publisher'),
            __('Craftify AI', 'craftify-ai-content-publisher'),
            'manage_options',
            'craftify-ai',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting(
            'craftify_ai_options_group',
            'craftify_ai_options',
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'craftify_ai_main_section',
            __('Connection Settings', 'craftify-ai-content-publisher'),
            array($this, 'section_callback'),
            'craftify-ai'
        );

        add_settings_field(
            'api_key',
            __('Craftify AI API Key', 'craftify-ai-content-publisher'),
            array($this, 'api_key_field_callback'),
            'craftify-ai',
            'craftify_ai_main_section'
        );

        add_settings_field(
            'site_name',
            __('Site Name', 'craftify-ai-content-publisher'),
            array($this, 'site_name_field_callback'),
            'craftify-ai',
            'craftify_ai_main_section'
        );

        add_settings_field(
            'default_status',
            __('Default Post Status', 'craftify-ai-content-publisher'),
            array($this, 'default_status_field_callback'),
            'craftify-ai',
            'craftify_ai_main_section'
        );

        add_settings_field(
            'default_author',
            __('Default Author', 'craftify-ai-content-publisher'),
            array($this, 'default_author_field_callback'),
            'craftify-ai',
            'craftify_ai_main_section'
        );

        add_settings_field(
            'connection_status',
            __('Connection Status', 'craftify-ai-content-publisher'),
            array($this, 'connection_status_field_callback'),
            'craftify-ai',
            'craftify_ai_main_section'
        );
    }

    /**
     * Sanitize options
     */
    public function sanitize_options($input) {
        $sanitized = array();

        if (isset($input['api_key'])) {
            $sanitized['api_key'] = sanitize_text_field($input['api_key']);
        }

        if (isset($input['site_name'])) {
            $sanitized['site_name'] = sanitize_text_field($input['site_name']);
        }

        if (isset($input['default_status'])) {
            $sanitized['default_status'] = in_array($input['default_status'], array('draft', 'publish', 'pending'))
                ? $input['default_status']
                : 'draft';
        }

        if (isset($input['default_author'])) {
            $sanitized['default_author'] = absint($input['default_author']);
        }

        // Register/update site with Craftify AI when API key changes
        $old_options = get_option('craftify_ai_options', array());
        if (
            !empty($sanitized['api_key']) &&
            (!isset($old_options['api_key']) || $old_options['api_key'] !== $sanitized['api_key'])
        ) {
            $this->register_site($sanitized['api_key'], $sanitized['site_name']);
        }

        return $sanitized;
    }

    /**
     * Section callback
     */
    public function section_callback() {
        echo '<p>' . esc_html__('Enter your Craftify AI API key to enable blog publishing from Craftify AI.', 'craftify-ai-content-publisher') . '</p>';
        echo '<p>' . sprintf(
            /* translators: %1$s: opening link tag, %2$s: closing link tag */
            esc_html__( 'You can find your API key in your %1$sCraftify AI Settings%2$s.', 'craftify-ai-content-publisher' ),
            '<a href="https://www.craftify.ai/signin?callbackUrl=/app/settings/integrations" target="_blank">',
            '</a>'
        ) . '</p>';
    }

    /**
     * API Key field callback
     */
    public function api_key_field_callback() {
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        echo '<input type="text" name="craftify_ai_options[api_key]" value="' . esc_attr($api_key) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('Enter the API key from your Craftify AI integrations page.', 'craftify-ai-content-publisher') . '</p>';
    }

    /**
     * Site name field callback
     */
    public function site_name_field_callback() {
        $site_name = isset($this->options['site_name']) ? $this->options['site_name'] : get_bloginfo('name');
        echo '<input type="text" name="craftify_ai_options[site_name]" value="' . esc_attr($site_name) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('This name will appear in Craftify AI when selecting a site to publish to.', 'craftify-ai-content-publisher') . '</p>';
    }

    /**
     * Default status field callback
     */
    public function default_status_field_callback() {
        $default_status = isset($this->options['default_status']) ? $this->options['default_status'] : 'draft';
        ?>
        <select name="craftify_ai_options[default_status]">
            <option value="draft" <?php selected($default_status, 'draft'); ?>><?php esc_html_e('Draft', 'craftify-ai-content-publisher'); ?></option>
            <option value="pending" <?php selected($default_status, 'pending'); ?>><?php esc_html_e('Pending Review', 'craftify-ai-content-publisher'); ?></option>
            <option value="publish" <?php selected($default_status, 'publish'); ?>><?php esc_html_e('Published', 'craftify-ai-content-publisher'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('The default status for posts published from Craftify AI.', 'craftify-ai-content-publisher'); ?></p>
        <?php
    }

    /**
     * Default author field callback
     */
    public function default_author_field_callback() {
        $default_author = isset($this->options['default_author']) ? $this->options['default_author'] : get_current_user_id();
        wp_dropdown_users(array(
            'name' => 'craftify_ai_options[default_author]',
            'selected' => $default_author,
            'capability' => 'edit_posts',
        ));
        echo '<p class="description">' . esc_html__('The default author for posts published from Craftify AI.', 'craftify-ai-content-publisher') . '</p>';
    }

    /**
     * Connection status field callback
     */
    public function connection_status_field_callback() {
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';

        if (empty($api_key)) {
            echo '<span class="craftify-status craftify-status-disconnected">' . esc_html__('Not Connected', 'craftify-ai-content-publisher') . '</span>';
            echo '<p class="description">' . esc_html__('Enter your API key and save settings to connect.', 'craftify-ai-content-publisher') . '</p>';
        } else {
            $is_valid = $this->verify_api_key($api_key);
            if ($is_valid) {
                echo '<span class="craftify-status craftify-status-connected">' . esc_html__('Connected', 'craftify-ai-content-publisher') . '</span>';
                echo '<p class="description">' . esc_html__('Your site is connected to Craftify AI.', 'craftify-ai-content-publisher') . '</p>';
            } else {
                echo '<span class="craftify-status craftify-status-error">' . esc_html__('Connection Error', 'craftify-ai-content-publisher') . '</span>';
                echo '<p class="description">' . esc_html__('Could not verify API key. Please check your key and try again.', 'craftify-ai-content-publisher') . '</p>';
            }
        }
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="craftify-ai-header">
                <img src="<?php echo esc_url(CRAFTIFY_AI_PLUGIN_URL . 'assets/craftify-logo.png'); ?>" alt="Craftify AI" class="craftify-logo" />
            </div>

            <form action="options.php" method="post">
                <?php
                settings_fields('craftify_ai_options_group');
                do_settings_sections('craftify-ai');
                submit_button(__('Save Settings', 'craftify-ai-content-publisher'));
                ?>
            </form>

            <div class="craftify-ai-info">
                <h2><?php esc_html_e('How it works', 'craftify-ai-content-publisher'); ?></h2>
                <ol>
                    <li><?php esc_html_e('Go to your Craftify AI Settings > Integrations and connect WordPress.', 'craftify-ai-content-publisher'); ?></li>
                    <li><?php esc_html_e('Copy the API key and paste it in the field above.', 'craftify-ai-content-publisher'); ?></li>
                    <li><?php esc_html_e('Save your settings to register this site with Craftify AI.', 'craftify-ai-content-publisher'); ?></li>
                    <li><?php esc_html_e('When publishing a blog from Craftify AI, select this WordPress site as the destination.', 'craftify-ai-content-publisher'); ?></li>
                </ol>
            </div>
        </div>
        <?php
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_craftify-ai' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'craftify-ai-admin',
            CRAFTIFY_AI_PLUGIN_URL . 'assets/admin.css',
            array(),
            CRAFTIFY_AI_VERSION
        );
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('craftify-ai/v1', '/publish', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_publish'),
            'permission_callback' => array($this, 'verify_request'),
            'args' => array(
                'title' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                    'validate_callback' => function( $param ) {
                        return is_string( $param ) && ! empty( $param );
                    },
                ),
                'content' => array(
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => function( $param ) {
                        return is_string( $param ) && ! empty( $param );
                    },
                ),
                'contentHtml' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'wp_kses_post',
                ),
                'excerpt' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_textarea_field',
                ),
                'featuredImageUrl' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'esc_url_raw',
                ),
                'seoTitle' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'seoDescription' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_textarea_field',
                ),
                'tags' => array(
                    'required' => false,
                    'type' => 'array',
                    'items' => array(
                        'type' => 'string',
                    ),
                    'sanitize_callback' => function( $param ) {
                        return is_array( $param ) ? array_map( 'sanitize_text_field', $param ) : array();
                    },
                ),
                'categories' => array(
                    'required' => false,
                    'type' => 'array',
                    'items' => array(
                        'type' => 'string',
                    ),
                    'sanitize_callback' => function( $param ) {
                        return is_array( $param ) ? array_map( 'sanitize_text_field', $param ) : array();
                    },
                ),
            ),
        ));

        register_rest_route('craftify-ai/v1', '/verify', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_verify'),
            'permission_callback' => array($this, 'verify_request'),
        ));
    }

    /**
     * Verify incoming request
     */
    public function verify_request($request) {
        $api_key = $request->get_header('X-Craftify-API-Key');
        $stored_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';

        return !empty($api_key) && !empty($stored_key) && hash_equals($stored_key, $api_key);
    }

    /**
     * Handle publish request
     */
    public function handle_publish( $request ) {
        // Get sanitized params (already sanitized via REST API args registration)
        $params = $request->get_params();

        // Prepare post data
        $post_content = ! empty( $params['contentHtml'] ) ? $params['contentHtml'] : wp_kses_post( $params['content'] );

        $post_data = array(
            'post_title'   => $params['title'],
            'post_content' => $post_content,
            'post_status'  => isset( $this->options['default_status'] ) ? $this->options['default_status'] : 'draft',
            'post_author'  => isset( $this->options['default_author'] ) ? absint( $this->options['default_author'] ) : get_current_user_id(),
            'post_type'    => 'post',
        );

        // Add excerpt if provided
        if ( ! empty( $params['excerpt'] ) ) {
            $post_data['post_excerpt'] = $params['excerpt'];
        }

        // Insert the post
        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return new WP_Error(
                'post_creation_failed',
                $post_id->get_error_message(),
                array('status' => 500)
            );
        }

        // Handle featured image
        if ( ! empty( $params['featuredImageUrl'] ) ) {
            $this->set_featured_image( $post_id, $params['featuredImageUrl'] );
        }

        // Handle SEO meta (if Yoast or RankMath is installed)
        if ( ! empty( $params['seoTitle'] ) ) {
            $seo_description = ! empty( $params['seoDescription'] ) ? $params['seoDescription'] : '';
            $this->set_seo_meta( $post_id, $params['seoTitle'], $seo_description );
        }

        // Handle tags
        if ( ! empty( $params['tags'] ) && is_array( $params['tags'] ) ) {
            wp_set_post_tags( $post_id, $params['tags'] );
        }

        // Handle categories
        if ( ! empty( $params['categories'] ) && is_array( $params['categories'] ) ) {
            $category_ids = array();
            foreach ( $params['categories'] as $category_name ) {
                $term = term_exists( $category_name, 'category' );
                if ( $term ) {
                    $category_ids[] = (int) $term['term_id'];
                } else {
                    $new_term = wp_insert_term( sanitize_text_field( $category_name ), 'category' );
                    if ( ! is_wp_error( $new_term ) ) {
                        $category_ids[] = (int) $new_term['term_id'];
                    }
                }
            }
            if ( ! empty( $category_ids ) ) {
                wp_set_post_categories( $post_id, $category_ids );
            }
        }

        return array(
            'success' => true,
            'post_id' => $post_id,
            'post_url' => get_permalink($post_id),
            'edit_url' => get_edit_post_link($post_id, 'raw'),
        );
    }

    /**
     * Handle verify request
     */
    public function handle_verify($request) {
        return array(
            'success' => true,
            'site_url' => get_site_url(),
            'site_name' => isset($this->options['site_name']) ? $this->options['site_name'] : get_bloginfo('name'),
            'version' => CRAFTIFY_AI_VERSION,
        );
    }

    /**
     * Set featured image from URL
     */
    private function set_featured_image($post_id, $image_url) {
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Download image
        $tmp = download_url($image_url);

        if (is_wp_error($tmp)) {
            return false;
        }

        $file_array = array(
            'name' => basename( wp_parse_url( $image_url, PHP_URL_PATH ) ),
            'tmp_name' => $tmp,
        );

        // Check file type
        $file_type = wp_check_filetype($file_array['name']);
        if (empty($file_type['ext'])) {
            $file_array['name'] .= '.jpg';
        }

        // Upload to media library
        $attachment_id = media_handle_sideload($file_array, $post_id);

        if (is_wp_error($attachment_id)) {
            wp_delete_file( $tmp );
            return false;
        }

        // Set as featured image
        set_post_thumbnail($post_id, $attachment_id);

        return $attachment_id;
    }

    /**
     * Set SEO meta (supports Yoast and RankMath)
     */
    private function set_seo_meta($post_id, $seo_title, $seo_description) {
        // Yoast SEO
        if (defined('WPSEO_VERSION')) {
            update_post_meta($post_id, '_yoast_wpseo_title', $seo_title);
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $seo_description);
        }

        // RankMath
        if (defined('RANK_MATH_VERSION')) {
            update_post_meta($post_id, 'rank_math_title', $seo_title);
            update_post_meta($post_id, 'rank_math_description', $seo_description);
        }

        // All in One SEO
        if (defined('AIOSEO_VERSION')) {
            update_post_meta($post_id, '_aioseo_title', $seo_title);
            update_post_meta($post_id, '_aioseo_description', $seo_description);
        }
    }

    /**
     * Register site with Craftify AI
     */
    private function register_site($api_key, $site_name) {
        $response = wp_remote_post(CRAFTIFY_AI_API_URL . '/integration/wordpress/subscribe', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-KEY' => $api_key,
            ),
            'body' => json_encode(array(
                'siteUrl' => get_site_url(),
                'siteName' => $site_name ?: get_bloginfo('name'),
                'webhookUrl' => rest_url('craftify-ai/v1/publish'),
            )),
            'timeout' => 30,
        ));

        if (is_wp_error($response)) {
            return false;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        return isset($body['status']) && $body['status'] === 'success';
    }

    /**
     * Unregister site from Craftify AI
     */
    private function unregister_site() {
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';

        if (empty($api_key)) {
            return;
        }

        // Note: We would need the subscription ID to unregister
        // For now, this is a placeholder for future implementation
    }

    /**
     * Verify API key with Craftify AI
     */
    private function verify_api_key($api_key) {
        $response = wp_remote_get(CRAFTIFY_AI_API_URL . '/integration/user', array(
            'headers' => array(
                'X-API-KEY' => $api_key,
            ),
            'timeout' => 15,
        ));

        if (is_wp_error($response)) {
            return false;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        return $status_code === 200;
    }
}

// Initialize the plugin
Craftify_AI::get_instance();
