<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class for managing the country code field in Elementor forms.
 *
 * This class adds functionality to include country codes in the telephone input fields of Elementor forms.
 *
 * @package ccfef
 *
 * @version 1.0.0
 */
class CFEFP_COUNTRY_CODE_FIELD {

	/**
	 * Plugin instance.
	 *
	 * @var CFEFP_COUNTRY_CODE_FIELD
	 *
	 * @access private
	 * @var null
	 */
	private static $instance = null;

	/**
	 * Get the plugin instance.
	 *
	 * @return CFEFP_COUNTRY_CODE_FIELD
	 * @static
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor method to initialize actions and assets.
	 */
	public function __construct() {
		$this->register_common_assets();
		add_action( 'elementor_pro/forms/render_field/tel', array( $this, 'elementor_form_tel_field_rendering' ), 9, 3 );
		add_action( 'hello_plus/forms/render_field/ehp-tel', array( $this, 'elementor_form_tel_field_rendering' ), 20, 3 );
		add_action( 'elementor/element/form/section_form_fields/before_section_end', array( $this, 'update_controls' ), 100, 2);
		add_action( 'elementor/element/ehp-form/section_form_fields/before_section_end', array( $this, 'update_controls' ), 100, 2);
		add_action( 'elementor/preview/init', array( $this, 'editor_inline_JS' ) );
		add_action( 'elementor/frontend/after_enqueue_scripts', array( $this, 'frontend_assets' ) );
		add_action( 'elementor/editor/before_enqueue_styles', array( $this, 'editor_assets') );
		add_action( 'wp_ajax_ccfef_elementor_review_notice', array( $this, 'ccfef_elementor_review_notice' ) );
	}

	public function elementor_form_tel_field_rendering( $item, $item_index, $form ) {
		if ( ( 'ehp-tel' === $item['field_type'] || 'tel' === $item['field_type'] ) && 'yes' === $item['ccfef-country-code-field'] ) {
			// Get and sanitize the default country.
			$default_country = $item['ccfef-country-code-default'];
			if ( preg_match( '/[^a-zA-Z]/', $default_country ) ) {
				$default_country = 'NAN';
			}
			
			$include_countries  = $item['ccfef-country-code-include'];
			$excluded_countries = $item['ccfef-country-code-exclude'];
			$dial_code_visibility = $item['ccfef-dial-code-visibility'];
			$strict_mode = $item['ccfef-strict-mode'];

			// Convert comma-separated strings to arrays if needed.
			if ( is_string( $include_countries ) ) {
				$include_countries = array_map( 'trim', explode( ',', $include_countries ) );
			}
			if ( is_string( $excluded_countries ) ) {
				$excluded_countries = array_map( 'trim', explode( ',', $excluded_countries ) );
			}
			
			// --- Added code to set data-common-countries ---
			$include_countries_orig  = $include_countries;
			$excluded_countries_orig = $excluded_countries;
			sort( $include_countries_orig );
			sort( $excluded_countries_orig );
			$commonAttr = ( $include_countries_orig === $excluded_countries_orig ) ? 'same' : '';
			// --- End of added code ---
		
			// Convert the include countries array back to a comma-separated string for the data attribute.
			$include_countries_str = implode( ',', $include_countries );
		
			echo '<span class="ccfef-editor-intl-input" data-id="form-field-' . esc_attr( $item['custom_id'] ) . '" data-field-id="' . esc_attr( $item['_id'] ) . '" data-default-country="' . esc_attr( $default_country ) . '" data-include-countries="' . esc_attr( $include_countries_str ) . '" data-exclude-countries="' . esc_attr( implode( ',', $excluded_countries ) ) . '" data-dial-code-visibility="' . esc_attr( $dial_code_visibility ) . '" data-strict-mode="' . esc_attr( $strict_mode ) . '" style="display: none;"></span>';
		}
		
	}
	
	
	

	public function editor_inline_JS() {
		wp_enqueue_script( 'ccfef-country-code-editor-script', CCFEF_PLUGIN_URL . 'assets/js/ccfef-content-template.min.js', array(), CCFEF_VERSION, true ); // for AOS animation
	}

	/**
	 * Register common assets for the plugin.
	 */
	public function register_common_assets() {
		// Define the errorMap constant at the top of your file
		$error_map = [
			__("The phone number you entered is not valid. Please check the format and try again.", "country-code-field-for-elementor-form"),
			__("The country code you entered is not recognized. Please ensure it is correct and try again.", "country-code-field-for-elementor-form"),
			__("The phone number you entered is too short. Please enter a complete phone number, including the country code.", "country-code-field-for-elementor-form"),
			__("The phone number you entered is too long. Please ensure it is in the correct format and try again.", "country-code-field-for-elementor-form"),
			__("The phone number you entered is not valid. Please check the format and try again.", "country-code-field-for-elementor-form")
		];

		wp_register_script( 'ccfef-country-code-library-script', CCFEF_PLUGIN_URL . 'assets/intl-tel-input/js/intlTelInput.js', array(), CCFEF_VERSION, true );
		wp_register_script( 'ccfef-country-code-script', CCFEF_PLUGIN_URL . 'assets/js/country-code-script.min.js', array( 'elementor-frontend', 'jquery', 'ccfef-country-code-library-script' ), CCFEF_VERSION, true );
		wp_register_script( 'ccfef-country-code-script-hello', CCFEF_PLUGIN_URL . 'assets/js/country-code-script-hello.min.js', array( 'elementor-frontend', 'jquery', 'ccfef-country-code-library-script' ), CCFEF_VERSION, true );
		wp_register_style( 'ccfef-country-code-library-style', CCFEF_PLUGIN_URL . 'assets/intl-tel-input/css/intlTelInput.min.css', array(), CCFEF_VERSION, 'all' );
		wp_register_style( 'ccfef-country-code-style', CCFEF_PLUGIN_URL . 'assets/css/country-code-style.min.css', array(), CCFEF_VERSION, 'all' );

		wp_localize_script(
			'ccfef-country-code-script',
			'CCFEFCustomData',
			array(
				'pluginDir' => CCFEF_PLUGIN_URL,
				'errorMap'  => $error_map, 
			)	
		);
		wp_localize_script(
			'ccfef-country-code-script-hello',
			'CCFEFCustomData',
			array(
				'pluginDir' => CCFEF_PLUGIN_URL,
				'errorMap'  => $error_map, 
			)	
		);
	}

	/**
	 * Enqueue frontend assets for the plugin.
	 */
	public function editor_assets() {
		wp_enqueue_style( 'ccfef-editor-style', CCFEF_PLUGIN_URL . 'assets/css/ccfef_editor.min.css', array(), CCFEF_VERSION, 'all' );
		wp_enqueue_script( 'ccfef-editor-script', CCFEF_PLUGIN_URL . 'assets/js/ccfef-editor.min.js', array( 'jquery' ), CCFEF_VERSION, true );
	}

	/**
	 * Enqueue frontend assets for the plugin.
	 */
	public function frontend_assets() {
		wp_enqueue_script( 'ccfef-country-code-library-script');
		wp_enqueue_script( 'ccfef-country-code-script');
		wp_enqueue_script( 'ccfef-country-code-script-hello');
		wp_enqueue_style( 'ccfef-country-code-library-style');
		wp_enqueue_style( 'ccfef-country-code-style');	    
	}

	/**
	 * Update form widget controls to include the country code control in tel field.
	 *
	 * Adds country code control to allow users to customize the country code field.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param \Elementor\Widget_Base $widget The form widget instance.
	 * @return void
	 */
	public function update_controls( $widget ) {
		$elementor = \Elementor\Plugin::instance();
		$control_data = $elementor->controls_manager->get_control_from_stack( $widget->get_unique_name(), 'form_fields' );
		if ( is_wp_error( $control_data ) ) {
				return;
		}

		$ccfef_default_desc = sprintf(
			"%s <b>'%s'</b> %s.",
			esc_html__( 'Set default country code in tel field, like', 'country-code-field-for-elementor-form' ),
			esc_html__( 'in', 'country-code-field-for-elementor-form' ),
			esc_html__( 'for India', 'country-code-field-for-elementor-form' ),
		);

		$ccfef_auto_detect_desc = sprintf(
			'%s <br> To use - <a target="__blank" href="https://coolformkit.com/pricing/?utm_source=ccfef_plugin&utm_medium=inside&utm_campaign=get_pro&utm_content=editor_panel">(UPGRADE TO PRO)</a>',
			esc_html__( 'Auto select user country using ipapi.co', 'country-code-field-for-elementor-form' )			
		);


		$ccfef_include_desc = sprintf(
			'%s - <b>%s</b>,<b>%s</b>,<b>%s</b>,<b>%s</b>',
			esc_html__( 'Display only these countries, add comma separated', 'country-code-field-for-elementor-form' ),
			esc_html__( 'ca', 'country-code-field-for-elementor-form' ),
			esc_html__( 'in', 'country-code-field-for-elementor-form' ),
			esc_html__( 'us', 'country-code-field-for-elementor-form' ), 
			esc_html__( 'gb', 'country-code-field-for-elementor-form' ),
		);

		$ccfef_prefer_desc = sprintf(
			'%s To use - <a target="__blank" href="https://coolformkit.com/pricing/?utm_source=ccfef_plugin&utm_medium=inside&utm_campaign=get_pro&utm_content=editor_panel">(UPGRADE TO PRO)</a>',
			esc_html__( 'The Specified countries will appear at the top of the list.', 'country-code-field-for-elementor-form' ),			
		);

		$ccfef_exclude_desc = sprintf(
			'%s - <b>%s</b>,<b>%s</b><br><br>%s - <a target="__blank" href="' . esc_url( 'https://www.iban.com/country-codes' ) . '">https://www.iban.com/country-codes</a>',
			esc_html__( 'Exclude some countries, add comma separated', 'country-code-field-for-elementor-form' ),
			esc_html__( 'af', 'country-code-field-for-elementor-form' ),
			esc_html__( 'pk', 'country-code-field-for-elementor-form' ),
			esc_html__( 'Check country codes alpha-2 list here', 'country-code-field-for-elementor-form' ),
		);

		$field_controls = array(
			'ccfef-country-code-field'   => array(
				'name'         => 'ccfef-country-code-field',
				'label'        => esc_html__( 'Country Code', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'country-code-field-for-elementor-form' ),
				'label_off'    => esc_html__( 'Hide', 'country-code-field-for-elementor-form' ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => array(
					'field_type' => array('tel', 'ehp-tel'),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			),

			'ccfef-country-code-default' => array(
				'name'         => 'ccfef-country-code-default',
				'label'        => esc_html__( 'Default Country', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'condition'    => array(
					'field_type'               => array('tel', 'ehp-tel'),
					'ccfef-country-code-field' => 'yes',
				),
				'description'  => $ccfef_default_desc,
				'default'      => 'in',
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
				'ai'           => array(
					'active' => false,
				),
			),

			'ccfef-country-code-include' => array(
				'name'         => 'ccfef-country-code-include',
				'label'        => esc_html__( 'Only country', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'description'  => $ccfef_include_desc,
				'condition'    => array(
					'field_type'               => array('tel', 'ehp-tel'),
					'ccfef-country-code-field' => 'yes',
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
				'ai'           => array(
					'active' => false,
				),
			),
		
			'ccfef-country-code-exclude' => array(
				'name'         => 'ccfef-country-code-exclude',
				'label'        => esc_html__( 'Exclude Countries', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'description'  => $ccfef_exclude_desc,
				'condition'    => array(
					'field_type'               => array('tel', 'ehp-tel'),
					'ccfef-country-code-field' => 'yes',
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
				'ai'           => array(
					'active' => false,
				),
			),
			'ccfef-dial-code-visibility' => array(
				'name'         => 'ccfef-dial-code-visibility',
				'label'        => esc_html__( 'Dial Code Visibility', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::CHOOSE,
				'options'      => array(
					'show'     => array(
						'title' => esc_html__( 'Show', 'country-code-field-for-elementor-form' ),
						'icon'  => 'far fa-eye',
					),
					'hide'     => array(
						'title' => esc_html__( 'Hide', 'country-code-field-for-elementor-form' ),
						'icon'  => 'far fa-eye-slash',
					),
					'separate' => array(
						'title' => esc_html__( 'Separate', 'country-code-field-for-elementor-form' ),
						'icon'  => 'fas fa-arrows-alt-h',
					),
				),
				'default'      => 'show',
				'condition'    => array(
					'field_type'               => array('tel', 'ehp-tel'),
					'ccfef-country-code-field' => 'yes',
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
				'ai'           => array(
					'active' => false,
				),
			),
			'ccfef-strict-mode' => array(
				'name'         => 'ccfef-strict-mode',
				'label'        => esc_html__( 'Strict Mode', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'country-code-field-for-elementor-form' ),
				'label_off'    => esc_html__( 'No', 'country-code-field-for-elementor-form' ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => array(
					'field_type'               => array('tel', 'ehp-tel'),
					'ccfef-country-code-field' => 'yes',
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
				'ai'           => array(
					'active' => false,
				),
			),

			'ccfef-country-code-auto-detect' => array(
		'name'         => 'ccfef-country-code-auto-detect',
		'label'        => esc_html__( 'Auto Detect Country', 'country-code-field-for-elementor-form' ),
		'type'         => \Elementor\Controls_Manager::SWITCHER,
		'label_on'     => esc_html__( 'Yes', 'country-code-field-for-elementor-form' ),
		'label_off'    => esc_html__( 'No', 'country-code-field-for-elementor-form' ),
		'return_value' => 'yes',
		'default'      => 'no',
		'description'  => $ccfef_auto_detect_desc,
		'condition'    => array(
			'field_type'               => array('tel', 'ehp-tel'),
			'ccfef-country-code-field' => 'yes',
		),
		'tab'          => 'content',
		'inner_tab'    => 'form_fields_content_tab',
		'tabs_wrapper' => 'form_fields_tabs',
		'ai'           => array(
			'active' => false,
		),
		'disabled'     => true, // This ensures the control is always disabled.
),


			'ccfef-country-code-prefer'      => array(
				'name'         => 'ccfef-country-code-prefer',
				'label'        => esc_html__( 'Preferred Countries', 'country-code-field-for-elementor-form' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'description'  => $ccfef_prefer_desc,
				'condition'    => array(
					'field_type'               => array('tel', 'ehp-tel'),
					'ccfef-country-code-field' => 'yes',
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
				'ai'           => array(
					'active' => false,
				),
			),

		);

		

		if ( ! get_option( 'ccfef_review_notice_dismiss' ) ) {
			$review_nonce = wp_create_nonce( 'ccfef_elementor_review' );
			$url          = admin_url( 'admin-ajax.php' );
			$html         = '<div class="ccfef_elementor_review_wrapper ccfef_custom_html">';
			$html        .= '<div id="ccfef_elementor_review_dismiss" data-url="' . esc_url( $url ) . '" data-nonce="' . esc_attr( $review_nonce ) . '">Close Notice X</div>
<div class="ccfef_elementor_review_msg">Hope this addon solved your problem! <br><a href="https://wordpress.org/support/plugin/country-code-field-for-elementor-form/reviews/#new-post" target="_blank"">Share the love with a ⭐⭐⭐⭐⭐ rating.</a><br><br></div>
<div class="ccfef_elementor_demo_btn"><a href="https://wordpress.org/support/plugin/country-code-field-for-elementor-form/reviews/#new-post" target="_blank">Submit Review</a></div>
</div>';

			$field_controls['ccfef_review_notice'] = array(
				'name'            => 'ccfef_review_notice',
				'type'            => \Elementor\Controls_Manager::RAW_HTML,
				'raw'             => $html,
				'content_classes' => 'ccfef_elementor_review_notice',
				'tab'             => 'content',
				'condition'       => array(
					'field_type'               => 'tel',
					'ccfef-country-code-field' => 'yes',
				),
				'inner_tab'       => 'form_fields_content_tab',
				'tabs_wrapper'    => 'form_fields_tabs',
			);
		}

		$control_data['fields'] = \array_merge( $control_data['fields'], $field_controls );
		$widget->update_control( 'form_fields', $control_data );
	}

	// Elementor Review notice ajax request function
	public function ccfef_elementor_review_notice() {
		if ( ! check_ajax_referer( 'ccfef_elementor_review', 'nonce', false ) ) {
			wp_send_json_error( __( 'Invalid security token sent.', 'country-code-field-for-elementor-form' ) );
			wp_die( '0', 400 );
		}

		if ( isset( $_POST['ccfef_notice_dismiss'] ) && 'true' === $_POST['ccfef_notice_dismiss'] ) {
			update_option( 'ccfef_review_notice_dismiss', 'yes' );
		}
		exit;
	}

}
