<?php

/**
 *
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

require_once __DIR__ . "/class-coralpay-auth.php";
require_once __DIR__ . "/class-coralpay-customer.php";
require_once __DIR__ . "/class-coralpay-merchant-profile.php";
require_once __DIR__ . "/class-coralpay-transaction.php";
require_once __DIR__ . "/class-coralpay-config.php";


class CoralpayPaymentGateway {

    /**
     * Authentication
     * @var CoralpayPaymentAuth
     */
    private CoralpayPaymentAuth $paymentAuth;

    /**
     * The Customer to pay
     * @var ?CoralPayCustomer
     */
    private ?CoralPayCustomer $customer;

    /**
     * Payment transaction
     * @var ?CoralPayTransaction
     */
    private ?CoralPayTransaction $payment;

    /**
     * Merchant profile - for payment page customization
     * @var ?CoralPayMerchantProfile
     */
    private ?CoralPayMerchantProfile $merchantProfile;

    /**
     * @param CoralPayTransaction $payment
     */
    public function setPayment(CoralPayTransaction $payment): CoralpayPaymentGateway
    {
        $this->payment = $payment;
        return $this;
    }

    /**
     * @param CoralPayCustomer $customer
     */
    public function setCustomer(CoralPayCustomer $customer): CoralpayPaymentGateway
    {
        $this->customer = $customer;
        return $this;
    }

    /**
     * @param CoralPayMerchantProfile $merchantProfile
     */
    public function setMerchantProfile(CoralPayMerchantProfile $merchantProfile): CoralpayPaymentGateway
    {
        $this->merchantProfile = $merchantProfile;
        return $this;
    }


    /**
     * @param string $username
     * @param string $password
     * @param string $url
     */
    public function __construct(string $username, string $password, string $url)
    {
        $this->paymentAuth = new CoralpayPaymentAuth($username, $password, $url);
    }


    /**
     * @throws Exception
     */
    public function processPayment($order): array
    {
        try {
            // Retrieve payment token
            $this->paymentAuth->authenticate();

            // Begin transaction
            $paymentLink = $this->createTransaction($order);

            return  [
                'result' => 'success',
                'redirect' => $paymentLink
            ];
        }catch (Exception $exception){

            $logger = wc_get_logger();
            $logger->info($exception->getMessage()." for order - ". $order->id);
            $logger->info($exception->getTraceAsString());

            throw $exception;
        }
    }

    /**
     * @param $order
     * @throws Exception
     */
    public function createTransaction($order){
        // Build payload
        $data = $this->buildPaymentPayload($order);

        // Declare woocommerce logger
        $logger = wc_get_logger();

        // Declare Argument
        $args = array(
            'headers' => $this->paymentAuth->getHeaders(),
            'body' => wp_json_encode($data)
        );

        // Send request for authentication
        $response = wp_remote_post(  $this->paymentAuth->url."/api/v1/InvokePayment", $args );

        // Validate response
        if ( ! is_wp_error( $response ) ) {
            $response = json_decode( $response['body'], true );

            return $response['payPageLink'];
        } else {

            // Log errors
            $logger->info(wp_json_encode($response));

            $msg = 'Unable to Connect to Coralpay.';
            throw new Exception(esc_html($msg));
        }
    }


    /**
     *
     * @param $order
     * @return array
     * @throws Exception
     */
    public function buildPaymentPayload($order): array
    {
        if(is_null($this->payment))
            throw new Exception("Unable to build payload, Payment instance is not set");

        if( is_null($this->merchantProfile))
            throw new Exception("Unable to build payload, Merchant Profile is not set");

        if (is_null($this->customer))
            throw new Exception("Unable to build payload, Customer Profile is not set");

        // Create the request header
        $requestHeader = $this->paymentAuth->getSignature(CoralpayConfig::getMerchantId(), $this->payment->traceId);

        return [
            "requestHeader"=> $requestHeader,
            "traceId"=> $this->payment->traceId,
            "productId"=> (String) $order->id,
            "amount"=> $this->payment->amount,
            "currency"=> $this->payment->currency,
            "feeBearer"=> "M",
            "returnUrl"=> $this->payment->redirectUrl,
            "customer"=> [
                "email"=> $this->customer->email,
                "name"=> $this->customer->name,
                "phone"=> $this->customer->phone,
            ],
            "customization"=> [
                "logoUrl"=> $this->merchantProfile->logo,
                "title"=> $this->merchantProfile->title,
                "description"=> $this->merchantProfile->description
            ]
        ];
    }

    /**
     * @throws Exception
     */
    public function queryTransaction($traceId, $order){

        // Retrieve payment token
        $this->paymentAuth->authenticate();

        // Declare woocommerce logger
        $logger = wc_get_logger();

        $requestHeader = $this->paymentAuth->getSignature(CoralpayConfig::getMerchantId(), $traceId);

        // Declare Argument
        $args = array(
            'headers' => $this->paymentAuth->getHeaders(),
            'body' => wp_json_encode([
                "requestHeader"=> $requestHeader,
                "traceId"=> (String) $traceId
            ])
        );
        $logger->info(wp_json_encode($args));

        // Send request for authentication
        $response = wp_remote_post(  $this->paymentAuth->url."/api/v1/TransactionQuery", $args );

        // Validate response
        if ( ! is_wp_error( $response ) ) {
            $response = json_decode( $response['body'], true );

            $logger->info(wp_json_encode($response));

            // Check if txn was successful
            if($response['responseCode'] != 00){
                $note = "Transaction failed. Please, try again";

                $order->update_status( 'failed' );
                $order->add_order_note( esc_html($note), 1 );
                $order->add_order_note( __("The transaction failed through coralpay", 'coralpay-gateway-for-woocommerce') );

                throw new Exception(esc_html($note));
            }

            // Check if the payment was meant for the order
            $orderId = explode('-', $traceId)[1];
            if(((string) $order->id) != $orderId){
                $note = "Transaction order mismatch";

                $order->update_status( 'failed' );
                $order->add_order_note( esc_html($note), 1 );
                $order->add_order_note( esc_html__('Transaction order mismatch', 'coralpay-gateway-for-woocommerce') );

                throw new Exception(esc_html($note));
            }

            // Compare txn amount and order total
            if((float) $response['amount'] < (float) $order->get_total() ) {
                $order->update_status('on-hold');
                $c_note = 'Order placed on-hold because payment received amount is ' . $order->get_currency() . ' ' . $response['amount'] . ', which is less than order total ' . $order->get_currency() . ' ' . $order->get_total();
                $order->add_order_note(esc_html($c_note), 1);

                wc_add_notice( esc_html($c_note), 'notice');
                return;
            }

            // Complete order if set
            $shouldCompleteOrder = CoralpayConfig::canAutoCompleteOrder();

            if($shouldCompleteOrder){
                $order->update_status( 'completed',  __( 'Order auto-completed after successful payment.', 'coralpay-gateway-for-woocommerce' ));
            }else {
                $order->update_status( 'processing',  __( 'Order auto-completed after successful payment.', 'coralpay-gateway-for-woocommerce' ));
            }
            
            $c_note = __( 'Payment successfully processed via CoralPay.', 'coralpay-gateway-for-woocommerce' );
            $order->add_order_note($c_note, 1);
            wc_add_notice( $c_note, 'notice' );

            //Empty cart
            WC()->cart->empty_cart();

            return;
        } else {

            // Log errors
            $logger->info(wp_json_encode($response));

            throw new Exception(esc_html($response["responseMessage"]));
        }
    }

}