<?php

/**
 *
 */
class CoralpayPaymentAuth {

    /**
     * Base url
     * @var string
     */
    public string $url;

    /**
     * The merchant username
     * @var string
     */
    private string $username;

    /**
     * The merchant password
     * @var string
     */
    private string $password;

    /**
     *  Auth token retrieved from authentication
     * @var string
     */
    private string $token;

    /**
     * Auth key
     * @var string
     */
    private string $key;

    /**
     * Timestamp
     * @var string
     */
    private string $timestamp;

    /**
     * @return string
     */
    public function getTimestamp(): string
    {
        return $this->timestamp;
    }

    /**
     * @return string
     */
    public function getToken(): string
    {
        return $this->token;
    }

    public function __construct(string $username, string $password, string $url)
    {
        $this->username = $username;
        $this->password = $password;
        $this->url = $url;
    }

    public function getHeaders(){
        return [
            'Content-Type' => 'application/json',
            'Authorization' => "Bearer ".$this->getToken()
        ];
    }

    /**
     * Authenticate merchant for payment
     * @throws Exception
     */
    public function authenticate(): void
    {
        $msg = 'Unable to Connect to Coralpay. Please contact the administrator';

        // Declare woocommerce logger
        $logger = wc_get_logger();

        // Declare payload
        $args = array(
            'method'  => 'POST',
            'headers' => [ 'Content-Type' => 'application/json'],
            'body' => wp_json_encode([
                'username' => $this->username,
                'password' => $this->password,
            ])
        );

        // Send request for authentication
        $response = wp_safe_remote_request(  $this->url."/api/v1/Authentication", $args );
        $logger->info(wp_json_encode($response));

        // Validate response
        if ( is_wp_error( $response )
            || wp_remote_retrieve_response_code($response) != 200
        ) {
            // Log errors
            $logger->info(wp_json_encode($response));
            throw new Exception(esc_html($msg));
        }

        // Retrieve token and key
        $response = json_decode( $response['body'], true );

        if( strtolower($response['status']) != "success")
        {
            $logger->info(wp_json_encode($response));
            throw new Exception(esc_html($msg));
        }

        $this->token = $response['token'];
        $this->key = $response['key'];
    }


    public function getSignature($merchantId, $traceId): array {
        // Get timestamp
        $this->timestamp = time();

        // Build the signature
        $signature = $merchantId. $traceId . $this->timestamp . $this->key;
        $hash = hash('sha256', ($signature));

        return [
            "merchantId"=> $merchantId,
            "timeStamp"=> $this->getTimestamp(),
            "signature"=> $hash
        ];
    }


}