<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

require_once __DIR__ . "/sdk/class-coralpay-payment-gateway-sdk.php";
require_once __DIR__ . "/sdk/class-coralpay-config.php";


class WC_CoralPay_Payment_Gateway extends WC_Payment_Gateway {

    protected CoralpayPaymentGateway $gateway;
    private array $options = [];

    public function __construct() {

        $this->options = CoralpayConfig::getSetttings();

        $this->id                 = 'coralpay';
        $this->icon               = plugins_url( 'assets/img/logo.svg', WC_CORALPAY_PLUGIN_FILE );
        $this->has_fields         = false;
        $this->method_title       = "Coralpay";
        $this->method_description = "CoralPay WooCommerce is a robust and secure payment gateway plugin that seamlessly integrates with WooCommerce to provide your customers with a smooth checkout experience.";
        $this->supports            = array('products');
        $this->enabled      =   $this->get_option('enabled', "no");

        $this->init_form_fields();
        $this->init_settings();


        $credentials = CoralpayConfig::getCredentials();

        $this->gateway = new CoralpayPaymentGateway(
            $credentials['username'],
            $credentials['password'],
            $credentials['url']
        );

        $this->title = $this->get_option('title', 'CoralPay');
        $this->description = $this->get_option('description', 'Official WooCommerce payment gateway for CoralPay.');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action( 'woocommerce_api_coralpay_gateway', array( $this, 'coralpay_verify_payment' ) );
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title'   => 'Enable/Disable',
                'type'    => 'checkbox',
                'label'   => 'Enable CoralPay Payment',
                'default' => 'no',
            ),
            'logo' => array(
                'title'       => 'Logo URL',
                'type'        => 'text',
                'description' => 'Your business logo, Optional',
                'default'     => '',
                'desc_tip'    => true,
            ),
            'business_name' => array(
                'title'       => 'Business Name',
                'type'        => 'text',
                'default'     => '',
            ),
            'business_description' => array(
                'title'       => 'Business Description',
                'type'        => 'text',
                'description' => 'Optional',
                'default'     => '',
            ),
            'title' => array(
                'title'       => 'Title',
                'type'        => 'text',
                'description' => 'Optional',
                'default'     => 'CoralPay Payment',
            ),
            'description' => array(
                'title'       => 'Description',
                'type'        => 'text',
                'description' => 'Optional',
            ),
            'merchant_id' => array(
                'title'       => 'Merchant ID ',
                'type'        => 'text',
                'description' => 'This controls the title users see during checkout.',
                'default'     => '',
                'desc_tip'    => true,
            ),
            'test_username' => array(
                'title'       => 'Test Username ',
                'type'        => 'text',
                'default'     => '',
            ),
            'test_password' => array(
                'title'       => 'Test Password ',
                'type'        => 'password',
                'default'     => '',
            ),
            'mode' => array(
                'title'   => 'Live Mode',
                'type'    => 'checkbox',
                'label'   => 'Live Mode',
                'default' => 'no',
            ),
            'live_username' => array(
                'title'       => 'Live Username',
                'type'        => 'text',
                'default'     => '',
            ),
            'live_password' => array(
                'title'       => 'Live Password',
                'type'        => 'password',
                'default'     => '',
            ),
            'autocomplete_order' => array(
                'title'   => 'Auto-Complete Order',
                'type'    => 'checkbox',
                'label'   => 'Auto Complete Order',
                'default' => 'no',
            ),
        );
    }

    public function getRequiredFields(bool $isLive){
        $fields = ["business_name", "logo", "merchant_id",];
        if($isLive){
            $fields = [...$fields, "live_username", "live_password"];
        }else{
            $fields = [...$fields, "test_username", "test_password"];
        }
        return $fields;
    }


    public function process_admin_options() {
        $hasError = false;
        $data = $this->get_post_data();

        // check if live mode
        $isLive = isset($data['woocommerce_' .$this->id.'_mode']);

        foreach ($this->getRequiredFields($isLive) as $field){
            $value = trim($data['woocommerce_' .$this->id.'_'.$field]);
            if (empty($value)) {
                $hasError = true;
                $error = str_replace("_"," ", ucfirst($field) .' is required.');
                WC_Admin_Settings::add_error($error);
            }
        }

        // if has any error ? don't save options
        if(!$hasError){
            parent::process_admin_options();
        }
    }

    public function validate_fields() {
        $isLive = $this->get_option('mode') == 'yes';
        foreach ($this->getRequiredFields($isLive) as $field){
            if (empty($this->get_option($field))) {
                wc_add_notice(__('CoralPay configuration are missing. Contact the admin to resolve this issue', 'coralpay-gateway-for-woocommerce'), 'error');
                return false;
            }
        }

        return true;
    }


    public function process_payment($order_id) {
        $order = wc_get_order($order_id);

         /* Initialise Customer info*/
        $first_name = $order->get_billing_first_name();
        $last_name = $order->get_billing_last_name();
        $customer = new CoralPayCustomer($order->get_billing_email(),  $first_name . ' ' . $last_name,  $order->get_billing_phone());

        /* Initialise Merchant info */
        $logo = $this->get_option('logo', "");
        $business_name = $this->get_option('business_name', "CoralPay Verge");
        $business_description = $this->get_option('business_description');
        $merchantProfile = new CoralPayMerchantProfile($logo, $business_name, $business_description);

        /* Create transaction & redirect Url */
        $payment = new CoralPayTransaction($order);

        $this->gateway->setCustomer($customer)
            ->setMerchantProfile($merchantProfile)
            ->setPayment($payment);


        return $this->gateway->processPayment($order);
    }

    public function coralpay_verify_payment(){

        /*
         * Unverified payment: something's not right
         * */
        if (
            ! isset( $_GET['_wpnonce'], $_GET['id'], $_GET['order_id'] ) ||
            ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'verify_payment' )
        ) {
            wc_add_notice( "Payment could not be verified", 'error' );
            wp_safe_redirect( wc_get_checkout_url() );
            exit;
        }


        /*
         * Payment failed
         * */
//        $status = $_GET['status'];
//        if ( !isset( $status ) || $status != 'success') {
//            wc_add_notice( "Payment failed. Please try again or use a different payment method", 'error' );
//            wp_redirect( wc_get_checkout_url() );
//            exit;
//        }


        /*
         * Verify payment
         * */
        $order = wc_get_order(absint( wp_unslash( $_GET['order_id'] ) ));
        $traceId = sanitize_text_field( wp_unslash( $_GET['id'] ) );
        
        try {
            $this->gateway->queryTransaction($traceId, $order);

            wp_redirect($this->get_return_url( $order ));
        }catch(Exception $error){
            $msg = !is_null($error->getMessage())? $error->getMessage() : 'Unable to verify transaction, Please try again';
            wc_add_notice( $msg, 'notice' );
            wp_redirect( wc_get_cart_url() );
        }

    }

    public function is_available(): bool
    {
        return $this->enabled == "yes";
    }
}
