<?php
/*
Plugin Name: Copy Pages and Posts by DMUK
Plugin URI: https://diversify.me.uk/copy-pages-and-posts-wordpress-plugin
Description: A plugin to copy pages and posts.
Version: 1.0.0
Author: Diversify.me.uk
Author URI: https://diversify.me.uk
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: copy-pages-and-posts-by-dmuk
*/


// File Name: copy-pages-and-posts-by-dmuk.php 
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Load the plugin text domain for translations
// load_plugin_textdomain( 'copy-pages-and-posts-by-dmuk', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );

// Require common functions to be included
// Included file below has been amended with if ( ! defined( 'ABSPATH' ) ) exit; at the top to address issue raised by WP
require_once plugin_dir_path(__FILE__) . 'dmuk-cpp-common-functions.php';

// Enqueue and localize script for handling the link click
add_action('admin_enqueue_scripts', 'dmuk_cpp_enqueue_admin_scripts');
function dmuk_cpp_enqueue_admin_scripts() {
    // Get the file modification time for cache busting
    $script_version = filemtime(plugin_dir_path(__FILE__) . 'dmuk_cpp_script.js');
    
    wp_enqueue_script('dmuk-cpp-js', plugin_dir_url(__FILE__) . 'dmuk_cpp_script.js', array('jquery'), $script_version, true);
    
    // Create nonce and pass it to JavaScript
    wp_localize_script('dmuk-cpp-js', 'dmukCppVars', array(
        'nonce' => wp_create_nonce('dmuk_cpp_copy_action') // Pass nonce to JavaScript
    ));
}


// FILTER HOOKS
// Hook into the post row actions for pages and posts and add copy link functions
add_filter('post_row_actions', 'dmuk_cpp_add_copy_post_link', 10, 2);
add_filter('page_row_actions', 'dmuk_cpp_add_copy_page_link', 10, 2);
function dmuk_cpp_add_copy_post_link($actions, $post) {
    if ($post->post_type === 'post') {
        $actions['copy_post'] = '<a href="#" class="dmuk-cpp-copy-post-link">Copy This Post</a>';
    }
    return $actions;
}

function dmuk_cpp_add_copy_page_link($actions, $post) {
    if ($post->post_type === 'page') {
        $actions['copy_page'] = '<a href="#" class="dmuk-cpp-copy-page-link">Copy This Page</a>';
    }
    return $actions;
}

// Enqueue the script for handling the link click
add_action('admin_footer', 'dmuk_cpp_admin_footer');
function dmuk_cpp_admin_footer() {
    $nonce = wp_create_nonce('dmuk_cpp_copy_action');
}

// Handle the request to show the copy form (pages and posts)
add_action('wp_ajax_dmuk_cpp_show_copy_form', 'dmuk_cpp_show_copy_form');

function dmuk_cpp_show_copy_form() {
    // Check required fields
    if (!isset($_POST['post_id'], $_POST['post_type'], $_POST['dmuk_cpp_nonce'])) {
        wp_send_json_error(['message' => 'Missing required parameters.']);
    }

    // Sanitize and verify nonce
    $nonce = sanitize_text_field(wp_unslash($_POST['dmuk_cpp_nonce']));
    if (!wp_verify_nonce($nonce, 'dmuk_cpp_copy_action')) {
        wp_send_json_error(['message' => 'Invalid nonce.']);
    }

    // Sanitize post data
    $post_id = absint($_POST['post_id']);
    $post_type = sanitize_text_field(wp_unslash($_POST['post_type']));
    
    // Validate post existence
    $post = get_post($post_id);
    if (!$post || $post->post_type !== $post_type) {
        wp_send_json_error(['message' => 'Post/Page not found.']);
    }

    // Prepare response
    $existing_name = $post->post_title;
    $new_name = $existing_name . ' ' . current_time('Y-m-d_H-i-s');

    wp_send_json_success([
        'existing_name' => esc_html($existing_name),
        'new_name' => esc_html($new_name),
        'type' => ucfirst(esc_html($post_type))
    ]);
}

// The central function to call relevant copy function, get result and display alert() 
// Handle the actual post/page copy
add_action('wp_ajax_dmuk_cpp_copy', 'dmuk_cpp_copy');
function dmuk_cpp_copy() {
    if (!isset($_POST['dmuk_cpp_nonce'])) {
        wp_send_json_error(['message' => 'Missing nonce.']);
    }
    
    $nonce = sanitize_text_field(wp_unslash($_POST['dmuk_cpp_nonce']));
    if (!wp_verify_nonce($nonce, 'dmuk_cpp_copy_action')) {
        wp_send_json_error(['message' => 'Invalid nonce.']);
    }
    
    if (!isset($_POST['post_id'], $_POST['new_name'], $_POST['post_type'])) {
        wp_send_json_error(['message' => 'Missing required fields.']);
    }
    
    $post_id   = absint($_POST['post_id']);
    $new_name  = sanitize_text_field(wp_unslash($_POST['new_name']));
    $post_type = sanitize_text_field(wp_unslash($_POST['post_type']));
    $copy_to_new_post = isset($_POST['copy_to_new_post']) ? intval($_POST['copy_to_new_post']) : 0;
    
    // Check if a post with the same name exists
    $existing_post = get_posts([
        'title'        => $new_name,
        'post_type'    => $post_type,
        'post_status'  => 'any',
        'numberposts'  => 1,
        'fields'       => 'ids',
    ]);
    
    if (!empty($existing_post)) {
        $new_name .= ' ' . current_time('Y-m-d_H-i-s');
    }
    
    $success_message = "";
    $failure_message = "";
    $new_post_id = null;
    
    if ($copy_to_new_post) {
        if ($post_type === 'page') {
            $success_message = "You have copied your PAGE to a POST. You will need to open your POSTS view to see it. ".
		"WARNING - POSTS can have different settings and linked content. Click on - Visit plugin Site - in the Dashboard / Plugins listing to see how to mitigate any issues that may arise.";
            $failure_message = "Failed to copy page to post. If this error persists ask your admin to enable error logging in wp-config.php and inspect the resulting debug.log.";
            $new_post_id = dmuk_cpp_copy_page_to_post($post_id, $new_name);
        } elseif ($post_type === 'post') {
            $success_message = "You have copied your POST to a PAGE. You will need to open your PAGES view to see it. ".
		"WARNING - PAGES have different settings and linked content. Click on - Visit plugin Site - in the Dashboard / Plugins listing to see how to mitigate any issues that may arise.";
            $failure_message = "Failed to copy post to page. If this error persists ask your admin to enable error logging in wp-config.php and inspect the resulting debug.log.";
            $new_post_id = dmuk_cpp_copy_post_to_page($post_id, $new_name);
        }
    } else {
        if ($post_type === 'page') {
            $success_message = "You will need to refresh this list to see it. ".
		"WARNING - Content has been copied but there may be custom settings. Click on - Visit plugin Site - in the Dashboard / Plugins listing to see how to mitigate any issues that may arise.";
             $failure_message = "Failed to copy page. If this error persists ask your admin to enable error logging in wp-config.php and inspect the resulting debug.log.";
            $new_post_id = dmuk_cpp_copy_page($post_id, $new_name);
        } elseif ($post_type === 'post') {
            $success_message = "You will need to refresh this list to see it. ".
		"WARNING - Content has been copied but there may be custom settings. Click on - Visit plugin Site - in the Dashboard / Plugins listing to see how to mitigate any issues that may arise.";
             $failure_message = "Failed to copy post. If this error persists ask your admin to enable error logging in wp-config.php and inspect the resulting debug.log.";
            $new_post_id = dmuk_cpp_copy_post($post_id, $new_name);
        }
    }
    
    if ($new_post_id) {
		wp_send_json_success([
        'message' => ucfirst($post_type) . ' copied successfully! New file name: ' . $new_name . ' (Saved as draft) - ' . $success_message
    ]);
	} else {
		wp_send_json_error([
        'message' => 'Error: ' . ucfirst($post_type) . ' not found. - ' . $failure_message
		]);
	}

}


// === Plugin Row Meta: Add Donate and How to Use Links ===
function dmuk_cpp_add_meta_links( $links, $file ) {
    // Add links if current plugin matches this file
    if ( plugin_basename( __FILE__ ) === $file ) {
        $donate_link = '<a href="' . esc_url( 'https://diversify.me.uk/donate/' ) . '" target="_blank">'
            . wp_kses_post( esc_html__( 'Donate', 'copy-pages-and-posts-by-dmuk' ) ) . '</a>';
        $how_to_use_link = '<a href="' . esc_url( 'https://diversify.me.uk/copy-pages-and-posts-wordpress-plugin/' ) . '" target="_blank">'
            . wp_kses_post( esc_html__( 'How to Use', 'copy-pages-and-posts-by-dmuk' ) ) . '</a>';

        array_push( $links, $donate_link, $how_to_use_link );
    }
    return $links;
}
add_filter( 'plugin_row_meta', 'dmuk_cpp_add_meta_links', 10, 2 );

// Enqueue CSS with timestamp
function dmuk_cpp_enqueue_styles() {
    $css_file = plugin_dir_url(__FILE__) . 'dmuk-cpp-style.css';
    $css_path = plugin_dir_path(__FILE__) . 'dmuk-cpp-style.css';
    $version = file_exists($css_path) ? filemtime($css_path) : time(); // Fallback to current time if file not found
    wp_enqueue_style('dmuk-cpp-style', $css_file, [], $version);
}

add_action('admin_enqueue_scripts', 'dmuk_cpp_enqueue_styles');
?>
