<?php
/**
 * Cool Payment Gateway Discounts for WooCommerce Discount
 *
 * @package Cool Payment Gateway Discounts for WooCommerce
 * @version 1.0.0
 */

namespace Manca\CoolPaymentGatewayDiscountsForWoo\Cart;

defined( 'ABSPATH' ) || exit;

/**
 * A class that holds all our discount logic
 */
class Discount {


	/**
	 * Initialize discount.
	 *
	 * @since 0.0.1
	 */
	public function __construct() {
		add_action( 'woocommerce_cart_calculate_fees', array( $this, 'apply_discount' ) );
		add_action( 'woocommerce_after_checkout_form', array( $this, 'add_jsscript_to_checkout' ) );
	}


	/**
	 * Get discount for the payment method.
	 *
	 * @param string $payment_method_id Payment method id.
	 * @since 0.0.1
	 * @return array Discount.
	 */
	public function get_discount_for_payment_method( $payment_method_id ) {
		$pct = get_option( 'cool_payment_gateway_discounts_for_woo_' . $payment_method_id . '_discount', 0 );
		$txt = get_option( 'cool_payment_gateway_discounts_for_woo_' . $payment_method_id . '_discount_text', '' );
		return array(
			'pct' => $pct,
			'txt' => $txt,
		);
	}

	/**
	 * Apply discount.
	 *
	 * @since 0.0.1
	 */
	public function apply_discount() {
		$payment_method = WC()->session->get( 'chosen_payment_method' );
		$discount       = $this->get_discount_for_payment_method( $payment_method );
		if ( ! empty( $discount['pct'] ) ) {
			$label  = $discount['txt'];
			$amount = -1 * floatval( WC()->cart->get_subtotal() * floatval( $discount['pct'] ) / 100 );
			WC()->cart->add_fee( $label, $amount, true, 'standard' );
		}
	}

	/**
	 * Add js script to checkout.
	 *
	 * @since 0.0.1
	 */
	public function add_jsscript_to_checkout() {
		wp_enqueue_script( 'cool-payment-gateway-discounts-for-woo-checkout' );
	}
}
