<?php
/**
 * Cool WC Logger - base class for Woo Loggers
 *
 * @package  Manca\CoolPaymentGatewayDiscountsForWoo\Dependencies\Manca\Coolwplib\Base
 */

namespace Manca\CoolPaymentGatewayDiscountsForWoo\Dependencies\Manca\Coolwplib\Base;

use Manca\CoolPaymentGatewayDiscountsForWoo\Dependencies\Manca\Coolwplib\Base\CoolPlugin;

defined( 'ABSPATH' ) || exit;

/**
 * Plugin's base Class
 */
abstract class CoolWCLogger {

	/**
	 * Instances of Loggers.
	 *
	 * @since 0.0.1
	 * @var array $instances array of CoolWCLoggers.
	 */
	protected static $instances = array();

	/**
	 * Instance of Logger.
	 *
	 * @since 0.0.1
	 * @var array $logger WCLogger.
	 */
	private static $logger;

	/**
	 * Get Logger Name.
	 *
	 * @since 0.0.1
	 * @return string Name to be used as File Name Identifier by Woo Logger.
	 */
	abstract protected function get_logger_name();

	/**
	 * Get Enabled?.
	 *
	 * @since 0.0.8
	 * @return bool Name to be used as File Name Identifier by Woo Logger.
	 */
	abstract protected function is_enabled();


	/**
	 * Get Instance - Singletton.
	 *
	 * @since 0.0.1
	 * @return CoolWCLogger Instance.
	 */
	public static function get_instance() {
		$cls = static::class;
		if ( ! isset( self::$instances[ $cls ] ) ) {
			self::$instances[ $cls ] = new static();
			self::$instances[ $cls ]->init();
		}

		return self::$instances[ $cls ];
	}

	/**
	 * Constructor.
	 *
	 * @since 0.0.1
	 */
	protected function __construct() { }

	/**
	 * Clone.
	 *
	 * @since 0.0.1
	 */
	protected function __clone() { }

	/**
	 * WakeUp.
	 *
	 * @since 0.0.1
	 * @throws \Exception Cannot WakeUp a Singleton.
	 */
	public function __wakeup() {
		throw new \Exception( 'Cannot unserialize a singleton.' );
	}

	/**
	 * Inits our logger singleton
	 *
	 * @return void
	 */
	public function init() {
		if ( function_exists( 'wc_get_logger' ) && ! isset( self::$logger ) ) {
			self::$logger = wc_get_logger();
		}
	}

	/**
	 * Logs an info message.
	 *
	 * @param mixed $msg Message to log.
	 * @return void
	 */
	public function log_info( $msg ) {
		if ( $this->is_enabled() ) {
			self::$logger->info( wc_print_r( $msg, true ), array( 'source' => $this->get_logger_name() ) );
		}
	}

	/**
	 * Logs an error message.
	 *
	 * @param mixed $msg Message to log.
	 * @return void
	 */
	public function log_error( $msg ) {
		if ( $this->is_enabled() ) {
			self::$logger->error( wc_print_r( $msg, true ), array( 'source' => $this->get_logger_name() ) );
		}
	}

	/**
	 * Logs an warning message.
	 *
	 * @param mixed $msg Message to log.
	 * @return void
	 */
	public function log_warning( $msg ) {
		if ( $this->is_enabled() ) {
			self::$logger->warning( wc_print_r( $msg, true ), array( 'source' => $this->get_logger_name() ) );
		}
	}

	/**
	 * Logs a debug message.
	 *
	 * @param mixed $msg Message to log.
	 * @return void
	 */
	public function log_debug( $msg ) {
		if ( $this->is_enabled() ) {
			self::$logger->debug( wc_print_r( $msg, true ), array( 'source' => $this->get_logger_name() ) );
		}
	}
}
