<?php
/**
 * Cool Cron - base class for crons
 *
 * @package  Manca\CoolPaymentGatewayDiscountsForWoo\Dependencies\Manca\Coolwplib\Base
 */

namespace Manca\CoolPaymentGatewayDiscountsForWoo\Dependencies\Manca\Coolwplib\Base;

defined( 'ABSPATH' ) || exit;

/**
 * Cron Processor's Main Class
 */
abstract class CoolCron {

	/**
	 * Get Cron Identifier.
	 *
	 * @since 0.0.1
	 * @return string Identifier.
	 */
	abstract protected function get_cron_id();

	/**
	 * Get Schedule Interval to run cron (in seconds).
	 *
	 * @since 0.0.1
	 * @return int Seconds.
	 */
	abstract protected function get_schedule_interval();

	/**
	 * Get Schedule Name.
	 *
	 * @since 0.0.1
	 * @return string Name.
	 */
	abstract protected function get_schedule_name();

	/**
	 * Run Cron Statements.
	 *
	 * @since 0.0.1
	 */
	abstract protected function run();

	/**
	 * Constructor.
	 *
	 * @since 0.0.1
	 */
	public function __construct() {
		add_filter( 'cron_schedules', array( $this, 'add_schedule' ) ); // phpcs:ignore WordPress.WP.CronInterval.ChangeDetected -- verified: for dynamics purposes.
		add_action( $this->get_cron_id(), array( $this, 'run_cron' ) );
	}

	/**
	 * Run Cron main Handler.
	 *
	 * @since 0.0.1
	 */
	public function run_cron() {
		$now = get_date_from_gmt( 'now', 'Y-m-d' );
		update_option( 'manca_coolwplib_' . $this->get_cron_id() . '_last-run', $now, 'yes' );
		$this->run();
	}

	/**
	 * Add New Schedule Time
	 *
	 * @since 0.0.1
	 * @param array $schedules Array of Schedules.
	 * @return array Schedules.
	 */
	public function add_schedule( $schedules ) {
		$schedules[ $this->get_schedule_name() ] = array(
			'interval' => $this->get_schedule_interval(),
			'display'  => $this->get_cron_id(),
		);
		return $schedules;
	}

	/**
	 * Cron Schedule Start.
	 *
	 * @since 0.0.1
	 */
	public function schedule_start() {
		if ( ! wp_next_scheduled( $this->get_cron_id() ) ) {
			wp_schedule_event( time(), $this->get_schedule_name(), $this->get_cron_id() );
		}
		wp_schedule_single_event( time(), $this->get_cron_id(), array() );
		spawn_cron();
	}

	/**
	 * Cron Schedule Stop.
	 *
	 * @since 0.0.1
	 */
	public function schedule_stop() {
		if ( wp_next_scheduled( $this->get_cron_id() ) ) {
			wp_clear_scheduled_hook( $this->get_cron_id() );
		}
	}
}
