<?php
/**
 * Settings management class
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Cookifi_Settings {

    private $option_group = 'cookifi_cmp_settings';
    private $page_slug = 'cookifi';

    /**
     * Initialize settings
     */
    public function init() {
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    /**
     * Add settings page to WordPress admin
     */
    public function add_settings_page() {
        add_options_page(
            __('Cookifi Settings', 'cookifi'),
            __('Cookifi', 'cookifi'),
            'manage_options',
            $this->page_slug,
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        // Register API Key setting
        register_setting(
            $this->option_group,
            'cookifi_cmp_api_key',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_api_key'),
                'default' => ''
            )
        );

        // Register Debug Mode setting
        register_setting(
            $this->option_group,
            'cookifi_cmp_debug_mode',
            array(
                'type' => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'default' => false
            )
        );

        // Register Hidden Mode setting
        register_setting(
            $this->option_group,
            'cookifi_cmp_hidden_mode',
            array(
                'type' => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'default' => false
            )
        );

        // Register Consent Provider Defaults setting
        register_setting(
            $this->option_group,
            'cookifi_cmp_consent_defaults',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_consent_defaults'),
                'default' => ''
            )
        );

        // Add settings section
        add_settings_section(
            'cookifi_cmp_main_section',
            __('Configuration', 'cookifi'),
            array($this, 'render_section_description'),
            $this->page_slug
        );

        // Add API Key field
        add_settings_field(
            'cookifi_cmp_api_key',
            __('API Key', 'cookifi'),
            array($this, 'render_api_key_field'),
            $this->page_slug,
            'cookifi_cmp_main_section'
        );

        // Add Debug Mode field
        add_settings_field(
            'cookifi_cmp_debug_mode',
            __('Debug Mode', 'cookifi'),
            array($this, 'render_debug_mode_field'),
            $this->page_slug,
            'cookifi_cmp_main_section'
        );

        // Add Hidden Mode field
        add_settings_field(
            'cookifi_cmp_hidden_mode',
            __('Hidden Mode', 'cookifi'),
            array($this, 'render_hidden_mode_field'),
            $this->page_slug,
            'cookifi_cmp_main_section'
        );

        // Add Consent Provider Defaults field
        add_settings_field(
            'cookifi_cmp_consent_defaults',
            __('Consent Provider Defaults', 'cookifi'),
            array($this, 'render_consent_defaults_field'),
            $this->page_slug,
            'cookifi_cmp_main_section'
        );
    }

    /**
     * Sanitize API Key
     */
    public function sanitize_api_key($value) {
        // Remove whitespace and validate format
        $value = trim($value);

        // Basic validation - adjust pattern as needed
        if (!empty($value) && !preg_match('/^COOKIFI_[A-Z0-9]+$/i', $value)) {
            add_settings_error(
                'cookifi_cmp_api_key',
                'invalid_api_key',
                __('Invalid API Key format. Expected format: COOKIFI_XXXXXXXXXX', 'cookifi'),
                'error'
            );
            return get_option('cookifi_cmp_api_key', '');
        }

        return sanitize_text_field($value);
    }

    /**
     * Sanitize Consent Provider Defaults
     */
    public function sanitize_consent_defaults($value) {
        // Trim whitespace
        $value = trim($value);

        // Remove PHP tags for security
        $value = str_replace(array('<?php', '<?', '?>'), '', $value);

        // Remove any <script> tags if user accidentally included them
        // The plugin will add <script> tags automatically during injection
        $value = preg_replace('/<\/?script[^>]*>/i', '', $value);
        $value = trim($value);

        // If not empty, validate basic structure
        if (!empty($value)) {
            // Check for expected consent mode patterns (gtag or uetq)
            $has_google = stripos($value, 'gtag') !== false && stripos($value, 'dataLayer') !== false;
            $has_uet = stripos($value, 'uetq') !== false;

            if (!$has_google && !$has_uet) {
                add_settings_error(
                    'cookifi_cmp_consent_defaults',
                    'invalid_consent_defaults',
                    __('Consent defaults must contain valid Google Consent Mode or Microsoft UET configuration. Please copy the script from your Cookifi dashboard.', 'cookifi'),
                    'error'
                );
                return '';
            }

            // Basic XSS prevention - check for suspicious patterns
            $dangerous_patterns = array(
                'javascript:',
                'onerror=',
                'onload=',
                '<iframe',
                '<embed',
                '<object'
            );

            foreach ($dangerous_patterns as $pattern) {
                if (stripos($value, $pattern) !== false) {
                    add_settings_error(
                        'cookifi_cmp_consent_defaults',
                        'suspicious_content',
                        __('Consent defaults contain potentially unsafe content. Please use only the script provided by Cookifi.', 'cookifi'),
                        'error'
                    );
                    return '';
                }
            }
        }

        return $value;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our settings page
        if ('settings_page_cookifi' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'cookifi-admin',
            COOKIFI_PLUGIN_URL . 'admin/assets/css/style.css',
            array(),
            COOKIFI_VERSION
        );
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Check if settings were saved
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WordPress core handles nonce for settings page
        if (isset($_GET['settings-updated'])) {
            add_settings_error(
                'cookifi_cmp_messages',
                'cookifi_cmp_message',
                __('Settings saved successfully.', 'cookifi'),
                'success'
            );
        }

        settings_errors('cookifi_cmp_messages');

        require_once COOKIFI_PLUGIN_DIR . 'admin/settings.php';
    }

    /**
     * Render section description
     */
    public function render_section_description() {
        echo '<p>' . esc_html__('Configure your Cookifi Consent Management Platform integration.', 'cookifi') . '</p>';
    }

    /**
     * Render API Key field
     */
    public function render_api_key_field() {
        $value = get_option('cookifi_cmp_api_key', '');
        ?>
        <input
            type="text"
            id="cookifi_cmp_api_key"
            name="cookifi_cmp_api_key"
            value="<?php echo esc_attr($value); ?>"
            class="regular-text"
            placeholder="COOKIFI_XXXXXXXXXX"
        />
        <p class="description">
            <?php
            /* translators: %s: URL to Cookifi admin dashboard */
            printf(
                esc_html__('Enter your Cookifi API Key. Don\'t have one? ', 'cookifi') . '<a href="%s" target="_blank">' . esc_html__('Sign up here', 'cookifi') . '</a>.',
                esc_url('https://admin.cooki.fi')
            );
            ?>
        </p>
        <?php
    }

    /**
     * Render Debug Mode field
     */
    public function render_debug_mode_field() {
        $value = get_option('cookifi_cmp_debug_mode', false);
        ?>
        <label>
            <input
                type="checkbox"
                id="cookifi_cmp_debug_mode"
                name="cookifi_cmp_debug_mode"
                value="1"
                <?php checked($value, true); ?>
            />
            <?php esc_html_e('Enable debug mode', 'cookifi'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, the consent dialog will show additional debugging information.', 'cookifi'); ?>
        </p>
        <?php
    }

    /**
     * Render Hidden Mode field
     */
    public function render_hidden_mode_field() {
        $value = get_option('cookifi_cmp_hidden_mode', false);
        ?>
        <label>
            <input
                type="checkbox"
                id="cookifi_cmp_hidden_mode"
                name="cookifi_cmp_hidden_mode"
                value="1"
                <?php checked($value, true); ?>
            />
            <?php esc_html_e('Enable hidden mode', 'cookifi'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, the consent dialog will be hidden (useful for testing).', 'cookifi'); ?>
        </p>
        <?php
    }

    /**
     * Render Consent Provider Defaults field
     */
    public function render_consent_defaults_field() {
        $value = get_option('cookifi_cmp_consent_defaults', '');
        ?>
        <textarea
            id="cookifi_cmp_consent_defaults"
            name="cookifi_cmp_consent_defaults"
            class="large-text code"
            rows="12"
            placeholder="<?php echo esc_attr( "<script>\nwindow.dataLayer = window.dataLayer || [];\nfunction gtag(){dataLayer.push(arguments);}\ngtag('consent', 'default', {...});\n</script>" ); ?>"
        ><?php echo esc_textarea($value); ?></textarea>
        <p class="description">
            <?php esc_html_e('Paste the consent provider defaults script from your Cookifi dashboard. This configures consent modes for your marketing and analytics integrations (e.g., Google Tag Manager, Microsoft Advertising, etc.). Leave empty if you don\'t use consent mode.', 'cookifi'); ?>
            <br>
            <strong><?php esc_html_e('Where to find this:', 'cookifi'); ?></strong>
            <?php
            printf(
                /* translators: %s: URL link to admin.cooki.fi */
                esc_html__('Log in to %s, go to your property settings, and copy the "Consent Provider Defaults" script.', 'cookifi'),
                '<a href="' . esc_url('https://admin.cooki.fi') . '" target="_blank">admin.cooki.fi</a>'
            );
            ?>
        </p>
        <?php
    }
}
