<?php
/**
 * Script injection class
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Cookifi_Script_Injector {

    /**
     * Initialize script injector
     */
    public function init() {
        // Hook into wp_enqueue_scripts to properly enqueue scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'), 1);
        // Add filter to customize script tag attributes
        add_filter('script_loader_tag', array($this, 'add_script_attributes'), 10, 3);
    }

    /**
     * Enqueue Cookifi scripts
     */
    public function enqueue_scripts() {
        // Get options
        $api_key = get_option('cookifi_cmp_api_key', '');
        $consent_defaults = get_option('cookifi_cmp_consent_defaults', '');

        // Don't enqueue if API key is not set
        if (empty($api_key)) {
            // Only show message to admins in HTML comments via wp_head
            if (current_user_can('manage_options')) {
                add_action('wp_head', function() {
                    echo '<!-- Cookifi: API Key not configured. Visit Settings > Cookifi to configure. -->' . "\n";
                }, 1);
            }
            return;
        }

        // Register and enqueue the Cookifi script
        wp_enqueue_script(
            'cookifi-cmp',
            'https://lib.cooki.fi/index.js',
            array(), // No dependencies
            COOKIFI_VERSION, // Use plugin version for cache busting
            false // Load in head (not footer)
        );

        // Add consent provider defaults as inline script BEFORE the main Cookifi script
        // This must execute before the Cookifi library initializes
        if (!empty($consent_defaults)) {
            wp_add_inline_script(
                'cookifi-cmp',
                $consent_defaults,
                'before'
            );
        }
    }

    /**
     * Add custom attributes to Cookifi script tag
     *
     * @param string $tag    The script tag HTML.
     * @param string $handle The script handle.
     * @param string $src    The script source URL.
     * @return string Modified script tag.
     */
    public function add_script_attributes($tag, $handle, $src) {
        // Only modify our Cookifi script
        if ('cookifi-cmp' !== $handle) {
            return $tag;
        }

        // Get options for custom attributes
        $api_key = get_option('cookifi_cmp_api_key', '');
        $debug_mode = get_option('cookifi_cmp_debug_mode', false);
        $hidden_mode = get_option('cookifi_cmp_hidden_mode', false);

        // Build custom attributes
        $custom_attrs = array();
        $custom_attrs[] = 'fetchpriority="high"';
        $custom_attrs[] = 'data-apikey="' . esc_attr($api_key) . '"';

        if ($debug_mode) {
            $custom_attrs[] = 'data-debug="true"';
        }

        if ($hidden_mode) {
            $custom_attrs[] = 'data-hidden="true"';
        }

        // Insert custom attributes before the src attribute
        $tag = str_replace(' src=', ' ' . implode(' ', $custom_attrs) . ' src=', $tag);

        return $tag;
    }

    /**
     * Get script status for admin notices
     */
    public function get_script_status() {
        $api_key = get_option('cookifi_cmp_api_key', '');
        return !empty($api_key);
    }
}
