<?php
/**
 * Cookie Opt for WordPress
 *
 * @package Cookie Optimizer
 * @author Optimizer Inc.
 * @copyright 2023- Optimizer Inc.
 * @license GNU General Public License, version 2
 *
 * @wordpress-plugin
 * Plugin Name: Cookie Optimizer
 * Plugin URI: https://manage.medipartner.jp
 * Description: Cookieをユーザーに見える化し、各法律にする準拠バナー同意ソリューション
 * Version: 1.0.19
 * Requires at least: 5.3
 * Requires PHP: 7.3
 * Author: Optimizer
 * Author URI: https://optimizer.vn/
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: cookie-opt
 * Domain Path: /languages
 */

use App\Includes\CookieOptBanner;
use App\Includes\CookieOptBannerPreview;
use App\Includes\CookieOptCookiePolicy;
use App\Includes\CookieOptDashboard;
use App\Includes\CookieOptLanguages;
use App\Includes\CookieOptOther;
use App\Includes\CookieOptPermission;
use App\Services\CookiePermissionService;
use App\Includes\CookieOptPublishSettings;
use App\Includes\CookieOptTagManagement;
use App\Includes\FeatureManager\CookieOptFeatureManager;
use App\Services\CookieOptBannerService;

// exit if accessed directly
if (!defined('ABSPATH')) {
	exit();
}

// Register the Composer autoloader...
require __DIR__ . '/vendor/autoload.php';

// Load constants
require_once __DIR__ . '/constants.php';

/**
 * Cookie Opt class.
 *
 * @class CookieOpt
 * @version	1.0.19
 */
class CookieOpt
{
	const DEFAULT_DEACTIVATION_TRUE = 1;
	private $network_admin = false;
	private $plugin_network_active = false;
	private static $_instance;

	public $network_options = [];
	public $tag_management;
	public $publishing_settings;
	public $cookie_opt_permission;
	public $cookie_opt_banner;
	public $cookie_opt_banner_preview;
	public $cookie_languages;
	public $cookie_opt_policy;
	public $cookie_opt_other;
	public $cookie_opt_dashboard;
	public $userPackage;
	public $limitNumber;
	/**
	 * @var $defaults
	 */
	public $defaults = [
		'version' => '1.0.19',
	];

	/**
	 * Disable object cloning.
	 *
	 * @return void
	 */
	public function __clone()
	{
	}

	/**
	 * Disable unserializing of the class.
	 *
	 * @return void
	 */
	public function __wakeup()
	{
	}

	/**
	 * Main plugin instance.
	 *
	 * @return object
	 */
	public static function instance()
	{
		if (self::$_instance === null) {
			self::$_instance = new self();

			//load đa ngôn ngữ trong file languages set up
			add_action('init', [self::$_instance, 'load_textdomain']);
			self::$_instance->userPackage = CookieOptFeatureManager::userPackage();
			self::$_instance->cookie_opt_dashboard = new CookieOptDashboard();
			self::$_instance->publishing_settings = new CookieOptPublishSettings();
			self::$_instance->tag_management = new CookieOptTagManagement();
			self::$_instance->cookie_opt_banner_preview = new CookieOptBannerPreview();
			self::$_instance->cookie_opt_permission = new CookieOptPermission();
			if (
				self::$_instance->userPackage->packageType() !== 'LOCK' &&
				!self::$_instance->userPackage->checkLimitPackage()
			) {
				self::$_instance->cookie_opt_banner = new CookieOptBanner();
			}
			if (self::$_instance->userPackage->checkSpLanguages()) {
				self::$_instance->cookie_languages = new CookieOptLanguages();
			}
			self::$_instance->cookie_opt_policy = new CookieOptCookiePolicy();
			self::$_instance->cookie_opt_other = new CookieOptOther();
		}

		return self::$_instance;
	}

	/**
	 * Class constructor.
	 *
	 * @return void
	 */
	public function __construct()
	{
		// activation hooks
		register_activation_hook(__FILE__, [$this, 'activation']);
		register_deactivation_hook(__FILE__, [$this, 'deactivation']);

		add_action('init', [$this, 'set_plugin_links']);

		add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_scripts']);
		add_action('wp_enqueue_scripts', [$this, 'enqueue_custom_styles']);
		add_action('wp_initialize_site', [$this, 'my_custom_function'], 10, 2);
	}

	/**
	 * Fires when a site's initialization routine should be executed.
	 *
	 * @param WP_Site $new_site New site object.
	 */
	public function my_custom_function($new_site, $args)
	{
		switch_to_blog($new_site->blog_id);
		$this->activate_site();
		$this->create_table_for_cookie_plugin();
		restore_current_blog();
	}

	public function enqueue_custom_styles()
	{
		$privacy_policy_page_id = get_option('wp_page_for_privacy_policy');
		if (is_page($privacy_policy_page_id)) {
			wp_register_style(
				'custom-styles-policy-user',
				COOKIE_OPT_URL . '/public/assets/css/page_policy_render.css',
				[],
				VERSION,
				false
			);
			wp_enqueue_style('custom-styles-policy-user');
		}
	}

	/**
	 * Check whether the current request is for the network administrative interface.
	 *
	 * @return bool
	 */
	public function is_network_admin()
	{
		return $this->network_admin;
	}

	/**
	 * Check whether the plugin is active for the entire network.
	 *
	 * @return bool
	 */
	public function is_plugin_network_active()
	{
		return $this->plugin_network_active;
	}

	/**
	 * Load textdomain.
	 *
	 * @return void
	 */
	public function load_textdomain()
	{
		//đường dẫn tới file languages
		load_plugin_textdomain(
			'cookie-opt',
			false,
			dirname(COOKIE_OPT_BASENAME) . '/languages/'
		);
	}

	/**
	 * Plugin activation.
	 *
	 * @global object $wpdb
	 *
	 * @param bool $network
	 * @return void
	 */
	public function activation($network)
	{
		// network activation?
		//$network = true khi là network admin
		if (is_multisite() && $network) {
			// có phải network admin hay không ?
			add_site_option('cookie_opt_version', $this->defaults['version']); // update cache js-css production

			global $wpdb;

			// get all available sites
			$blogs_ids = $wpdb->get_col("SELECT blog_id FROM {$wpdb->blogs}"); // phpcs:ignore WordPress.DB

			foreach ($blogs_ids as $blog_id) {
				// change to another site
				switch_to_blog((int) $blog_id);

				// run current site activation process
				$this->activate_site(); //chuyển từng site để thêm option
				$this->create_table_for_cookie_plugin();

				restore_current_blog();
			}
		} else {
			$this->activate_site(); //thêm các option liên quan
			//create table
			$this->create_table_for_cookie_plugin();
		}

		$cookie_Opt_Cookie_Policy = new CookieOptCookiePolicy();
		$CookieOptBannerService = new CookieOptBannerService();
		$cookie_opt_permission_service = new CookiePermissionService();

		$CookieOptBannerService->scanCookieDefaultData();
		$cookie_opt_permission_service->generateDoNotSellPage();
		$cookie_Opt_Cookie_Policy->renderPolicyDefault();
	}

	/**
	 * Single site activation.
	 *
	 * @return void
	 */
	public function activate_site()
	{
		add_option('cookie_opt_id', '', '', false);
		add_option('cookie_opt_version', $this->defaults['version'], '', false);
		$day = current_datetime()->format('d');
		add_option(
			'cookie_opt_views',
			[
				$day => 0,
			],
			'',
			false
		);
	}

	/**
	 * Plugin deactivation.
	 *
	 * @global object $wpdb
	 *
	 * @param bool $network
	 * @return void
	 */
	public function deactivation($network)
	{
		// network deactivation?
		if (is_multisite() && $network) {
			global $wpdb;

			// get all available sites
			$blogs_ids = $wpdb->get_col("SELECT blog_id FROM {$wpdb->blogs}"); // phpcs:ignore WordPress.DB

			foreach ($blogs_ids as $blog_id) {
				// change to another site
				switch_to_blog((int) $blog_id);

				// run current site deactivation process
				$this->drop_table_for_cookie_plugin();

				restore_current_blog();
			}
		} else {
			$this->drop_table_for_cookie_plugin();
		}

		$this->dropCron(); // chưa test
	}

	public function dropCron()
	{
		if (wp_next_scheduled('opt_cookie_my_daily_event')) {
			$timestamp = wp_next_scheduled('opt_cookie_my_daily_event');
			wp_unschedule_event($timestamp, 'opt_cookie_my_daily_event');
		}
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @param string $page
	 * @return void
	 */
	public function admin_enqueue_scripts($page)
	{
		// plugins page?
		if ($page === 'plugins.php') {
			add_thickbox();
		}
		// -----
		wp_register_style(
			'custom-styles-policy-admin_',
			vite('resources/scss/app.scss'),
			[],
			VERSION,
			false
		);
		$privacy_policy_page_id = get_option('wp_page_for_privacy_policy');
		if (is_page($privacy_policy_page_id)) {
			wp_register_style(
				'custom-styles-policy-admin',
				COOKIE_OPT_URL . '/public/assets/css/page_policy_render.css',
				[],
				VERSION,
				false
			);
		}
		wp_register_style(
			'custom-styles-notice-cookie-opt',
			COOKIE_OPT_URL . '/public/assets/css/cookie_notice.css',
			[],
			VERSION,
			false
		);
		wp_register_script(
			'custom-script-cookie-opt',
			vite('resources/js/pages/notice.js'),
			['jquery'],
			VERSION,
			true
		);

		wp_register_script(
			'custom-script-cookie-opt_',
			vite('resources/js/app.js'),
			[],
			VERSION,
			true
		);

		// ----
		wp_enqueue_script('custom-script-cookie-opt');
		wp_enqueue_script('custom-script-cookie-opt_');
		wp_enqueue_style('custom-styles-notice-cookie-opt');
		wp_enqueue_style('custom-styles-policy-admin');
		wp_enqueue_style('custom-styles-policy-admin_');

		do_action('cookie_plugin_language_library', $page);
		do_action('cookie_plugin_cookie_permission_library', $page);
		do_action('cookie_plugin_policy_library', $page);
		do_action('cookie_plugin_dashboard_library', $page);
		do_action('cookie_plugin_cookie_banner_preview_library', $page);
	}

	/**
	 * Set plugin links.
	 *
	 * @return void
	 */

	public function set_plugin_links()
	{
		// filters
		add_filter('plugin_action_links', [$this, 'plugin_action_links'], 10, 2);
		add_filter(
			'network_admin_plugin_action_links',
			[$this, 'plugin_action_links'],
			10,
			2
		);
	}

	/**
	 * Add links to settings page.
	 *
	 * @param array $links
	 * @param string $file
	 * @return array
	 */
	public function plugin_action_links($links, $file)
	{
		if (
			!current_user_can(
				apply_filters('co_manage_cookie_notice_cap', 'manage_options')
			)
		) {
			return $links;
		}
		if ($file === COOKIE_OPT_BASENAME) {
			$url = $this->is_network_admin()
				? network_admin_url('admin.php?page=publishing_settings')
				: admin_url('admin.php?page=publishing_settings');
			$text_name = $this->userPackage->packageType();
			if ($text_name !== 'PRO') {
				$text_name = esc_html__('Upgrade', 'cookie-opt');
			}
			$links[] = sprintf(
				'<a href="%s" style="color: #20C19E; font-weight: bold">%s</a>',
				esc_url($url),
				$text_name
			);
		}

		return $links;
	}

	public function create_table_for_cookie_plugin()
	{
		require_once plugin_dir_path(__FILE__) . 'table/create_table.php';
		if (class_exists('Table')) {
			Table::create_table_for_cookie_plugin();
		}
	}
	public function drop_table_for_cookie_plugin()
	{
		$status = get_option(
			'cookie_opt_deactivation',
			CookieOpt::DEFAULT_DEACTIVATION_TRUE
		);
		if ($status == CookieOpt::DEFAULT_DEACTIVATION_TRUE) {
			require_once plugin_dir_path(__FILE__) . 'table/create_table.php';
			if (class_exists('Table')) {
				Table::drop_table_for_cookie_plugin();
			}
		}
	}
}

/**
 * Initialize Cookie Opt.
 *
 * @return object
 */
function CookieOpt()
{
	static $instance;

	// first call to instance() initializes the plugin
	if ($instance === null || !($instance instanceof CookieOpt)) {
		$instance = CookieOpt::instance();
	}

	return $instance;
}

if ($_SERVER['REQUEST_URI'] == '/favicon.ico') { //phpcs:ignore WordPress.Security
	return false;
}

global $cookie_opt;
$cookie_opt = CookieOpt();

if (!wp_doing_ajax()) {
	//=================================================
	// 管理画面に「Cookie Opt」を追加登録する
	//=================================================
	add_action('admin_menu', function () use ($cookie_opt) {
		//---------------------------------
		// メインメニュー① ※実際のページ表示はサブメニュー①
		//---------------------------------
		add_menu_page(
			'Cookie Optimizer', // ページのタイトルタグ<title>に表示されるテキスト
			esc_html__('Cookie Optimizer', 'cookie-opt'), // 左メニューとして表示されるテキスト
			'manage_options', // 必要な権限 manage_options は通常 administrator のみに与えられた権限
			esc_attr('cookie_opt'), // 左メニューのスラッグ名 →URLのパラメータに使われる /wp-admin/admin.php?page=toriaezu_menu
			'', // メニューページを表示する際に実行される関数(サブメニュー①の処理をする時はこの値は空にする)
			COOKIE_OPT_URL . '/public/assets/images/plugin_icon.svg', // メニューのアイコンを指定 https://developer.wordpress.org/resource/dashicons/#awards
			80 // メニューが表示される位置のインデックス(0が先頭) 5=投稿,10=メディア,20=固定ページ,25=コメント,60=テーマ,65=プラグイン,70=ユーザー,75=ツール,80=設定
		);

		//---------------------------------
		// サブメニュー①
		//---------------------------------
		add_submenu_page(
			'cookie_opt', // 親メニューのスラッグ
			esc_html__('Startup Settings', 'cookie-opt'), // ページのタイトルタグ<title>に表示されるテキスト
			esc_html__('Startup Settings', 'cookie-opt'), // サブメニューとして表示されるテキスト
			'manage_options', // 必要な権限 manage_options は通常 administrator のみに与えられた権限
			'cookie_opt', //サブメニューのスラッグ名
			'cookie_permission', //（任意）このページのコンテンツを出力するために呼び出される関数
			80
		);

		//---------------------------------
		// サブメニュー③
		//---------------------------------
		add_submenu_page(
			'cookie_opt', // 親メニューのスラッグ
			esc_html__('Cookie List Setting', 'cookie-opt'), // ページのタイトルタグ<title>に表示されるテキスト
			esc_html__('Cookie List Setting', 'cookie-opt'), // サブメニューとして表示されるテキスト
			'manage_options', // 必要な権限 manage_options は通常 administrator のみに与えられた権限
			'cookie_policy', //サブメニューのスラッグ名
			'cookie_policy', //（任意）このページのコンテンツを出力するために呼び出される関数
			81
		);

		//---------------------------------
		// サブメニュー ※事実上の親メニュー②
		//---------------------------------
		add_submenu_page(
			'cookie_opt',
			esc_html__('Tag Management Settings', 'cookie-opt'),
			esc_html__('Tag Management Settings', 'cookie-opt'),
			'manage_options',
			'tag_management',
			'tag_management',
			82
		);

		//---------------------------------
		// サブメニュー④
		//---------------------------------
		if ($cookie_opt->userPackage->checkSpLanguages()) {
			add_submenu_page(
				'cookie_opt', // 親メニューのスラッグ
				esc_html__('Language Setting', 'cookie-opt'), // ページのタイトルタグ<title>に表示されるテキスト
				esc_html__('Language Setting', 'cookie-opt'), // サブメニューとして表示されるテキスト
				'manage_options', // 必要な権限 manage_options は通常 administrator のみに与えられた権限
				'co_languages', //サブメニューのスラッグ名
				'co_languages', //（任意）このページのコンテンツを出力するために呼び出される関数
				83
			);
		}
		//---------------------------------
		// サブメニュー⑤
		//---------------------------------
		add_submenu_page(
			'cookie_opt', // 親メニューのスラッグ
			esc_html__('Publishing Settings', 'cookie-opt'), // ページのタイトルタグ<title>に表示されるテキスト
			esc_html__('Publishing Settings', 'cookie-opt'), // サブメニューとして表示されるテキスト
			'manage_options', // 必要な権限 manage_options は通常 administrator のみに与えられた権限
			'publishing_settings', //サブメニューのスラッグ名
			'publishing_settings', //（任意）このページのコンテンツを出力するために呼び出される関数
			84
		);
		//---------------------------------
		// サブメニュー⑥
		//---------------------------------
		add_submenu_page(
			'cookie_opt', // 親メニューのスラッグ
			esc_html__('External collaboration', 'cookie-opt'), // ページのタイトルタグ<title>に表示されるテキスト
			esc_html__('External collaboration', 'cookie-opt'), // サブメニューとして表示されるテキスト
			'manage_options', // 必要な権限 manage_options は通常 administrator のみに与えられた権限
			'co_others', //サブメニューのスラッグ名
			'co_others', //（任意）このページのコンテンツを出力するために呼び出される関数
			85
		);
	});
}

//=================================================
// メインメニューページ内容の表示・更新処理
//=================================================
function tag_management()
{
	global $cookie_opt;
	$cookie_opt->tag_management->run();
}

add_action('init', 'do_output_buffer');
/**
 * Initialize output buffering.
 */
function do_output_buffer()
{
	ob_start();
}
//=================================================
// サブメニュー①ページ内容の表示・更新処理
//=================================================
function cookie_permission()
{
	global $cookie_opt;
	$cookie_opt->cookie_opt_permission->run();
}

//=================================================
// サブメニュー②ページ内容の表示・更新処理
//=================================================
function cookie_policy()
{
	global $cookie_opt;
	$cookie_opt->cookie_opt_policy->run();
}

//=================================================
// サブメニュー③ページ内容の表示・更新処理
//=================================================
function co_languages()
{
	global $cookie_opt;
	$cookie_opt->cookie_languages->run();
}

//=================================================
// サブメニュー④ページ内容の表示・更新処理
//=================================================
function publishing_settings()
{
	global $cookie_opt;
	$cookie_opt->publishing_settings->run();
}

//=================================================
// サブメニュー⑤ページ内容の表示・更新処理
//=================================================
function co_others()
{
	global $cookie_opt;
	$cookie_opt->cookie_opt_other->run();
}
