<?php

namespace App\Tables;

use App\Services\CookieOptLanguagesService;
use App\Services\CookieOptPolicyServices;

class CookieOptCookieListTable extends CookieOptBaseTable
{
	public $policy_services;
	public $languages_service;

	public function __construct()
	{
		parent::__construct([
			'singular' => 'item',
			'plural' => 'items',
			'ajax' => false,
		]);

		$this->policy_services = new CookieOptPolicyServices();
		$this->languages_service = new CookieOptPolicyServices();
	}

	public function get_columns()
	{
		$columns = [
			'cb' => '<input type="checkbox" />',
			'name' => esc_html__('Name', 'cookie-opt'),
			'category_name' => esc_html__('Category', 'cookie-opt'),
			'publisher' => esc_html__('Publisher', 'cookie-opt'),
			'description' => esc_html__('Description', 'cookie-opt'),
		];

		return $columns;
	}

	protected function get_sortable_columns()
	{
		$sortable_columns = [
			'name' => ['name', true],
			'category_name' => ['category_name', true],
			'publisher' => ['publisher', true],
			'description' => ['description', true],
		];

		return $sortable_columns;
	}

	protected function column_default($item, $column_name)
	{
		switch ($column_name) {
			case 'name':
			case 'category_name':
			case 'publisher':
			case 'description':
				return $item[$column_name];
			default:
				return print_r($item, true); //phpcs:ignore WordPress.PHP
		}
	}

	protected function column_cb($item)
	{
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			$this->_args['singular'],
			$item['id']
		);
	}

	protected function column_name($item)
	{
		$page = wp_unslash($_REQUEST['page']); //phpcs:ignore WordPress.Security

		$edit_query_args = [
			'page' => $page,
			'nonce_action' => wp_create_nonce('action_policy'),
			'action' => 'edit',
			'id' => $item['id'],
		];

		$actions['edit'] = sprintf(
			'<a href="%1$s">%2$s</a>',
			esc_url(wp_nonce_url(add_query_arg($edit_query_args, 'admin.php'), 'edit_' . $item['id'])),
			__('Edit', 'cookie-opt')
		);

		$delete_query_args = [
			'page' => $page,
			'index' => 'list-cookie',
			'verify' => wp_create_nonce('nds_delete_cookie_policy'),
			'nonce_action' => wp_create_nonce('action_policy'),
			'action' => 'delete',
			'id' => $item['id'],
		];

		$actions['delete'] = sprintf(
			'<a href="%1$s">%2$s</a>',
			esc_url(wp_nonce_url(add_query_arg($delete_query_args, 'admin.php'), 'delete_' . $item['id'])),
			__('Delete', 'cookie-opt')
		);

		return sprintf(
			'%1$s %2$s',
			$item['name'],
			$this->row_actions($actions)
		);
	}

	protected function get_bulk_actions()
	{
		$actions = [
			'delete_many' => __('Delete', 'cookie-opt'),
		];

		return $actions;
	}

	protected function process_bulk_action()
	{
		if ('delete_many' === $this->current_action()) {
			try {
				$this->policy_services->deleteManyCookieList($_REQUEST['item'] ?? []); //phpcs:ignore WordPress.Security
				$nonce_action = wp_create_nonce('action_policy');

				set_transient(
					'msg-cookie-opt',
					esc_html__('Deleted cookies list successfully', 'cookie-opt')
				);
				wp_redirect(
					admin_url(
						'admin.php?page=cookie_policy&index=list-cookie&nonce_action=' .
						$nonce_action
					)
				);
				die();
			} catch (\Exception $e) {
			}
		}
	}

	public function get_table_data()
	{
		$languages_service = new CookieOptLanguagesService();
		$res_by_index = $languages_service->getLanguagesByStatus();
		$search = sanitize_text_field($_REQUEST['s'] ?? '') ?? ''; //phpcs:ignore WordPress.Security
		return $this->policy_services->indexCookieList($search, $res_by_index->default_language);
	}

	public function prepare_items()
	{
		global $wpdb;

		$per_page = 5;
		$columns = $this->get_columns();
		$hidden = [];
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = [$columns, $hidden, $sortable];

		$this->process_bulk_action();

		$data = $this->get_table_data();

		usort($data, [$this, 'usort_reorder']);

		$current_page = $this->get_pagenum();

		$total_items = count($data);

		$data = array_slice($data, ($current_page - 1) * $per_page, $per_page);

		$this->items = $data;

		$this->set_pagination_args([
			'total_items' => $total_items,
			'per_page' => $per_page,
			'total_pages' => ceil($total_items / $per_page),
		]);
	}
	protected function usort_reorder($a, $b)
	{
		// If no sort, default to title.
		$orderby = !empty($_REQUEST['orderby']) ? wp_unslash($_REQUEST['orderby']) : 'name'; //phpcs:ignore WordPress.Security

		// If no order, default to asc.
		$order = !empty($_REQUEST['order']) ? wp_unslash($_REQUEST['order']) : 'asc'; //phpcs:ignore WordPress.Security

		// Determine sort order.
		$result = strcmp($a[$orderby], $b[$orderby]);

		return ('asc' === $order) ? $result : -$result;
	}

	public function getCustomParamsSortable()
	{
		return [
			'index' => 'list-cookie',
			'nonce_action' => wp_create_nonce('action_policy')
		];
	}	
}