<?php

namespace App\Services;

use App\Services\BaseService\CookieOptBaseService;
use App\Services\CookieOptPostService;
use App\Services\CookieOptCookieCategoryService;
use App\Services\CookieOptBannerService;
use App\Includes\FeatureManager\CookieOptFeatureManager;

class CookiePermissionService extends CookieOptBaseService
{
	const COOKIE_OPT_REGULATION_LIST = ['eu', 'us', 'ja'];
	const COOKIE_OPT_REGULATION_DEFAULT = 'ja';
	const POSITION_CSS_CLASS_LIST = [
		'box-bottom-left' => 'coop-position__box-bt-l',
		'box-bottom-right' => 'coop-position__box-bt-r',
		'box-top-left' => 'coop-position__box-t-l',
		'box-top-right' => 'coop-position__box-t-r',
		'classic-bottom' => 'coop-position__classic-bt',
		'classic-top' => 'coop-position__classic-t',
		'banner-bottom' => 'coop-position__banner-bt',
		'banner-top' => 'coop-position__banner-t',
		'popup' => 'coop-position__popup'
	];

	const ANIMATION_CSS_CLASS_LIST = [
		'none' => '',
		'fade' => 'coop__animate--fade-in',
		'slide-right-to-left' => 'coop__animate--slide-r-l',
		'slide-left-to-right' => 'coop__animate--slide-l-r',
		'slide-top-to-bottom' => 'coop__animate--slide-t-bt',
		'slide-bottom-to-top' => 'coop__animate--slide-bt-t',
	];

	public $data_setting_tab = [
		'title' => '',
		'description' => '',
		'buttons' => [
			'accept' => [
				'status' => 1,
				'content' => ''
			],
			'reject' => [
				'status' => '',
				'content' => ''
			],
			'pri_policy' => [
				'status' => '',
				'content' => '',
				'link' => '',
				'link_type' => '',
				'link_position' => '',
				'link_target' => '',
			],
			'cookie_policy' => [
				'status' => '',
				'content' => ''
			],
			'term_of_use' => [
				'status' => '',
				'content' => '',
			],
		],
		'customize' => [
			'status' => '',
			'content' => '',
			'title' => '',
			'description' => '',
			'alway_active' => '',
			'here' => '',
			'save_btn' => '',
		],
		'on_scroll' => [
			'status' => 1,
			'value' => 100,
		],
		'on_click' => 1,
		'expired' => 'infinity',
		'conditional' => [
			'status' => 0,
			'rules' => [
				'show' => [

				],
				'hide' => [

				],
			],
		],
		'all_page' => [

		],
		'all_post_type' => [

		],
		'geolocation' => [
			'status' => 'checked',
			'type' => [
				'auto' => 'selected',
				'manual' => '',
			],
			'setting' => [
				'banner_display' => [
					'eu' => 'checked',
					'us' => 'checked',
					'ja' => 'checked',
					'other' => 'checked',
				],
				'pri_policy_link' => [
					'eu' => 'checked',
					'us' => 'checked',
					'ja' => 'checked',
					'other' => 'checked',
				],
				'term_of_use' => [
					'us' => 'checked',
				],
			],
		],
	];

	public $data_layout_tab = [
		'position' => [
			'box-bottom-right' => 'selected',
			'box-bottom-left' => '',
			'box-top-left' => '',
			'box-top-right' => '',
			'classic-bottom' => '',
			'classic-top' => '',
			'banner-bottom' => '',
			'banner-top' => '',
			'popup' => ''
		],
		'animation' => [
			'none' => 'checked',
			'fade' => '',
			'slide' => '',
		],
		'slide' => [
			'slide-right-to-left' => 'selected',
			'slide-left-to-right' => '',
			'slide-top-to-bottom' => '',
			'slide-bottom-to-top' => '',
		],
		'text_color' => '#333',
		'button_color' => '#84daff',
		'bar_color' => '#fff',
		'bar_opacity' => '100'
	];

	public $data_banner = [
		'regulation' => '',
		'title' => '',
		'description' => '',
		'buttons' => [
			'accept_button' => '',
			'reject_button' => '',
			'customize_button' => '',
			'cookie_policy_button' => '',
			'privacy_policy_button' => '',
			'do_not_sell_page_button' => '',
		],
		'buttons_link' => [
			'accept_button' => '',
			'reject_button' => '',
			'customize_button' => '',
			'cookie_policy_button' => '',
			'privacy_policy_button' => '',
			'do_not_sell_page_button' => '',
		],
		'customize' => [
			'title' => '',
			'description' => '',
			'always_active' => '',
			'here' => '',
			'button_save' => '',
		],
		'category_list' => [

		],
		'style_banner' => '',
		'style_button' => '',

		'position' => '',
		'animation' => '',
		'target' => '_blank',
		'hidden' => false
	];

	protected $post_service;
	protected $cookie_category_service;
	protected $CookieOptBannerService;
	protected $default_language;

	public function __construct()
	{
		parent::__construct();
		$this->post_service = new CookieOptPostService();
		$this->cookie_category_service = new CookieOptCookieCategoryService();
		$this->CookieOptBannerService = new CookieOptBannerService();
		$this->default_language = CookieOptFeatureManager::userPackage();
	}

	public function getTable()
	{
		return 'cookie_opt_banner';
	}

	//---------------------------------------------------------------------------------------------------
	// For Banner
	//---------------------------------------------------------------------------------------------------

	/**
	 * Get setting banner data
	 *
	 * @return array
	 */
	public function getSettingBanner()
	{
		global $wpdb;

		$result = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$this->table} WHERE status = %d ", 1), ARRAY_A); // phpcs:ignore WordPress.DB

		$result['setting'] = json_decode($result['setting'], true);
		$result['setting']['regulation'] = $result['regulation'];

		return $result['setting'];
	}

	/**
	 * Get Notify for CookieScan js file by locale
	 * @return array{process: string, success: string, successDescription: string}
	 */
	public function getNotifyCookieScanByLocale(){
		$notify = [
			'progress' => __('Scan in progress...','cookie-opt'),
			'success' => __('Cookies scan complete', 'cookie-opt'),
			'successDescription' => __('Scanned cookies are automatically added to the cookie list.', 'cookie-opt')
		];
		return $notify;
	}

	//---------------------------------------------------------------------------------------------------
	// For Cookie Permission
	//---------------------------------------------------------------------------------------------------

	/**
	 * Get Data for Preview Banner
	 * @param mixed $data array info about regulation, position, animation, text color, bar color, button color, bar opacity
	 * @return array
	 */
	public function getDataBanner($data)
	{
		global $wpdb;
		$prepare_query = $wpdb->prepare("SELECT * FROM {$this->table} WHERE status = %d ", 1); //phpcs:ignore WordPress.DB
		if (isset($_POST['regulation']) && $data['regulation'] !== null) { //phpcs:ignore WordPress.Security
			$prepare_query = $wpdb->prepare("SELECT * FROM {$this->table} WHERE regulation = %s ", $data['regulation']); //phpcs:ignore WordPress.DB
		}
		$result = $wpdb->get_row($prepare_query, ARRAY_A); //phpcs:ignore WordPress.DB

		//Convert Fields To Array
		$result['setting'] = json_decode($result['setting'], true);
		$result['content'] = json_decode($result['contents'], true);
		$lang = (isset($data['language']) && $data['language'] !== null && in_array($data['language'], array_keys($result['content']))) ? $data['language'] : $result['default_language'];

		$this->data_banner['regulation'] = $result['regulation'];
		$this->data_banner['title'] = $result['content'][$lang]['notice']['title'];
		$this->data_banner['description'] = $result['content'][$lang]['notice']['description'];

		foreach ($this->data_banner['buttons'] as $key => $value) {
			if (isset($result['setting']['buttons'][$key]) && $result['setting']['buttons'][$key] === 1) {
				$this->data_banner['buttons'][$key] = $result['content'][$lang]['notice']['buttons'][$key];
			} else {
				unset($this->data_banner['buttons'][$key]);
			}
		}

		$this->data_banner['customize'] = $result['content'][$lang]['customize'];
		$this->data_banner['position'] = self::POSITION_CSS_CLASS_LIST[$result['setting']['position']];

		//Button links
		$policy_page_link = !empty($result['setting']['privacy_policy']['link']) ? wp_parse_url($result['setting']['privacy_policy']['link'], PHP_URL_PATH) : wp_parse_url(get_permalink(get_option('wp_page_for_privacy_policy')), PHP_URL_PATH);
		$term_of_use_page_link = wp_parse_url(get_permalink(get_option('cookie_opt_do_not_sell_page')), PHP_URL_PATH);
		$this->data_banner['buttons_link']['cookie_policy_button'] = !empty($policy_page_link) ? "href={$policy_page_link}" : '';
		$this->data_banner['buttons_link']['privacy_policy_button'] = !empty($policy_page_link) ? "href={$policy_page_link}" : '';
		$this->data_banner['buttons_link']['do_not_sell_page_button'] = !empty($term_of_use_page_link) ? "href={$term_of_use_page_link}" : '';

		//Validate input data
		$btn_color = $result['setting']['colors']['button-color'];
		$bar_color = $result['setting']['colors']['bar-color'];
		$bar_opacity = $result['setting']['colors']['bar-opacity'];
		$text_color = $result['setting']['colors']['text-color'];

		$this->data_banner['style_banner'] = sprintf('background: %s; opacity: %s; color: %s;', $bar_color, $bar_opacity, $text_color);
		$this->data_banner['style_button'] = sprintf('background: %s; color: %s;', $btn_color, $text_color);

		$this->data_banner['category_list'] = $this->cookie_category_service->getDataCategory($lang);

		$this->data_banner['target'] = !empty($result['setting']['privacy_policy']['link_target']) ? $result['setting']['privacy_policy']['link_target'] : $this->data_banner['target'];

		return $this->data_banner;
	}

	/**
	 * Get Data for Banner to Show
	 * @param mixed $data array info about regulation, position, animation, text color, bar color, button color, bar opacity
	 * @return array
	 */
	public function getDataBannerToShow()
	{
		global $wpdb;
		$prepare_query = $wpdb->prepare("SELECT * FROM {$this->table} WHERE status = %d ", 1); //phpcs:ignore WordPress.DB
		$result = $wpdb->get_row($prepare_query, ARRAY_A); //phpcs:ignore WordPress.DB

		//Convert Fields To Array
		$result['setting'] = json_decode($result['setting'], true);
		$result['content'] = json_decode($result['contents'], true);
		$lang = $result['default_language'];
		// dd($result['setting']['privacy_policy']['link_target']);
		$this->data_banner['regulation'] = $result['regulation'];
		$this->data_banner['title'] = $result['content'][$lang]['notice']['title'];
		$this->data_banner['description'] = $result['content'][$lang]['notice']['description'];

		foreach ($this->data_banner['buttons'] as $key => $value) {
			if (isset($result['setting']['buttons'][$key]) && $result['setting']['buttons'][$key] === 1) {
				$this->data_banner['buttons'][$key] = $result['content'][$lang]['notice']['buttons'][$key];
			} else {
				unset($this->data_banner['buttons'][$key]);
			}
		}

		$this->data_banner['customize'] = $result['content'][$lang]['customize'];
		$this->data_banner['position'] = self::POSITION_CSS_CLASS_LIST[$result['setting']['position']];

		$policy_page_link = !empty($result['setting']['privacy_policy']['link']) ? wp_parse_url($result['setting']['privacy_policy']['link'], PHP_URL_PATH) : wp_parse_url(get_permalink(get_option('wp_page_for_privacy_policy')), PHP_URL_PATH);
		$term_of_use_page_link = wp_parse_url(get_permalink(get_option('cookie_opt_do_not_sell_page')), PHP_URL_PATH);
		$this->data_banner['buttons_link']['cookie_policy_button'] = !empty($policy_page_link) ? "href={$policy_page_link}" : '';
		$this->data_banner['buttons_link']['privacy_policy_button'] = !empty($policy_page_link) ? "href={$policy_page_link}" : '';
		$this->data_banner['buttons_link']['do_not_sell_page_button'] = !empty($term_of_use_page_link) ? "href={$term_of_use_page_link}" : '';

		//Validate input data
		$btn_color = $result['setting']['colors']['button-color'];
		$bar_color = $result['setting']['colors']['bar-color'];
		$bar_opacity = $result['setting']['colors']['bar-opacity'];
		$text_color = $result['setting']['colors']['text-color'];

		//Add style
		$this->data_banner['style_banner'] = sprintf('background: %s; opacity: %s; color: %s;', $bar_color, $bar_opacity, $text_color);
		$this->data_banner['style_button'] = sprintf('background: %s; color: %s;', $btn_color, $text_color);
		$this->data_banner['animation'] =  self::ANIMATION_CSS_CLASS_LIST[$result['setting']['animation']];

		$this->data_banner['category_list'] = $this->cookie_category_service->getDataCategory($lang);

		$this->data_banner['target'] = !empty($result['setting']['privacy_policy']['link_target']) ? $result['setting']['privacy_policy']['link_target'] : $this->data_banner['target'];
		$this->data_banner['conditional'] = $result['setting']['conditional'];

		return $this->data_banner;
	}

	public function getDataRegulationDefault(){
		global $wpdb;
		$prepare_query = $wpdb->prepare("SELECT regulation FROM {$this->table} WHERE status = %d ", 1); //phpcs:ignore WordPress.DB
		$result = $wpdb->get_row($prepare_query);
		$regulation = 'ja';
		if (isset($result) && !empty($result->regulation)){
			$regulation = $result->regulation;
		}
		return $regulation;
	}

	/**
	 * Get data for setting tab
	 * @param mixed $regulation regulation selected default is ja
	 * @return array
	 */
	public function getDataSettingTab($regulation)
	{
		global $wpdb;
		$result = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$this->table} WHERE regulation = %s", $regulation), ARRAY_A); // phpcs:ignore WordPress.DB

		$all_pages = $this->post_service->loadAllPage();
		$all_post_types = get_post_types(['public' => true], 'names', 'or');
		// unset($all_post_types['attachment']);
		$geolocation_option = get_option('cookie_opt_geolocation_setting');
		$lang = $result['default_language'];
		$result['contents'] = json_decode($result['contents'], true);
		$result['setting'] = json_decode($result['setting'], true);

		// Title + Description
		$this->data_setting_tab['title'] = $result['contents'][$lang]['notice']['title'];
		$this->data_setting_tab['description'] = $result['contents'][$lang]['notice']['description'];

		//Buttons
		$this->data_setting_tab['buttons']['accept']['status'] = $result['setting']['buttons']['accept_button'];
		$this->data_setting_tab['buttons']['accept']['content'] = $result['contents'][$lang]['notice']['buttons']['accept_button'];

		$this->data_setting_tab['buttons']['reject']['status'] = (isset($result['setting']['buttons']['reject_button']) && $result['setting']['buttons']['reject_button'] == 1) ? 'checked' : '';
		$this->data_setting_tab['buttons']['reject']['content'] = $result['contents'][$lang]['notice']['buttons']['reject_button'];

		$this->data_setting_tab['buttons']['cookie_policy']['status'] = (isset($result['setting']['buttons']['cookie_policy_button']) && $result['setting']['buttons']['cookie_policy_button'] == 1) ? 'checked' : '';
		$this->data_setting_tab['buttons']['cookie_policy']['content'] = $result['contents'][$lang]['notice']['buttons']['cookie_policy_button'];

		//Privacy policy
		$this->data_setting_tab['buttons']['pri_policy']['status'] = $result['setting']['buttons']['privacy_policy_button'] == 1 ? 'checked' : '';
		$this->data_setting_tab['buttons']['pri_policy']['content'] = $result['contents'][$lang]['notice']['buttons']['privacy_policy_button'];
		$this->data_setting_tab['buttons']['pri_policy']['link'] = url_to_postid($result['setting']['privacy_policy']['link']);
		$this->data_setting_tab['buttons']['pri_policy']['link_type'] = $result['setting']['privacy_policy']['link_type'];
		$this->data_setting_tab['buttons']['pri_policy']['link_target'] = $result['setting']['privacy_policy']['link_target'];

		//Term of useata_setting_tab['buttons']['pri_policy']['link'];die;
		$this->data_setting_tab['buttons']['term_of_use']['status'] = (isset($result['setting']['buttons']['do_not_sell_page_button']) && $result['setting']['buttons']['do_not_sell_page_button'] == 1) ? 'checked' : '';
		$this->data_setting_tab['buttons']['term_of_use']['content'] = $result['contents'][$lang]['notice']['buttons']['do_not_sell_page_button'];

		//Customize
		$this->data_setting_tab['customize'] = $result['contents'][$lang]['customize'];
		$this->data_setting_tab['customize']['status'] = (isset($result['setting']['buttons']['customize_button']) && $result['setting']['buttons']['customize_button'] == 1) ? 'checked' : '';
		$this->data_setting_tab['customize']['content'] = $result['contents'][$lang]['notice']['buttons']['customize_button'];

		//Geolocation Setting
		if ($geolocation_option) {
			$data_format = json_decode($geolocation_option, true);
			$this->data_setting_tab['geolocation'] = $data_format;
		}

		//Setting scroll + click
		$this->data_setting_tab['on_scroll'] = $result['setting']['onScroll'];
		$this->data_setting_tab['on_scroll']['status'] = $result['setting']['onScroll']['status'] == 1 ? 'checked' : '';
		$this->data_setting_tab['on_click'] = $result['setting']['onClick'] == 1 ? 'checked' : '';

		//Expired
		$this->data_setting_tab['expired'] = $result['setting']['acceptedExpiry'];

		//Conditional display
		$this->data_setting_tab['conditional'] = $result['setting']['conditional'];
		$this->data_setting_tab['conditional']['status'] = $this->data_setting_tab['conditional']['status'] == 1 ? 'checked' : '';
		$this->data_setting_tab['all_page'] = $all_pages;
		$this->data_setting_tab['all_post_type'] = $all_post_types;

		return $this->data_setting_tab;
	}

	/**
	 * Get data for banner layout tab
	 * @param mixed $regulation regulation selected, default is ja
	 * @return array
	 */
	public function getDataBannerLayout($regulation = 'ja')
	{
		global $wpdb;
		$result = $wpdb->get_row($wpdb->prepare("SELECT setting FROM {$this->table} WHERE regulation = %s", $regulation), ARRAY_A); // phpcs:ignore WordPress.DB
		$result = json_decode($result['setting'], true);

		$this->data_layout_tab['position']['box-bottom-left'] = ($result['position'] == 'box-bottom-left') ? 'checked' : '';
		$this->data_layout_tab['position']['box-bottom-right'] = ($result['position'] == 'box-bottom-right') ? 'checked' : '';
		$this->data_layout_tab['position']['box-top-left'] = ($result['position'] == 'box-top-left') ? 'checked' : '';
		$this->data_layout_tab['position']['box-top-right'] = ($result['position'] == 'box-top-right') ? 'checked' : '';
		$this->data_layout_tab['position']['classic-bottom'] = ($result['position'] == 'classic-bottom') ? 'checked' : '';
		$this->data_layout_tab['position']['classic-top'] = ($result['position'] == 'classic-top') ? 'checked' : '';
		$this->data_layout_tab['position']['banner-bottom'] = ($result['position'] == 'banner-bottom') ? 'checked' : '';
		$this->data_layout_tab['position']['banner-top'] = ($result['position'] == 'banner-top') ? 'checked' : '';
		$this->data_layout_tab['position']['popup'] = ($result['position'] == 'popup') ? 'checked' : '';

		$this->data_layout_tab['animation']['none'] = ($result['animation'] == 'none') ? 'checked' : '';
		$this->data_layout_tab['animation']['fade'] = ($result['animation'] == 'fade') ? 'checked' : '';
		$this->data_layout_tab['animation']['slide'] = ($result['animation'] == 'slide') ? 'checked' : '';

		$this->data_layout_tab['slide']['slide-right-to-left'] = ($result['animation'] == 'slide-right-to-left') ? 'selected' : '';
		$this->data_layout_tab['slide']['slide-left-to-right'] = ($result['animation'] == 'slide-left-to-right') ? 'selected' : '';
		$this->data_layout_tab['slide']['slide-top-to-bottom'] = ($result['animation'] == 'slide-top-to-bottom') ? 'selected' : '';
		$this->data_layout_tab['slide']['slide-bottom-to-top'] = ($result['animation'] == 'slide-bottom-to-top') ? 'selected' : '';

		$this->data_layout_tab['text_color'] = $result['colors']['text-color'];
		$this->data_layout_tab['button_color'] = $result['colors']['button-color'];
		$this->data_layout_tab['bar_color'] = $result['colors']['bar-color'];
		$this->data_layout_tab['bar_opacity'] = $result['colors']['bar-opacity'];

		return $this->data_layout_tab;
	}


	/**
	 * Update data for Startup tab
	 * @param mixed $regulation regulation selected
	 * @param mixed $data_update data setting to update
	 * @return bool|int
	 */
	public function updateDataStartup($regulation, $data_update)
	{
		global $wpdb;
		$result = $wpdb->get_row($wpdb->prepare("SELECT setting, default_language FROM {$this->table} WHERE regulation = %s ", $regulation), ARRAY_A); //phpcs:ignore WordPress.DB

		$result['default_language'] = $data_update['coop-start__language'];
		// $result['default_language'] = $data_update['coop-start__regulation'];
		$result['setting'] = json_decode($result['setting'], true);

		$result['setting']['position'] = $data_update['coop-start__position'];
		$result['setting']['animation'] = $data_update['coop-start__animation'];

		$result['setting']['colors']['text-color'] = $data_update['coop-start__text-color'];
		$result['setting']['colors']['button-color'] = $data_update['coop-start__btn-color'];
		$result['setting']['colors']['bar-color'] = $data_update['coop-start__bar-color'];
		$result['setting']['colors']['bar-opacity'] = $data_update['coop-start__bar-opacity'];

		// //Parse to Json
		$result['setting'] = wp_json_encode($result['setting']);

		$update = $wpdb->update($this->table, ['setting' => $result['setting'], 'default_language' => $result['default_language']], ['regulation' => $regulation]); // phpcs:ignore WordPress.DB

		$this->updateDefaultBanner($regulation);

		update_option('cookie_opt_first_setting', 1);

		return $update;
	}

	/**
	 * Update data for Setting tab
	 * @param mixed $regulation regulation selected
	 * @param mixed $data_update data setting to update
	 * @return void
	 */
	public function updateDataSetting($regulation, $data_update)
	{
		global $wpdb;
		$result = $wpdb->get_row($wpdb->prepare("SELECT setting, contents, default_language FROM {$this->table} WHERE regulation = %s ", $regulation), ARRAY_A);  //phpcs:ignore WordPress.DB
		$lang = $result['default_language'];
		$result['setting'] = json_decode($result['setting'], true);
		$result['contents'] = json_decode($result['contents'], true);

		//Geolocation
		$this->updateOptionGeolocation($data_update);

		//Title + Description
		$result['contents'][$lang]['notice']['title'] = $data_update['coop-setting__title'];
		$result['contents'][$lang]['notice']['description'] = $data_update['coop-setting__description'];

		//Accept Button
		$result['contents'][$lang]['notice']['buttons']['accept_button'] = $data_update['coop-setting__accept-btn'];

		//Privacy Policy (Button + Setting)
		$result['setting']['buttons']['privacy_policy_button'] = $data_update['coop-setting__pri-policy-status'];
		if ($data_update['coop-setting__pri-policy-status'] == 1) {
			$result['contents'][$lang]['notice']['buttons']['privacy_policy_button'] = $data_update['coop-setting__pri-policy-btn'];
			$result['setting']['privacy_policy']['link'] = get_permalink($data_update['coop-setting__pri-policy-page-id']);
			$result['setting']['privacy_policy']['link_target'] = $data_update['coop-setting__pri-policy-target'];
		}

		//On click
		$result['setting']['onClick'] = $data_update['coop-setting__on-click'];

		//On scroll
		$result['setting']['onScroll']['status'] = $data_update['coop-setting__onscroll-status'];
		if ($data_update['coop-setting__onscroll-status'] == 1) {
			$result['setting']['onScroll']['value'] = $data_update['coop-setting__onscroll-value'];
		}

		//Expired
		$result['setting']['acceptedExpiry'] = $data_update['coop-setting__expired'];

		//Conditional Display
		$result['setting']['conditional']['status'] = $data_update['coop-setting__conditional-status'];
		if ($data_update['coop-setting__conditional-status']) {
			$option_value = $_POST['coop-setting__conditional-rule']; // phpcs:ignore WordPress.Security
			$value = [];
			if (isset($_POST['coop-setting__conditional']['rule'][$option_value]) && !empty($_POST['coop-setting__conditional']['rule'][$option_value])) // phpcs:ignore WordPress.Security
				$value = $_POST['coop-setting__conditional']['rule'][$option_value]; // phpcs:ignore WordPress.Security
			$result['setting']['conditional']['rules'][$option_value] = $value;
		}

		//Special Fields for regulation ja
		if ($regulation == 'ja') {
			//Cookie policy
			$result['contents'][$lang]['notice']['buttons']['cookie_policy_button'] = $data_update['coop-setting__cookie-policy-btn'];
		}

		//Special Fields for regulation eu + us
		if ($regulation == 'eu' || $regulation == 'us') {
			//reject
			$result['setting']['buttons']['reject_button'] = $data_update['coop-setting__reject-status'];
			if ($data_update['coop-setting__reject-status'] == 1) {
				$result['contents'][$lang]['notice']['buttons']['reject_button'] = $data_update['coop-setting__reject-btn'];
			}

			//Customize
			$result['setting']['buttons']['customize_button'] = $data_update['coop-setting__cus-status'];
			if ($data_update['coop-setting__cus-status'] == 1) {
				$result['contents'][$lang]['notice']['buttons']['customize_button'] = $data_update['coop-setting__cus-btn'];
				$result['contents'][$lang]['customize']['title'] = $data_update['coop-setting__cus-title'];
				$result['contents'][$lang]['customize']['description'] = $data_update['coop-setting__cus-des'];
				$result['contents'][$lang]['customize']['here'] = $data_update['coop-setting__cus-here'];
				$result['contents'][$lang]['customize']['button_save'] = $data_update['coop-setting__cus-save-btn'];
			}
		}

		//Special Fields for regulation us
		if ($regulation == 'us') {
			//Term of use
			$result['setting']['buttons']['do_not_sell_page_button'] = $data_update['coop-setting__term-of-use-status'];
			if ($data_update['coop-setting__term-of-use-status'] == 1) {
				$result['contents'][$lang]['notice']['buttons']['do_not_sell_page_button'] = $data_update['coop-setting__term-of-use-btn'];
			}
		}

		//Parse to Json
		$result['setting'] = wp_json_encode($result['setting']);
		$result['contents'] = wp_json_encode($result['contents']);

		$update = $wpdb->update($this->table, ['setting' => $result['setting'], 'contents' => $result['contents']], ['regulation' => $regulation]); // phpcs:ignore WordPress.DB

		$this->updateDefaultBanner($regulation);

		//Update banner active
		$is_banner_active_option = get_option('cookie_opt_banner_active');
		if ($is_banner_active_option != -1) {
			$is_banner_active = isset($data_update['coop-setting__banner-active']) ? 1 : 0;
			update_option('cookie_opt_banner_active', $is_banner_active);
		}

		return $update;
	}

	/**
	 * Update option Geolocation setting
	 * @param mixed $data_update
	 * @return void
	 */
	public function updateOptionGeolocation($data_update)
	{
		$value['status'] = isset($data_update['coop-setting__geolocation-status']) ? 'checked' : '';

		$value['type']['manual'] = ($data_update['coop-setting__geolocation-value'] == 'manual') ? 'selected' : '';
		$value['type']['auto'] = ($data_update['coop-setting__geolocation-value'] == 'auto') ? 'selected' : '';

		$value['setting']['banner_display']['eu'] = isset($data_update['coop-setting__display-banner-eu']) ? 'checked' : '';
		$value['setting']['banner_display']['us'] = isset($data_update['coop-setting__display-banner-us']) ? 'checked' : '';
		$value['setting']['banner_display']['ja'] = isset($data_update['coop-setting__display-banner-ja']) ? 'checked' : '';
		$value['setting']['banner_display']['other'] = isset($data_update['coop-setting__display-banner-other']) ? 'checked' : '';

		$value['setting']['pri_policy_link']['eu'] = isset($data_update['coop-setting__pri-policy-link-eu']) ? 'checked' : '';
		$value['setting']['pri_policy_link']['us'] = isset($data_update['coop-setting__pri-policy-link-us']) ? 'checked' : '';
		$value['setting']['pri_policy_link']['ja'] = isset($data_update['coop-setting__pri-policy-link-ja']) ? 'checked' : '';
		$value['setting']['pri_policy_link']['other'] = isset($data_update['coop-setting__pri-policy-link-other']) ? 'checked' : '';

		$value['setting']['term_of_use']['us'] = isset($data_update['coop-setting__term-of-use-us']) ? 'checked' : '';
		$data_format = wp_json_encode($value);

		//Update
		$option = get_option('cookie_opt_geolocation_setting');
		if (!isset($option)) {
			add_option('cookie_opt_geolocation_setting', $data_format, '', true);
		} else {
			update_option('cookie_opt_geolocation_setting', $data_format);
		}
	}

	/**
	 * Update data for banner layout tab
	 * @param mixed $regulation regulation selected
	 * @param mixed $data_update data setting to update
	 * @return bool|int
	 */
	public function updateDataBannerLayout($regulation, $data_update)
	{
		global $wpdb;
		$result = $wpdb->get_row($wpdb->prepare("SELECT setting FROM {$this->table} WHERE regulation = %s ", $regulation), ARRAY_A); // phpcs:ignore WordPress.DB
		$result['setting'] = json_decode($result['setting'], true);

		$result['setting']['position'] = $data_update['coop-layout__position'];
		$result['setting']['animation'] = $data_update['coop-layout__animation'];

		$result['setting']['colors']['text-color'] = $data_update['coop-layout__text-color'];
		$result['setting']['colors']['button-color'] = $data_update['coop-layout__btn-color'];
		$result['setting']['colors']['bar-color'] = $data_update['coop-layout__bar-color'];
		$result['setting']['colors']['bar-opacity'] = $data_update['coop-layout__bar-opacity'];
		// //Parse to Json
		$result['setting'] = wp_json_encode($result['setting']);

		$update = $wpdb->update($this->table, ['setting' => $result['setting']], ['regulation' => $regulation]); // phpcs:ignore WordPress.DB

		$this->updateDefaultBanner($regulation);

		return $update;
	}

	/**
	 * Gets the current default regulation (where status = 1).
	 *
	 * @return string|null The regulation code if found, or null if not set.
	 */
	public function getDefaultRegulation()
	{
		global $wpdb;

		$regulation = $wpdb->get_var(
			"SELECT regulation FROM {$this->table} WHERE status = 1 LIMIT 1"
		); // phpcs:ignore WordPress.DB

		return $regulation ?: self::COOKIE_OPT_REGULATION_DEFAULT;
	}

	/**
	 * Updates the default banner by:
	 * - Setting all `status` values in the table to 0
	 * - Setting `status = 1` for the row that matches the given regulation
	 *
	 * @param string $regulation The regulation code to set as default.
	 *                           Valid values: 'eu', 'us', 'ja'.
	 *                           Defaults to 'ja' if the value is not valid.
	 *
	 * @return bool Always returns true after the update.
	 */
	public function updateDefaultBanner($regulation = self::COOKIE_OPT_REGULATION_DEFAULT)
	{
		global $wpdb;
		if (!in_array($regulation, self::COOKIE_OPT_REGULATION_LIST, true)) {
			$regulation = self::COOKIE_OPT_REGULATION_DEFAULT;
		}
		$wpdb->query("UPDATE $this->table SET status = 0"); // phpcs:ignore WordPress.DB

		$wpdb->update(
			$this->table, // phpcs:ignore WordPress.DB
			['status' => 1],
			['regulation' => $regulation],
			['%d'],
			['%s']
		);

		return true;
	}

	/**
	 * Reset data setting startup tab
	 * @return bool
	 */
	public function resetDataStartup()
	{
		update_option('cookie_opt_first_setting', 0);
		return true;
	}


	/**
	 * General do not sell pageư
	 * @return int|\WP_Error
	 */
	public function generateDoNotSellPage()
	{
		global $wpdb;

		$result = $wpdb->get_row($wpdb->prepare("SELECT default_language FROM {$this->table} WHERE regulation = %s ", 'us'), ARRAY_A); // phpcs:ignore WordPress.DB
		$data = $this->renderDataDoNotSellPage($result['default_language']);
		$page = $this->createPageDoNotSell($data);

		$new_page_id = wp_insert_post([
			'comment_status' => 'close',
			'ping_status' => 'close',
			'post_author' => 1,
			'post_title' => $data['title'],
			'post_name' => 'term-of-use',
			'post_status' => 'publish',
			'post_content' => $page,
			'post_type' => 'page',
			'tax_input' => array(
				'post_tag' => array('Do Not Sell Information Page')
			)
		]);

		$option_name = 'cookie_opt_do_not_sell_page';
		$old_page_id = get_option($option_name);
		if (!empty($old_page_id)) {
			wp_delete_post($old_page_id, true);
		}
		update_option($option_name, $new_page_id);

		return $new_page_id;
	}

	/**
	 * Create content for Do not sell page
	 * @param mixed $data
	 * @return bool|string
	 */
	public function createPageDoNotSell($data)
	{
		$title = $data['content']['title'];
		$heading = $data['content']['heading_1'];
		$paragraph_1 = $data['content']['paragraph_1'];
		$title_paragraph_2 = $data['content']['heading_2'];
		$paragraph_2 = $data['content']['paragraph_2'];
		$footer = $data['content']['footer'];
		ob_start();
		?>
		<!-- <h2 style="text-align: center;"><?php echo esc_attr(
			$title
		); ?><br></h2> -->
		<h4><?php echo esc_attr($heading); ?><br></h4>
		<p><?php echo esc_attr($paragraph_1); ?><br></p>
		<h5><?php echo esc_attr($title_paragraph_2); ?><br></h5>
		<p><?php echo esc_attr($paragraph_2); ?><br></p>
		<p><?php echo esc_attr($footer); ?></p>
		<?php return ob_get_clean();
	}

	/**
	 * Render content for Do not sell page
	 * @param mixed $lang
	 * @return array
	 */
	public function renderDataDoNotSellPage($lang)
	{
		$data = [
			'en' => [
				'title' => 'Do not sell my personal information',
				'content' => [
					'title' => 'Do not sell my personal information',

					'heading_1' =>
						'Your rights under the California Consumer Privacy Act',

					'paragraph_1' => "The California Consumer Privacy Act (CCPA) provides you with rights regarding how your data or personal information is treated.
                    Under the legislation, California residents can choose to opt out of the “sale” of their personal information to third parties.
                    Based on the CCPA definition, “sale” refers to data collection for the purpose of creating advertising and other communications. Learn more about CCPA and your privacy rights.
                    Learn more about CCPA and your privacy rights.",

					'heading_2' => 'How to opt out',

					'paragraph_2' => "By clicking on the link below, we will no longer collect or sell your personal information. 
                    This applies to both third-parties and the data we collect to help personalize your experience on our website or through other communications. 
                    For more information, view our privacy policy.",

					'footer' =>
						'To be eligible to opt-out, you must be browsing from California.',
				],
			],
			'ja' => [
				'title' => 'Do not sell my personal information',
				'content' => [
					'title' => '私の個人情報を販売しないでください',

					'heading_1' =>
						'カリフォルニア州消費者プライバシー法に基づくあなたの権利',

					'paragraph_1' => "カリフォルニア州消費者プライバシー法 (CCPA) は、データまたは個人情報の取り扱いに関する権利をお客様に与えます。
                    この法律に基づき、カリフォルニア州の住民は、第三者への個人情報の「販売」をオプトアウトすることを選択できます。
                    CCPA の定義に基づくと、「販売」とは、広告やその他のコミュニケーションを作成する目的でのデータ収集を指します。
                    CCPA とプライバシー権について詳しくは、こちらをご覧ください。",

					'heading_2' => 'オプトアウトする方法',

					'paragraph_2' => "以下のリンクをクリックすると、当社がお客様の個人情報を収集または販売することはなくなります。
                    これは、サードパーティと、当社の Web サイトまたはその他のコミュニケーションを通じてお客様のエクスペリエンスをパーソナライズするために当社が収集するデータの両方に適用されます。
                    詳細については、当社のプライバシー ポリシーをご覧ください。",

					'footer' =>
						'オプトアウトするには、カリフォルニア州から閲覧している必要があります。',
				],
			],
		];

		return $data[$lang];
	}


	/**
	 * Render data for conditional param value
	 * @return array{all_page_types: array{front: string, home: string, all_pages: mixed, all_post_types: array<string|\WP_Post_Type>, user_types: array{guest: string, logged_in: string}}}
	 */
	public function dataConditionalParamValue()
	{
		$all_pages = $this->post_service->loadAllPage();
		$all_post_types = get_post_types(['public' => true], 'names', 'or');
		unset($all_post_types['attachment']);

		$data = [
			'all_page_types' => ['front' => 'Front Page', 'home' => 'Home Page'],
			'all_pages' => json_decode(wp_json_encode($all_pages), true),
			'all_post_types' => $all_post_types,
			'user_types' => ['logged_in' => 'Login', 'guest' => 'Guest'],
		];

		return $data;
	}

	/**
	 * Check param value for conditional display banner
	 * @param mixed $rule
	 * @return bool
	 */
	public function checkParam($rule, $page_id, $page_type, $post_type, $user_type)
	{
		switch ($rule['type']) {
			case 'page':
				$check = $this->handlePageValueCondition(
					$rule['operator'],
					$rule['value'],
					$page_id
				);
				break;
			case 'post_type':
				$check = $this->handlePostTypeValueCondition(
					$rule['operator'],
					$rule['value'],
					$post_type
				);
				break;
			case 'user_type':
				$check = $this->handleUserTypeValueCondition(
					$rule['operator'],
					$rule['value'],
					$user_type
				);
				break;
			default:
				$check = $this->handlePageTypeValueCondition(
					$rule['operator'],
					$rule['value'],
					$page_type
				);
				break;
		}
		return $check;
	}

	/**
	 * check post type value for conditional display
	 * @param mixed $operator
	 * @param mixed $value
	 * @return bool
	 */
	public function handlePostTypeValueCondition($operator, $value, $post_type)
	{
		if ($operator === 'equal') {
			return $value === $post_type;
		}
		return $value !== $post_type;
	}

	/**
	 * Check page value for conditional display
	 * @param mixed $operator
	 * @param mixed $value
	 * @return bool
	 */
	public function handlePageValueCondition($operator, $value, $page_id)
	{
		if ($operator === 'equal') {
			return $page_id === $value;
		}
		return $page_id !== $value;
	}

	/**
	 * Check page type value for conditional display
	 * @param mixed $operator
	 * @param mixed $value
	 * @return bool
	 */
	public function handlePageTypeValueCondition($operator, $value, $page_type)
	{
		if ($operator === 'equal') {
			return $page_type === $value;
		}
		return $page_type !== $value;
	}

	/**
	 * Check user type value for conditional status
	 * @param mixed $operator
	 * @param mixed $value
	 * @return bool
	 */
	public function handleUserTypeValueCondition($operator, $value, $user_type)
	{

		if ($operator === 'equal') {
			return $user_type === $value;
		}
		return $user_type !== $value;
	}
}
