<?php

namespace App\Services;

use App\Includes\CookieOptPublishSettings;
use App\Includes\FeatureManager\CookieOptFeatureManager;
use App\Services\BaseService\CookieOptBaseService;
use App\Validates\PublishSettingsService;
use DateTimeZone;
use WP_Error;

class CookieOptBannerService extends CookieOptBaseService
{
	const COOKIE_OPT_DEFAULT_DATA = [
		'plan' => '----',
		'sp_language' => false,
		'limit_banner' => false,
		'number_limit' => 0,
		'1' => 0,
		'2' => 0,
		'3' => 0,
		'4' => 0,
		'created_time' => '',
		'expired_time' => ''
	];

	public function getTable()
	{
		return 'cookie_opt_cookie_list';
	}

	public function saveCookie($data)
	{
		global $wpdb;
		$all_data = [];
		$all_data = array_merge($all_data, array_keys($data));

		$data_cookie['en']['Google'] = $all_data;
		$data_cookie['cdb_license'] = '123456789';
		$results = $this->getCookieInfo($data_cookie);

		if (!empty($results)) {
			$wpdb->query($wpdb->prepare("DELETE FROM {$this->table} WHERE type = %d", 0)); // phpcs:ignore WordPress.DB
			foreach ($results->en as $result) {
				$category_id = $this->checkCategory($result->purposeID);
				if (in_array($category_id, [1, 2, 3, 4])) {
					$cookie['name'] = $result->name;
					$cookie['categoryID'] = $this->checkCategory($result->purposeID);
					$cookie['description'] = $result->cookieFunction;
					$cookie['publisher'] = $result->service;

					$wpdb->insert($this->table, $cookie); // phpcs:ignore WordPress.DB
				}
			}
		}
	}

	public function scanCookieDefaultData()
	{
		global $wpdb;

		$all_data = [];
		$all_data = array_filter(array_keys($_COOKIE), function ($cookie_name) {
			return !preg_match('/^(PHPSESSID|wordpress_)/', $cookie_name);
		});

		if (count($all_data) != 0) {
			$data_cookie['en']['Google'] = $all_data;
			$data_cookie['cdb_license'] = '123456789';
			$results = $this->getCookieInfo($data_cookie);

			if (!empty($results)) {
				$wpdb->query( // phpcs:ignore WordPress.DB
					"DELETE FROM {$this->table}" // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				);

				foreach ($results->en as $result) {
					$cookie['name'] = $result->name;
					$cookie['categoryID'] = $this->checkCategory($result->purposeID);
					$cookie['description'] = $result->cookieFunction;
					$cookie['publisher'] = $result->service;

					$wpdb->insert($this->table, $cookie); // phpcs:ignore WordPress.DB
				}
			}
		}
	}

	public function getCookieInfo($data)
	{
		$json = wp_json_encode($data);
		$endpoint = trailingslashit(MY_PREFIX_COOKIE_DATABASE_URL) . 'v1/cookies/';

		$args = [
			'body' => $json,
			'headers' => [
				'Content-Type' => 'application/json',
			],
			'timeout' => 60,
			'redirection' => 5,
			'blocking' => true,
			'httpversion' => '1.0',
			'sslverify' => false,
		];

		$response = wp_remote_post($endpoint, $args);

		if (is_wp_error($response)) {
			$error_message = $response->get_error_message();
			return new WP_Error(
				'api_error',
				"Could not connect to cookiedatabase.org: $error_message"
			);
		} else {
			$body = wp_remote_retrieve_body($response);
			$result = json_decode($body);

			if (isset($result->data->error)) {
				return new WP_Error('api_error', $result->data->error);
			} else {
				return $result->data;
			}
		}
	}

	public function checkCategory($purpose_id)
	{
		$category_id = '';
		if ($purpose_id == 61) {
			$category_id = 3;
		}
		if ($purpose_id == 62 || $purpose_id == 63) {
			$category_id = 2;
		}
		return $category_id;
	}

	/**
	 * Statistic Banner
	 * @param int $type
	 * @return void
	 */
	public function statisticBanner($type = 4)
	{
		$key = get_option('cookie_opt_id', '');
		$statistics = get_option('cookie_opt_statistics', '');
		if (empty($key) || empty($statistics)) {
			return;
		}
		$statistics = d2e4c8a1_9f0b_4c3d_8e72_1a5f9bd0c6a4($key, $statistics);
		$statistics[$type]++;
		$data_encrypted = e8e9f2b_5c41_4a67_b1d3_9e2f0c7a84e5($key, $statistics);
		update_option('cookie_opt_statistics', $data_encrypted);
		if($statistics['4'] == $statistics['number_limit']){
			do_action('cookie_opt_check_daily_event');
			return;
		}
	}

	public static function getStatisticBanner()
	{
		$key = get_option('cookie_opt_id', '');
		if (empty($key)) {
			return self::COOKIE_OPT_DEFAULT_DATA;
		}
		$statistics = get_option('cookie_opt_statistics', '');
		if (!empty($statistics)) {
			return d2e4c8a1_9f0b_4c3d_8e72_1a5f9bd0c6a4($key, $statistics);
		}
		$statistics = self::cookie_opt_sync_statistics(true);
		return $statistics;
	}

	/**
	 * Sync statistic with API.
	 *
	 * @param bool $return_decrypted  true
	 * @return bool|array true
	 */
	public static function cookie_opt_sync_statistics($return_decrypted = false)
	{
		$key = get_option('cookie_opt_id', '');
		if (empty($key)) {
			return false;
		}

		$api_url = COOKIE_OPT_API_MEMBER . 'daily-statistics';
		$data_statistic = get_option('cookie_opt_statistics', '');

		$statistics = ! empty($data_statistic)
			? $data_statistic
			: e8e9f2b_5c41_4a67_b1d3_9e2f0c7a84e5($key, self::COOKIE_OPT_DEFAULT_DATA);

		$domain   = wp_parse_url(home_url());
		$headers  = [
			'Content-Type'  => 'application/json',
			'Authorization' => 'Basic Y29va2llb3B0X2FkbV91c2VyOmNWVExKaDlwQ3hSWg==',
			'Referer'       => isset($domain['host']) ? $domain['host'] : '',
		];
		$payload  = [
			'key'        => $key,
			'statistics' => $statistics,
		];

		$response = wp_remote_post($api_url, [
			'headers' => $headers,
			'body'    => wp_json_encode($payload),
			'method'  => 'POST',
			'timeout' => 20,
		]);

		if (is_wp_error($response)) {
			return self::COOKIE_OPT_DEFAULT_DATA;
		}
		if (200 !== wp_remote_retrieve_response_code($response)) {
			return self::COOKIE_OPT_DEFAULT_DATA;
		}

		$body   = wp_remote_retrieve_body($response);
		$json   = json_decode($body, true);
		$data_response   = isset($json['data']) ? $json['data'] : null;

		if (empty($data_response) || ! is_array($data_response)) {
			return self::COOKIE_OPT_DEFAULT_DATA;
		}

		update_option('cookie_opt_statistics', $data_response['statistics']);

		(new CookieOptPublishSettingsService())->cookie_opt_schedule_daily_once($data_response['schedule_time']);

		if ($return_decrypted) {
			return d2e4c8a1_9f0b_4c3d_8e72_1a5f9bd0c6a4($key, $data_response['statistics']);
		}

		return true;
	}
}
