<?php

namespace App\Includes;

// exit if accessed directly
if (!defined('ABSPATH')) {
	exit();
}

use App\Includes\CookieOptBaseInc;
use App\Services\CookieOptPostService;
use App\Services\CookieOptTagInfoService;
use App\Services\CookieOptTagManagementService;
use App\Services\CookieOptTagManagementListTable;
use App\Validates\TagManagementRequest;

/**
 * CookieOptTagManagement class.
 *
 * @class CookieOptTagManagement
 */
class CookieOptTagManagement extends CookieOptBaseInc
{
	public $parameters = [];
	public $operators = [];
	public $conditional_display_types = [];
	public $positions = [];
	public $styles = [];
	public $revoke_opts = [];
	public $links = [];
	public $link_targets = [];
	public $link_positions = [];
	public $colors = [];
	public $times = [];
	public $effects = [];
	public $script_placements = [];
	public $level_names = [];
	public $text_strings = [];
	private $analytics_app_data = [];

	/**
	 * Class constructor.
	 *
	 * @return void
	 */
	public function __construct()
	{
		// actions
		add_action('wp_print_footer_scripts', [$this, 'renderPositionScripts']);
		add_action('wp_print_styles', [$this, 'renderPositionStyles']);
		add_action('admin_init', [$this, 'admin_hook_page']);
	}

	public function renderPositionScripts()
	{
		$this->renderPosition(1);
	}

	public function renderPositionStyles()
	{
		$this->renderPosition(0);
	}

	public function renderPosition($position)
	{
		$page_id = get_the_ID();
		$Post_Service = new CookieOptPostService();
		$Tag_Service = new CookieOptTagManagementService();

		$tag = $Tag_Service->getAllTagPosition($position);
		foreach ($tag as $key => $value) {
			if ($value['installation_page'] === 'all') {
				$listPage = $Post_Service->getAllPage();
				$listArrayId = wp_list_pluck($listPage, 'ID');
				$tag[$key]['listpage'] = $listArrayId;
			} else {
				$tag[$key]['listpage'] = json_decode($value['installation_page'], true);
			}
		}

		foreach ($tag as $key => $value) {
			if (in_array($page_id, $value['listpage'])) {
				echo htmlspecialchars_decode($value['tag']); //phpcs:ignore WordPress.Security
			}
		}
	}

	public function admin_hook_page()
	{
		add_action('admin_post_handle_add_tag', [$this, 'handle_tag']);
		add_action('admin_post_handle_update_tag', [$this, 'handle_tag_update']);
	}

	public function run()
	{
		$this->action = 'index';
		if (
			isset($_GET['tag_nonce']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['tag_nonce'])), 'tag_action')
		) {
			$this->action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : 'index';
			$this->param['id'] = isset($_GET['id']) ? sanitize_text_field(wp_unslash($_GET['id'])) : null;
			$this->param['verify'] = isset($_GET['verify']) ? sanitize_text_field(wp_unslash($_GET['verify'])) : null;
		}

		switch ($this->action) {
			case 'add':
				$this->tag_management_add();
				break;
			case 'edit':
				if (!empty($this->param['id'])) {
					$this->tag_management_edit($this->param['id']);
					break;
				}
			case 'delete':
				if (!empty($this->param['id'])) {
					$this->handle_delete($this->param['id']);
					break;
				}
			default:
				$this->index();
				break;
		}
	}

	/**
	 * Get data for the Add Tag view.
	 *
	 * @return array - Data for the Add Tag view.
	 */
	public function tag_management_add()
	{
		//Get all Tag Info
		$tag_info_service = new CookieOptTagInfoService();
		$all_tag = $tag_info_service->get();

		//Get all Page
		$post_service = new CookieOptPostService();
		$all_page = $post_service->getAllPage();

		$data = [
			'all_tag' => $all_tag,
			'all_page' => $all_page,
		];
		$this->view('tag_managements/tag_management_add', $data);
	}
	/**
	 * Get data for editing a tag.
	 *
	 * @param int $id - The ID of the tag to edit.
	 * @return array - Data for editing the tag.
	 */
	function tag_management_edit($id)
	{
		//Get Tag Management
		$tag_management_service = new CookieOptTagManagementService();
		$tag_management = $tag_management_service->find($id);

		if ($tag_management == null) {
			wp_redirect(admin_url('admin.php?page=tag_management'));
			die();
		}

		//Get Tag ID & Name
		$tag_info_service = new CookieOptTagInfoService();
		$all_tag = $tag_info_service->get();

		$tag_id = $tag_management->tagid;
		if ($tag_id == null) {
			$tag_name = 'Other';
		} else {
			$tag = $tag_info_service->find($tag_id);
			$tag_name = $tag->name;
		}

		//Get Pages
		$post_service = new CookieOptPostService();
		$all_page = $post_service->getAllPage();

		$installation_page = json_decode($tag_management->installation_page, true);
		if ($tag_management->installation_page == 'all') {
			$installation_page = wp_list_pluck($all_page, 'ID');
			$installation_page[] = 'all';
		}

		$data = [
			'id' => $id,
			'name' => $tag_management->name,
			'tag_id' => $tag_id,
			'tag_name' => $tag_name,
			'tag' => $tag_management->tag,
			'position' => $tag_management->position,
			'installation_page' => $installation_page,
			'all_page' => $all_page,
			'active' => $tag_management->active,
			'all_tag' => $all_tag,
		];
		$this->view('tag_managements/tag_management_edit', $data);
	}

	/**
	 * Handle tag deletion.
	 */
	public function handle_delete($id)
	{
		global $wpdb;
		$checkExist = TagManagementRequest::ruleCheckExistTag($id);
		if (empty($checkExist)) {
			wp_redirect(admin_url('admin.php?page=tag_management'));
			exit();
		}
		$table_name = $wpdb->prefix . 'cookie_opt_tag_management';
		$wpdb->delete($table_name, ['id' => $id]); // phpcs:ignore WordPress.DB
		set_transient('successDelate', true);
		set_transient(
			'msg-cookie-opt',
			esc_html__('Delete successfully', 'cookie-opt')
		);
		wp_redirect(admin_url('admin.php?page=tag_management'));
		exit();
	}

	/**
	 * Get data for the Tag Management view.
	 *
	 * @return array - Data for the Tag Management view.
	 */
	public function index()
	{
		$add_url = wp_nonce_url(
			admin_url('admin.php?page=tag_management&action=add'),
			'tag_action',
			'tag_nonce'
		);
		$table = new CookieOptTagManagementListTable();

		$data = [
			'table' => $table,
			'add_url' => $add_url,
		];
		$this->view('tag_managements/tag_management', $data);
	}
	public function handle_tag_update()
	{
		if (
			isset($_POST['nds_add_user_meta_nonce']) &&
			wp_verify_nonce(
				sanitize_text_field(wp_unslash($_POST['nds_add_user_meta_nonce'])),
				'nds_add_user_meta_form_nonce'
			)
		) {
			global $wpdb;
			$nonce_action = isset($_POST['tag_action']) ? sanitize_text_field(wp_unslash($_POST['tag_action'])) : '';
			$errors = TagManagementRequest::validateUpdate($_POST, sanitize_text_field(wp_unslash($_GET['id']))); //phpcs:ignore WordPress.Security
			if ($errors->has_errors()) {
				set_transient('errorsForUpdate', $errors);
			} else {
				$data_tag = [];
				$data_tag['tag'] = htmlspecialchars(trim(wp_unslash($_POST['tag']))); //phpcs:ignore WordPress.Security
				$data_tag['active'] = sanitize_text_field(wp_unslash($_POST['active'])); //phpcs:ignore WordPress.Security
				$data_tag['position'] = sanitize_text_field(wp_unslash($_POST['position'])); //phpcs:ignore WordPress.Security
				$data_tag['installation_page'] = 'all';
				if (!in_array('all', $_POST['installation_page'])) { //phpcs:ignore WordPress.Security
					$data_tag['installation_page'] = wp_json_encode($_POST['installation_page']); //phpcs:ignore WordPress.Security
				}

				if (sanitize_text_field(wp_unslash($_POST['tag_id'])) !== 'null') { //phpcs:ignore WordPress.Security
					$tag_info_service = new CookieOptTagInfoService();
					$tag_info = $tag_info_service->show_by_id($_POST['tag_id']); //phpcs:ignore WordPress.Security

					$data_tag['name'] = $tag_info->name;
					$data_tag['tagid'] = sanitize_text_field(wp_unslash($_POST['tag_id'])); //phpcs:ignore WordPress.Security
				} else {
					$data_tag['tagid'] = null;
					$data_tag['name'] = sanitize_text_field(wp_unslash($_POST['name'])); //phpcs:ignore WordPress.Security
				}
				$table_tag_management = $wpdb->prefix . 'cookie_opt_tag_management';
				$wpdb->update($table_tag_management, $data_tag, ['id' => $_GET['id']]); // phpcs:ignore WordPress
				set_transient('successUpdate', true);
				set_transient(
					'msg-cookie-opt',
					esc_html__('Update successfully', 'cookie-opt')
				);
				wp_redirect(
					admin_url(
						'admin.php?page=tag_management&action=edit&tag_nonce=' .
						$nonce_action
					)
				);
				exit();
			}

			wp_redirect(
				admin_url(
					'admin.php?page=tag_management&action=edit&id=' .
					sanitize_text_field(wp_unslash($_GET['id'])) . //phpcs:ignore WordPress.Security
					'&tag_nonce=' .
					$nonce_action
				)
			);
			exit();
		} else {
			wp_die('Forbidden', 'Forbidden', ['response' => 403]);
		}
	}
	/**
	 * Handle tag creation.
	 */
	public function handle_tag()
	{
		if (
			isset($_POST['nds_add_user_meta_nonce']) &&
			wp_verify_nonce(
				sanitize_text_field(wp_unslash($_POST['nds_add_user_meta_nonce'])),
				'nds_add_user_meta_form_nonce'
			)
		) {
			global $wpdb;
			$errors = TagManagementRequest::validateCreate($_POST);
			$nonce_action = isset($_POST['add_tag_nonce'])
				? sanitize_text_field(wp_unslash($_POST['add_tag_nonce']))
				: '';
			if ($errors->has_errors()) {
				set_transient('errorsForCreate', $errors);
			} else {
				$data_tag = [];
				$data_tag['tag'] = htmlspecialchars(wp_unslash($_POST['tag'])); //phpcs:ignore WordPress.Security
				$data_tag['installation_page'] = 'all';
				if (!in_array('all', $_POST['installation_page'])) { //phpcs:ignore WordPress.Security
					$data_tag['installation_page'] = wp_json_encode(
						$_POST['installation_page'] //phpcs:ignore WordPress.Security
					);
				}

				if (sanitize_text_field(wp_unslash($_POST['tag_id'])) !== 'null') {  //phpcs:ignore WordPress.Security
					$tag_info_service = new CookieOptTagInfoService();
					$tag_info = $tag_info_service->show_by_id(sanitize_text_field(wp_unslash($_POST['tag_id']))); //phpcs:ignore WordPress.Security

					$data_tag['name'] = $tag_info->name;
					$data_tag['position'] = $tag_info->position;
					$data_tag['tagid'] = sanitize_text_field(wp_unslash($_POST['tag_id'])); //phpcs:ignore WordPress.Security
				} else {
					$data_tag['name'] = sanitize_text_field(wp_unslash($_POST['name'])); //phpcs:ignore WordPress.Security
					$data_tag['position'] = sanitize_text_field(wp_unslash($_POST['position']));  //phpcs:ignore WordPress.Security
				}
				$table_tag_management = $wpdb->prefix . 'cookie_opt_tag_management';
				$wpdb->insert($table_tag_management, $data_tag); // phpcs:ignore WordPress.DB
				set_transient('successCreate', true);
				set_transient(
					'msg-cookie-opt',
					esc_html__('Add successfully', 'cookie-opt')
				);
				wp_redirect(
					admin_url('admin.php?page=tag_management&tag_nonce=' . $nonce_action)
				);
				exit();
			}
			wp_redirect(
				admin_url(
					'admin.php?page=tag_management&action=add&tag_nonce=' . $nonce_action
				)
			);
			exit();
		} else {
			wp_die('Forbidden', 'Forbidden', ['response' => 403]);
		}
	}
}
