<?php

namespace App\Includes;

// exit if accessed directly
if (!defined('ABSPATH')) {
	exit();
}

use App\Includes\CookieOptBaseInc;
use App\Includes\FeatureManager\CookieOptFeatureManager;
use App\Services\CookieOptBannerService;
use App\Services\CookieOptOptionService;
use App\Validates\PublishSettingsRequest;
use App\Services\CookieOptDashboardService;
use App\Services\CookieOptPublishSettingsService;
use DateTime;
use DateTimeZone;

/**
 * CookieOptPublishSettings class.
 *
 * @class CookieOptPublishSettings
 */
class CookieOptPublishSettings extends CookieOptBaseInc
{
	private $publish_setting_service;
	private $page;
	public function __construct()
	{
		$this->publish_setting_service = new CookieOptPublishSettingsService();

		add_action('admin_init', [$this, 'admin_hook_page']);
		add_action('cookie_opt_check_daily_event', [$this, 'handle_cookie_opt_check_daily']);
	}

	public function admin_hook_page()
	{
		add_action('admin_post_handle_add_publish_settings', [$this, 'handle_add_publish_settings',]);
		add_action('admin_post_handle_update_package_user', [$this, 'handle_update_package_user',]);
	}

	public function handle_cookie_opt_check_daily()
	{
		CookieOptBannerService::cookie_opt_sync_statistics();
		return;
	}

	/**
	 * First Active Cookie Opt Key
	 * @return void
	 */
	public function handle_add_publish_settings()
	{
		if (
			isset($_POST['nds_add_publish_settings']) &&
			wp_verify_nonce(
				sanitize_text_field(wp_unslash($_POST['nds_add_publish_settings'])),
				'nds_add_publish_settings'
			)
		) {
			$errors = PublishSettingsRequest::publishSettingsError($_POST);
			if ($errors->has_errors()) {
				set_transient('validate-err', $errors);
				wp_redirect(admin_url('admin.php?page=publishing_settings'));
				die();
			}
			$option_value = sanitize_text_field(wp_unslash($_POST['cookie_opt_id'])); //phpcs:ignore WordPress.Security
			$body = [
				'key' => $option_value,
			];
			$domain = wp_parse_url(home_url());
			$response = wp_remote_post(COOKIE_OPT_API_MEMBER . 'first-activation-key', [
				'body' => wp_json_encode($body),
				'headers' => [
					'Content-Type' => 'application/json',
					'Authorization' =>
					'Basic Y29va2llb3B0X2FkbV91c2VyOmNWVExKaDlwQ3hSWg==',
					'Referer' => $domain['host'],
				],
			]);
			// Check result
			if (is_wp_error($response)) {
				//notify error
				wp_redirect(admin_url('admin.php?page=publishing_settings'));
				die();
			} else {
				// handel response
				$response_body = wp_remote_retrieve_body($response);
				$array_response = json_decode($response_body, true);
				if (isset($array_response['status']) && $array_response['status'] == true) {
					$option_value = $array_response['data']['key_value'];
					update_option('cookie_opt_id', $option_value);
					update_option('cookie_opt_banner_active', 1);
					update_option('cookie_opt_statistics', $array_response['data']['statistics']);
					$this->publish_setting_service->cookie_opt_schedule_daily_once($array_response['data']['schedule_time']);
				} else {
					//notify updated failed
					$errors->add('publish_settings', esc_html__('Wrong key.', 'cookie-opt'));+
					set_transient('validate-err', $errors);
					if (false !== get_transient('packageUser')) {
						delete_transient('packageUser');
					}
					wp_redirect(admin_url('admin.php?page=publishing_settings'));
					die();
				}
			}

			is_multisite() ? switch_to_blog(get_current_blog_id()) : ''; // Switch current site when was multiple site

			set_transient(
				'msg-cookie-opt',
				esc_html__('Add successfully', 'cookie-opt')
			);
			if (false !== get_transient('packageUser')) {
				delete_transient('packageUser');
			}

			is_multisite() ? restore_current_blog() : ''; // Return to previous site when was multiple site

			wp_redirect(admin_url('admin.php?page=publishing_settings'));
			die();
		}
		wp_die('Forbidden', 'Forbidden', ['response' => 403]);
	}

	public function handle_update_package_user()
	{
		if (
			isset($_POST['nds_add_publish_settings']) &&
			wp_verify_nonce(
				sanitize_text_field(wp_unslash($_POST['nds_add_publish_settings'])),
				'nds_add_publish_settings'
			)
		) {
			$plan = CookieOptFeatureManager::userPackage()->packageType();
			$url = 'https://cookieoptimizer.net/login/';
			if ($plan === 'PRO') {
				$url = admin_url('admin.php?page=publishing_settings');
			}
			wp_redirect($url);
			die();
		}
		wp_die('Forbidden', 'Forbidden', ['response' => 403]);
	}

	public function run()
	{
		$this->index();
	}
	public function index()
	{
		is_multisite() ? switch_to_blog(get_current_blog_id()) : ''; // Switch current site when was multiple site
		$userPackage = CookieOptFeatureManager::userPackage();
		$plan = $userPackage->packageType();
		$limitPackage = null;
		$remaining_amount = null;

		$option = get_option('cookie_opt_id', '');

		if ($plan == 'LOCK') {
			$plan = '----';
			$limitPackage = '----';
			$remaining_amount = '----';
			$createTime = '----';
		} else {
			if ($plan == 'PRO') {
				$limitPackage = esc_html__('Unlimited', 'cookie-opt');
				$remaining_amount = esc_html__('Unlimited', 'cookie-opt');
			}
			$createTime = $userPackage->getCreateTime();
		}

		is_multisite() ? restore_current_blog() : ''; // Return to previous site when was multisite

		// if (CookieOptFeatureManager::userPackage()->checkCurrentLimit()) {
		// 	$this->view('module/notice-limit', ['plan' => $plan]);
		// }
		$this->view('publish_setting/index', [
			'option' => $option,
			'page' => $this->page,
			'plan' => $plan,
			'limit_package' => $limitPackage,
			'remaining_amount' => $remaining_amount,
			'created_time' => $createTime,
		]);
	}
}
