<?php

namespace App\Includes;

// exit if accessed directly
if (!defined('ABSPATH')) {
	exit();
}

use App\Includes\CookieOptBaseInc;
use App\Services\CookieOptLanguagesService;
use App\Services\CookieOptPostService;
use App\Services\CookieOptPolicyServices;
use App\Services\CookiePermissionService;
use App\Validates\PolicyRequestCategoryRequest;
use App\Validates\PolicyRequestListRequest;
/**
 * Cookie_Opt_Cookie_Permission class.
 *
 * @class Cookie_Opt_Cookie_Permission
 */
class CookieOptCookiePolicy extends CookieOptBaseInc
{
	const COOKIE_OPT_STATUS_ADD_COOKIE_LIST = 1;
	public $policy_services;
	public $post_services;
	public $page_index;
	public$permission_service;
	public function __construct()
	{
		$this->policy_services = new CookieOptPolicyServices();
		$this->post_services = new CookieOptPostService();
		$this->permission_service = new CookiePermissionService();
		add_action('admin_init', [$this, 'admin_hook_page']);
	}

	public function admin_hook_page()
	{
		add_action('admin_post_handle_add_category', [
			$this,
			'handle_add_category',
		]);
		add_action('cookie_plugin_policy_library', [
			$this,
			'cookie_plugin_policy_library',
		]);
		add_action('admin_post_handle_add_cookie_list', [
			$this,
			'handle_add_cookie_list',
		]);
		add_action('admin_post_handle_edit_cookie_list', [
			$this,
			'handle_edit_cookie_list',
		]);
	}

	public function cookie_plugin_policy_library($page)
	{
		if ($page == 'cookie-optimizer_page_cookie_policy') {
			$notify_cookie_scan_by_locale = $this->permission_service->getNotifyCookieScanByLocale();
			wp_register_style(
				'cookie-opt-custom-css',
				COOKIE_OPT_URL . '/public/assets/css/co_languages.css',
				[],
				VERSION
			);

			wp_register_script_es6(
				'cookie-opt-policy-js-custom',
				vite('resources/js/pages/cookie-policy.js'),
				[],
				VERSION,
				true
			);
			wp_localize_script('cookie-opt-policy-js-custom', 'dataCookiePolicy', [
				'notifyCookieScan' => $notify_cookie_scan_by_locale
			]);

			//apply register
			wp_enqueue_style('cookie-opt-custom-css');
			wp_enqueue_style('cookie-opt-policy-custom-css');
			wp_enqueue_script('cookie-opt-policy-js-custom');
		}
	}

	public function run()
	{
		$this->action = 'index';
		$this->page_index = 'category';
		if (
			isset($_GET['nonce_action']) &&
			!empty($_GET['nonce_action']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce_action'])), 'action_policy')
		) {
			if (isset($_GET['index']) && !empty($_GET['index'])) {
				$page_index = sanitize_key($_GET['index']);
				if ($page_index == 'list-cookie') {
					$this->page_index = 'list-cookie';
				}
				if ($page_index == 'policy') {
					$this->page_index = 'policy';
				}
			}
			$this->action = isset($_GET['action'])
				? sanitize_key($_GET['action'])
				: null;
			$this->param['id'] = isset($_GET['id'])
				? sanitize_key($_GET['id'])
				: null;
			$this->param['verify'] = isset($_GET['verify'])
				? sanitize_key($_GET['verify'])
				: null;
		}

		switch ($this->action) {
			case 'delete':
				if (!empty($this->param['id']) && !empty($this->param['verify'])) {
					$this->delete($this->param['id'], $this->param['verify']);
				}
				break;
			case 'edit':
				if (!empty($this->param['id'])) {
					$this->edit($this->param['id']);
				}
				break;
			case 'policy':
				$this->renderPolicy();
				break;
			default:
				$this->index();
				break;
		}
	}

	public function edit($id)
	{
		$languages_service = new CookieOptLanguagesService();
		$res_by_index = $languages_service->getLanguagesByStatus();
		$Category = $languages_service->getCategory();
		$data_category = $this->policy_services->getDataCategory($Category);

		$list_by_id = $this->policy_services->showEditByEdit($id);
		if (is_null($list_by_id)) {
			wp_redirect(admin_url('admin.php?page=cookie_policy&index=list-cookie'));
			die();
		}
		$this->view('cookie_policy/cookie-edit-list', [
			'data_category' => $data_category,
			'default_language' => $res_by_index->default_language,
			'list_by_id' => $list_by_id,
		]);
	}

	public function index()
	{
		$languages_service = new CookieOptLanguagesService();
		$res_by_index = $languages_service->getLanguagesByStatus();
		$category = $languages_service->getCategory();
		$data_category = $this->policy_services->getDataCategory($category);
		$data_policy = $this->policy_services->indexCookieList($res_by_index->default_language);
		$cookie_list_table = $this->policy_services->getCookieListTable();
		$id_page_policy = get_option('wp_page_for_privacy_policy');
		$id_page_term_of_use = get_option('cookie_opt_do_not_sell_page');
		$all_page = [
			'policy' => [
				'ID' => 149,
				'post_title' => get_the_title($id_page_policy),
				'preview_link' => get_preview_post_link($id_page_policy),
				'edit_link' => get_edit_post_link($id_page_policy),
			],
			'term_of_use' => [
				'ID' => 150,
				'post_title' => get_the_title($id_page_term_of_use),
				'preview_link' => get_preview_post_link($id_page_term_of_use),
				'edit_link' => get_edit_post_link($id_page_term_of_use),
			]
		];

		$this->view('cookie_policy/index', [
			'page_index' => $this->page_index,
			'data_category' => $data_category,
			'default_language' => $res_by_index->default_language,
			'data_policy' => $data_policy,
			'all_pages' => $all_page,
			'cookie_list_table' => $cookie_list_table
		]);
	}

	public function handle_add_category()
	{
		if (
			isset($_POST['nds_add_category']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nds_add_category'])), 'nds_add_category')
		) {
			$nonce_action = isset($_POST['nonce_action'])
				? sanitize_text_field(wp_unslash($_POST['nonce_action']))
				: '';
			$errors = PolicyRequestCategoryRequest::validateCategoryEdit($_POST);
			if ($errors->has_errors()) {
				set_transient('validate-err', $errors);
				wp_redirect(
					admin_url(
						'admin.php?page=cookie_policy&nonce_action=' . $nonce_action
					)
				);
				die();
			}
			$dataBannerAndCategory = $this->policy_services->getDefaultLanguage();

			$this->policy_services->addCategory(
				$_POST['name'], //phpcs:ignore WordPress.Security
				$_POST['description'], //phpcs:ignore WordPress.Security
				$dataBannerAndCategory
			);
			set_transient(
				'msg-cookie-opt',
				esc_html__('Update successfully', 'cookie-opt')
			);
			wp_redirect(
				admin_url('admin.php?page=cookie_policy&nonce_action=' . $nonce_action)
			);
			die();
		}
		wp_die('Forbidden', 'Forbidden', ['response' => 403]);
	}

	public function handle_add_cookie_list()
	{
		if (
			isset($_POST['nds_add_cookie_list']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nds_add_cookie_list'])), 'nds_add_cookie_list')
		) {
			$nonce_action = isset($_POST['nonce_action'])
				? sanitize_text_field(wp_unslash($_POST['nonce_action']))
				: '';
			$errors = PolicyRequestListRequest::validateListAdd($_POST);
			if ($errors->has_errors()) {
				set_transient('validate-err', $errors);
				wp_redirect(
					admin_url(
						'admin.php?page=cookie_policy&index=list-cookie&nonce_action=' .
						$nonce_action
					)
				);
				die();
			}
			$data = [
				'name' => sanitize_text_field(wp_unslash($_POST['name'])), //phpcs:ignore WordPress.Security
				'categoryID' => sanitize_text_field(wp_unslash($_POST['category_id'])), //phpcs:ignore WordPress.Security
				'publisher' => sanitize_text_field(wp_unslash($_POST['publisher'])), //phpcs:ignore WordPress.Security
				'description' => sanitize_text_field(wp_unslash($_POST['description'])), //phpcs:ignore WordPress.Security
				'type' => self::COOKIE_OPT_STATUS_ADD_COOKIE_LIST
			];
			$this->policy_services->addCookieList($data);
			$this->policy_services->storePolicyPage();
			set_transient(
				'msg-cookie-opt',
				esc_html__('Create cookies list successfully', 'cookie-opt')
			);
			wp_redirect(
				admin_url(
					'admin.php?page=cookie_policy&index=list-cookie&nonce_action=' .
					$nonce_action
				)
			);
			die();
		}
		wp_die('Forbidden', 'Forbidden', ['response' => 403]);
	}

	public function handle_edit_cookie_list()
	{
		if (
			isset($_POST['nds_edit_cookie_list']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nds_edit_cookie_list'])), 'nds_edit_cookie_list')
		) {
			$nonce_action = isset($_POST['nonce_action'])
				? sanitize_text_field(wp_unslash($_POST['nonce_action']))
				: '';
			$errors = PolicyRequestListRequest::validateListAdd($_POST);
			if ($errors->has_errors()) {
				set_transient('validate-err', $errors);
				wp_redirect(
					admin_url(
						'admin.php?page=cookie_policy&action=edit&id=' .
						sanitize_text_field(wp_unslash($_GET['id'])) . //phpcs:ignore WordPress.Security
						'&nonce_action=' .
						$nonce_action
					)
				);
				die();
			}
			$data = [
				'name' => sanitize_text_field(wp_unslash($_POST['name'])), //phpcs:ignore WordPress.Security
				'categoryID' => sanitize_text_field(wp_unslash($_POST['category_id'])), //phpcs:ignore WordPress.Security
				'description' => sanitize_text_field(wp_unslash($_POST['description'])), //phpcs:ignore WordPress.Security
			];

			$this->policy_services->editCookieList($data, sanitize_text_field(wp_unslash($_GET['id']))); //phpcs:ignore WordPress.Security
			set_transient(
				'msg-cookie-opt',
				esc_html__('Update cookies list successfully', 'cookie-opt')
			);
			wp_redirect(
				admin_url(
					'admin.php?page=cookie_policy&index=list-cookie&nonce_action=' .
					$nonce_action
				)
			);
			die();
		}
		wp_die('Forbidden', 'Forbidden', ['response' => 403]);
	}
	public function delete($id, $verify)
	{
		if (wp_verify_nonce($verify, 'nds_delete_cookie_policy')) {
			$nonce_action = isset($_GET['nonce_action']) ? sanitize_text_field(wp_unslash($_GET['nonce_action'])) : '';
			$this->policy_services->deleteCookieList($id);
			set_transient(
				'msg-cookie-opt',
				esc_html__('Deleted cookies list successfully', 'cookie-opt')
			);
			wp_redirect(
				admin_url(
					'admin.php?page=cookie_policy&index=list-cookie&nonce_action=' .
					$nonce_action
				)
			);
			die();
		}
		wp_die('Forbidden', 'Forbidden', ['response' => 403]);
	}

	public function renderPolicy()
	{
		$languages_service = new CookieOptLanguagesService();

		$dataRender = $this->policy_services->getDataForRender();
		$res_by_index = $languages_service->getLanguagesByStatus();
		$html = $this->policy_services->renderPolicy(
			$dataRender,
			$res_by_index->default_language
		);

		$new_page = [
			'post_title' => 'Cookie Policy',
			'post_content' => $html,
			'post_status' => 'publish',
			'post_type' => 'page',
			'post_name' => 'cookie-policy',
		];
		// Tạo trang mới
		$new_page_id = wp_insert_post($new_page);
		update_option('wp_page_for_privacy_policy', $new_page_id);
		$edit_page_url = get_permalink($new_page_id);
		if ($edit_page_url) {
			wp_redirect($edit_page_url);
		} else {
			wp_redirect(admin_url('edit.php?post_type=page&paged=1'));
		}
		die();
	}
	public function add_type_attribute($attributes)
	{
		if (
			isset($attributes['id']) &&
			$attributes['id'] === 'cookie-opt-policy-js-custom-js'
		) {
			$attributes['type'] = 'module';
		}

		return $attributes;
	}

	public function renderPolicyDefault()
	{
		$dataRender = $this->policy_services->getDataForRender();
		$html = $this->policy_services->renderPolicy($dataRender, 'ja');
		$new_page = [
			'post_title' => 'Cookie Policy',
			'post_content' => $html,
			'post_status' => 'publish',
			'post_type' => 'page',
			'post_name' => 'cookie-policy',
		];
		// Tạo trang mới
		$new_page_id = wp_insert_post($new_page);
		update_option('wp_page_for_privacy_policy', $new_page_id);
	}
}
