<?php

namespace App\Includes;

use App\Includes\FeatureManager\CookieOptFeatureManager;
use App\Services\CookieOptBannerService;
use App\Services\CookieOptDashboardService;
use App\Services\CookiePermissionService;

if (!defined('ABSPATH')) {
	exit();
}

class CookieOptBanner extends CookieOptBaseInc
{
	public $CookiePermissionService;
	public $CookieOptBanner_service;
	public $server_dashboard;
	public function __construct()
	{
		$this->CookiePermissionService = new CookiePermissionService();
		$this->CookieOptBanner_service = new CookieOptBannerService();
		$this->server_dashboard = new CookieOptDashboardService();
		if (!wp_doing_ajax() && !wp_doing_cron()) {
			$package = CookieOptFeatureManager::userPackage();
			$package_user = $package->packageType();
			if ($package_user !== 'LOCK' && !$package->checkLimitPackage()) {
				add_action('wp_enqueue_scripts', [$this, 'register_enqueue_script']);
				add_action('send_headers', [$this, 'set_cookie']);
			}
		}
		if (wp_doing_ajax()) {
			add_action('wp_ajax_nopriv_handle_banner', [$this, 'handle_banner']);
			add_action('wp_ajax_handle_banner', [$this, 'handle_banner']);

			add_action('wp_ajax_nopriv_handle_view_banner', [$this, 'handle_view_banner',]);
			add_action('wp_ajax_handle_view_banner', [$this, 'handle_view_banner']);

			add_action('wp_ajax_get_parse_cookie_opt', [$this, 'handle_get_parse_cookie_opt']);
			add_action('wp_ajax_nopriv_get_parse_cookie_opt', [$this, 'handle_get_parse_cookie_opt']);

			add_action('wp_ajax_set_cookie_opt_server', [$this, 'handle_set_cookie_opt_server']);
			add_action('wp_ajax_nopriv_set_cookie_opt_server', [$this, 'handle_set_cookie_opt_server']);

			add_action('wp_ajax_nopriv_handle_render_banner', [$this, 'handle_render_banner']);
			add_action('wp_ajax_handle_render_banner', [$this, 'handle_render_banner']);
		}
	}

	public function set_cookie()
	{
		$is_banner_active = get_option('cookie_opt_banner_active');
		if ($is_banner_active != 1) {
			return;
		}

		if (isset($_COOKIE[COOKIE_OPT_NAME])) {
			return;
		}

		$regulation = $this->CookiePermissionService->getDataRegulationDefault();

		// default cookie values
		$data_cookie = [
			'action'      => 0,
			'type'        => $regulation === 'ja' ? 1 : '',
			'necessary'   => 1,
			'functional'  => 0,
			'socialmedia' => 0,
			'performance' => 0,
		];

		$value = wp_json_encode($data_cookie);
		$expires = $regulation === 'ja' ? time() + 31556926 : 0;
		$secure = is_ssl();

		setcookie(COOKIE_OPT_NAME, $value, $expires, '/', '', $secure, true);

		$_COOKIE[COOKIE_OPT_NAME] = $value;
		cookie_log_current_state('send_headers');
	}
	public function handle_view_banner()
	{
		if (
			isset($_POST['none']) &&
			wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['none'])), 'nds_view_banner')
		) {
			$this->CookieOptBanner_service->statisticBanner();
			return wp_send_json(
				[
					'status' => 200,
					'data' => [],
					'msg' => 'Success',
				],
				200
			);
		}
		return wp_send_json(
			[
				'status' => 401,
				'data' => [],
				'msg' => 'oop',
			],
			401
		);
	}
	public function handle_banner()
	{
		if (isset($_POST['none']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['none'])), 'none_handle_banner')) {
			$type = sanitize_text_field(wp_unslash($_POST['type'])); //phpcs:ignore WordPress.Security
			$this->CookieOptBanner_service->statisticBanner($type);
			return wp_send_json(
				[
					'status' => 200,
					'msg' => 'Success',
				],
				200
			);
		}
	}

	public function handle_get_parse_cookie_opt()
	{
		$cookieRaw = isset($_COOKIE[COOKIE_OPT_NAME]) ? $_COOKIE[COOKIE_OPT_NAME] : null;
		$dataCookie = null;

		if ($cookieRaw) {
			$dataCookie = urldecode(stripslashes(sanitize_text_field(wp_unslash($cookieRaw))));
		}

		if (!$dataCookie) {
			$dataCookie = [
				'action'      => 0,
				'type'        => 0,
				'necessary'   => 1,
				'functional'  => 0,
				'socialmedia' => 0,
				'performance' => 0,
			];
		}

		wp_send_json_success([
			'message' => 'Get cookie success',
			'cookie' => $dataCookie
		]);
	}

	public function handle_set_cookie_opt_server()
	{
		if (isset($_POST['type']) && isset($_POST['cookie_value'])) {
			$cookie_value = sanitize_text_field($_POST['cookie_value']);
			$type = isset($_POST['type']) ? intval($_POST['type']) : 1;

			$expires = ($type === 3) ? 0 : time() + (365 * 24 * 60 * 60); // 1 year

			$value    = wp_unslash($cookie_value);
			$secure   = is_ssl();

			setcookie(COOKIE_OPT_NAME, $value, $expires, '/', '', $secure, true);
			cookie_log_current_state('client_send_cookie_value');

			wp_send_json_success([
				'message' => 'Cookie update success',
				'name' => COOKIE_OPT_NAME,
				'value' => $value,
			]);
			die;
		}

		wp_send_json_error(['message' => 'Missing cookie name or value']);
	}

	public function register_enqueue_script()
	{
		$data_events = $this->prepare_data_localize();
		wp_register_script(
			'cookie-custom-banner-js',
			vite('resources/js/pages/banner.js'),
			['jquery'],
			VERSION,
			true
		);
		wp_localize_script('cookie-custom-banner-js', 'dataEvent', $data_events);

		wp_enqueue_style(
			'custom-styles-banner-content_',
			vite('resources/scss/banner.scss'),
			[],
			VERSION

		);

		wp_enqueue_script('cookie-custom-banner-js');
	}

	public function handle_render_banner()
	{
		$html_banner = $this->data_banner_html();
		$is_display_banner = $this->handle_display_banner($_GET['pageId'], $_GET['pageType'], $_GET['postType'], $_GET['userType']);

		wp_send_json_success([
			'message' => 'Get cookie success',
			'dataBannerRender' => $html_banner,
			'isDisplayBanner' => $is_display_banner
		]);
	}

	public function prepare_data_localize()
	{
		global $post;
		$setting = $this->CookiePermissionService->getSettingBanner();
		$page_id = (is_singular() && isset($post->ID)) ? (int) $post->ID : 0;
		$page_type = is_front_page() ? 'front' : 'home';
		$post_type = is_singular() ? get_post_type() : '';
		$user_type  = is_user_logged_in() ? 'login' : 'guest';

		return [
			'none_handle_banner' => wp_create_nonce('none_handle_banner'),
			'url' => admin_url('admin-ajax.php'),
			'regulation' => $setting['regulation'],
			'eventOnclick' => $setting['onClick'],
			'eventOnScroll' => $setting['onScroll'],
			'eventReload' => $setting['reloading'],
			'pageId'   => $page_id,
			'pageType' => $page_type,
			'postType' => $post_type,
			'userType' => $user_type,
		];
	}

	public function data_banner_html()
	{
		$data = $this->CookiePermissionService->getDataBannerToShow();
		ob_start();
		include COOKIE_OPT_PATH . '/templates/banner/banner_main.php';
		return ob_get_clean();
	}

	/**
	 * Summary of handle_display_banner
	 * @return bool true is display, false is hide
	 */
	public function handle_display_banner($page_id, $page_type, $post_type, $user_type)
	{
		$is_banner_active = get_option('cookie_opt_banner_active');
		if ($is_banner_active != 1)
			return false;

		$cookie_raw = $_COOKIE[COOKIE_OPT_NAME] ?? null;

		if (!$cookie_raw) {
			return false;
		}

		$decoded = urldecode(wp_unslash($cookie_raw));
		$data_cookie = json_decode($decoded, true);
		if (!isset($data_cookie['action'])) {
			return false;
		}

		if ($data_cookie['action'] === 1) {
			return false;
		}

		if ($data_cookie['action'] === 0) {

			$banner = $this->CookiePermissionService->getDataBannerToShow();
			$condition_status = (bool)($banner['conditional']['status'] ?? false);
			$rules_show = $banner['conditional']['rules']['show'] ?? [];
			$rules_hide = $banner['conditional']['rules']['hide'] ?? [];

			$anyMatch = function (array $rules) use($page_id, $page_type, $post_type, $user_type): bool {
				foreach ($rules as $rule) {
					if ($this->CookiePermissionService->checkParam($rule, $page_id, $page_type, $post_type, $user_type)) {
						return true;
					}
				}
				return false;
			};

			if (!$condition_status) {
				return true;
			}

			if(empty($rules_hide) && empty($rules_show)){
				return true;
			}

			if (!empty($rules_hide) && $anyMatch($rules_hide)) {
				return false;
			}

			if (empty($rules_show) || $anyMatch($rules_show)) {
				return true;
			}
		}

		return false;
	}
}
