<?php
/**
 * AJAX handlers for ContentWeaver AI Lite
 *
 * @package ContentWeaverAILite
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Ensure we can use the spintax utility.
 */
require_once plugin_dir_path( __FILE__ ) . 'spintax.php';

/**
 * Parse AI suggestions into a clean array of strings.
 * Handles code fences, invalid JSON with pipe separators, and basic delimiters.
 *
 * @param string $content Raw AI content string.
 * @return array<int,string> List of alternatives.
 */
function contentweaver_ai_lite_parse_ai_suggestions( string $content ): array {
    $raw = trim( $content );

    // Extract content inside triple backticks if present (```json ... ``` or ``` ... ```)
    if ( preg_match( '/```[a-zA-Z]*\s*([\s\S]*?)```/m', $raw, $m ) === 1 ) {
        $raw = trim( (string) $m[1] );
    }

    // If looks like a JSON array but uses pipes instead of commas, normalize
    if ( ( '' !== $raw ) && ( '[' === $raw[0] ) && ( ']' === substr( $raw, -1 ) ) ) {
        if ( strpos( $raw, '|' ) !== false && strpos( $raw, ',' ) === false ) {
            $raw = str_replace( '|', ',', $raw );
        }
        $decoded = json_decode( $raw, true );
        if ( is_array( $decoded ) ) {
            $items = array_map( static function ( $s ): string {
                if ( is_string( $s ) ) {
                    $s = trim( $s );
                    // Strip surrounding quotes if any lingering
                    $s = preg_replace( '/^\"|\"$/', '', $s );
                    $s = preg_replace( "/^'|'$/", '', $s );
                    return (string) $s;
                }
                return '';
            }, $decoded );
            $items = array_values( array_filter( $items, static fn( $v ): bool => '' !== $v ) );
            return array_values( array_unique( $items ) );
        }
    }

    // Fallback: remove wrapping [ ] if present and split on common delimiters
    $unwrapped = $raw;
    if ( ( '' !== $unwrapped ) && ( '[' === $unwrapped[0] ) && ( ']' === substr( $unwrapped, -1 ) ) ) {
        $unwrapped = trim( substr( $unwrapped, 1, -1 ) );
    }
    // Normalize vertical bars to commas when no commas exist
    if ( strpos( $unwrapped, ',' ) === false && strpos( $unwrapped, '|' ) !== false ) {
        $unwrapped = str_replace( '|', ',', $unwrapped );
    }

    $parts = preg_split( '/[\n,;]+/', $unwrapped );
    if ( ! is_array( $parts ) ) {
        $parts = [ $unwrapped ];
    }
    $clean = array_map( static function ( $s ): string {
        $s = is_string( $s ) ? trim( $s ) : '';
        $s = preg_replace( '/^\"|\"$/', '', $s );
        $s = preg_replace( "/^'|'$/", '', $s );
        return (string) $s;
    }, $parts );
    $clean = array_values( array_filter( $clean, static fn( $v ): bool => '' !== $v ) );
    return array_values( array_unique( $clean ) );
}

/**
 * Process basic spintax and return multiple variations.
 * Expected POST: nonce, spintax, num_spins (int), max_variations (unused here but accepted)
 */
function contentweaver_ai_lite_ajax_process_basic_spintax(): void {
	check_ajax_referer( 'spintax_generation', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'contentweaver-ai-lite' ) ] );
	}

	$spintax       = isset( $_POST['spintax'] ) && is_string( $_POST['spintax'] ) ? sanitize_textarea_field( wp_unslash( $_POST['spintax'] ) ) : '';
	$num_spins     = isset( $_POST['num_spins'] ) ? (int) $_POST['num_spins'] : 5;
	$num_spins     = max( 1, min( 10, $num_spins ) );
	$max_variations = isset( $_POST['max_variations'] ) ? (int) $_POST['max_variations'] : 5;
	$max_variations = max( 1, min( 10, $max_variations ) );

	if ( '' === trim( $spintax ) ) {
		wp_send_json_error( [ 'message' => __( 'No spintax provided.', 'contentweaver-ai-lite' ) ] );
	}

	// If AI suggestions requested, scan for {*AI*} markers and return alternatives.
	if ( isset( $_POST['generate_ai'] ) && 'true' === $_POST['generate_ai'] ) {
		$options  = get_option( CONTENTWEAVER_AI_LITE_OPTION );
		$api_key  = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
		$provider = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
		$model    = is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini';
		$base_url = 'openai' === $provider ? 'https://api.openai.com/v1' : 'https://api.deepseek.com/v1';

		if ( '' === $api_key ) {
			wp_send_json_error( [ 'message' => __( 'API key not set. Please configure it on the Lite Dashboard.', 'contentweaver-ai-lite' ) ] );
		}

		// Find all groups that contain *AI* inside braces
		$pattern = '/\{([^}]*\*AI\*[^}]*)\}/i';
		preg_match_all( $pattern, $spintax, $matches, PREG_SET_ORDER );

		$suggestions = [];
		foreach ( $matches as $match ) {
			$content = $match[1];
			$parts   = array_filter( array_map( 'trim', explode( '|', $content ) ) );
			// Remove *AI*
			$parts   = array_values( array_filter( $parts, static function ( $p ): bool { return '*AI*' !== strtoupper( $p ); } ) );

			// Determine the reference word: first non-AI part or fallback to 'example'
			$word = isset( $parts[0] ) ? $parts[0] : 'example';

			$endpoint = untrailingslashit( $base_url ) . '/chat/completions';
			$prompt   = 'Provide ' . (string) $max_variations . ' natural synonyms or small phrase alternatives for the following word or short phrase. Respond ONLY as a JSON array of strings (no code block fences, no extra keys). Word: ' . $word;
			$body     = [
				'model'       => $model,
				'messages'    => [ [ 'role' => 'user', 'content' => $prompt ] ],
				'temperature' => 0.7,
			];
			$args = [
				'headers' => [ 'Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $api_key ],
				'timeout' => 30,
				'body'    => wp_json_encode( $body ),
			];

			$response = wp_remote_post( $endpoint, $args );
			if ( is_wp_error( $response ) ) {
				wp_send_json_error( [ 'message' => $response->get_error_message() ] );
			}
			$status  = wp_remote_retrieve_response_code( $response );
			$raw     = wp_remote_retrieve_body( $response );
			$decoded = json_decode( $raw, true );
			if ( 200 !== $status || ! is_array( $decoded ) || ! isset( $decoded['choices'][0]['message']['content'] ) ) {
				$detail = is_string( $raw ) ? $raw : '';
				if ( is_array( $decoded ) && isset( $decoded['error']['message'] ) && is_string( $decoded['error']['message'] ) ) {
					$detail = (string) $decoded['error']['message'];
				}
				/* translators: %s: API error detail returned by the provider */
				wp_send_json_error( [ 'message' => sprintf( __( 'API error: %s', 'contentweaver-ai-lite' ), wp_kses_post( $detail ) ) ] );
			}

			$content_json = (string) $decoded['choices'][0]['message']['content'];
			$alts         = contentweaver_ai_lite_parse_ai_suggestions( $content_json );
			$alts         = array_values( array_filter( $alts, static fn( string $s ): bool => '' !== $s ) );
			$alts = array_slice( $alts, 0, $max_variations );

			$suggestions[] = [
				'original'     => $word,
				'alternatives' => $alts,
			];
		}

		wp_send_json_success( [ 'suggestions' => $suggestions ] );
	}

	// Default path: local-only spintax expansion
	$variations = [];
	for ( $i = 0; $i < $num_spins; $i++ ) {
		$variations[] = contentweaver_ai_lite_spin_text( $spintax );
	}

	wp_send_json_success( [ 'variations' => $variations ] );
}
add_action( 'wp_ajax_cw_lite_process_basic_spintax', 'contentweaver_ai_lite_ajax_process_basic_spintax' );

/**
 * Generate word alternatives using an OpenAI-compatible API.
 * Expected POST: nonce, word, max_variations (int)
 */
function contentweaver_ai_lite_ajax_generate_word_alternatives(): void {
	check_ajax_referer( 'spintax_generation', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'contentweaver-ai-lite' ) ] );
	}

	$word           = isset( $_POST['word'] ) && is_string( $_POST['word'] ) ? sanitize_text_field( wp_unslash( $_POST['word'] ) ) : '';
	$max_variations = isset( $_POST['max_variations'] ) ? (int) $_POST['max_variations'] : 5;
	$max_variations = max( 1, min( 10, $max_variations ) );

	if ( '' === $word ) {
		wp_send_json_error( [ 'message' => __( 'No word provided.', 'contentweaver-ai-lite' ) ] );
	}

	$options  = get_option( CONTENTWEAVER_AI_LITE_OPTION );
	$api_key  = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
	$provider = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
	$model    = is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini';

	$base_url = 'openai' === $provider ? 'https://api.openai.com/v1' : 'https://api.deepseek.com/v1';

	if ( '' === $api_key ) {
		wp_send_json_error( [ 'message' => __( 'API key not set. Please configure it on the Lite Dashboard.', 'contentweaver-ai-lite' ) ] );
	}

	$endpoint = untrailingslashit( $base_url ) . '/chat/completions';
	$prompt   = 'Provide ' . (string) $max_variations . ' natural synonyms or small phrase alternatives for the following word or short phrase. Respond ONLY as a JSON array of strings (no code block fences, no extra keys). Word: ' . $word;

	$body = [
		'model'    => $model,
		'messages' => [
			[ 'role' => 'user', 'content' => $prompt ],
		],
		'temperature' => 0.7,
	];

	$args = [
		'headers' => [
			'Content-Type'  => 'application/json',
			'Authorization' => 'Bearer ' . $api_key,
		],
		'timeout' => 30,
		'body'    => wp_json_encode( $body ),
	];

	$response = wp_remote_post( $endpoint, $args );
	if ( is_wp_error( $response ) ) {
		wp_send_json_error( [ 'message' => $response->get_error_message() ] );
	}
	$status  = wp_remote_retrieve_response_code( $response );
	$raw     = wp_remote_retrieve_body( $response );
	$decoded = json_decode( $raw, true );

	if ( 200 !== $status || ! is_array( $decoded ) || ! isset( $decoded['choices'][0]['message']['content'] ) ) {
		$detail = '';
		if ( is_array( $decoded ) && isset( $decoded['error']['message'] ) && is_string( $decoded['error']['message'] ) ) {
			$detail = (string) $decoded['error']['message'];
		}
		if ( '' === $detail && is_string( $raw ) ) {
			$detail = $raw;
		}
		/* translators: %s: API error detail returned by the provider */
		wp_send_json_error( [ 'message' => sprintf( __( 'API error: %s', 'contentweaver-ai-lite' ), wp_kses_post( $detail ) ) ] );
	}
	$content = (string) $decoded['choices'][0]['message']['content'];
	$parsed  = contentweaver_ai_lite_parse_ai_suggestions( $content );
	// Remove duplicates and the original word itself (case-insensitive)
	$parsed = array_values( array_filter( $parsed, static function ( string $s ) use ( $word ): bool {
		return '' !== $s && 0 !== strcasecmp( $s, $word );
	} ) );
	$alternatives = array_slice( $parsed, 0, $max_variations );
	$spintax      = '{' . $word;
	foreach ( $alternatives as $alt ) {
		$spintax .= '|' . $alt;
	}
	$spintax .= '}';

	wp_send_json_success( [
		'spintax'      => $spintax,
		'alternatives' => $alternatives,
		'original'     => $word,
	] );
}
add_action( 'wp_ajax_cw_lite_generate_word_alternatives', 'contentweaver_ai_lite_ajax_generate_word_alternatives' );

/**
 * Generate article content from pasted text using OpenAI-compatible API.
 * Expected POST: nonce, article_text, model (optional to override), post_status (ignored in Lite)
 */
function contentweaver_ai_lite_ajax_generate_article(): void {
	check_ajax_referer( 'contentweaver_generate_article', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'contentweaver-ai-lite' ) ] );
	}

	$article_text = isset( $_POST['article_text'] ) && is_string( $_POST['article_text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['article_text'] ) ) : '';
    $model_param  = isset( $_POST['model'] ) && is_string( $_POST['model'] ) ? sanitize_text_field( wp_unslash( $_POST['model'] ) ) : '';

	if ( '' === $article_text ) {
		wp_send_json_error( [ 'message' => __( 'Please paste article text.', 'contentweaver-ai-lite' ) ] );
	}

	$options  = get_option( CONTENTWEAVER_AI_LITE_OPTION );
	$api_key  = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
	$provider = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
	$model    = '' !== $model_param ? $model_param : ( is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini' );
	$base_url = 'openai' === $provider ? 'https://api.openai.com/v1' : 'https://api.deepseek.com/v1';

	if ( '' === $api_key ) {
		wp_send_json_error( [ 'message' => __( 'API key not set. Please configure it on the Lite Dashboard.', 'contentweaver-ai-lite' ) ] );
	}

	$endpoint = untrailingslashit( $base_url ) . '/chat/completions';
	$prompt   = 'Rewrite and enhance the following article for clarity and coherence. Keep the meaning, avoid promotional tone, and output Markdown. Article:\n\n' . $article_text;

	$body = [
		'model'    => $model,
		'messages' => [
			[ 'role' => 'user', 'content' => $prompt ],
		],
		'temperature' => 0.7,
	];

	$args = [
		'headers' => [
			'Content-Type'  => 'application/json',
			'Authorization' => 'Bearer ' . $api_key,
		],
		'timeout' => 45,
		'body'    => wp_json_encode( $body ),
	];

	$response = wp_remote_post( $endpoint, $args );
	if ( is_wp_error( $response ) ) {
		wp_send_json_error( [ 'message' => $response->get_error_message() ] );
	}

	$status  = wp_remote_retrieve_response_code( $response );
	$raw     = wp_remote_retrieve_body( $response );
	$decoded = json_decode( $raw, true );

	if ( 200 !== $status || ! is_array( $decoded ) || ! isset( $decoded['choices'][0]['message']['content'] ) ) {
		$detail = '';
		if ( is_array( $decoded ) && isset( $decoded['error']['message'] ) && is_string( $decoded['error']['message'] ) ) {
			$detail = (string) $decoded['error']['message'];
		}
		if ( '' === $detail && is_string( $raw ) ) {
			$detail = $raw;
		}
		/* translators: %s: API error detail returned by the provider */
		wp_send_json_error( [ 'message' => sprintf( __( 'API error: %s', 'contentweaver-ai-lite' ), wp_kses_post( $detail ) ) ] );
	}

	$content = (string) $decoded['choices'][0]['message']['content'];
	wp_send_json_success( [ 'content' => $content ] );
}
add_action( 'wp_ajax_cw_lite_generate_article', 'contentweaver_ai_lite_ajax_generate_article' );
add_action( 'wp_ajax_contentweaver_generate_article', 'contentweaver_ai_lite_ajax_generate_article' );

/**
 * Manual Generate: Create a draft post from either a URL (fetch + extract) or pasted text.
 * Expected POST: nonce, article_url (optional), article_text (optional), model (optional),
 *                voice_id (optional), featured_image_mode (optional: none|default|site_icon|cc0), default_image_id (optional)
 */
function contentweaver_ai_lite_ajax_manual_generate_post(): void {
    check_ajax_referer( 'contentweaver_generate_article', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'contentweaver-ai-lite' ) ] );
    }

    $input_url   = isset( $_POST['article_url'] ) && is_string( $_POST['article_url'] ) ? esc_url_raw( wp_unslash( $_POST['article_url'] ) ) : '';
    $input_text  = isset( $_POST['article_text'] ) && is_string( $_POST['article_text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['article_text'] ) ) : '';
    $model_in    = isset( $_POST['model'] ) && is_string( $_POST['model'] ) ? sanitize_text_field( wp_unslash( $_POST['model'] ) ) : '';
    $voice_in    = isset( $_POST['voice_id'] ) ? (int) $_POST['voice_id'] : 0;
    $fim_in      = isset( $_POST['featured_image_mode'] ) && is_string( $_POST['featured_image_mode'] ) ? sanitize_key( wp_unslash( $_POST['featured_image_mode'] ) ) : '';
    $def_img_in  = isset( $_POST['default_image_id'] ) ? (int) $_POST['default_image_id'] : 0;

    if ( '' === $input_url && '' === $input_text ) {
        wp_send_json_error( [ 'message' => __( 'Provide a URL or paste article text.', 'contentweaver-ai-lite' ) ] );
    }

    $options   = get_option( CONTENTWEAVER_AI_LITE_OPTION );
    $api_key   = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
    $provider  = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
    $model     = '' !== $model_in ? $model_in : ( is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini' );
    $base_url  = is_array( $options ) && isset( $options['base_url'] ) && is_string( $options['base_url'] ) && '' !== trim( $options['base_url'] )
        ? (string) $options['base_url']
        : ( 'deepseek' === $provider ? 'https://api.deepseek.com/v1' : 'https://api.openai.com/v1' );
    $voice_id  = $voice_in > 0 ? $voice_in : ( is_array( $options ) && isset( $options['voice_id'] ) ? (int) $options['voice_id'] : 10 );

    if ( '' === $api_key ) {
        wp_send_json_error( [ 'message' => __( 'API key not set. Configure it on the Lite Dashboard.', 'contentweaver-ai-lite' ) ] );
    }

    // Acquire source article text
    $article_text = $input_text;
    $source_name  = '';
    $homepage     = '';
    $link         = '';
    if ( '' === $article_text && '' !== $input_url ) {
        $link = esc_url_raw( $input_url );
        if ( '' !== $link ) {
            $f = function_exists( 'contentweaver_ai_lite_fetch_url' ) ? contentweaver_ai_lite_fetch_url( $link ) : [ 'status' => 0, 'type' => '', 'body' => '' ];
            if ( is_array( $f ) && isset( $f['status'], $f['body'] ) && (int) $f['status'] >= 200 && (int) $f['status'] < 400 && is_string( $f['body'] ) ) {
                $page_html = (string) $f['body'];
                $extracted = function_exists( 'contentweaver_ai_lite_extract_article_html' ) ? contentweaver_ai_lite_extract_article_html( $link, $page_html ) : '';
                $article_text = '' !== trim( (string) $extracted ) ? (string) $extracted : wp_strip_all_tags( $page_html );
            }
            $host        = (string) ( wp_parse_url( $link, PHP_URL_HOST ) );
            $source_name = '' !== $host && function_exists( 'contentweaver_ai_lite_source_name_from_host' ) ? contentweaver_ai_lite_source_name_from_host( $host ) : '';
            $homepage    = '' !== $host ? ( 'https://' . $host ) : '';
        }
    }

    if ( '' === trim( $article_text ) ) {
        wp_send_json_error( [ 'message' => __( 'Could not obtain article content from URL or text.', 'contentweaver-ai-lite' ) ] );
    }

    // Voice context
    $voices = function_exists( 'contentweaver_ai_lite_get_voices' ) ? contentweaver_ai_lite_get_voices() : [];
    $voice  = isset( $voices[ $voice_id ] ) ? $voices[ $voice_id ] : ( isset( $voices[10] ) ? $voices[10] : null );
    $voice_name = is_array( $voice ) && isset( $voice['name'] ) ? (string) $voice['name'] : 'Journalistic Expert';
    $voice_desc = is_array( $voice ) && isset( $voice['description'] ) ? (string) $voice['description'] : '';
    $voice_settings = is_array( $voice ) && isset( $voice['summary_guidelines'] ) ? (string) $voice['summary_guidelines'] : '';
    $intro_examples = is_array( $voice ) && isset( $voice['intro_examples'] ) && is_array( $voice['intro_examples'] ) ? implode( "\n", array_map( 'sanitize_text_field', $voice['intro_examples'] ) ) : '';
    $closing_examples = is_array( $voice ) && isset( $voice['closing_examples'] ) && is_array( $voice['closing_examples'] ) ? implode( "\n", array_map( 'sanitize_text_field', $voice['closing_examples'] ) ) : '';

    // Build prompt context
    $user_id = get_current_user_id();
    $user    = $user_id > 0 ? get_userdata( $user_id ) : false;
    $author_name = ( $user && isset( $user->display_name ) && is_string( $user->display_name ) ) ? (string) $user->display_name : 'Editor';
    $ctx = [
        'original_title' => '',
        'original_guid'  => $link,
        'source_name'    => $source_name,
        'homepage'       => $homepage,
        'article_text'   => wp_strip_all_tags( $article_text ),
        'voice_name'     => $voice_name,
        'voice_description' => $voice_desc,
        'voice_settings_description' => $voice_settings,
        'author_name'    => $author_name,
        'intro_examples' => $intro_examples,
        'closing_examples' => $closing_examples,
    ];

    $min_words = 400;
    $prompt = function_exists( 'contentweaver_ai_lite_build_cwai_prompt' ) ? contentweaver_ai_lite_build_cwai_prompt( $ctx, $min_words ) : ( 'Rewrite this article to at least 400 words:\n\n' . wp_strip_all_tags( $article_text ) );

    // Call provider
    $endpoint = untrailingslashit( $base_url ) . '/chat/completions';
    $body     = [ 'model' => $model, 'messages' => [ [ 'role' => 'user', 'content' => $prompt ] ], 'temperature' => 0.5 ];
    $args     = [ 'headers' => [ 'Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $api_key ], 'timeout' => 60, 'body' => wp_json_encode( $body ) ];
    $response = wp_remote_post( $endpoint, $args );
    if ( is_wp_error( $response ) ) {
        wp_send_json_error( [ 'message' => $response->get_error_message() ] );
    }
    $status  = (int) wp_remote_retrieve_response_code( $response );
    $raw_out = 200 === $status ? (string) wp_remote_retrieve_body( $response ) : '';
    if ( '' !== $raw_out ) {
        $decoded = json_decode( $raw_out, true );
        if ( is_array( $decoded ) && isset( $decoded['choices'][0]['message']['content'] ) && is_string( $decoded['choices'][0]['message']['content'] ) ) {
            $raw_out = (string) $decoded['choices'][0]['message']['content'];
        } else {
            $raw_out = '';
        }
    }

    $new_title   = '';
    $summary_html = '';
    $sources_txt = '';
    $tags_list   = '';
    $meta_desc   = '';
    $wrap_txt    = '';
    $holder_txt  = '';
    $image_query = '';
    if ( '' !== $raw_out && function_exists( 'contentweaver_ai_lite_parse_cwai_sections' ) ) {
        $sections = contentweaver_ai_lite_parse_cwai_sections( $raw_out );
        if ( isset( $sections['New Title'] ) ) { $new_title = wp_strip_all_tags( (string) $sections['New Title'] ); }
        if ( isset( $sections['Summary'] ) ) { $summary_html = wp_kses_post( (string) $sections['Summary'] ); }
        if ( isset( $sections['Sources'] ) ) { $sources_txt = sanitize_text_field( (string) $sections['Sources'] ); }
        if ( isset( $sections['Tags'] ) ) { $tags_list = (string) $sections['Tags']; }
        if ( isset( $sections['Meta Description'] ) ) { $meta_desc = sanitize_text_field( (string) $sections['Meta Description'] ); }
        if ( isset( $sections['TheWrap'] ) ) { $wrap_txt = sanitize_text_field( (string) $sections['TheWrap'] ); }
        if ( isset( $sections['Holder'] ) ) { $holder_txt = sanitize_text_field( (string) $sections['Holder'] ); }
        if ( isset( $sections['ImageQuery'] ) ) { $image_query = sanitize_text_field( (string) $sections['ImageQuery'] ); }
    }

    if ( '' === $summary_html ) {
        $summary_html = '<p>' . esc_html( wp_strip_all_tags( $article_text ) ) . '</p>';
    }
    if ( '' === $new_title ) {
        $new_title = ( '' !== $link && '' !== $source_name ) ? ( 'Imported: ' . (string) $source_name ) : __( 'Manual Generated Article', 'contentweaver-ai-lite' );
    }

    $postarr = [ 'post_title' => $new_title, 'post_content' => $summary_html, 'post_status' => 'draft', 'post_type' => 'post' ];
    $post_id = wp_insert_post( $postarr );
    if ( ! is_int( $post_id ) || $post_id <= 0 ) {
        wp_send_json_error( [ 'message' => __( 'Failed to create post.', 'contentweaver-ai-lite' ) ] );
    }

    if ( '' !== $link ) { add_post_meta( $post_id, '_cw_lite_source', esc_url_raw( $link ), true ); }
    if ( '' !== $sources_txt ) { update_post_meta( $post_id, '_cw_lite_sources', $sources_txt ); }
    if ( '' !== $wrap_txt ) { update_post_meta( $post_id, '_cw_lite_wrap', $wrap_txt ); }
    if ( '' !== $holder_txt ) { update_post_meta( $post_id, '_cw_lite_holder', $holder_txt ); }
    if ( '' !== $meta_desc ) { update_post_meta( $post_id, '_cw_lite_meta_description', $meta_desc ); }

    // Featured Image Mode
    $fim = '' !== $fim_in ? $fim_in : ( is_array( $options ) && isset( $options['featured_image_mode'] ) ? (string) $options['featured_image_mode'] : 'none' );
    $thumb_set = false;
    $cc0_candidates = [];
    $cc0_used_candidate = '';
    $selected_image_desc = '';
    if ( 'default' === $fim ) {
        $aid = $def_img_in > 0 ? $def_img_in : ( isset( $options['default_image_id'] ) ? (int) $options['default_image_id'] : 0 );
        if ( $aid > 0 && wp_attachment_is_image( $aid ) ) {
            set_post_thumbnail( $post_id, $aid );
            $thumb_set = true;
            $selected_image_desc = 'default_image:' . (string) $aid;
        } else {
            $site_icon_id = (int) get_option( 'site_icon' );
            if ( $site_icon_id > 0 && wp_attachment_is_image( $site_icon_id ) ) {
                set_post_thumbnail( $post_id, $site_icon_id );
                $thumb_set = true;
                $selected_image_desc = 'site_icon:' . (string) $site_icon_id;
            }
        }
    } elseif ( 'site_icon' === $fim ) {
        $site_icon_id = (int) get_option( 'site_icon' );
        if ( $site_icon_id > 0 && wp_attachment_is_image( $site_icon_id ) ) {
            set_post_thumbnail( $post_id, $site_icon_id );
            $thumb_set = true;
            $selected_image_desc = 'site_icon:' . (string) $site_icon_id;
        }
    } elseif ( 'cc0' === $fim ) {
        // Prefer ImageQuery terms, then title
        $candidates = [];
        if ( '' !== $image_query ) {
            $img_terms_raw = array_slice( array_map( static function( $s ): string { return trim( (string) $s ); }, explode( ',', (string) $image_query ) ), 0, 3 );
            foreach ( $img_terms_raw as $term ) { if ( '' !== $term ) { $candidates[] = preg_replace( '/\s+/', ' ', $term ); } }
            $combined = preg_replace( '/\s*,\s*/', ' ', (string) $image_query );
            $combined = is_string( $combined ) ? preg_replace( '/\s+/', ' ', trim( (string) $combined ) ) : '';
            if ( '' !== $combined ) { $candidates[] = $combined; }
        }
        $title_for_search = get_the_title( $post_id );
        if ( '' !== $title_for_search ) { $candidates[] = $title_for_search; }
        // Expose candidates for response
        $cc0_candidates = $candidates;

        $results_cc0 = [];
        foreach ( $candidates as $cand ) {
            $results_cc0 = function_exists( 'contentweaver_ai_lite_openverse_search_cc0' ) ? contentweaver_ai_lite_openverse_search_cc0( (string) $cand, 5 ) : [];
            if ( ! empty( $results_cc0 ) ) { $cc0_used_candidate = (string) $cand; break; }
        }
        if ( empty( $results_cc0 ) && '' !== $title_for_search ) {
            $results_cc0 = function_exists( 'contentweaver_ai_lite_openverse_search_cc0' ) ? contentweaver_ai_lite_openverse_search_cc0( (string) $title_for_search, 5 ) : [];
            if ( ! empty( $results_cc0 ) && '' === $cc0_used_candidate ) { $cc0_used_candidate = (string) $title_for_search; }
        }
        if ( ! empty( $results_cc0 ) ) {
            $tried = 0;
            foreach ( $results_cc0 as $res_cc0 ) {
                $tried++;
                $att_id = function_exists( 'contentweaver_ai_lite_download_image_to_media' ) ? contentweaver_ai_lite_download_image_to_media( isset( $res_cc0['url'] ) ? (string) $res_cc0['url'] : '', $post_id ) : 0;
                if ( $att_id > 0 && wp_attachment_is_image( $att_id ) ) {
                    set_post_thumbnail( $post_id, $att_id );
                    if ( is_array( $res_cc0 ) ) {
                        if ( isset( $res_cc0['landing'] ) ) { update_post_meta( $post_id, '_cw_image_source_url', sanitize_text_field( (string) $res_cc0['landing'] ) ); }
                        if ( isset( $res_cc0['creator'] ) ) { update_post_meta( $post_id, '_cw_image_creator', sanitize_text_field( (string) $res_cc0['creator'] ) ); }
                        if ( isset( $res_cc0['title'] ) ) { update_post_meta( $post_id, '_cw_image_title', sanitize_text_field( (string) $res_cc0['title'] ) ); }
                    }
                    update_post_meta( $post_id, '_cw_image_source', 'openverse' );
                    update_post_meta( $post_id, '_cw_image_license', 'CC0' );
                    $thumb_set = true;
                    $selected_image_desc = 'cc0:' . ( isset( $res_cc0['url'] ) ? (string) $res_cc0['url'] : '' );
                    break;
                }
                if ( $tried >= 3 ) { break; }
            }
        }
        if ( ! $thumb_set ) {
            $aid = $def_img_in > 0 ? $def_img_in : ( isset( $options['default_image_id'] ) ? (int) $options['default_image_id'] : 0 );
            if ( $aid > 0 && wp_attachment_is_image( $aid ) ) {
                set_post_thumbnail( $post_id, $aid );
                $thumb_set = true;
                $selected_image_desc = 'default_image:' . (string) $aid;
            } else {
                $site_icon_id = (int) get_option( 'site_icon' );
                if ( $site_icon_id > 0 && wp_attachment_is_image( $site_icon_id ) ) {
                    set_post_thumbnail( $post_id, $site_icon_id );
                    $thumb_set = true;
                    $selected_image_desc = 'site_icon:' . (string) $site_icon_id;
                }
            }
        }
    }

    /* translators: 1: post ID, 2: post title */
    $msg = sprintf( __( 'Created draft post #%1$d: %2$s', 'contentweaver-ai-lite' ), (int) $post_id, esc_html( get_the_title( $post_id ) ) );
    // Collect Openverse last-request debug if available
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals
    $ov_debug = ( 'cc0' === $fim && isset( $GLOBALS['cw_lite_last_cc0_debug'] ) && is_array( $GLOBALS['cw_lite_last_cc0_debug'] ) ) ? $GLOBALS['cw_lite_last_cc0_debug'] : [];

    $result = [
        'message'        => $msg,
        'post_id'        => (int) $post_id,
        'title'          => (string) get_the_title( $post_id ),
        'content'        => (string) $summary_html,
        'voice'          => (string) $voice_name,
        'debug'          => [
            'featured_image_mode' => (string) $fim,
            'thumbnail_set'       => (bool) $thumb_set,
            'selected_image'      => (string) $selected_image_desc,
            'image_query'         => (string) $image_query,
            'cc0'                 => [
                'candidates'     => array_values( array_map( 'strval', $cc0_candidates ) ),
                'used_candidate' => (string) $cc0_used_candidate,
                'openverse'      => $ov_debug,
            ],
        ],
    ];
    wp_send_json_success( $result );
}
add_action( 'wp_ajax_cw_lite_manual_generate_post', 'contentweaver_ai_lite_ajax_manual_generate_post' );
