<?php
/**
 * Admin pages for ContentWeaver AI Lite
 *
 * @package ContentWeaverAILite
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Safely fetch options with defaults.
 *
 * @return array<string,string>
 */
function contentweaver_ai_lite_get_options(): array {
	$defaults = [
		'provider'      => 'openai',
		'api_key'       => '',
		'base_url'      => 'https://api.openai.com/v1',
		'model'         => 'gpt-4o-mini',
		'rss_feed_url'  => '',
		'voice_id'      => 10,
        'items_per_run' => 3,
		'featured_image_mode' => 'none',
		'default_image_id'    => 0,
	];

	$opts = get_option( CONTENTWEAVER_AI_LITE_OPTION );
	if ( ! is_array( $opts ) ) {
		return $defaults;
	}
	// Preserve scalar types from saved options; merge with defaults for stability.
	return array_merge( $defaults, $opts );
}

/**
 * Helper: render admin notice from querystring.
 */
function contentweaver_ai_lite_render_query_notice(): void {
	if ( isset( $_GET['cw_lite_notice'] ) && is_string( $_GET['cw_lite_notice'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$message = sanitize_text_field( wp_unslash( $_GET['cw_lite_notice'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		echo '<div class="notice notice-info is-dismissible"><p>' . esc_html( $message ) . '</p></div>';
	}
}

/**
 * Dashboard page (provider + API key + model).
 */
function contentweaver_ai_lite_render_dashboard_page(): void {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	$options  = contentweaver_ai_lite_get_options();
	$provider = $options['provider'];
	$base_url = 'openai' === $provider ? 'https://api.openai.com/v1' : 'https://api.deepseek.com/v1';
	?>
	<div class="wrap">
		<h1><?php echo esc_html__( 'ContentWeaver AI Lite', 'contentweaver-ai-lite' ); ?></h1>
		<?php contentweaver_ai_lite_render_query_notice(); ?>

		<form action="<?php echo esc_url( admin_url( 'options.php' ) ); ?>" method="post">
			<?php settings_fields( 'contentweaver_ai_lite' ); ?>
			<?php $opts = contentweaver_ai_lite_get_options(); ?>
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row"><label for="cw_lite_provider"><?php echo esc_html__( 'Provider', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<select id="cw_lite_provider" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[provider]">
								<option value="openai" <?php selected( $opts['provider'], 'openai' ); ?>><?php echo esc_html__( 'OpenAI', 'contentweaver-ai-lite' ); ?></option>
								<option value="deepseek" <?php selected( $opts['provider'], 'deepseek' ); ?>><?php echo esc_html__( 'DeepSeek', 'contentweaver-ai-lite' ); ?></option>
							</select>
							<p class="description"><?php echo esc_html__( 'Only OpenAI-compatible providers are supported in Lite.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="cw_lite_api_key"><?php echo esc_html__( 'API Key', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<input type="password" id="cw_lite_api_key" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[api_key]" value="<?php echo esc_attr( $opts['api_key'] ); ?>" class="regular-text" autocomplete="off" />
							<p class="description"><?php echo esc_html__( 'Stored locally; never sent to ContentWeaver servers.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="cw_lite_model"><?php echo esc_html__( 'Model', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<input type="text" id="cw_lite_model" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[model]" value="<?php echo esc_attr( $opts['model'] ); ?>" class="regular-text" />
							<p class="description"><?php echo esc_html__( 'Example (OpenAI): gpt-4o-mini. Example (DeepSeek): deepseek-chat.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Base URL', 'contentweaver-ai-lite' ); ?></th>
						<td>
							<input type="text" value="<?php echo esc_attr( $base_url ); ?>" class="regular-text" readonly />
							<p class="description"><?php echo esc_html__( 'Determined by provider selection.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
				</tbody>
			</table>
			<?php submit_button( __( 'Save Settings', 'contentweaver-ai-lite' ) ); ?>
		</form>
	</div>
	<?php
}

/**
 * RSS page: single feed + manual run button.
 */
function contentweaver_ai_lite_render_rss_page(): void {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	$options = contentweaver_ai_lite_get_options();
	?>
	<div class="wrap">
		<h1><?php echo esc_html__( 'RSS Lite', 'contentweaver-ai-lite' ); ?></h1>
		<?php contentweaver_ai_lite_render_query_notice(); ?>

		<form action="<?php echo esc_url( admin_url( 'options.php' ) ); ?>" method="post" style="margin-bottom:20px;">
			<?php settings_fields( 'contentweaver_ai_lite' ); ?>
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row"><label for="cw_lite_rss_url"><?php echo esc_html__( 'Feed URL', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<input type="url" id="cw_lite_rss_url" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[rss_feed_url]" value="<?php echo esc_attr( $options['rss_feed_url'] ); ?>" class="regular-text code" placeholder="https://example.com/feed" />
							<p class="description"><?php echo esc_html__( 'Lite supports a single RSS feed. Use the button below to import up to 3 items as drafts.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="cw_lite_voice_id"><?php echo esc_html__( 'Voice', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<?php $voices = function_exists( 'contentweaver_ai_lite_get_voices' ) ? contentweaver_ai_lite_get_voices() : []; $vid = isset( $options['voice_id'] ) ? (int) $options['voice_id'] : 10; ?>
							<select id="cw_lite_voice_id" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[voice_id]">
								<?php foreach ( $voices as $id => $voice ) : if ( ! is_array( $voice ) || ! isset( $voice['name'] ) ) { continue; } ?>
									<option value="<?php echo esc_attr( (string) $id ); ?>" <?php selected( $vid, (int) $id ); ?>><?php echo esc_html( (string) $voice['name'] ); ?></option>
								<?php endforeach; ?>
							</select>
							<p class="description"><?php echo esc_html__( 'Choose the voice to apply when rewriting full articles (default: Journalistic Expert).', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="cw_lite_items_per_run"><?php echo esc_html__( 'Items Per Run', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<?php $ipr = isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3; if ( $ipr < 1 ) { $ipr = 1; } if ( $ipr > 3 ) { $ipr = 3; } ?>
							<input type="number" id="cw_lite_items_per_run" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[items_per_run]" value="<?php echo esc_attr( (string) $ipr ); ?>" min="1" max="3" step="1" />
							<p class="description"><?php echo esc_html__( 'How many items to import per manual run (1–3).', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"></th>
						<td>
							<p class="description"><strong><?php echo esc_html__( 'Original publisher images are not used.', 'contentweaver-ai-lite' ); ?></strong> <?php echo esc_html__( 'Choose a default image, use your Site Icon, select Public Domain (Openverse), or select none.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Featured Image Mode', 'contentweaver-ai-lite' ); ?></th>
						<td>
							<?php $fim = isset( $options['featured_image_mode'] ) ? (string) $options['featured_image_mode'] : 'none'; $def_id = isset( $options['default_image_id'] ) ? (int) $options['default_image_id'] : 0; ?>
							<fieldset>
								<label><input type="radio" id="cw-lite-fimg-none" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[featured_image_mode]" value="none" <?php checked( $fim, 'none' ); ?>/> <?php echo esc_html__( 'None', 'contentweaver-ai-lite' ); ?></label><br />
								<label><input type="radio" id="cw-lite-fimg-default" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[featured_image_mode]" value="default" <?php checked( $fim, 'default' ); ?>/> <?php echo esc_html__( 'Default Image (single image used for all imports)', 'contentweaver-ai-lite' ); ?></label><br />
								<div id="cw-lite-default-image-controls" style="margin:8px 0 0 22px; <?php echo ( 'default' === $fim ) ? '' : 'display:none;'; ?>">
									<input type="hidden" id="cw_lite_default_image_id" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[default_image_id]" value="<?php echo esc_attr( (string) $def_id ); ?>" />
									<button type="button" class="button" id="cw-lite-select-default-image"><?php echo esc_html__( 'Select Image', 'contentweaver-ai-lite' ); ?></button>
									<button type="button" class="button link-delete" id="cw-lite-clear-default-image" <?php echo $def_id > 0 ? '' : 'disabled'; ?>><?php echo esc_html__( 'Clear', 'contentweaver-ai-lite' ); ?></button>
									<div id="cw-lite-default-image-preview" style="margin-top:10px;">
										<?php if ( $def_id > 0 ) { echo wp_kses_post( wp_get_attachment_image( $def_id, 'thumbnail' ) ); } ?>
									</div>
									<p class="description"><?php echo esc_html__( 'If no image is selected, we will fall back to the Site Icon (if available).', 'contentweaver-ai-lite' ); ?></p>
								</div>
								<label><input type="radio" id="cw-lite-fimg-siteicon" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[featured_image_mode]" value="site_icon" <?php checked( $fim, 'site_icon' ); ?>/> <?php echo esc_html__( 'Use Site Icon', 'contentweaver-ai-lite' ); ?></label><br />
								<label><input type="radio" id="cw-lite-fimg-cc0" name="<?php echo esc_attr( CONTENTWEAVER_AI_LITE_OPTION ); ?>[featured_image_mode]" value="cc0" <?php checked( $fim, 'cc0' ); ?>/> <?php echo esc_html__( 'Public Domain images (Openverse)', 'contentweaver-ai-lite' ); ?></label><br />
								<div id="cw-lite-cc0-disclaimer" class="description" style="margin:6px 0 0 22px; <?php echo ( 'cc0' === $fim ) ? '' : 'display:none;'; ?>">
								<p><?php echo wp_kses_post( __( 'When selected, this mode uses the Openverse API (api.openverse.engineering) to search for public domain images using your post title as the query. No API key is required, and no data is sent to ContentWeaver servers; requests go directly from your WordPress site to Openverse.', 'contentweaver-ai-lite' ) ); ?></p>
								<p style="margin-top:8px;">&nbsp;</p>
								<p><?php echo wp_kses_post( __( 'Public Domain (Creative Commons Zero, CC0) means the creator has waived their rights to the work. You may copy, modify, and use the image, including for commercial purposes, without permission or attribution. Always review the image for appropriateness for your site.', 'contentweaver-ai-lite' ) ); ?></p>
								<p style="margin-top:8px;"><?php echo wp_kses_post( __( 'If no suitable public domain image is found or the download fails, the importer falls back in this order: Default Image (if selected) → Site Icon → No image.', 'contentweaver-ai-lite' ) ); ?></p>
							</div>
						</fieldset>
					</td>
				</tr>
				</tbody>
			</table>
			<?php submit_button( __( 'Save Feed URL', 'contentweaver-ai-lite' ) ); ?>
		</form>

		<form id="cw-lite-run-form" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" method="post">
			<?php wp_nonce_field( 'contentweaver_ai_lite_rss_run' ); ?>
			<input type="hidden" name="action" value="contentweaver_ai_lite_rss_run" />
			<?php $btn_n = isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3; if ( $btn_n < 1 ) { $btn_n = 1; } if ( $btn_n > 3 ) { $btn_n = 3; } ?>
			<?php submit_button( sprintf( /* translators: %d: number of items */ __( 'Run Now (Import up to %d)', 'contentweaver-ai-lite' ), (int) $btn_n ), 'secondary' ); ?>
		</form>

		<!-- AJAX overlay and results container (match Pro styles) -->
		<div class="contentweaver-loading-overlay" style="display: none;" aria-hidden="true">
			<div class="contentweaver-spinner" aria-label="spinner"></div>
			<div class="contentweaver-loading-message"></div>
		</div>
		<div id="cw-lite-results" class="cw-lite-results" aria-live="polite" aria-atomic="true"></div>
	</div>
	<?php
}

/**
 * Spintax page: local processing, no network.
 */
function contentweaver_ai_lite_render_spintax_page(): void {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	?>
	<div class="wrap">
		<h2><?php echo esc_html__( 'Spintax Generator', 'contentweaver-ai-lite' ); ?></h2>

		<!-- Loading overlay -->
		<div class="contentweaver-loading-overlay" style="display: none;">
			<div class="contentweaver-loading-spinner"></div>
			<div class="contentweaver-loading-message">Processing...</div>
		</div>

		<div class="spintax-help">
			<h3><?php echo esc_html__( 'How to Use Spintax', 'contentweaver-ai-lite' ); ?></h3>
			<p><?php echo esc_html__( 'Create variations of your content using spintax syntax:', 'contentweaver-ai-lite' ); ?></p>
			<ul>
				<li><code>{word1|word2|word3}</code> - <?php echo esc_html__( 'Basic spintax with manual alternatives', 'contentweaver-ai-lite' ); ?></li>
				<li><code>{word|*AI*}</code> - <?php echo esc_html__( 'AI will suggest alternatives for this word', 'contentweaver-ai-lite' ); ?></li>
			</ul>
			<p class="description"><strong><?php echo esc_html__( 'Highlight any word', 'contentweaver-ai-lite' ); ?></strong> <?php echo esc_html__( 'to generate AI alternatives', 'contentweaver-ai-lite' ); ?></p>
		</div>

		<form method="post" action="" id="spintax-generation-form">
			<?php wp_nonce_field( 'spintax_generation', 'spintax_nonce' ); ?>

			<table class="form-table">
				<tr>
					<th scope="row"><label for="spintax_input"><?php echo esc_html__( 'Content', 'contentweaver-ai-lite' ); ?></label></th>
					<td>
						<textarea name="spintax_input" id="spintax_input" rows="10" class="large-text code"></textarea>
						<p class="description"><?php echo esc_html__( 'Enter your content with spintax markup. Use {word1|word2|*AI*} to get AI suggestions.', 'contentweaver-ai-lite' ); ?></p>
						<div class="button-group">
							<button type="button" class="button" id="generate-ai"><?php echo esc_html__( 'Generate AI Spintax', 'contentweaver-ai-lite' ); ?></button>
						</div>
					</td>
				</tr>

				<tr id="ai-suggestions-row" style="display: none;">
					<th scope="row"><?php echo esc_html__( 'AI Suggestions', 'contentweaver-ai-lite' ); ?></th>
					<td>
						<div id="ai-suggestions" class="ai-suggestions-box"></div>
						<div class="button-group">
							<button type="button" class="button button-primary" id="apply-suggestions"><?php echo esc_html__( 'Apply Selected', 'contentweaver-ai-lite' ); ?></button>
							<button type="button" class="button" id="cancel-suggestions"><?php echo esc_html__( 'Cancel', 'contentweaver-ai-lite' ); ?></button>
						</div>
					</td>
				</tr>

				<tr>
					<th scope="row"><label for="max_variations"><?php echo esc_html__( 'Number of Suggestions', 'contentweaver-ai-lite' ); ?></label></th>
					<td>
						<div class="slider-container">
							<div id="variations-slider"></div>
							<div id="variations-value">5</div>
						</div>
						<input type="hidden" id="max_variations" value="5">
						<p class="description"><?php echo esc_html__( 'How many AI-generated suggestions to request for each highlighted word.', 'contentweaver-ai-lite' ); ?></p>
					</td>
				</tr>

				<tr>
					<th scope="row"><label for="num_spins"><?php echo esc_html__( 'Number of Spins', 'contentweaver-ai-lite' ); ?></label></th>
					<td>
						<div class="slider-container">
							<div id="spins-slider"></div>
							<div id="spins-value">5</div>
						</div>
						<input type="hidden" id="num_spins" value="5">
						<p class="description"><?php echo esc_html__( 'How many unique paragraphs to generate locally using the spintax (does not affect AI requests).', 'contentweaver-ai-lite' ); ?></p>
					</td>
				</tr>

				<tr>
					<th scope="row"><?php echo esc_html__( 'Preview', 'contentweaver-ai-lite' ); ?></th>
					<td>
						<textarea id="preview" rows="8" style="width:100%; resize:vertical; min-height:120px; max-height:400px; overflow:auto; background:#fff; color:#222; border:1px solid #ccc;" readonly></textarea>
						<div style="display:flex; gap:10px; margin-top:8px;">
							<button type="button" id="process-spintax" class="button button-primary" style="order:1;">Process Spintax</button>
							<button type="button" id="copy-spintax" class="button" style="order:2;">Copy All</button>
						</div>
					</td>
				</tr>
			</table>
		</form>
	</div>

	<?php
}
 /**
  * Manual Generation (text) page.
  */
function contentweaver_ai_lite_render_manual_page(): void {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    $options = contentweaver_ai_lite_get_options();
	?>
	<div class="wrap">
		<h1><?php echo esc_html__( 'Manual Content Generation', 'contentweaver-ai-lite' ); ?></h1>

		<!-- Loading overlay -->
		<div class="contentweaver-loading-overlay" style="display: none;">
			<div class="contentweaver-loading-spinner"></div>
			<div class="contentweaver-loading-message">Processing...</div>
		</div>

		

		        <form id="manual-generation-form" method="post">
            <?php wp_nonce_field( 'contentweaver_generate_article', 'nonce' ); ?>

			<div class="generation-type-toggle" style="margin-bottom: 20px;">
				<label class="switch">
					<input type="radio" name="generation_type" value="article" checked> <?php echo esc_html__( 'Article Generation', 'contentweaver-ai-lite' ); ?>
				</label>
				<label class="switch" style="margin-left: 20px;">
					<input type="radio" name="generation_type" value="image" disabled> <?php echo esc_html__( 'Image Generation (Pro only)', 'contentweaver-ai-lite' ); ?>
				</label>
				<p class="description" style="margin-top: 10px;">
					<?php echo esc_html__( 'Lite generates text with your OpenAI-compatible API key. Images and post creation are available in Pro.', 'contentweaver-ai-lite' ); ?>
				</p>
			</div>

			<div id="article-fields">
				<table class="form-table">
					<tr>
						<th scope="row"><label for="article_url"><?php echo esc_html__( 'Article URL', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<input type="url" name="article_url" id="article_url" class="regular-text">
							<p class="description"><?php echo esc_html__( 'Provide a URL OR paste the article text below. If both are provided, pasted text takes priority.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="article_text"><?php echo esc_html__( 'Paste Article Text', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<textarea name="article_text" id="article_text" rows="10" class="large-text"></textarea>
							<p class="description"><?php echo esc_html__( 'Paste the text to rewrite and enhance. URL fetch is not available in Lite.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="voice"><?php echo esc_html__( 'Voice', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<?php $voices = function_exists( 'contentweaver_ai_lite_get_voices' ) ? contentweaver_ai_lite_get_voices() : []; $vid = isset( $options['voice_id'] ) ? (int) $options['voice_id'] : 10; ?>
							<select name="voice_id" id="voice" class="regular-text">
								<?php foreach ( $voices as $id => $voice ) : if ( ! is_array( $voice ) || ! isset( $voice['name'] ) ) { continue; } ?>
									<option value="<?php echo esc_attr( (string) $id ); ?>" <?php selected( $vid, (int) $id ); ?>><?php echo esc_html( (string) $voice['name'] ); ?></option>
								<?php endforeach; ?>
							</select>
							<p class="description"><?php echo esc_html__( 'Same voices as RSS. Used to guide tone and structure.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="post_status"><?php echo esc_html__( 'Post Status', 'contentweaver-ai-lite' ); ?></label></th>
						<td>
							<select name="post_status" id="post_status" class="regular-text" disabled>
								<option value="draft"><?php echo esc_html__( 'Draft', 'contentweaver-ai-lite' ); ?></option>
								<option value="publish"><?php echo esc_html__( 'Publish Immediately', 'contentweaver-ai-lite' ); ?></option>
							</select>
							<p class="description"><?php echo esc_html__( 'Publishing from Manual is available in Pro.', 'contentweaver-ai-lite' ); ?></p>
						</td>
					</tr>
				</table>
			</div>

			<h2 style="margin-top:16px;"><?php echo esc_html__( 'Featured Image Mode', 'contentweaver-ai-lite' ); ?></h2>
			<?php $fim = isset( $options['featured_image_mode'] ) ? (string) $options['featured_image_mode'] : 'none'; $def_id = isset( $options['default_image_id'] ) ? (int) $options['default_image_id'] : 0; ?>
			<fieldset>
				<label><input type="radio" id="cw-lite-fimg-none" name="featured_image_mode" value="none" <?php checked( $fim, 'none' ); ?>/> <?php echo esc_html__( 'None', 'contentweaver-ai-lite' ); ?></label><br />
				<label><input type="radio" id="cw-lite-fimg-default" name="featured_image_mode" value="default" <?php checked( $fim, 'default' ); ?>/> <?php echo esc_html__( 'Default Image (single image used for all imports)', 'contentweaver-ai-lite' ); ?></label><br />
				<div id="cw-lite-default-image-controls" style="margin:8px 0 0 22px; <?php echo ( 'default' === $fim ) ? '' : 'display:none;'; ?>">
					<input type="hidden" id="cw_lite_default_image_id" name="default_image_id" value="<?php echo esc_attr( (string) $def_id ); ?>" />
					<button type="button" class="button" id="cw-lite-select-default-image"><?php echo esc_html__( 'Select Image', 'contentweaver-ai-lite' ); ?></button>
					<button type="button" class="button link-delete" id="cw-lite-clear-default-image" <?php echo $def_id > 0 ? '' : 'disabled'; ?>><?php echo esc_html__( 'Clear', 'contentweaver-ai-lite' ); ?></button>
					<div id="cw-lite-default-image-preview" style="margin-top:10px;">
						<?php if ( $def_id > 0 ) { echo wp_kses_post( wp_get_attachment_image( $def_id, 'thumbnail' ) ); } ?>
					</div>
					<p class="description"><?php echo esc_html__( 'If no image is selected, we will fall back to the Site Icon (if available).', 'contentweaver-ai-lite' ); ?></p>
				</div>
				<label><input type="radio" id="cw-lite-fimg-siteicon" name="featured_image_mode" value="site_icon" <?php checked( $fim, 'site_icon' ); ?>/> <?php echo esc_html__( 'Use Site Icon', 'contentweaver-ai-lite' ); ?></label><br />
				<label><input type="radio" id="cw-lite-fimg-cc0" name="featured_image_mode" value="cc0" <?php checked( $fim, 'cc0' ); ?>/> <?php echo esc_html__( 'Public Domain images (Openverse)', 'contentweaver-ai-lite' ); ?></label><br />
				<div id="cw-lite-cc0-disclaimer" class="description" style="margin:6px 0 0 22px; <?php echo ( 'cc0' === $fim ) ? '' : 'display:none;'; ?>">
					<p><?php echo wp_kses_post( __( 'When selected, this mode uses the Openverse API (api.openverse.engineering) to search for public domain images using your content as the query. No API key is required, and no data is sent to ContentWeaver servers; requests go directly from your WordPress site to Openverse.', 'contentweaver-ai-lite' ) ); ?></p>
					<p style="margin-top:8px;">&nbsp;</p>
					<p><?php echo wp_kses_post( __( 'Public Domain (Creative Commons Zero, CC0) means the creator has waived their rights to the work. You may copy, modify, and use the image, including for commercial purposes, without permission or attribution. Always review the image for appropriateness for your site.', 'contentweaver-ai-lite' ) ); ?></p>
					<p style="margin-top:8px;">&nbsp;</p>
					<p><?php echo wp_kses_post( __( 'If no suitable public domain image is found or the download fails, the importer falls back in this order: Default Image (if selected) → Site Icon → No image.', 'contentweaver-ai-lite' ) ); ?></p>
				</div>
			</fieldset>

			<p class="submit" style="margin-top:16px;">
				<input type="submit" id="generate-button" class="button button-primary" value="<?php echo esc_attr__( 'Generate Article', 'contentweaver-ai-lite' ); ?>">
			</p>

			<!-- Results/Diagnostics panel (same placement as RSS) -->
			<div id="cw-lite-results" class="cw-lite-results" aria-live="polite" aria-atomic="true"></div>
		</form>
	</div>

    <?php
}

/**
 * Upgrade page.
 */
function contentweaver_ai_lite_render_upgrade_page(): void {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    ?>
    <div class="wrap cw-upgrade-wrap">

        <div class="cw-up-hero" role="region" aria-label="<?php echo esc_attr__( 'Upgrade to Pro', 'contentweaver-ai-lite' ); ?>">
            <div class="cw-up-hero-inner">
                <div>
                    <h1><?php echo esc_html__( 'Upgrade to ContentWeaver AI Pro', 'contentweaver-ai-lite' ); ?></h1>
                    <p><?php echo esc_html__( 'Automate your pipeline with multi-feed scheduling, server-side generation, image creation, analytics, and more—powered by ContentWeaver-managed providers, models, and keys with generous token limits so you don\'t burn your own.', 'contentweaver-ai-lite' ); ?></p>
                    <div class="cw-up-badges" aria-hidden="true">
                        <span class="cw-badge"><span class="dashicons dashicons-schedule"></span><?php echo esc_html__( 'Scheduling', 'contentweaver-ai-lite' ); ?></span>
                        <span class="cw-badge"><span class="dashicons dashicons-rss"></span><?php echo esc_html__( 'Multiple Feeds', 'contentweaver-ai-lite' ); ?></span>
                        <span class="cw-badge"><span class="dashicons dashicons-performance"></span><?php echo esc_html__( 'Server‑Side Speed', 'contentweaver-ai-lite' ); ?></span>
                        <span class="cw-badge"><span class="dashicons dashicons-format-image"></span><?php echo esc_html__( 'AI Images', 'contentweaver-ai-lite' ); ?></span>
                        <span class="cw-badge"><span class="dashicons dashicons-lock"></span><?php echo esc_html__( 'Managed Keys', 'contentweaver-ai-lite' ); ?></span>
                    </div>
                    <div class="cw-up-ctas">
                        <?php
                        $upgrade_url = 'https://contentweaverai.com/getting-started/';
                        $compare_id  = 'cw-compare';
                        ?>
                        <a class="button button-primary cw-btn cw-btn-primary" href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Upgrade to Pro', 'contentweaver-ai-lite' ); ?></a>
                        <a class="button cw-btn cw-btn-outline" href="#<?php echo esc_attr( $compare_id ); ?>"><?php echo esc_html__( 'Compare Lite vs Pro', 'contentweaver-ai-lite' ); ?></a>
                    </div>
                    <div class="cw-affiliate">
                        <span class="dashicons dashicons-megaphone" aria-hidden="true"></span>
                        <span><?php echo esc_html__( 'Affiliate-ready: add your own affiliate links and callouts directly into generated posts.', 'contentweaver-ai-lite' ); ?></span>
                    </div>
                </div>
                <div>
                    <div class="cw-card" style="background:linear-gradient(135deg,#fdf2f8 0%, #eff6ff 100%); border-color:#e5e7eb;">
                        <h3><span class="dashicons dashicons-star-filled" aria-hidden="true"></span><?php echo esc_html__( 'Built for teams that ship daily', 'contentweaver-ai-lite' ); ?></h3>
                        <p><?php echo esc_html__( 'Queue content, keep posts moving, and free your editors to choose winners, not wrangle feeds.', 'contentweaver-ai-lite' ); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <div class="cw-up-grid" role="list" aria-label="<?php echo esc_attr__( 'Why upgrade', 'contentweaver-ai-lite' ); ?>">
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-schedule" aria-hidden="true"></span><?php echo esc_html__( 'Scheduling & Queues', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Turn feeds into a content queue—draft or publish on a schedule without manual clicks.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-rss" aria-hidden="true"></span><?php echo esc_html__( 'Multiple RSS Feeds', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Import from multiple sources with per-feed rules, prioritization, and easy overrides.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-performance" aria-hidden="true"></span><?php echo esc_html__( 'Server‑Side Generation', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Fewer timeouts and faster processing—heavy lifting moves off the browser to the server.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-format-image" aria-hidden="true"></span><?php echo esc_html__( 'Image Generation', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Use model images when public domain isn’t enough—on-brand visuals every time. Includes 7 image models across 5+ providers.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-admin-network" aria-hidden="true"></span><?php echo esc_html__( 'Models & Providers', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Access 9+ text models and 7 image models across 5+ providers using ContentWeaver-managed keys and negotiated rates—generous token limits, no need to spend your own.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-chart-bar" aria-hidden="true"></span><?php echo esc_html__( 'Analytics & Logs', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'See imports, rewrites, image outcomes, and failures—optimize your pipeline with data.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-megaphone" aria-hidden="true"></span><?php echo esc_html__( 'Affiliate Callouts Built‑In', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Add Amazon, ClickBank, and digital product affiliate callouts directly into generated posts—written in your chosen voice, no extra plugins required.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-admin-customizer" aria-hidden="true"></span><?php echo esc_html__( 'Templates, Voices, SEO Controls', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Enforce style and structure with reusable templates, voice defaults, and SEO tuning.', 'contentweaver-ai-lite' ); ?></p>
            </div>
            <div class="cw-card" role="listitem">
                <h3><span class="dashicons dashicons-sos" aria-hidden="true"></span><?php echo esc_html__( 'Priority Support', 'contentweaver-ai-lite' ); ?></h3>
                <p><?php echo esc_html__( 'Get priority responses and faster fixes so your workflow never stalls.', 'contentweaver-ai-lite' ); ?></p>
            </div>
        </div>

        <div class="cw-up-strip" id="<?php echo esc_attr( $compare_id ); ?>">
            <strong><?php echo esc_html__( 'What you gain with Pro', 'contentweaver-ai-lite' ); ?></strong>
            <div>
                <a class="button button-primary cw-btn cw-btn-primary" href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Upgrade Now', 'contentweaver-ai-lite' ); ?></a>
            </div>
        </div>

        <div class="cw-up-compare">
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__( 'Feature', 'contentweaver-ai-lite' ); ?></th>
                        <th><?php echo esc_html__( 'Lite', 'contentweaver-ai-lite' ); ?></th>
                        <th class="pro"><?php echo esc_html__( 'Pro', 'contentweaver-ai-lite' ); ?> <span class="cw-pro-badge"><?php echo esc_html__( 'Most Popular', 'contentweaver-ai-lite' ); ?></span></th>
                    </tr>
                </thead>
                <tbody>
                    <tr class="cw-compare-group"><th colspan="3"><span class="cw-compare-label"><span class="dashicons dashicons-rss"></span><?php echo esc_html__( 'Content Power', 'contentweaver-ai-lite' ); ?></span></th></tr>
                    <tr>
                        <td><?php echo esc_html__( 'RSS Feeds', 'contentweaver-ai-lite' ); ?> <span class="cw-mini-badge"><?php echo esc_html__( 'Scale with your site', 'contentweaver-ai-lite' ); ?></span></td>
                        <td class="no">1</td>
                        <td class="yes pro">5+ <?php echo esc_html__( '(with per‑feed rules)', 'contentweaver-ai-lite' ); ?></td>
                    </tr>
                    <tr>
                        <td><?php echo esc_html__( 'Run Mode', 'contentweaver-ai-lite' ); ?></td>
                        <td class="no"><?php echo esc_html__( 'Manual only', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'Manual + Scheduled', 'contentweaver-ai-lite' ); ?></td>
                    </tr>
                    <tr>
                        <td><?php echo esc_html__( 'Post Status', 'contentweaver-ai-lite' ); ?></td>
                        <td class="no"><?php echo esc_html__( 'Draft', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'Draft / Publish', 'contentweaver-ai-lite' ); ?></td>
                    </tr>

                    <tr class="cw-compare-group"><th colspan="3"><span class="cw-compare-label"><span class="dashicons dashicons-performance"></span><?php echo esc_html__( 'Performance', 'contentweaver-ai-lite' ); ?></span></th></tr>
                    <tr>
                        <td><?php echo esc_html__( 'Processing', 'contentweaver-ai-lite' ); ?> <span class="cw-mini-badge"><?php echo esc_html__( 'No timeouts', 'contentweaver-ai-lite' ); ?></span></td>
                        <td class="no"><?php echo esc_html__( 'Browser', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'Server‑Side (faster, reliable)', 'contentweaver-ai-lite' ); ?></td>
                    </tr>
                    <tr>
                        <td><?php echo esc_html__( 'API Keys & Usage', 'contentweaver-ai-lite' ); ?></td>
                        <td class="no"><?php echo esc_html__( 'Your own provider keys, your usage.', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'ContentWeaver-managed keys with generous token limits at negotiated rates.', 'contentweaver-ai-lite' ); ?></td>
                    </tr>

                    <tr class="cw-compare-group"><th colspan="3"><span class="cw-compare-label"><span class="dashicons dashicons-art"></span><?php echo esc_html__( 'Creative Control', 'contentweaver-ai-lite' ); ?></span></th></tr>
                    <tr>
                        <td><?php echo esc_html__( 'Images', 'contentweaver-ai-lite' ); ?></td>
                        <td class="no"><?php echo esc_html__( 'Public Domain only (Openverse)', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'Public Domain + AI Images', 'contentweaver-ai-lite' ); ?></td>
                    </tr>
                    <tr>
                        <td><?php echo esc_html__( 'Templates / Voices', 'contentweaver-ai-lite' ); ?></td>
                        <td class="no">—</td>
                        <td class="yes pro"><?php echo esc_html__( 'Advanced controls', 'contentweaver-ai-lite' ); ?></td>
                    </tr>

                    <tr class="cw-compare-group"><th colspan="3"><span class="cw-compare-label"><span class="dashicons dashicons-chart-bar"></span><?php echo esc_html__( 'Support & Insights', 'contentweaver-ai-lite' ); ?></span></th></tr>
                    <tr>
                        <td><?php echo esc_html__( 'Diagnostics', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes"><?php echo esc_html__( 'Basic', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'Advanced (analytics, logs)', 'contentweaver-ai-lite' ); ?></td>
                    </tr>
                    <tr>
                        <td><?php echo esc_html__( 'Support', 'contentweaver-ai-lite' ); ?></td>
                        <td class="no"><?php echo esc_html__( 'Community', 'contentweaver-ai-lite' ); ?></td>
                        <td class="yes pro"><?php echo esc_html__( 'Priority', 'contentweaver-ai-lite' ); ?></td>
                    </tr>
                </tbody>
            </table>
            <div class="cw-cta-sticky">
                <a class="button button-primary cw-btn cw-btn-primary" href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Upgrade to Pro and unlock it all →', 'contentweaver-ai-lite' ); ?></a>
            </div>
        </div>

        <div class="cw-faq" role="region" aria-label="<?php echo esc_attr__( 'Frequently asked questions', 'contentweaver-ai-lite' ); ?>">
            <h2 style="margin-top:18px;"><?php echo esc_html__( 'FAQs', 'contentweaver-ai-lite' ); ?></h2>
            <details>
                <summary><span class="dashicons dashicons-migrate" aria-hidden="true"></span> <?php echo esc_html__( 'Will my Lite settings carry over?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'Most content settings carry over. API keys do not: Lite uses your own OpenAI‑compatible key, while Pro runs server‑side with ContentWeaver‑managed keys.', 'contentweaver-ai-lite' ); ?></p>
            </details>
            <details>
                <summary><span class="dashicons dashicons-performance" aria-hidden="true"></span> <?php echo esc_html__( 'Is server‑side processing faster?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'In practice, yes—especially for larger batches. It reduces browser timeouts and improves throughput.', 'contentweaver-ai-lite' ); ?></p>
            </details>
            <details>
                <summary><span class="dashicons dashicons-money" aria-hidden="true"></span> <?php echo esc_html__( 'Can I cancel anytime?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'Yes. Manage your plan from your account. No long‑term contracts.', 'contentweaver-ai-lite' ); ?></p>
            </details>
            <details>
                <summary><span class="dashicons dashicons-lock" aria-hidden="true"></span> <?php echo esc_html__( 'Do I need new API keys?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'No. Pro does not require your API keys; it uses ContentWeaver‑managed provider keys on the server.', 'contentweaver-ai-lite' ); ?></p>
            </details>

            <details>
                <summary><span class="dashicons dashicons-clock" aria-hidden="true"></span> <?php echo esc_html__( 'I only post once a week. Is Pro worth it?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'Yes. Scheduling and server‑side generation still save time and reduce errors. Queue posts to go live automatically and avoid manual runs, timeouts, or browser waits.', 'contentweaver-ai-lite' ); ?></p>
            </details>
            <details>
                <summary><span class="dashicons dashicons-admin-users" aria-hidden="true"></span> <?php echo esc_html__( 'Can I use Pro on client projects?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'Yes. Pro supports per‑site configurations, multiple feeds with rules, and reliable server‑side runs—ideal for agency and client workflows.', 'contentweaver-ai-lite' ); ?></p>
            </details>
            <details>
                <summary><span class="dashicons dashicons-image-rotate" aria-hidden="true"></span> <?php echo esc_html__( 'What if I change my mind?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'You can cancel anytime. Downgrading to Lite keeps your posts and settings; Pro‑only scheduling and advanced features simply turn off.', 'contentweaver-ai-lite' ); ?></p>
            </details>
            <details>
                <summary><span class="dashicons dashicons-chart-bar" aria-hidden="true"></span> <?php echo esc_html__( 'What’s the #1 reason users upgrade?', 'contentweaver-ai-lite' ); ?></summary>
                <p><?php echo esc_html__( 'Reliability and automation. Pro turns feeds into a hands‑off queue with server‑side speed, generous model access, and fewer manual touchpoints—so you publish more with less effort.', 'contentweaver-ai-lite' ); ?></p>
            </details>
        </div>
    </div>
    <?php
}
