<?php
/**
 * Plugin Name:       Content Weaver AI Lite — by Barking Llama
 * Plugin URI: https://wordpress.org/plugins/contentweaver-ai-lite/
 * Description:       Lite edition for WordPress.org: OpenAI-compatible manual text generation with your own API key, simple RSS import (1 feed, 3 items per run, manual only), and local Spintax utilities. No calls to ContentWeaver servers.
 * Version:           1.0.2
 * Author:            Barking Llama
 * Author URI:        https://contentweaverai.com/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       contentweaver-ai-lite
 * Domain Path:       /languages
 */

// phpcs:ignoreFile -- This file follows WP coding standards to the extent possible while adding type hints and comments for clarity.

if ( ! defined( 'ABSPATH' ) ) {
	// Prevent direct access.
	exit;
}

/**
 * Search Openverse for CC0 images by query.
 *
 * @param string $query Search term (typically the post title).
 * @param int    $page_size Number of results to request (1-5 recommended).
 * @return array<int,array<string,mixed>>
 */
function contentweaver_ai_lite_openverse_search_cc0( string $query, int $page_size = 3 ): array {
    // Expose last-request debug info for UI verification.
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals
    global $cw_lite_last_cc0_debug;
    $query = trim( wp_strip_all_tags( $query ) );
    if ( '' === $query ) {
        $cw_lite_last_cc0_debug = [ 'url' => '', 'status' => 0, 'error' => 'empty_query', 'attempts' => [] ];
        return [];
    }
    $page_size = max( 1, min( 5, $page_size ) );
    $attempts = [];

    $do_request = static function( string $q ) use ( $page_size ): array {
        $url = add_query_arg(
            [
                'q'           => $q, // add_query_arg handles encoding
                'license'     => 'cc0',
                'page_size'   => (string) $page_size,
                'mature'      => 'false',
                'extension'   => '',
            ],
            'https://api.openverse.engineering/v1/images/'
        );
        $args = [
            'timeout'    => 10,
            'headers'    => [ 'User-Agent' => contentweaver_ai_lite_get_feed_user_agent() ],
            'sslverify'  => true,
            'redirection'=> 3,
        ];
        $resp = wp_remote_get( $url, $args );
        return [ $url, $resp ];
    };

    // Attempt 1: full title
    [ $url1, $resp1 ] = $do_request( $query );
    if ( is_wp_error( $resp1 ) ) {
        $attempts[] = [ 'url' => $url1, 'status' => 0, 'error' => (string) $resp1->get_error_message() ];
    } else {
        $code1 = (int) wp_remote_retrieve_response_code( $resp1 );
        $body1 = (string) wp_remote_retrieve_body( $resp1 );
        $err   = '';
        $results1 = [];
        if ( $code1 < 200 || $code1 >= 300 ) {
            $err = 'http_error';
        } elseif ( '' === $body1 ) {
            $err = 'empty_body';
        } else {
            $json1 = json_decode( $body1, true );
            if ( is_array( $json1 ) && isset( $json1['results'] ) && is_array( $json1['results'] ) ) {
                $results1 = $json1['results'];
            } else {
                $err = 'bad_json';
            }
        }
        $attempts[] = [ 'url' => $url1, 'status' => $code1, 'error' => $err ];
        if ( empty( $err ) && ! empty( $results1 ) ) {
            $cw_lite_last_cc0_debug = [ 'url' => $url1, 'status' => $code1, 'error' => '', 'attempts' => $attempts ];
            $out = [];
            foreach ( $results1 as $r ) {
                if ( ! is_array( $r ) ) { continue; }
                $img_url  = isset( $r['url'] ) && is_string( $r['url'] ) ? $r['url'] : '';
                $thumb    = isset( $r['thumbnail'] ) && is_string( $r['thumbnail'] ) ? $r['thumbnail'] : '';
                $landing  = isset( $r['foreign_landing_url'] ) && is_string( $r['foreign_landing_url'] ) ? $r['foreign_landing_url'] : '';
                $creator  = isset( $r['creator'] ) && is_string( $r['creator'] ) ? $r['creator'] : '';
                $title    = isset( $r['title'] ) && is_string( $r['title'] ) ? $r['title'] : '';
                if ( '' === $img_url ) { continue; }
                $out[] = [ 'url' => $img_url, 'thumb' => $thumb, 'landing' => $landing, 'creator' => $creator, 'title' => $title ];
            }
            return $out;
        }
    }

    // Attempt 2: simplified keywords fallback
    $q2 = strtolower( $query );
    $q2 = preg_replace( '/[^a-z\s]/', ' ', $q2 ); // remove punctuation/numbers
    $parts = preg_split( '/\s+/', trim( (string) $q2 ) );
    $stop = [ 'the','a','an','and','or','but','with','without','to','of','for','our','your','on','in','at','by','from','this','that','these','those','best','top','new','how','are','is','be','as','it','its','their','reader','readers','people','make','made' ];
    $keywords = [];
    if ( is_array( $parts ) ) {
        foreach ( $parts as $p ) {
            $p = trim( (string) $p );
            if ( '' === $p || strlen( $p ) < 3 ) { continue; }
            if ( in_array( $p, $stop, true ) ) { continue; }
            $keywords[] = $p;
        }
    }
    $keywords = array_slice( array_values( array_unique( $keywords ) ), 0, 4 );
    $simplified = implode( ' ', $keywords );
    if ( '' !== $simplified && $simplified !== $query ) {
        [ $url2, $resp2 ] = $do_request( $simplified );
        if ( is_wp_error( $resp2 ) ) {
            $attempts[] = [ 'url' => $url2, 'status' => 0, 'error' => (string) $resp2->get_error_message() ];
        } else {
            $code2 = (int) wp_remote_retrieve_response_code( $resp2 );
            $body2 = (string) wp_remote_retrieve_body( $resp2 );
            $err2  = '';
            $results2 = [];
            if ( $code2 < 200 || $code2 >= 300 ) {
                $err2 = 'http_error';
            } elseif ( '' === $body2 ) {
                $err2 = 'empty_body';
            } else {
                $json2 = json_decode( $body2, true );
                if ( is_array( $json2 ) && isset( $json2['results'] ) && is_array( $json2['results'] ) ) {
                    $results2 = $json2['results'];
                } else {
                    $err2 = 'bad_json';
                }
            }
            $attempts[] = [ 'url' => $url2, 'status' => $code2, 'error' => $err2 ];
            if ( empty( $err2 ) && ! empty( $results2 ) ) {
                $cw_lite_last_cc0_debug = [ 'url' => $url2, 'status' => $code2, 'error' => '', 'attempts' => $attempts ];
                $out = [];
                foreach ( $results2 as $r ) {
                    if ( ! is_array( $r ) ) { continue; }
                    $img_url  = isset( $r['url'] ) && is_string( $r['url'] ) ? $r['url'] : '';
                    $thumb    = isset( $r['thumbnail'] ) && is_string( $r['thumbnail'] ) ? $r['thumbnail'] : '';
                    $landing  = isset( $r['foreign_landing_url'] ) && is_string( $r['foreign_landing_url'] ) ? $r['foreign_landing_url'] : '';
                    $creator  = isset( $r['creator'] ) && is_string( $r['creator'] ) ? $r['creator'] : '';
                    $title    = isset( $r['title'] ) && is_string( $r['title'] ) ? $r['title'] : '';
                    if ( '' === $img_url ) { continue; }
                    $out[] = [ 'url' => $img_url, 'thumb' => $thumb, 'landing' => $landing, 'creator' => $creator, 'title' => $title ];
                }
                return $out;
            }
        }
    }

    // No results; record attempts and return empty.
    $last_attempt = end( $attempts );
    $cw_lite_last_cc0_debug = [
        'url'      => is_array( $last_attempt ) && isset( $last_attempt['url'] ) ? (string) $last_attempt['url'] : '',
        'status'   => is_array( $last_attempt ) && isset( $last_attempt['status'] ) ? (int) $last_attempt['status'] : 0,
        'error'    => is_array( $last_attempt ) && isset( $last_attempt['error'] ) ? (string) $last_attempt['error'] : 'no_results',
        'attempts' => $attempts,
    ];
    return [];
}

/**
 * Download an image to the Media Library and return the attachment ID.
 *
 * @param string $image_url Remote image URL.
 * @param int    $post_id   Post ID used for sideload context.
 * @return int Attachment ID or 0 on failure.
 */
function contentweaver_ai_lite_download_image_to_media( string $image_url, int $post_id ): int {
    if ( '' === $image_url ) { return 0; }
    if ( ! function_exists( 'media_sideload_image' ) ) {
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';
    }
    $result = media_sideload_image( $image_url, $post_id, '', 'id' );
    if ( is_wp_error( $result ) ) {
        return 0;
    }
    return is_numeric( $result ) ? (int) $result : 0;
}

/**
 * Extract the first <img> src from HTML.
 *
 * @param string $html HTML content.
 * @return string URL or empty.
 */
function contentweaver_ai_lite_extract_first_image_src( string $html ): string {
    if ( '' === trim( $html ) ) { return ''; }
    if ( preg_match( '/<img[^>]+src=["\']([^"\']+)["\']/i', $html, $m ) ) {
        return esc_url_raw( (string) $m[1] );
    }
    return '';
}

/**
 * Define plugin constants.
 */
const CONTENTWEAVER_AI_LITE_VERSION    = '1.0.2';
const CONTENTWEAVER_AI_LITE_TEXTDOMAIN = 'contentweaver-ai-lite';
const CONTENTWEAVER_AI_LITE_OPTION     = 'contentweaver_ai_lite_options';

/**
 * Autoload includes.
 * We keep a very small footprint: one includes file for admin pages and one for spintax utilities.
 */
require_once plugin_dir_path( __FILE__ ) . 'includes/spintax.php';
require_once plugin_dir_path( __FILE__ ) . 'includes/admin-pages.php';
require_once plugin_dir_path( __FILE__ ) . 'includes/ajax.php';
require_once plugin_dir_path( __FILE__ ) . 'includes/voices.php';

/**
 * Translations:
 * WordPress.org auto-loads translations for plugins since WP 4.6 when distributed via the directory.
 * No explicit load_plugin_textdomain() call is required here.
 */

/**
 * Enqueue admin scripts/styles needed for sliders and UI components.
 */
function contentweaver_ai_lite_admin_enqueue( string $hook ): void {
    // Only load on our plugin pages.
    if ( false === strpos( $hook, 'contentweaver-ai-lite' ) ) {
        return;
    }

    wp_enqueue_style( 'wp-components' );
    wp_enqueue_script( 'jquery-ui-slider' );
    // Ensure media frame is available for default image picker.
    if ( function_exists( 'wp_enqueue_media' ) ) { wp_enqueue_media(); }

    // Enqueue RSS assets on all plugin pages to ensure availability (lightweight files).
    wp_enqueue_style( 'cw-lite-rss-admin', plugins_url( 'assets/admin-rss.css', __FILE__ ), [], CONTENTWEAVER_AI_LITE_VERSION );
    wp_enqueue_script( 'cw-lite-rss-admin', plugins_url( 'assets/admin-rss.js', __FILE__ ), [ 'jquery' ], CONTENTWEAVER_AI_LITE_VERSION, true );
    wp_localize_script( 'cw-lite-rss-admin', 'cwLiteRSS', [
        'ajaxUrl' => admin_url( 'admin-ajax.php' ),
        'nonce'   => wp_create_nonce( 'contentweaver_ai_lite_rss_run' ),
        'i18n'    => [
            'running' => __( 'Processing...', 'contentweaver-ai-lite' ),
            'error'   => __( 'An error occurred while running the import.', 'contentweaver-ai-lite' ),
        ],
    ] );

    // Determine which subpage we are on to conditionally enqueue assets.
    // Hooks typically look like:
    // - 'toplevel_page_contentweaver-ai-lite'
    // - 'contentweaver-ai-lite_page_contentweaver-ai-lite-spintax'
    // - 'contentweaver-ai-lite_page_contentweaver-ai-lite-manual'
    // - 'contentweaver-ai-lite_page_contentweaver-ai-lite-upgrade'
    $is_spintax = ( false !== strpos( $hook, 'contentweaver-ai-lite-spintax' ) );
    $is_manual  = ( false !== strpos( $hook, 'contentweaver-ai-lite-manual' ) );
    $is_upgrade = ( false !== strpos( $hook, 'contentweaver-ai-lite-upgrade' ) );

    if ( $is_spintax ) {
        wp_enqueue_style( 'cw-lite-spintax', plugins_url( 'assets/admin-spintax.css', __FILE__ ), [], CONTENTWEAVER_AI_LITE_VERSION );
        wp_enqueue_script( 'cw-lite-spintax', plugins_url( 'assets/admin-spintax.js', __FILE__ ), [ 'jquery', 'jquery-ui-slider' ], CONTENTWEAVER_AI_LITE_VERSION, true );
        // Provide ajax URL via localize for consistency if needed later.
        wp_localize_script( 'cw-lite-spintax', 'cwLiteSpintax', [
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
        ] );
    }

    if ( $is_manual ) {
        wp_enqueue_style( 'cw-lite-manual', plugins_url( 'assets/admin-manual.css', __FILE__ ), [], CONTENTWEAVER_AI_LITE_VERSION );
        wp_enqueue_script( 'cw-lite-manual', plugins_url( 'assets/admin-manual.js', __FILE__ ), [ 'jquery' ], CONTENTWEAVER_AI_LITE_VERSION, true );
        wp_localize_script( 'cw-lite-manual', 'cwLiteManual', [
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
        ] );
    }

    if ( $is_upgrade ) {
        wp_enqueue_style( 'cw-lite-upgrade', plugins_url( 'assets/admin-upgrade.css', __FILE__ ), [], CONTENTWEAVER_AI_LITE_VERSION );
    }
}
add_action( 'admin_enqueue_scripts', 'contentweaver_ai_lite_admin_enqueue' );

/**
 * User agent used for RSS fetches to avoid 403s from stricter feeds (e.g., TMZ).
 *
 * @return string
 */
function contentweaver_ai_lite_get_feed_user_agent(): string {
    // Mimic Pro: fixed Chrome UA for reliability against stricter feeds.
    return 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_0) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/78.0.3904.97 Safari/537.36';
}

/**
 * Set SimplePie options for fetch_feed: user agent, timeout, and force feed if needed.
 *
 * @param SimplePie $feed SimplePie instance.
 * @param string    $url  Feed URL.
 */
function contentweaver_ai_lite_feed_options( $feed, string $url ): void { // phpcs:ignore
    if ( is_object( $feed ) && method_exists( $feed, 'set_useragent' ) ) {
        $feed->set_useragent( contentweaver_ai_lite_get_feed_user_agent() );
    }
    if ( is_object( $feed ) && method_exists( $feed, 'set_timeout' ) ) {
        $feed->set_timeout( 20 );
    }
    if ( is_object( $feed ) && method_exists( $feed, 'force_feed' ) ) {
        // Some feeds return text/html while still being valid RSS; accept anyway.
        $feed->force_feed( true );
    }
    if ( is_object( $feed ) && method_exists( $feed, 'set_stupidly_fast' ) ) {
        // Relax parsing checks to tolerate slightly invalid feeds.
        $feed->set_stupidly_fast( true );
    }
}
add_action( 'wp_feed_options', 'contentweaver_ai_lite_feed_options', 10, 2 );

/**
 * Ensure HTTP requests to the configured RSS URL have browser-like headers to avoid 403s.
 *
 * @param array<string,mixed> $args Request args.
 * @param string              $url  Request URL.
 * @return array<string,mixed>
 */
function contentweaver_ai_lite_feed_http_headers( array $args, string $url ): array {
    // Apply during active RSS runs to all requests (covers redirects/CDNs).
    $is_fetching = isset( $GLOBALS['cw_lite_fetching_feed'] ) && true === $GLOBALS['cw_lite_fetching_feed'];
    if ( $is_fetching ) {
        $headers = isset( $args['headers'] ) && is_array( $args['headers'] ) ? $args['headers'] : [];
        // Mimic Pro: only set a fixed User-Agent for feed requests.
        $headers['User-Agent'] = contentweaver_ai_lite_get_feed_user_agent();
        $args['headers']             = $headers;
        $args['timeout']             = isset( $args['timeout'] ) ? max( (int) $args['timeout'], 20 ) : 20;
    }
    return $args;
}
add_filter( 'http_request_args', 'contentweaver_ai_lite_feed_http_headers', 10, 2 );

/**
 * During manual runs, disable feed caching for freshness.
 *
 * @return int
 */
function contentweaver_ai_lite_feed_cache_lifetime(): int {
    return 0; // seconds
}

/**
 * Fetch a URL with browser-like headers.
 *
 * @param string $url Target URL.
 * @return array{status:int,type:string,body:string}
 */
function contentweaver_ai_lite_fetch_url( string $url ): array {
    $GLOBALS['cw_lite_fetching_feed'] = true; // Ensure headers filter applies during fetch.
    $response = wp_remote_get( $url, [
        'headers' => [
            'User-Agent'      => contentweaver_ai_lite_get_feed_user_agent(),
            'Accept'          => 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language' => 'en-US,en;q=0.9',
            'Referer'         => $url,
        ],
        'timeout' => 20,
    ] );
    $status = 0;
    $type   = '';
    $body   = '';
    if ( ! is_wp_error( $response ) ) {
        $status = (int) wp_remote_retrieve_response_code( $response );
        $type   = (string) wp_remote_retrieve_header( $response, 'content-type' );
        $body   = (string) wp_remote_retrieve_body( $response );
    }
    return [ 'status' => $status, 'type' => $type, 'body' => $body ];
}

/**
 * Extract main article HTML from a page using simple heuristics.
 *
 * @param string $page_url The page URL (used for resolving relative links if needed).
 * @param string $html     Raw HTML.
 * @return string Cleaned article HTML or empty string on failure.
 */
function contentweaver_ai_lite_extract_article_html( string $page_url, string $html ): string { // phpcs:ignore
    if ( '' === trim( $html ) ) {
        return '';
    }
    // Suppress libxml warnings for malformed HTML.
    $dom = new DOMDocument();
    libxml_use_internal_errors( true );
    $loaded = $dom->loadHTML( $html );
    libxml_clear_errors();
    if ( false === $loaded ) {
        return '';
    }

    // Remove script and style tags.
    $remove_tags = [ 'script', 'style', 'noscript' ];
    foreach ( $remove_tags as $tag ) {
        $nodes = $dom->getElementsByTagName( $tag );
        for ( $i = $nodes->length - 1; $i >= 0; $i-- ) {
            $node = $nodes->item( $i );
            if ( $node && $node->parentNode ) {
                $node->parentNode->removeChild( $node );
            }
        }
    }

    // Candidate selectors by common article containers.
    $candidates = [ 'article',
        "//*[@role='main']",
        "//*[@id='article']",
        "//*[contains(concat(' ', normalize-space(@class), ' '), ' article-body ')]",
        "//*[contains(concat(' ', normalize-space(@class), ' '), ' entry-content ')]",
        "//*[contains(concat(' ', normalize-space(@class), ' '), ' post-content ')]",
        "//*[contains(concat(' ', normalize-space(@class), ' '), ' story-body ')]",
        "//*[contains(concat(' ', normalize-space(@class), ' '), ' main-content ')]",
        "//*[@id='content']",
        "//*[contains(concat(' ', normalize-space(@class), ' '), ' content ')]",
    ];

    $xpath = new DOMXPath( $dom );
    $best_html = '';
    $best_len  = 0;
    foreach ( $candidates as $sel ) {
        if ( 'article' === $sel ) {
            $nodes = $dom->getElementsByTagName( 'article' );
            foreach ( $nodes as $node ) {
                $text = trim( $node->textContent );
                $len  = strlen( $text );
                if ( $len > $best_len ) {
                    $best_len  = $len;
                    $best_html = $dom->saveHTML( $node ) ?: '';
                }
            }
        } else {
            $nodes = $xpath->query( $sel );
            if ( $nodes instanceof DOMNodeList ) {
                foreach ( $nodes as $node ) {
                    $text = trim( $node->textContent );
                    $len  = strlen( $text );
                    if ( $len > $best_len ) {
                        $best_len  = $len;
                        $best_html = $dom->saveHTML( $node ) ?: '';
                    }
                }
            }
        }
    }

    // Fallback: join all paragraph tags.
    if ( '' === $best_html ) {
        $paras = $dom->getElementsByTagName( 'p' );
        $chunks = [];
        foreach ( $paras as $p ) {
            $txt = trim( $p->textContent );
            if ( '' !== $txt ) {
                $chunks[] = '<p>' . esc_html( $txt ) . '</p>';
            }
        }
        $best_html = implode( "\n", $chunks );
    }

    return is_string( $best_html ) ? $best_html : '';
}

/**
 * Rewrite RSS item content using the configured OpenAI-compatible provider.
 * Falls back to original content when API key is missing or on error.
 *
 * @param string $title   Item title (used to guide rewriting).
 * @param string $content Original HTML content from the feed.
 * @return string Rewritten HTML content or original on failure.
 */
function contentweaver_ai_lite_rss_rewrite_content( string $title, string $content ): string {
    $options = get_option( CONTENTWEAVER_AI_LITE_OPTION );
    $url     = is_array( $options ) && isset( $options['rss_feed_url'] ) ? (string) $options['rss_feed_url'] : '';
    // Per-run limits and buffer. Initialize counters used in discovery logic.
    $limit    = is_array( $options ) && isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3;
    if ( $limit < 1 ) { $limit = 1; }
    if ( $limit > 3 ) { $limit = 3; }
    $fetch_max = (int) min( 20, max( $limit * 4, $limit ) );
    $maxitems = 0;
    $items    = [];
    // Configure per-run limits and pre-initialize counters/arrays used in fallbacks.
    $limit    = is_array( $options ) && isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3;
    if ( $limit < 1 ) { $limit = 1; }
    if ( $limit > 3 ) { $limit = 3; }
    $fetch_max = (int) min( 20, max( $limit * 4, $limit ) );
    $maxitems = 0;
    $items    = [];
    // Items per run (1–3) and fetch_max buffer for skipping duplicates/extraction failures.
    $limit   = is_array( $options ) && isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3;
    if ( $limit < 1 ) { $limit = 1; }
    if ( $limit > 3 ) { $limit = 3; }
    $fetch_max = (int) min( 20, max( $limit * 4, $limit ) );
    $limit   = is_array( $options ) && isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3;
    if ( $limit < 1 ) { $limit = 1; }
    if ( $limit > 3 ) { $limit = 3; }
    // Fetch more items than the final import cap so we can skip duplicates or failed extractions
    // while still producing up to $limit imported drafts from the top of the feed.
    $fetch_max = (int) min( 20, max( $limit * 4, $limit ) );
    $api_key  = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
    $provider = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
    $model    = is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini';
    $base_url = is_array( $options ) && isset( $options['base_url'] ) && is_string( $options['base_url'] ) && '' !== trim( $options['base_url'] )
        ? (string) $options['base_url']
        : ( 'deepseek' === $provider ? 'https://api.deepseek.com/v1' : 'https://api.openai.com/v1' );

    if ( '' === $api_key ) {
        return $content; // No API key configured; keep original content.
    }

    $endpoint = untrailingslashit( $base_url ) . '/chat/completions';
    $prompt   = 'Rewrite and lightly enhance the following article for clarity, coherence, and neutrality. '
        . 'Avoid promotional language. Maintain factual accuracy. Output clean HTML only, no markdown, no code fences.\n\n'
        . 'Title: ' . $title . "\n\n"
        . 'Content (HTML): ' . wp_strip_all_tags( $content );

    $body = [
        'model'       => $model,
        'messages'    => [ [ 'role' => 'user', 'content' => $prompt ] ],
        'temperature' => 0.5,
    ];

    $args = [
        'headers' => [
            'Content-Type'  => 'application/json',
            'Authorization' => 'Bearer ' . $api_key,
        ],
        'timeout' => 40,
        'body'    => wp_json_encode( $body ),
    ];

    $response = wp_remote_post( $endpoint, $args );
    if ( is_wp_error( $response ) ) {
        return $content;
    }

    $status  = wp_remote_retrieve_response_code( $response );
    $raw     = wp_remote_retrieve_body( $response );
    $decoded = json_decode( $raw, true );
    if ( 200 !== $status || ! is_array( $decoded ) || ! isset( $decoded['choices'][0]['message']['content'] ) || ! is_string( $decoded['choices'][0]['message']['content'] ) ) {
        return $content;
    }

    $rewritten = trim( (string) $decoded['choices'][0]['message']['content'] );
    if ( '' === $rewritten ) {
        return $content;
    }
    return $rewritten;
}

/**
 * Check if the Pro plugin is active.
 *
 * @return bool True if Pro plugin is active.
 */
function contentweaver_ai_lite_is_pro_active(): bool {
	if ( ! function_exists( 'is_plugin_active' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	return is_plugin_active( 'contentweaver-ai/contentweaver-ai.php' );
}

/**
 * Admin notice shown when Pro is active alongside Lite.
 */
function contentweaver_ai_lite_admin_notice_pro_active(): void {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	if ( contentweaver_ai_lite_is_pro_active() ) {
		echo '<div class="notice notice-warning is-dismissible"><p>' . esc_html__( 'ContentWeaver AI Pro is active. For best experience, use Pro as your primary UI. You can deactivate Lite or keep it for standalone testing.', 'contentweaver-ai-lite' ) . '</p></div>';
	}
}
add_action( 'admin_notices', 'contentweaver_ai_lite_admin_notice_pro_active' );

/**
 * Register plugin settings using the Settings API.
 * We store all options in a single array for simplicity.
 */
function contentweaver_ai_lite_register_settings(): void {
    register_setting(
        'contentweaver_ai_lite',
        CONTENTWEAVER_AI_LITE_OPTION,
        [
            'type'              => 'array',
            'sanitize_callback' => 'contentweaver_ai_lite_sanitize_options',
            'default'           => [
                'provider'      => 'openai',
                'api_key'       => '',
                'base_url'      => 'https://api.openai.com/v1',
                'model'         => 'gpt-4o-mini',
                'rss_feed_url'  => '',
                'voice_id'      => 10,
                'items_per_run' => 3,
                'featured_image_mode' => 'none',
                'default_image_id'    => 0,
            ],
        ]
    );
}
add_action( 'admin_init', 'contentweaver_ai_lite_register_settings' );

/**
 * Sanitize options before saving.
 *
 * @param mixed $input Raw input.
 * @return array<string,string> Sanitized options array.
 */
function contentweaver_ai_lite_sanitize_options( $input ): array { // phpcs:ignore
    // Start from existing options to avoid blanking unrelated fields when saving a subset form.
    $existing = get_option( CONTENTWEAVER_AI_LITE_OPTION );
    $sanitized = [
        'provider'     => 'openai',
        'api_key'      => '',
        'base_url'     => 'https://api.openai.com/v1',
        'model'        => 'gpt-4o-mini',
        'rss_feed_url' => '',
        'voice_id'     => 10,
        'items_per_run'=> 3,
        'featured_image_mode' => 'none',
        'default_image_id'    => 0,
    ];
    if ( is_array( $existing ) ) {
        foreach ( $sanitized as $k => $v ) {
            // Preserve any existing value (string or numeric) to avoid resetting fields on partial saves.
            if ( array_key_exists( $k, $existing ) ) {
                $sanitized[ $k ] = $existing[ $k ];
            }
        }
    }

    if ( is_array( $input ) ) {
        if ( isset( $input['provider'] ) ) {
            $provider              = is_string( $input['provider'] ) ? strtolower( $input['provider'] ) : 'openai';
            $sanitized['provider'] = in_array( $provider, [ 'openai', 'deepseek' ], true ) ? $provider : 'openai';
        }

        if ( isset( $input['api_key'] ) && is_string( $input['api_key'] ) ) {
            $sanitized['api_key'] = trim( $input['api_key'] );
        }

        if ( isset( $input['base_url'] ) && is_string( $input['base_url'] ) ) {
            $base_url = trim( $input['base_url'] );
            // Allow only http/https URLs.
            if ( preg_match( '/^https?:\/\/[A-Za-z0-9\-_.:]+\/[A-Za-z0-9\-_.\/]*$/', $base_url ) === 1 ) {
                $sanitized['base_url'] = untrailingslashit( $base_url );
            }
        }

        if ( isset( $input['model'] ) && is_string( $input['model'] ) ) {
            $sanitized['model'] = sanitize_text_field( $input['model'] );
        }

        if ( isset( $input['rss_feed_url'] ) && is_string( $input['rss_feed_url'] ) ) {
            $sanitized['rss_feed_url'] = esc_url_raw( trim( $input['rss_feed_url'] ) );
        }
        if ( isset( $input['voice_id'] ) ) {
            $vid = is_numeric( $input['voice_id'] ) ? (int) $input['voice_id'] : 10;
            $voices = function_exists( 'contentweaver_ai_lite_get_voices' ) ? contentweaver_ai_lite_get_voices() : [];
            $sanitized['voice_id'] = ( isset( $voices[ $vid ] ) ) ? $vid : 10;
        }
        if ( isset( $input['items_per_run'] ) ) {
            $n = is_numeric( $input['items_per_run'] ) ? (int) $input['items_per_run'] : 3;
            if ( $n < 1 ) { $n = 1; }
            if ( $n > 3 ) { $n = 3; }
            $sanitized['items_per_run'] = $n;
        }
        if ( isset( $input['featured_image_mode'] ) && is_string( $input['featured_image_mode'] ) ) {
            $mode = strtolower( trim( $input['featured_image_mode'] ) );
            $allowed = [ 'none', 'default', 'site_icon', 'cc0' ];
            $sanitized['featured_image_mode'] = in_array( $mode, $allowed, true ) ? $mode : 'none';
        }
        if ( isset( $input['default_image_id'] ) ) {
            $aid = is_numeric( $input['default_image_id'] ) ? (int) $input['default_image_id'] : 0;
            if ( $aid > 0 ) {
                $att = get_post( $aid );
                if ( $att && 'attachment' === $att->post_type && wp_attachment_is_image( $aid ) ) {
                    $sanitized['default_image_id'] = $aid;
                } else {
                    $sanitized['default_image_id'] = 0;
                }
            } else {
                $sanitized['default_image_id'] = 0;
            }
        }
    }

    // Provider-specific default base URLs.
    if ( 'deepseek' === $sanitized['provider'] ) {
        $sanitized['base_url'] = 'https://api.deepseek.com/v1';
    } else {
        $sanitized['base_url'] = 'https://api.openai.com/v1';
    }

    return $sanitized;
}

/**
 * Build a human-friendly source name from a URL host.
 *
 * @param string $host Hostname.
 * @return string
 */
function contentweaver_ai_lite_source_name_from_host( string $host ): string {
    $h = strtolower( $host );
    $h = preg_replace( '/^www\./', '', $h );
    $parts = explode( '.', $h );
    if ( count( $parts ) >= 2 ) {
        $base = $parts[ count( $parts ) - 2 ];
        return ucwords( str_replace( '-', ' ', $base ) );
    }
    return ucwords( str_replace( '-', ' ', $h ) );
}

/**
 * Build CWAI base prompt using provided context and selected voice.
 *
 * @param array<string,string> $ctx Context with keys: original_title, original_guid, source_name, homepage, article_text, voice_name, voice_description, voice_settings_description, author_name.
 * @param int $min_words Minimum words (Lite: 400).
 * @return string
 */
function contentweaver_ai_lite_build_cwai_prompt( array $ctx, int $min_words ): string {
    $seo_guidance         = '';
    $excluded_words_str   = '';
    $formatted_intro_examples   = isset( $ctx['intro_examples'] ) ? (string) $ctx['intro_examples'] : '';
    $formatted_closing_examples = isset( $ctx['closing_examples'] ) ? (string) $ctx['closing_examples'] : '';

    $tpl = "You are a professional content writer. Write a unique article based on the following source content.\n\n"
        . "The original title is: {original_title}  \n"
        . "The original GUID is: {original_guid}\n\n"
        . "Write in the style of {voice_name}, who is described as: {voice_description}  \n"
        . "⚠️ DO NOT include or refer to the voice name or description inside the article itself.\n\n"
        . "The summary must be at least {min_words} SEO-friendly words. Do not stop early.\n\n"
        . "Source: {source_name}  \n"
        . "Homepage: {homepage}  \n"
        . "Original Title: {original_title}  \n"
        . "Article: {article_text}\n\n"
        . "⚠️ REQUIRED FORMAT: You must include **all sections in this exact order**.  \n"
        . "⚠️ DO NOT return numbered lists, bullet points, or step-by-step formatting of any kind.  \n"
        . "Only use standard paragraph structure and plain text.  \n"
        . "Numbered, bulleted, or step-formatting will result in rejection.\n\n"
        . "---\n\n"
        . "<SECTION:Subject>\n[A concise 3–5 word subject, listing the most famous person first for categorization.]\n</SECTION>\n\n"
        . "<SECTION:ImageQuery>\n\n[Return only 2–4 plain ASCII words (no punctuation, numbers, or quotes) that best describe an image to illustrate this story. Use concrete nouns and short modifiers. Examples: \"chicken recipes\", \"stainless steel pan\", \"fresh herbs\". Do not include words like readers, people, make, made, the, a, an.]\n\n</SECTION>\n\n"
        . "<SECTION:New Title>\n\n[A new, engaging, SEO-optimized headline that reflects the story clearly.]  \n{seo_guidance}\n\n</SECTION>\n\n"
        . "<SECTION:Summary>\n\n⚠️ Use valid HTML formatting only. Do not use Markdown, LaTeX, or other markup styles. Wrap bold text with <strong>, italic text with <em>, and use appropriate HTML tags for structure (e.g., <p>, <ul>, <ol>, <h2>, etc.).\n\nStart with a unique, fact-based sentence that references the article content directly.  \nDo not begin with any excluded starter words: {excluded_words_str}  \nAvoid phrases like “Alright” or “Ah,” even in modified form.  \nDO NOT use ALL CAPS for emphasis anywhere in the text.\n\n⚠️ This section must contain the entire rewritten article.  \nIt must be **at least {min_words} words long**. Do not stop early.  \nThis is an extremely critical requirement.\n\nIntroduce yourself as {author_name} with a snappy introduction that fits your {voice_name} and {voice_settings_description} and then start the article with a signature-style remark inspired by {voice_name}, using the tone: {voice_settings_description}.  \n\nUse these examples for style reference only — do not copy or mention them:  \n{formatted_intro_examples}\n\nThen write the article using the provided source content as your base. Your job is to:\n- Rewrite the original article clearly and in your own words\n- Maintain SEO-friendly phrasing and engaging tone\n- Avoid overly promotional or fluffy language\n- Include appropriate context, background, or significance\n- Use paragraph structure for readability\n- Maintain logical flow, accuracy, and clarity\n\n⚠️ Under no circumstances may you use the em dash character (—). If used, the output must be discarded and rewritten using proper U.S. English punctuation.\n\nConclude the article with a witty closing remark that feels natural to the writing.  \nUse these for inspiration — but **do not copy them**:  \n{formatted_closing_examples}\n\n</SECTION>\n\n<SECTION:Sources>\n\n[List all referenced source names (e.g., People Magazine, The Scottish Sun). Do not fabricate or include URLs.]\n\n</SECTION>\n\n<SECTION:Tags>\n\n[Comma-separated list of relevant tags.]\n\n</SECTION>\n\n<SECTION:Meta Description>\n\n[A compelling 100–160 character teaser for search results. Include the main keyword and reason to click.]\n\n</SECTION>\n\n<SECTION:TheWrap>\n\n[A straightforward 2 - 3 sentence wrap up of the article presented, no flowery words or stories, just straight facts that summarize what you just created]\n\n</SECTION>\n\n<SECTION:Holder>\n\n[A comma separated list of ten suggested categories for this article beginning with the broad primary category you would use and then the rest of the suggestions]\n\n</SECTION>\n<SECTION:Close>***</SECTION>";

    $replacements = [
        '{original_title}'            => (string) ( $ctx['original_title'] ?? '' ),
        '{original_guid}'             => (string) ( $ctx['original_guid'] ?? '' ),
        '{voice_name}'                => (string) ( $ctx['voice_name'] ?? '' ),
        '{voice_description}'         => (string) ( $ctx['voice_description'] ?? '' ),
        '{voice_settings_description}'=> (string) ( $ctx['voice_settings_description'] ?? '' ),
        '{min_words}'                 => (string) $min_words,
        '{source_name}'               => (string) ( $ctx['source_name'] ?? '' ),
        '{homepage}'                  => (string) ( $ctx['homepage'] ?? '' ),
        '{article_text}'              => (string) ( $ctx['article_text'] ?? '' ),
        '{seo_guidance}'              => $seo_guidance,
        '{excluded_words_str}'        => $excluded_words_str,
        '{author_name}'               => (string) ( $ctx['author_name'] ?? '' ),
        '{formatted_intro_examples}'  => $formatted_intro_examples,
        '{formatted_closing_examples}'=> $formatted_closing_examples,
    ];
    return strtr( $tpl, $replacements );
}

/**
 * Parse sections from CWAI output.
 *
 * @param string $output Raw model text.
 * @return array<string,string>
 */
function contentweaver_ai_lite_parse_cwai_sections( string $output ): array {
    $sections = [];
    $pattern = '/<SECTION:([^>]+)>\s*([\s\S]*?)\s*<\/SECTION>/i';
    if ( preg_match_all( $pattern, $output, $m, PREG_SET_ORDER ) ) {
        foreach ( $m as $match ) {
            $key = trim( (string) $match[1] );
            $val = trim( (string) $match[2] );
            $sections[ $key ] = $val;
        }
    }
    return $sections;
}

/**
 * Try to extract og:image from HTML.
 *
 * @param string $html HTML source.
 * @return string URL or empty.
 */
function contentweaver_ai_lite_extract_og_image( string $html ): string {
    if ( '' === trim( $html ) ) { return ''; }
    if ( preg_match( '/<meta[^>]+property=["\']og:image["\'][^>]+content=["\']([^"\']+)["\']/i', $html, $m ) ) {
        return esc_url_raw( (string) $m[1] );
    }
    if ( preg_match( '/<meta[^>]+name=["\']twitter:image["\'][^>]+content=["\']([^"\']+)["\']/i', $html, $m2 ) ) {
        return esc_url_raw( (string) $m2[1] );
    }
    return '';
}

/**
 * Set featured image from URL via sideload.
 *
 * @param int    $post_id Post ID.
 * @param string $image_url Image URL.
 */
function contentweaver_ai_lite_set_featured_image( int $post_id, string $image_url ): void {
    if ( $post_id <= 0 || '' === $image_url ) { return; }
    require_once ABSPATH . 'wp-admin/includes/file.php';
    require_once ABSPATH . 'wp-admin/includes/media.php';
    require_once ABSPATH . 'wp-admin/includes/image.php';
    $att_id = 0;
    $html = media_sideload_image( $image_url, $post_id, null, 'html' );
    if ( ! is_wp_error( $html ) ) {
        // Try to resolve the attachment by scanning attachments for this post.
        $attachments = get_posts( [ 'post_type' => 'attachment', 'posts_per_page' => 1, 'post_parent' => $post_id, 'orderby' => 'date', 'order' => 'DESC' ] );
        if ( is_array( $attachments ) && isset( $attachments[0] ) && isset( $attachments[0]->ID ) ) {
            $att_id = (int) $attachments[0]->ID;
        }
    }
    if ( $att_id > 0 ) {
        set_post_thumbnail( $post_id, $att_id );
    }
}

/**
 * Add admin menus for Lite.
 */
function contentweaver_ai_lite_register_admin_menu(): void {
	$capability = 'manage_options';
	$slug       = 'contentweaver-ai-lite';

	// If Pro is active, hide Lite menus to reduce UX friction (Phase 1 coexistence).
	if ( contentweaver_ai_lite_is_pro_active() ) {
		return;
	}

	add_menu_page(
		__( 'ContentWeaver AI Lite', 'contentweaver-ai-lite' ),
		__( 'ContentWeaver AI Lite', 'contentweaver-ai-lite' ),
		$capability,
		$slug,
		'contentweaver_ai_lite_render_dashboard_page',
		'dashicons-admin-generic',
		56
	);

	add_submenu_page( $slug, __( 'Dashboard', 'contentweaver-ai-lite' ), __( 'Dashboard', 'contentweaver-ai-lite' ), $capability, $slug, 'contentweaver_ai_lite_render_dashboard_page' );
	add_submenu_page( $slug, __( 'RSS Lite', 'contentweaver-ai-lite' ), __( 'RSS Lite', 'contentweaver-ai-lite' ), $capability, 'contentweaver-ai-lite-rss', 'contentweaver_ai_lite_render_rss_page' );
	add_submenu_page( $slug, __( 'Spintax', 'contentweaver-ai-lite' ), __( 'Spintax', 'contentweaver-ai-lite' ), $capability, 'contentweaver-ai-lite-spintax', 'contentweaver_ai_lite_render_spintax_page' );
	add_submenu_page( $slug, __( 'Manual Generation', 'contentweaver-ai-lite' ), __( 'Manual Generation', 'contentweaver-ai-lite' ), $capability, 'contentweaver-ai-lite-manual', 'contentweaver_ai_lite_render_manual_page' );
	add_submenu_page( $slug, __( 'Upgrade to Pro', 'contentweaver-ai-lite' ), __( 'Upgrade to Pro', 'contentweaver-ai-lite' ), $capability, 'contentweaver-ai-lite-upgrade', 'contentweaver_ai_lite_render_upgrade_page' );
}
add_action( 'admin_menu', 'contentweaver_ai_lite_register_admin_menu' );

/**
 * Handle the manual RSS run action (manual only, up to 5 items).
 */
function contentweaver_ai_lite_handle_rss_run(): void {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Unauthorized.', 'contentweaver-ai-lite' ) );
	}

	if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
		check_ajax_referer( 'contentweaver_ai_lite_rss_run' );
	} else {
		check_admin_referer( 'contentweaver_ai_lite_rss_run' );
	}

	$options = get_option( CONTENTWEAVER_AI_LITE_OPTION );
	$url     = is_array( $options ) && isset( $options['rss_feed_url'] ) ? (string) $options['rss_feed_url'] : '';

	// Initialize per-run limits and predeclare variables used in later logic.
	$limit = is_array( $options ) && isset( $options['items_per_run'] ) ? (int) $options['items_per_run'] : 3;
	if ( $limit < 1 ) { $limit = 1; }
	if ( $limit > 3 ) { $limit = 3; }
	$fetch_max = (int) min( 20, max( $limit * 4, $limit ) );
	$maxitems  = 0;
	$items     = [];

    if ( '' === $url ) {
        wp_safe_redirect( add_query_arg( 'cw_lite_notice', rawurlencode( __( 'Please set an RSS feed URL first.', 'contentweaver-ai-lite' ) ), admin_url( 'admin.php?page=contentweaver-ai-lite-rss' ) ) );
        exit;
    }

    // Disable caching for this manual fetch then restore default.
    add_filter( 'wp_feed_cache_transient_lifetime', 'contentweaver_ai_lite_feed_cache_lifetime' );
    $used_alt_url       = false;
    $used_raw_fallback  = false;
    $discovered_alt_url = false;
    $http_status        = 0;
    $content_type       = '';
    $body_snippet       = '';
    $GLOBALS['cw_lite_fetching_feed'] = true;
    // Per WP.org guidance: if we need to load feed.php, require it and immediately call a function from it.
    if ( function_exists( 'fetch_feed' ) ) {
        $feed = fetch_feed( $url );
    } else {
        require_once ABSPATH . WPINC . '/feed.php';
        $feed = fetch_feed( $url );
    }
    remove_filter( 'wp_feed_cache_transient_lifetime', 'contentweaver_ai_lite_feed_cache_lifetime' );

    // Fallback 1: try alternate .xml URL if /rss appears invalid
    if ( is_wp_error( $feed ) ) {
        $err_msg = $feed->get_error_message();
        $alt_url = '';
        if ( preg_match( '/\/rss\/?$/i', $url ) === 1 ) {
            $alt_url = rtrim( $url, '/' ) . '.xml';
        }
        if ( '' !== $alt_url ) {
            add_filter( 'wp_feed_cache_transient_lifetime', 'contentweaver_ai_lite_feed_cache_lifetime' );
            $feed = fetch_feed( $alt_url );
            remove_filter( 'wp_feed_cache_transient_lifetime', 'contentweaver_ai_lite_feed_cache_lifetime' );
            $used_alt_url = true;
        }
    }

    // Fallback 2 removed: avoid direct SimplePie includes per WP.org guidance.
    if ( is_wp_error( $feed ) ) {
        $request = wp_remote_get( $url, [
            'headers' => [ 'User-Agent' => contentweaver_ai_lite_get_feed_user_agent() ],
            'timeout' => 20,
        ] );
        if ( ! is_wp_error( $request ) ) {
            $http_status  = (int) wp_remote_retrieve_response_code( $request );
            $content_type = (string) wp_remote_retrieve_header( $request, 'content-type' );
            $body         = (string) wp_remote_retrieve_body( $request );
            $body_snippet = substr( wp_strip_all_tags( $body ), 0, 180 );
        }
    }

    // Discovery fallback: if parsed zero items and we likely got HTML, try to discover <link rel="alternate" ...>
    if ( ( 0 === $maxitems || empty( $items ) ) && '' !== $body_snippet && ( false !== stripos( (string) $content_type, 'text/html' ) || false !== stripos( $body_snippet, '<html' ) ) ) {
        $discovered_href = '';
        if ( isset( $body ) && is_string( $body ) ) {
            if ( preg_match( '/<link[^>]+rel=["\']alternate["\'][^>]+type=["\']application\/(?:rss|atom)\+xml["\'][^>]+href=["\']([^"\']+)["\']/i', $body, $m ) === 1 ) {
                $discovered_href = html_entity_decode( (string) $m[1] );
            }
        }
        if ( '' !== $discovered_href ) {
            // Resolve relative URLs
            if ( 0 === strpos( $discovered_href, '//' ) ) {
                $discovered_href = 'https:' . $discovered_href;
            } elseif ( 0 === strpos( $discovered_href, '/' ) ) {
                $parsed = wp_parse_url( $url );
                if ( is_array( $parsed ) && isset( $parsed['scheme'], $parsed['host'] ) ) {
                    $discovered_href = $parsed['scheme'] . '://' . $parsed['host'] . $discovered_href;
                }
            }
            add_filter( 'wp_feed_cache_transient_lifetime', 'contentweaver_ai_lite_feed_cache_lifetime' );
            $feed2 = fetch_feed( $discovered_href );
            remove_filter( 'wp_feed_cache_transient_lifetime', 'contentweaver_ai_lite_feed_cache_lifetime' );
            if ( ! is_wp_error( $feed2 ) ) {
                $maxitems = (int) ( method_exists( $feed2, 'get_item_quantity' ) ? $feed2->get_item_quantity( $fetch_max ) : 0 );
                $items    = method_exists( $feed2, 'get_items' ) ? $feed2->get_items( 0, $maxitems ) : [];
                $feed     = $feed2;
                $discovered_alt_url = true;
            }
        }
    }

    if ( is_wp_error( $feed ) ) {
        wp_safe_redirect( add_query_arg( 'cw_lite_notice', rawurlencode( sprintf( /* translators: %s: error message */ __( 'RSS error: %s', 'contentweaver-ai-lite' ), $feed->get_error_message() ) ), admin_url( 'admin.php?page=contentweaver-ai-lite-rss' ) ) );
        exit;
    }

    $maxitems = (int) ( method_exists( $feed, 'get_item_quantity' ) ? $feed->get_item_quantity( $fetch_max ) : 0 );
    $items    = method_exists( $feed, 'get_items' ) ? $feed->get_items( 0, $maxitems ) : [];

    // If no items parsed but no WP error, perform raw-fetch fallback once more.
    if ( 0 === $maxitems || empty( $items ) ) {
        $request = wp_remote_get( $url, [
            'headers' => [
                'User-Agent'      => contentweaver_ai_lite_get_feed_user_agent(),
                'Accept'          => 'application/rss+xml, application/xml;q=0.9, text/xml;q=0.8, */*;q=0.5',
                'Accept-Language' => 'en-US,en;q=0.9',
                'Referer'         => $url,
            ],
            'timeout' => 20,
        ] );
        if ( ! is_wp_error( $request ) ) {
            $http_status  = (int) wp_remote_retrieve_response_code( $request );
            $content_type = (string) wp_remote_retrieve_header( $request, 'content-type' );
            $body         = (string) wp_remote_retrieve_body( $request );
            $body_snippet = substr( wp_strip_all_tags( $body ), 0, 180 );
            // No raw SimplePie fallback here to avoid direct core includes.
        }
    }

    $imported = 0;
    $duplicates_skipped = 0;
    $imported           = 0;
    $fetched_full_pages = 0;
    $rewrites_done      = 0;
    $titles_set         = 0;
    $images_set         = 0;
    $tags_assigned      = 0;
    $cc0_searches       = 0;
    $cc0_downloads      = 0;
    $cc0_failures       = 0;
    // Per-item CC0 debug details for UI (attempt URLs/status and selection info).
    $cc0_item_debug     = [];
    // Collect the first few item titles for display in the results panel.
    $first_titles       = [];
    $opts_for_ai = get_option( CONTENTWEAVER_AI_LITE_OPTION );
    $ai_enabled  = is_array( $opts_for_ai ) && ! empty( $opts_for_ai['api_key'] );
    // For debug: requested count and voice label from saved options.
    $requested_n = (int) $limit;
    $voice_id_for_debug = is_array( $opts_for_ai ) && isset( $opts_for_ai['voice_id'] ) ? (int) $opts_for_ai['voice_id'] : 10;
    $all_voices = function_exists( 'contentweaver_ai_lite_get_voices' ) ? contentweaver_ai_lite_get_voices() : [];
    $voice_label = ( is_array( $all_voices ) && isset( $all_voices[ $voice_id_for_debug ] ) && is_array( $all_voices[ $voice_id_for_debug ] ) && isset( $all_voices[ $voice_id_for_debug ]['name'] ) ) ? (string) $all_voices[ $voice_id_for_debug ]['name'] : 'Journalistic Expert';
    foreach ( $items as $item ) {
        $title   = is_object( $item ) && method_exists( $item, 'get_title' ) ? wp_strip_all_tags( (string) $item->get_title() ) : '';
        $content = is_object( $item ) && method_exists( $item, 'get_content' ) ? (string) $item->get_content() : '';
        if ( '' === trim( $content ) && is_object( $item ) && method_exists( $item, 'get_description' ) ) {
            $content = (string) $item->get_description();
        }
        $link    = is_object( $item ) && method_exists( $item, 'get_link' ) ? (string) $item->get_link() : '';

        if ( count( $first_titles ) < 3 && '' !== $title ) {
            $first_titles[] = $title;
        }

        // Skip duplicates by source link.
        if ( '' !== $link ) {
            $existing = get_posts( [
                'post_type'      => 'post',
                'post_status'    => 'any',
                'fields'         => 'ids',
                'posts_per_page' => 1,
                'meta_query'     => [
                    [ 'key' => '_cw_lite_source', 'value' => $link, 'compare' => '=' ],
                ],
            ] );
            if ( is_array( $existing ) && count( $existing ) > 0 ) {
                $duplicates_skipped++;
                continue; // Duplicate; skip.
            }
        } elseif ( '' !== $title ) {
            // If no link is present, use exact title match via get_page_by_title as a weaker fallback.
            $existing_post = get_page_by_title( $title, OBJECT, 'post' );
            if ( $existing_post && isset( $existing_post->ID ) ) {
                $duplicates_skipped++;
                continue;
            }
        }

        // Always attempt to retrieve and extract the full article page when a link is available.
        $page_html = '';
        $got_full  = false;
        if ( '' !== $link ) {
            $f = contentweaver_ai_lite_fetch_url( $link );
            if ( $f['status'] >= 200 && $f['status'] < 400 && is_string( $f['body'] ) && '' !== trim( $f['body'] ) ) {
                $page_html = (string) $f['body'];
                $extracted = contentweaver_ai_lite_extract_article_html( $link, $page_html );
                if ( '' !== $extracted ) {
                    $content = $extracted;
                    $fetched_full_pages++;
                    $got_full = true;
                }
            }
        }
        // If we failed to extract full article, skip this item by default (quality first).
        if ( ! $got_full ) {
            continue;
        }

        // Build CWAI prompt context
        $guid = is_object( $item ) && method_exists( $item, 'get_id' ) ? (string) $item->get_id() : $link;
        $host = ( '' !== $link ) ? (string) parse_url( $link, PHP_URL_HOST ) : '';
        $source_name = '' !== $host ? contentweaver_ai_lite_source_name_from_host( $host ) : '';
        $homepage = '' !== $host ? ( 'https://' . $host ) : '';
        $user_id = get_current_user_id();
        $user    = $user_id > 0 ? get_userdata( $user_id ) : false;
        $author_name = ( $user && isset( $user->display_name ) && is_string( $user->display_name ) ) ? (string) $user->display_name : 'Editor';

        $voice_id = is_array( $opts_for_ai ) && isset( $opts_for_ai['voice_id'] ) ? (int) $opts_for_ai['voice_id'] : 10;
        $voices = function_exists( 'contentweaver_ai_lite_get_voices' ) ? contentweaver_ai_lite_get_voices() : [];
        $voice  = isset( $voices[ $voice_id ] ) ? $voices[ $voice_id ] : ( isset( $voices[10] ) ? $voices[10] : null );
        $voice_name = is_array( $voice ) && isset( $voice['name'] ) ? (string) $voice['name'] : 'Journalistic Expert';
        $voice_desc = is_array( $voice ) && isset( $voice['description'] ) ? (string) $voice['description'] : '';
        $voice_settings = is_array( $voice ) && isset( $voice['summary_guidelines'] ) ? (string) $voice['summary_guidelines'] : '';
        $intro_examples = is_array( $voice ) && isset( $voice['intro_examples'] ) && is_array( $voice['intro_examples'] ) ? implode( "\n", array_map( 'sanitize_text_field', $voice['intro_examples'] ) ) : '';
        $closing_examples = is_array( $voice ) && isset( $voice['closing_examples'] ) && is_array( $voice['closing_examples'] ) ? implode( "\n", array_map( 'sanitize_text_field', $voice['closing_examples'] ) ) : '';

        $ctx = [
            'original_title' => $title,
            'original_guid'  => $guid,
            'source_name'    => $source_name,
            'homepage'       => $homepage,
            'article_text'   => wp_strip_all_tags( $content ),
            'voice_name'     => $voice_name,
            'voice_description' => $voice_desc,
            'voice_settings_description' => $voice_settings,
            'author_name'    => $author_name,
            'intro_examples' => $intro_examples,
            'closing_examples' => $closing_examples,
        ];

        $min_words = 400; // Lite requirement
        $prompt = contentweaver_ai_lite_build_cwai_prompt( $ctx, $min_words );

        // Call provider using CWAI prompt
        $options  = get_option( CONTENTWEAVER_AI_LITE_OPTION );
        $api_key  = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
        $provider = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
        $model    = is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini';
        $base_url = is_array( $options ) && isset( $options['base_url'] ) && is_string( $options['base_url'] ) && '' !== trim( $options['base_url'] )
            ? (string) $options['base_url']
            : ( 'deepseek' === $provider ? 'https://api.deepseek.com/v1' : 'https://api.openai.com/v1' );

        $endpoint = untrailingslashit( $base_url ) . '/chat/completions';
        $body     = [
            'model'       => $model,
            'messages'    => [ [ 'role' => 'user', 'content' => $prompt ] ],
            'temperature' => 0.5,
        ];
        $args = [
            'headers' => [ 'Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $api_key ],
            'timeout' => 60,
            'body'    => wp_json_encode( $body ),
        ];
        $response = wp_remote_post( $endpoint, $args );
        $raw_out  = '';
        if ( ! is_wp_error( $response ) ) {
            $status  = wp_remote_retrieve_response_code( $response );
            $raw_out = ( 200 === (int) $status ) ? (string) wp_remote_retrieve_body( $response ) : '';
            if ( '' !== $raw_out ) {
                $decoded = json_decode( $raw_out, true );
                if ( is_array( $decoded ) && isset( $decoded['choices'][0]['message']['content'] ) && is_string( $decoded['choices'][0]['message']['content'] ) ) {
                    $raw_out = (string) $decoded['choices'][0]['message']['content'];
                } else {
                    $raw_out = '';
                }
            }
        }

        $new_title = $title;
        $summary_html = wp_kses_post( $content );
        $sources_txt = '';
        $tags_list   = '';
        $meta_desc   = '';
        $wrap_txt    = '';
        $holder_txt  = '';
        $image_query = '';
        if ( '' !== $raw_out ) {
            $sections = contentweaver_ai_lite_parse_cwai_sections( $raw_out );
            if ( isset( $sections['New Title'] ) && is_string( $sections['New Title'] ) ) { $new_title = wp_strip_all_tags( (string) $sections['New Title'] ); }
            if ( isset( $sections['Summary'] ) && is_string( $sections['Summary'] ) ) { $summary_html = wp_kses_post( (string) $sections['Summary'] ); }
            if ( isset( $sections['Sources'] ) ) { $sources_txt = sanitize_text_field( (string) $sections['Sources'] ); }
            if ( isset( $sections['Tags'] ) ) { $tags_list = (string) $sections['Tags']; }
            if ( isset( $sections['Meta Description'] ) ) { $meta_desc = sanitize_text_field( (string) $sections['Meta Description'] ); }
            if ( isset( $sections['TheWrap'] ) ) { $wrap_txt = sanitize_text_field( (string) $sections['TheWrap'] ); }
            if ( isset( $sections['Holder'] ) ) { $holder_txt = sanitize_text_field( (string) $sections['Holder'] ); }
            if ( isset( $sections['ImageQuery'] ) ) { $image_query = sanitize_text_field( (string) $sections['ImageQuery'] ); }
            $rewrites_done++;
        }

        // Enforce 400+ words minimal by very basic check; optionally could re-ask to extend.
        $wc = str_word_count( wp_strip_all_tags( $summary_html ) );
        if ( $wc < 400 ) {
            // Best-effort: do not loop forever; leave as-is in Lite to avoid long waits.
        }

        $postarr = [
            'post_title'   => $new_title,
            'post_content' => $summary_html,
            'post_status'  => 'draft',
            'post_type'    => 'post',
        ];

        $post_id = wp_insert_post( $postarr );
        if ( is_int( $post_id ) && $post_id > 0 ) {
            $imported++;
            if ( '' !== $link ) {
                add_post_meta( $post_id, '_cw_lite_source', esc_url_raw( $link ), true );
            }
            if ( '' !== $sources_txt ) { update_post_meta( $post_id, '_cw_lite_sources', $sources_txt ); }
            if ( '' !== $wrap_txt ) { update_post_meta( $post_id, '_cw_lite_wrap', $wrap_txt ); }
            if ( '' !== $holder_txt ) { update_post_meta( $post_id, '_cw_lite_holder', $holder_txt ); }
            if ( '' !== $meta_desc ) { update_post_meta( $post_id, '_cw_lite_meta_description', $meta_desc ); }

            // Assign tags from model output, auto-creating post tags.
            if ( '' !== $tags_list ) {
                $tags = array_filter( array_map( static function( $t ): string { return sanitize_text_field( trim( (string) $t ) ); }, explode( ',', $tags_list ) ), static function( $t ): bool { return '' !== $t; } );
                if ( ! empty( $tags ) ) {
                    wp_set_post_terms( $post_id, $tags, 'post_tag', false );
                    $tags_assigned++;
                }
            }

            // Featured image: Phase 1+ (CC0 optional). Options: none | default | site_icon | cc0
            $mode = is_array( $options ) && isset( $options['featured_image_mode'] ) ? (string) $options['featured_image_mode'] : 'none';
            $thumb_set = false;
            if ( 'default' === $mode ) {
                $aid = isset( $options['default_image_id'] ) ? (int) $options['default_image_id'] : 0;
                if ( $aid > 0 && wp_attachment_is_image( $aid ) ) {
                    set_post_thumbnail( $post_id, $aid );
                    $images_set++;
                    $thumb_set = true;
                } else {
                    // Fallback to site icon if present.
                    $site_icon_id = (int) get_option( 'site_icon' );
                    if ( $site_icon_id > 0 && wp_attachment_is_image( $site_icon_id ) ) {
                        set_post_thumbnail( $post_id, $site_icon_id );
                        $images_set++;
                        $thumb_set = true;
                    }
                }
            } elseif ( 'site_icon' === $mode ) {
                $site_icon_id = (int) get_option( 'site_icon' );
                if ( $site_icon_id > 0 && wp_attachment_is_image( $site_icon_id ) ) {
                    set_post_thumbnail( $post_id, $site_icon_id );
                    $images_set++;
                    $thumb_set = true;
                }
            } elseif ( 'cc0' === $mode ) {
                // Try Openverse CC0 using multiple candidate queries (Simplicity First):
                // 1) Tags phrase (first 2–3 tags)  2) Meta Description  3) TheWrap  4) Holder
                // 5) New Title  6) Original Title. Each call also tries a simplified-keyword fallback internally.
                $post_title_for_search = get_the_title( $post_id );
                $cc0_searches++;

                // Build query candidates from already-parsed sections.
                $candidates = [];
                if ( isset( $image_query ) && is_string( $image_query ) && '' !== trim( $image_query ) ) {
                    // 1) Try each comma-separated term individually first (max 3)
                    $img_terms_raw = array_slice( array_map( static function( $s ): string { return trim( (string) $s ); }, explode( ',', (string) $image_query ) ), 0, 3 );
                    foreach ( $img_terms_raw as $term ) {
                        if ( '' === $term ) { continue; }
                        // Normalize internal whitespace
                        $term_norm = preg_replace( '/\s+/', ' ', $term );
                        if ( is_string( $term_norm ) && '' !== trim( $term_norm ) ) {
                            $candidates[] = (string) $term_norm;
                        }
                    }
                    // 2) Then try the combined phrase (commas -> spaces)
                    $combined = preg_replace( '/\s*,\s*/', ' ', (string) $image_query );
                    $combined = is_string( $combined ) ? preg_replace( '/\s+/', ' ', trim( (string) $combined ) ) : '';
                    if ( is_string( $combined ) && '' !== $combined ) {
                        $candidates[] = (string) $combined;
                    }
                }
                // Tags: join first 3 tags into a phrase
                if ( isset( $tags_list ) && is_string( $tags_list ) && '' !== trim( $tags_list ) ) {
                    $tparts = array_filter( array_map( static function( $t ): string { return sanitize_text_field( trim( (string) $t ) ); }, explode( ',', $tags_list ) ), static function( $t ): bool { return '' !== $t; } );
                    if ( ! empty( $tparts ) ) {
                        $candidates[] = implode( ' ', array_slice( $tparts, 0, 3 ) );
                    }
                }
                if ( isset( $meta_desc ) && is_string( $meta_desc ) && '' !== trim( $meta_desc ) ) { $candidates[] = $meta_desc; }
                if ( isset( $wrap_txt ) && is_string( $wrap_txt ) && '' !== trim( $wrap_txt ) ) { $candidates[] = $wrap_txt; }
                if ( isset( $holder_txt ) && is_string( $holder_txt ) && '' !== trim( $holder_txt ) ) { $candidates[] = $holder_txt; }
                if ( isset( $new_title ) && is_string( $new_title ) && '' !== trim( $new_title ) ) { $candidates[] = $new_title; }
                if ( is_string( $post_title_for_search ) && '' !== trim( $post_title_for_search ) ) { $candidates[] = $post_title_for_search; }

                // Aggregate attempt URLs/status across all candidate queries for UI debug.
                $agg_attempts = [];
                $results_cc0 = [];
                $used_candidate = '';
                foreach ( $candidates as $cand ) {
                    $results_cc0 = contentweaver_ai_lite_openverse_search_cc0( (string) $cand, 5 );
                    if ( isset( $GLOBALS['cw_lite_last_cc0_debug'] ) && is_array( $GLOBALS['cw_lite_last_cc0_debug'] ) && isset( $GLOBALS['cw_lite_last_cc0_debug']['attempts'] ) && is_array( $GLOBALS['cw_lite_last_cc0_debug']['attempts'] ) ) {
                        $agg_attempts = array_merge( $agg_attempts, $GLOBALS['cw_lite_last_cc0_debug']['attempts'] );
                    }
                    if ( ! empty( $results_cc0 ) ) {
                        $used_candidate = (string) $cand;
                        break; // Use first candidate that yields results
                    }
                }
                // Update last debug with aggregated attempts for display
                if ( isset( $GLOBALS['cw_lite_last_cc0_debug'] ) && is_array( $GLOBALS['cw_lite_last_cc0_debug'] ) ) {
                    $GLOBALS['cw_lite_last_cc0_debug']['attempts'] = $agg_attempts;
                }

                $selected_image_url = '';
                if ( ! empty( $results_cc0 ) ) {
                    $tried = 0;
                    foreach ( $results_cc0 as $res_cc0 ) {
                        $tried++;
                        $att_id = contentweaver_ai_lite_download_image_to_media( isset( $res_cc0['url'] ) ? (string) $res_cc0['url'] : '', $post_id );
                        if ( $att_id > 0 && wp_attachment_is_image( $att_id ) ) {
                            set_post_thumbnail( $post_id, $att_id );
                            // Store minimal provenance.
                            if ( is_array( $res_cc0 ) ) {
                                if ( isset( $res_cc0['landing'] ) ) { update_post_meta( $post_id, '_cw_image_source_url', sanitize_text_field( (string) $res_cc0['landing'] ) ); }
                                if ( isset( $res_cc0['creator'] ) ) { update_post_meta( $post_id, '_cw_image_creator', sanitize_text_field( (string) $res_cc0['creator'] ) ); }
                                if ( isset( $res_cc0['title'] ) ) { update_post_meta( $post_id, '_cw_image_title', sanitize_text_field( (string) $res_cc0['title'] ) ); }
                            }
                            update_post_meta( $post_id, '_cw_image_source', 'openverse' );
                            update_post_meta( $post_id, '_cw_image_license', 'CC0' );
                            $selected_image_url = isset( $res_cc0['url'] ) && is_string( $res_cc0['url'] ) ? (string) $res_cc0['url'] : '';
                            $images_set++;
                            $cc0_downloads++;
                            $thumb_set = true;
                            break;
                        }
                        if ( $tried >= 3 ) { break; }
                    }
                    if ( ! $thumb_set ) { $cc0_failures++; }
                } else {
                    $cc0_failures++;
                }

                // Record per-item CC0 debug info for UI visibility.
                $cc0_item_debug[] = [
                    'post_id'        => (int) $post_id,
                    'title'          => (string) get_the_title( $post_id ),
                    'image_query'    => (string) $image_query,
                    'attempts'       => $agg_attempts,
                    'used_candidate' => (string) $used_candidate,
                    'selected_image' => (string) $selected_image_url,
                ];
                if ( ! $thumb_set ) {
                    // Fallback chain: Default -> Site Icon.
                    $aid = isset( $options['default_image_id'] ) ? (int) $options['default_image_id'] : 0;
                    if ( $aid > 0 && wp_attachment_is_image( $aid ) ) {
                        set_post_thumbnail( $post_id, $aid );
                        $images_set++;
                        $thumb_set = true;
                    } else {
                        $site_icon_id = (int) get_option( 'site_icon' );
                        if ( $site_icon_id > 0 && wp_attachment_is_image( $site_icon_id ) ) {
                            set_post_thumbnail( $post_id, $site_icon_id );
                            $images_set++;
                            $thumb_set = true;
                        }
                    }
                }
            } else {
                // 'none' => no featured image.
            }
            // Respect hard cap according to Items Per Run setting.
            if ( $imported >= $limit ) {
                break;
            }
        }
    }
    // Unset global flag to avoid affecting other requests post-run.
    if ( isset( $GLOBALS['cw_lite_fetching_feed'] ) ) {
        unset( $GLOBALS['cw_lite_fetching_feed'] );
    }

    $debug_suffix = sprintf( ' Requested:%d posts, Voice:%s | Parsed:%d Dups:%d Mode:%s%s%s Status:%s Type:%s Fetched:%d Rewrote:%d AI:%s Titles:%d Images:%d Tags:%d First:%s',
        (int) $requested_n,
        esc_html( $voice_label ),
        (int) ( is_array( $items ) ? count( $items ) : 0 ),
        (int) $duplicates_skipped,
        $used_alt_url ? 'alt' : 'main',
        $used_raw_fallback ? '+raw' : '',
        $discovered_alt_url ? '+disc' : '',
        $http_status ? (string) $http_status : '-',
        $content_type ? esc_html( substr( $content_type, 0, 40 ) ) : '-',
        (int) $fetched_full_pages,
        (int) $rewrites_done,
        $ai_enabled ? 'on' : 'off',
        (int) $titles_set,
        (int) $images_set,
        (int) $tags_assigned,
        esc_html( implode( ' | ', array_map( 'wp_strip_all_tags', $first_titles ) ) )
    );
    $msg = sprintf( /* translators: %d: number imported */ __( 'Imported %d item(s) as drafts.', 'contentweaver-ai-lite' ), $imported ) . $debug_suffix;

    // Build a structured result for AJAX or for pretty rendering later.
    // Expose lightweight Openverse debug (last request only) for verification in UI.
    $cc0_dbg = [];
    if ( isset( $GLOBALS['cw_lite_last_cc0_debug'] ) && is_array( $GLOBALS['cw_lite_last_cc0_debug'] ) ) {
        $cc0_dbg = [
            'url'    => isset( $GLOBALS['cw_lite_last_cc0_debug']['url'] ) ? (string) $GLOBALS['cw_lite_last_cc0_debug']['url'] : '',
            'status' => isset( $GLOBALS['cw_lite_last_cc0_debug']['status'] ) ? (int) $GLOBALS['cw_lite_last_cc0_debug']['status'] : 0,
            'error'  => isset( $GLOBALS['cw_lite_last_cc0_debug']['error'] ) ? (string) $GLOBALS['cw_lite_last_cc0_debug']['error'] : '',
        ];
    }

    $result = [
        'message'           => $msg,
        'requested'         => (int) $requested_n,
        'voice'             => (string) $voice_label,
        'imported'          => (int) $imported,
        'parsed'            => (int) ( is_array( $items ) ? count( $items ) : 0 ),
        'duplicates'        => (int) $duplicates_skipped,
        'fetched_full'      => (int) $fetched_full_pages,
        'rewrites'          => (int) $rewrites_done,
        'ai'                => (bool) $ai_enabled,
        'titles_set'        => (int) $titles_set,
        'images_set'        => (int) $images_set,
        'tags_assigned'     => (int) $tags_assigned,
        'cc0_searches'      => (int) $cc0_searches,
        'cc0_downloads'     => (int) $cc0_downloads,
        'cc0_failures'      => (int) $cc0_failures,
        'cc0_debug'         => $cc0_dbg,
        'cc0_item_debug'    => $cc0_item_debug,
        'first_titles'      => $first_titles,
        'http_status'       => (int) $http_status,
        'content_type'      => (string) $content_type,
        'mode'              => [ 'alt' => (bool) $used_alt_url, 'raw' => (bool) $used_raw_fallback, 'disc' => (bool) $discovered_alt_url ],
    ];

    if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
        wp_send_json_success( $result );
    } else {
        wp_safe_redirect( add_query_arg( 'cw_lite_notice', rawurlencode( $msg ), admin_url( 'admin.php?page=contentweaver-ai-lite-rss' ) ) );
        exit;
    }
}

// Register the RSS manual run handler.
add_action( 'admin_post_contentweaver_ai_lite_rss_run', 'contentweaver_ai_lite_handle_rss_run' );
add_action( 'wp_ajax_contentweaver_ai_lite_rss_run', 'contentweaver_ai_lite_handle_rss_run' );

function contentweaver_ai_lite_handle_manual_generation(): void {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Unauthorized.', 'contentweaver-ai-lite' ) );
	}

	check_admin_referer( 'contentweaver_ai_lite_manual_run' );

	$options = get_option( CONTENTWEAVER_AI_LITE_OPTION );
	// Sanitize early per WP.org guidelines.
    $prompt  = isset( $_POST['cw_lite_prompt'] ) && is_string( $_POST['cw_lite_prompt'] ) ? sanitize_textarea_field( wp_unslash( $_POST['cw_lite_prompt'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing

	$provider = is_array( $options ) && isset( $options['provider'] ) ? (string) $options['provider'] : 'openai';
	$api_key  = is_array( $options ) && isset( $options['api_key'] ) ? (string) $options['api_key'] : '';
	$base_url = is_array( $options ) && isset( $options['base_url'] ) ? (string) $options['base_url'] : 'https://api.openai.com/v1';
	$model    = is_array( $options ) && isset( $options['model'] ) ? (string) $options['model'] : 'gpt-4o-mini';

	if ( '' === trim( $prompt ) ) {
		wp_safe_redirect( add_query_arg( 'cw_lite_notice', rawurlencode( __( 'Please enter a prompt.', 'contentweaver-ai-lite' ) ), admin_url( 'admin.php?page=contentweaver-ai-lite-manual' ) ) );
		exit;
	}
	if ( '' === $api_key ) {
		wp_safe_redirect( add_query_arg( 'cw_lite_notice', rawurlencode( __( 'Please set your API key in Settings.', 'contentweaver-ai-lite' ) ), admin_url( 'admin.php?page=contentweaver-ai-lite' ) ) );
		exit;
	}

	$endpoint = untrailingslashit( $base_url ) . '/chat/completions';
	$body     = [
		'model'    => $model,
		'messages' => [
			[ 'role' => 'user', 'content' => $prompt ],
		],
	];

	$args = [
		'headers' => [
			'Content-Type'  => 'application/json',
			'Authorization' => 'Bearer ' . $api_key,
		],
		'timeout' => 30,
		'body'    => wp_json_encode( $body ),
	];

	$response = wp_remote_post( $endpoint, $args );
	$output   = '';

	if ( is_wp_error( $response ) ) {
		$output = sprintf( /* translators: %s: error message */ __( 'Request error: %s', 'contentweaver-ai-lite' ), $response->get_error_message() );
	} else {
		$status  = wp_remote_retrieve_response_code( $response );
		$raw     = wp_remote_retrieve_body( $response );
		$decoded = json_decode( $raw, true );

		if ( 200 === $status && is_array( $decoded ) && isset( $decoded['choices'][0]['message']['content'] ) && is_string( $decoded['choices'][0]['message']['content'] ) ) {
			$output = (string) $decoded['choices'][0]['message']['content'];
		} else {
			$detail = '';
			if ( is_array( $decoded ) && isset( $decoded['error']['message'] ) && is_string( $decoded['error']['message'] ) ) {
				$detail = (string) $decoded['error']['message'];
			}
			if ( '' === $detail && is_string( $raw ) ) {
				$detail = $raw;
			}
			$output = sprintf( /* translators: %1$d: status, %2$s: details */ __( 'API returned status %1$d. Details: %2$s', 'contentweaver-ai-lite' ), $status, wp_kses_post( $detail ) );
		}
	}

	$user_id = get_current_user_id();
	$key     = 'contentweaver_ai_lite_manual_output_' . (string) $user_id;
	set_transient( $key, $output, 5 * MINUTE_IN_SECONDS );

	wp_safe_redirect( add_query_arg( 'cw_lite_manual_output', '1', admin_url( 'admin.php?page=contentweaver-ai-lite-manual' ) ) );
	exit;
}
add_action( 'admin_post_contentweaver_ai_lite_manual_run', 'contentweaver_ai_lite_handle_manual_generation' );

/**
 * Add Settings link on the Plugins screen for Lite (only when Pro is not active).
 *
 * @param array<int,string> $links Existing action links.
 * @return array<int,string> Modified action links.
 */
function contentweaver_ai_lite_plugin_action_links( array $links ): array {
    if ( contentweaver_ai_lite_is_pro_active() ) {
        return $links;
    }

    $settings_url  = admin_url( 'admin.php?page=contentweaver-ai-lite' );
    $settings_link = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'contentweaver-ai-lite' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'contentweaver_ai_lite_plugin_action_links' );
