<?php
/**
 * Plugin Name: ContentPulse – Content Review Reminder
 * Plugin URI: https://wordpress.org/plugins/contentpulse/
 * Description: Notify users to update their old posts and articles with review scheduling and email notifications.
 * Version: 1.0.0
 * Author: innovatrixx
 * Author URI: 
 * Text Domain: contentpulse
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Network: true 
 *
 * ContentPulse is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * ContentPulse is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CONTENTPULSE_VERSION', '1.0.0');
define('CONTENTPULSE_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CONTENTPULSE_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('CONTENTPULSE_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('CONTENTPULSE_MINIMUM_WP_VERSION', '5.0');
define('CONTENTPULSE_MINIMUM_PHP_VERSION', '7.4');

/**
 * Main ContentPulse class
 */
class ContentPulse
{

    /**
     * Instance of this class
     *
     * @var ContentPulse
     */
    private static $instance;

    /**
     * Get the singleton instance
     *
     * @return ContentPulse
     */
    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        add_action('init', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Check minimum requirements
        add_action('admin_init', array($this, 'check_version'));
    }

    /**
     * Check WordPress and PHP version requirements
     */
    public function check_version()
    {
        // Check WordPress version
        if (!$this->compatible_version()) {
            if (is_plugin_active(CONTENTPULSE_PLUGIN_BASENAME)) {
                deactivate_plugins(CONTENTPULSE_PLUGIN_BASENAME);
                add_action('admin_notices', array($this, 'disabled_notice'));
                if (isset($_GET['activate'])) {
                    // This is a safe usage during plugin activation, no form processing occurs here.
                    unset($_GET['activate']);
                }
            }
        }
    }

    /**
     * Check if the environment is compatible
     *
     * @return bool
     */
    private function compatible_version()
    {
        if (version_compare($GLOBALS['wp_version'], CONTENTPULSE_MINIMUM_WP_VERSION, '<')) {
            return false;
        }

        if (version_compare(phpversion(), CONTENTPULSE_MINIMUM_PHP_VERSION, '<')) {
            return false;
        }

        return true;
    }

    /**
     * Display disabled notice
     */
    /* translators: 1: Required WordPress version, 2: Required PHP version */
    public function disabled_notice()
    {
        echo '<div class="notice notice-error"><p><strong>' . esc_html__('ContentPulse', 'contentpulse') . '</strong> ' .
            esc_html(sprintf(
                /* translators: 1: Required WordPress version, 2: Required PHP version */
                __('requires WordPress %1$s or higher and PHP %2$s or higher. Please upgrade and try again.', 'contentpulse'),
                esc_html(CONTENTPULSE_MINIMUM_WP_VERSION),
                esc_html(CONTENTPULSE_MINIMUM_PHP_VERSION)
            )) . '</p></div>';
    }

    /**
     * Initialize the plugin
     */
    public function init()
    {
        // Load text domain for translations
        load_plugin_textdomain('contentpulse', false, dirname(CONTENTPULSE_PLUGIN_BASENAME) . '/languages');

        // Add dashboard widget
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));

        // Add settings page
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));

        // Add meta box to post editor
        add_action('add_meta_boxes', array($this, 'add_review_meta_box'));
        add_action('save_post', array($this, 'save_review_meta'));

        // Schedule daily email notifications
        add_action('wp', array($this, 'schedule_daily_email'));
        add_action('contentpulse_daily_email', array($this, 'send_daily_email'));

        // Enqueue scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

        // Add settings link to plugins page
        add_filter('plugin_action_links_' . CONTENTPULSE_PLUGIN_BASENAME, array($this, 'add_settings_link'));
    }

    /**
     * Plugin activation
     */
    public function activate()
    {
        // Check version compatibility
        if (!$this->compatible_version()) {
            wp_die(
                esc_html(sprintf(
                    /* translators: 1: Required WordPress version, 2: Required PHP version */
                    __('ContentPulse requires WordPress %1$s or higher and PHP %2$s or higher.', 'contentpulse'),
                    esc_html(CONTENTPULSE_MINIMUM_WP_VERSION),
                    esc_html(CONTENTPULSE_MINIMUM_PHP_VERSION)
                )),
                esc_html__('Plugin Activation Error', 'contentpulse'),
                array('back_link' => true)
            );
        }

        // Schedule daily email check
        if (!wp_next_scheduled('contentpulse_daily_email')) {
            wp_schedule_event(time(), 'daily', 'contentpulse_daily_email');
        }

        // Set default options
        add_option('contentpulse_review_frequency', 30);
        add_option('contentpulse_email_notifications', '0');
        add_option('contentpulse_notification_email', get_option('admin_email'));
        add_option('contentpulse_version', CONTENTPULSE_VERSION);

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate()
    {
        wp_clear_scheduled_hook('contentpulse_daily_email');
        flush_rewrite_rules();
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_scripts($hook)
    {
        if (in_array($hook, array('post.php', 'post-new.php'), true)) {
            // Enqueue jQuery UI datepicker
            wp_enqueue_script('jquery-ui-datepicker');
            
            // Enqueue jQuery UI CSS for proper styling
            wp_enqueue_style(
                'jquery-ui-css',
                'https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css',
                array(),
                '1.13.2'
            );
            
            // Alternative: Use WordPress bundled jQuery UI CSS if available
            if (wp_style_is('jquery-ui-core', 'registered')) {
                wp_enqueue_style('jquery-ui-core');
            }

            wp_add_inline_script('jquery-ui-datepicker', '
                jQuery(document).ready(function($) {
                    try {
                        if ($("#contentpulse_next_review_date").length) {
                            // Destroy any existing datepicker to prevent conflicts
                            try {
                                $("#contentpulse_next_review_date").datepicker("destroy");
                            } catch(e) {
                                // Ignore errors if no datepicker exists
                            }
                            
                            // Initialize datepicker with enhanced options
                            $("#contentpulse_next_review_date").datepicker({
                                dateFormat: "yy-mm-dd",
                                minDate: 0,
                                showAnim: "fadeIn",
                                changeMonth: true,
                                changeYear: true,
                                yearRange: "-10:+10",
                                beforeShow: function(input, inst) {
                                    // Ensure the datepicker appears above other elements
                                    setTimeout(function() {
                                        if (inst.dpDiv) {
                                            inst.dpDiv.css({
                                                "z-index": "999999",
                                                "position": "absolute"
                                            });
                                        }
                                    }, 0);
                                },
                                onClose: function(dateText, inst) {
                                    // Ensure proper cleanup
                                    $(this).focus();
                                }
                            });
                            
                            // Add click handler as fallback
                            $("#contentpulse_next_review_date").on("click", function() {
                                if (!$(this).hasClass("hasDatepicker")) {
                                    $(this).datepicker("show");
                                }
                            });
                            
                            // Handle potential conflicts with other plugins
                            $(document).on("click", "#contentpulse_next_review_date", function(e) {
                                e.stopPropagation();
                            });
                        }
                    } catch(error) {
                        console.warn("ContentPulse datepicker initialization error:", error);
                    }
                });
                
                // Additional fallback for dynamic content
                jQuery(document).on("DOMNodeInserted", function(e) {
                    try {
                        if (e.target.id === "contentpulse_next_review_date" && !jQuery(e.target).hasClass("hasDatepicker")) {
                            setTimeout(function() {
                                if (jQuery("#contentpulse_next_review_date").length && !jQuery("#contentpulse_next_review_date").hasClass("hasDatepicker")) {
                                    jQuery("#contentpulse_next_review_date").datepicker({
                                        dateFormat: "yy-mm-dd",
                                        minDate: 0,
                                        showAnim: "fadeIn",
                                        changeMonth: true,
                                        changeYear: true,
                                        yearRange: "-10:+10"
                                    });
                                }
                            }, 100);
                        }
                    } catch(error) {
                        console.warn("ContentPulse dynamic datepicker error:", error);
                    }
                });
            ');
        }

        // Enqueue admin styles
        wp_enqueue_style(
            'contentpulse-admin-style',
            CONTENTPULSE_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            CONTENTPULSE_VERSION
        );
    }

    /**
     * Add settings link to plugins page
     *
     * @param array $links Existing links
     * @return array Modified links
     */
    public function add_settings_link($links)
    {
        $settings_link = '<a href="' . admin_url('options-general.php?page=contentpulse-settings') . '">' . esc_html__('Settings', 'contentpulse') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget()
    {
        wp_add_dashboard_widget(
            'contentpulse_widget',
            esc_html__('Content Review Needed', 'contentpulse'),
            array($this, 'dashboard_widget_content')
        );
    }

    /**
     * Dashboard widget content
     */
    public function dashboard_widget_content()
    {
        $posts_needing_review = $this->get_posts_needing_review();

        if (empty($posts_needing_review)) {
            echo '<p>' . esc_html__('All your content is up to date! 🎉', 'contentpulse') . '</p>';
            return;
        }

        echo '<div class="contentpulse-widget">';
        echo '<p><strong>' . sprintf(
            /* translators: %d: Number of posts that need review */
            esc_html(_n('%d post needs review:', '%d posts need review:', count($posts_needing_review), 'contentpulse')),
            count($posts_needing_review)
        ) . '</strong></p>';
        echo '<div style="max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;">';
        echo '<ul style="margin: 0; padding: 0; list-style: none;">';

        foreach ($posts_needing_review as $post) {
            $last_updated = get_the_modified_date('Y-m-d', $post->ID);
            $last_reviewed = get_post_meta($post->ID, '_contentpulse_last_reviewed', true);

            echo '<li style="margin-bottom: 15px; padding: 8px; background: #f9f9f9; border-left: 3px solid #dc3232;">';
            echo '<strong><a href="' . esc_url(get_edit_post_link($post->ID)) . '">' . esc_html($post->post_title) . '</a></strong><br>';
            echo '<small>';
            echo esc_html__('Last Updated:', 'contentpulse') . ' ' . ($last_updated ? esc_html($last_updated) : esc_html__('Never', 'contentpulse')) . '<br>';
            echo esc_html__('Last Reviewed:', 'contentpulse') . ' ' . ($last_reviewed ? esc_html($last_reviewed) : esc_html__('Never', 'contentpulse')) . '<br>';
            echo '<span style="color: #dc3232;">' . esc_html__('Status: Needs Review', 'contentpulse') . '</span>';
            echo '</small>';
            echo '</li>';
        }

        echo '</ul>';
        echo '</div>';
        echo '<p><a href="' . esc_url(admin_url('options-general.php?page=contentpulse-settings')) . '">' . esc_html__('Settings', 'contentpulse') . '</a></p>';
        echo '</div>';
    }

    /**
     * Get posts that need review
     *
     * @return array Array of post objects
     */
    public function get_posts_needing_review()
    {
        $review_frequency = absint(get_option('contentpulse_review_frequency', 30));
        $today = current_time('Y-m-d');

        // Get all published posts only
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_contentpulse_next_review_date',
                    'value' => $today,
                    'compare' => '<='
                ),
                array(
                    'key' => '_contentpulse_next_review_date',
                    'compare' => 'NOT EXISTS'
                )
            )
        );

        $query = new WP_Query($args);
        $posts_needing_review = array();

        foreach ($query->posts as $post) {
            $needs_review = false;

            // Check if post has a custom next review date
            $next_review_date = get_post_meta($post->ID, '_contentpulse_next_review_date', true);

            if ($next_review_date) {
                // If custom review date is today or past, it needs review
                if ($next_review_date <= $today) {
                    $needs_review = true;
                }
            } else {
                // Use frequency-based logic
                $last_reviewed = get_post_meta($post->ID, '_contentpulse_last_reviewed', true);

                if ($last_reviewed) {
                    // Check if it's been reviewed and enough time has passed since last review
                    $review_due_date = gmdate('Y-m-d', strtotime($last_reviewed . ' + ' . $review_frequency . ' days'));
                    if ($today >= $review_due_date) {
                        $needs_review = true;
                    }
                } else {
                    // Never been reviewed, check based on last modified date
                    $last_modified = get_the_modified_date('Y-m-d', $post->ID);
                    if ($last_modified) {
                        $review_due_date = gmdate('Y-m-d', strtotime($last_modified . ' + ' . $review_frequency . ' days'));
                        if ($today >= $review_due_date) {
                            $needs_review = true;
                        }
                    }
                }
            }

            if ($needs_review) {
                $posts_needing_review[] = $post;
            }
        }

        wp_reset_postdata();
        return $posts_needing_review;
    }

    /**
     * Add settings page
     */
    public function add_settings_page()
    {
        add_options_page(
            esc_html__('ContentPulse Settings', 'contentpulse'),
            esc_html__('ContentPulse', 'contentpulse'),
            'manage_options',
            'contentpulse-settings',
            array($this, 'settings_page_content')
        );
    }

    /**
     * Register settings
     */
    public function register_settings()
    {
        register_setting('contentpulse_settings', 'contentpulse_review_frequency', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 30
        ));
        register_setting('contentpulse_settings', 'contentpulse_email_notifications', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => '0'
        ));
        register_setting('contentpulse_settings', 'contentpulse_notification_email', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => get_option('admin_email')
        ));

        add_settings_section(
            'contentpulse_main_section',
            esc_html__('Review Settings', 'contentpulse'),
            array($this, 'settings_section_callback'),
            'contentpulse_settings'
        );

        add_settings_field(
            'review_frequency',
            esc_html__('Review Frequency (days)', 'contentpulse'),
            array($this, 'review_frequency_callback'),
            'contentpulse_settings',
            'contentpulse_main_section'
        );

        add_settings_field(
            'email_notifications',
            esc_html__('Enable Email Notifications', 'contentpulse'),
            array($this, 'email_notifications_callback'),
            'contentpulse_settings',
            'contentpulse_main_section'
        );

        add_settings_field(
            'notification_email',
            esc_html__('Notification Email', 'contentpulse'),
            array($this, 'notification_email_callback'),
            'contentpulse_settings',
            'contentpulse_main_section'
        );
    }

    /**
     * Sanitize checkbox value
     *
     * @param mixed $value Input value
     * @return string
     */
    public function sanitize_checkbox($value)
    {
        return ('1' === $value) ? '1' : '0';
    }

    /**
     * Settings section callback
     */
    public function settings_section_callback()
    {
        echo '<p>' . esc_html__('Configure how ContentPulse monitors your content for reviews.', 'contentpulse') . '</p>';
    }

    /**
     * Review frequency field callback
     */
    public function review_frequency_callback()
    {
        $value = absint(get_option('contentpulse_review_frequency', 30));
        echo '<input type="number" name="contentpulse_review_frequency" value="' . esc_attr($value) . '" min="1" max="365" />';
        echo '<p class="description">' . esc_html__('Content older than this many days will be marked for review.', 'contentpulse') . '</p>';
    }

    /**
     * Email notifications field callback
     */
    public function email_notifications_callback()
    {
        $value = get_option('contentpulse_email_notifications', '0');
        echo '<input type="checkbox" name="contentpulse_email_notifications" value="1" ' . checked($value, '1', false) . ' />';
        echo '<label>' . esc_html__('Send daily email notifications for content needing review', 'contentpulse') . '</label>';
    }

    /**
     * Notification email field callback
     */
    public function notification_email_callback()
    {
        $value = sanitize_email(get_option('contentpulse_notification_email', get_option('admin_email')));
        echo '<input type="email" name="contentpulse_notification_email" value="' . esc_attr($value) . '" class="regular-text" required />';
        echo '<p class="description">' . esc_html__('Email address to receive notifications.', 'contentpulse') . '</p>';
    }

    /**
     * Settings page content
     */
    public function settings_page_content()
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'contentpulse'));
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('contentpulse_settings');
                do_settings_sections('contentpulse_settings');
                submit_button();
                ?>
            </form>

            <div class="card">
                <h2><?php esc_html_e('Content Review Statistics', 'contentpulse'); ?></h2>
                <?php
                $posts_needing_review = $this->get_posts_needing_review();
                $total_posts = $this->get_total_published_content();
                ?>
                <p><strong><?php esc_html_e('Posts needing review:', 'contentpulse'); ?></strong>
                    <?php echo absint(count($posts_needing_review)); ?></p>
                <p><strong><?php esc_html_e('Total published content:', 'contentpulse'); ?></strong>
                    <?php echo absint($total_posts); ?></p>
                <p><strong><?php esc_html_e('Content up to date:', 'contentpulse'); ?></strong>
                    <?php echo absint($total_posts - count($posts_needing_review)); ?></p>


            </div>
        </div>
        <?php
    }

    /**
     * Add review meta box
     */
    public function add_review_meta_box()
    {
        add_meta_box(
            'contentpulse_review_box',
            esc_html__('Content Review', 'contentpulse'),
            array($this, 'review_meta_box_content'),
            'post',
            'side',
            'high'
        );
    }

    /**
     * Review meta box content
     *
     * @param WP_Post $post Current post object
     */
    public function review_meta_box_content($post)
    {
        wp_nonce_field('contentpulse_review_nonce', 'contentpulse_review_nonce');

        $last_reviewed = get_post_meta($post->ID, '_contentpulse_last_reviewed', true);
        $next_review_date = get_post_meta($post->ID, '_contentpulse_next_review_date', true);

        echo '<div class="contentpulse-meta-box">';

        if ($last_reviewed) {
            echo '<p><strong>' . esc_html__('Last Reviewed:', 'contentpulse') . '</strong> ' . esc_html($last_reviewed) . '</p>';
        }

        echo '<p>';
        echo '<label>';
        echo '<input type="checkbox" name="contentpulse_mark_reviewed" value="1" /> ';
        echo esc_html__('Mark as Reviewed', 'contentpulse');
        echo '</label>';
        echo '</p>';

        echo '<div style="margin-top: 15px; margin-bottom: 15px;">';
        echo '<label for="contentpulse_next_review_date"><strong>' . esc_html__('Next Review Date (Optional):', 'contentpulse') . '</strong></label><br>';
        echo '<input type="text" id="contentpulse_next_review_date" name="contentpulse_next_review_date" value="' . esc_attr($next_review_date) . '" placeholder="YYYY-MM-DD" style="width: 100%; max-width: 200px;" />';
        echo '<p class="description">' . esc_html__('Leave blank to use default review frequency. Select a date to override the automatic scheduling.', 'contentpulse') . '</p>';
        echo '</div>';

        echo '</div>';
    }

    /**
     * Save review meta data
     *
     * @param int $post_id Post ID
     */
    public function save_review_meta($post_id)
    {
        if (
            !isset($_POST['contentpulse_review_nonce']) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['contentpulse_review_nonce'])), 'contentpulse_review_nonce')
        ) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Handle next review date first
        if (isset($_POST['contentpulse_next_review_date'])) {
            $next_review_date = sanitize_text_field(wp_unslash($_POST['contentpulse_next_review_date']));
            if (!empty($next_review_date) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $next_review_date)) {
                update_post_meta($post_id, '_contentpulse_next_review_date', $next_review_date);
            } else {
                delete_post_meta($post_id, '_contentpulse_next_review_date');
            }
        }

        // Handle "Mark as Reviewed" checkbox
        if (isset($_POST['contentpulse_mark_reviewed']) && '1' === $_POST['contentpulse_mark_reviewed']) {
            update_post_meta($post_id, '_contentpulse_last_reviewed', current_time('Y-m-d'));

            // If marked as reviewed and no custom date is set, remove any existing custom date
            if (empty($_POST['contentpulse_next_review_date'])) {
                delete_post_meta($post_id, '_contentpulse_next_review_date');
            }
        }
    }

    /**
     * Schedule daily email
     */
    public function schedule_daily_email()
    {
        if (!wp_next_scheduled('contentpulse_daily_email')) {
            wp_schedule_event(time(), 'daily', 'contentpulse_daily_email');
        }
    }

    /**
     * Get total published content count
     *
     * @return int
     */
    public function get_total_published_content()
    {
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );

        $query = new WP_Query($args);
        $count = $query->found_posts;
        wp_reset_postdata();
        return $count;
    }

    /**
     * Send daily email notification
     *
     * @param bool $force_send Force send email regardless of settings
     * @return bool
     */
    public function send_daily_email($force_send = false)
    {
        $email_enabled = get_option('contentpulse_email_notifications', '0');

        if (!$force_send && '1' !== $email_enabled) {
            return false;
        }

        $posts_needing_review = $this->get_posts_needing_review();

        if (empty($posts_needing_review)) {
            return false;
        }

        $email = sanitize_email(get_option('contentpulse_notification_email', get_option('admin_email')));
        $site_name = get_bloginfo('name');
        $subject = sprintf(
            /* translators: 1: Site name, 2: Number of posts that need attention */
            esc_html__('[%1$s] Content Review Reminder - %2$d posts need attention', 'contentpulse'),
            $site_name,
            count($posts_needing_review)
        );
        /* translators: 1: New line character, 2: Number of posts, 3: Site name */
        $message = sprintf(esc_html__('Hello,%1$s%1$sYou have %2$d posts that need review on %3$s:%1$s%1$s', 'contentpulse'), "\n", count($posts_needing_review), $site_name);

        foreach ($posts_needing_review as $post) {
            $last_updated = get_the_modified_date('Y-m-d', $post->ID);
            $last_reviewed = get_post_meta($post->ID, '_contentpulse_last_reviewed', true);

            $message .= sprintf(
                /* translators: 1: Post title, 2: New line character, 3: Edit link, 4: Last updated date, 5: Last reviewed date */
                esc_html__('• %1$s%2$s  Edit: %3$s%2$s  Last Updated: %4$s%2$s  Last Reviewed: %5$s%2$s  Status: Needs Review%2$s%2$s', 'contentpulse'),
                $post->post_title,
                "\n",
                get_edit_post_link($post->ID),
                $last_updated ? $last_updated : esc_html__('Never', 'contentpulse'),
                $last_reviewed ? $last_reviewed : esc_html__('Never', 'contentpulse')
            );
        }

        $message .= sprintf(
            /* translators: 1: New line character, 2: Dashboard URL */
            esc_html__('%1$sVisit your dashboard to review these posts: %2$s%1$s%1$sBest regards,%1$sContentPulse Plugin', 'contentpulse'),
            "\n",
            admin_url('index.php')
        );

        $headers = array('Content-Type: text/plain; charset=UTF-8');

        $mail_sent = wp_mail($email, $subject, $message, $headers);


        return $mail_sent;
    }
}

// Initialize the plugin
ContentPulse::get_instance();