<?php
/**
 * Plugin Name: Content Refresh Manager
 * Plugin URI: https://www.wpsupportspecialists.com/content-refresh-manager/
 * Description: Sends automated reminders to update specific content based on predefined criteria.
 * Version: 1.0.1
 * Author: WP Support Specialists
 * Author URI: https://www.wpsupportspecialists.com
 * Text Domain: content-refresh-manager
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class Content_Refresh_Manager {
    private $batch_size = 5;
    private $version = '1.1';

    public function __construct() {
        // Hook into WordPress
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp', array($this, 'schedule_reminders'));
        add_action('contrm_send_reminders', array($this, 'send_reminders'));
        add_action('wp_footer', array($this, 'add_powered_by_badge'));
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_init', array($this, 'add_content_age_column'));
        add_action('init', array($this, 'load_textdomain'));
        add_action('wp_ajax_contrm_send_manual_reminder', array($this, 'ajax_send_manual_reminder'));
        add_filter('bulk_actions-edit-post', array($this, 'register_bulk_update_action'));
        add_filter('handle_bulk_actions-edit-post', array($this, 'handle_bulk_update_action'), 10, 3);
    }

    public function load_textdomain() {
        load_plugin_textdomain('content-refresh-manager', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    public function add_content_age_column() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $post_types = get_option('contrm_post_types', array('post'));
        if (!is_array($post_types)) {
            $post_types = array($post_types);
        }

        foreach ($post_types as $post_type) {
            add_filter("manage_{$post_type}_posts_columns", array($this, 'add_age_column'));
            add_action("manage_{$post_type}_posts_custom_column", array($this, 'show_age_column'), 10, 2);
        }
    }

    public function add_admin_menu() {
      add_menu_page(
        esc_html__('Content Refresh Manager', 'content-refresh-manager'),
        esc_html__('Content Refresh', 'content-refresh-manager'),
        'manage_options',
        'crm-settings',
        array($this, 'settings_page'),
        	'dashicons-backup',  // You can change this icon
        	30  // Position in menu
        );
  }

  public function enqueue_admin_scripts() {
    $screen = get_current_screen();
    
    // Always enqueue dashicons
    wp_enqueue_style('dashicons');

    wp_enqueue_style(
        'contrm-admin-styles',
        plugin_dir_url(__FILE__) . 'assets/css/admin-styles.css',
        [],
        '1.0.0'
    );

    wp_enqueue_script(
        'contrm-admin-scripts',
        plugin_dir_url(__FILE__) . 'assets/js/admin-script.js',
        ['jquery'],
        '1.0.0',
        true
    );

    wp_localize_script(
        'contrm-admin-scripts',
        'ContrmScriptData',
        array(
            'sendingText' => esc_html__('Sending...', 'content-refresh-manager'),
            'nonce' => wp_create_nonce('contrm_manual_reminder'),
            'reminderText' => esc_html__('Reminder sent successfully!', 'content-refresh-manager'),
            'errorText' => esc_html__('Error sending reminder. Please try again.', 'content-refresh-manager'),
            'sendText' => esc_html__('Send Reminder Now', 'content-refresh-manager')
        )
    );
    
}

public function register_settings() {
    register_setting(
        'contrm_settings',
        'contrm_reminder_interval',
        array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_interval'),
            'default' => 30
        )
    );

    register_setting(
        'contrm_settings',
        'contrm_recipient_email',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => get_option('admin_email')
        )
    );

    register_setting(
        'contrm_settings',
        'contrm_show_powered_by',
        array(
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => array($this, 'sanitize_post_types'),
        )
    );

    register_setting(
        'contrm_settings',
        'contrm_post_types',
        array(
            'type' => 'array',
            'sanitize_callback' => array($this, 'sanitize_post_types'),
            'default' => array('post')
        )
    );

    register_setting(
        'contrm_settings',
        'contrm_categories',
        array(
            'type' => 'array',
            'sanitize_callback' => array($this, 'sanitize_terms'),
            'default' => array()
        )
    );

    register_setting(
        'contrm_settings',
        'contrm_tags',
        array(
            'type' => 'array',
            'sanitize_callback' => array($this, 'sanitize_terms'),
            'default' => array()
        )
    );

    register_setting(
        'contrm_settings',
        'contrm_email_format',
        array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_email_format'),
            'default' => 'html'
        )
    );
}

public function sanitize_boolean($value) {
    return filter_var($value, FILTER_VALIDATE_BOOLEAN);
}

public function sanitize_interval($value) {
    return absint($value) ?: 30;
}

public function sanitize_post_types($input) {
    if (!is_array($input)) {
        return array('post');
    }
    $valid_post_types = get_post_types(array('public' => true));
    return array_intersect($input, $valid_post_types);
}

public function sanitize_terms($input) {
    if (!is_array($input)) {
        return array();
    }
    return array_map('absint', $input);
}

public function sanitize_email_format($input) {
    return ($input === 'plain') ? 'plain' : 'html';
}

public function settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'content-refresh-manager'));
    }
    
    // Remove all notices and plugin messages
    remove_all_actions('admin_notices');
    remove_all_actions('all_admin_notices');
    remove_all_actions('elementor/admin/after_create_settings/elementor');

    // Get stats for header
    $outdated_count = $this->get_outdated_content_count();
    $total_monitored = $this->get_total_monitored_content();
    $next_reminder = wp_next_scheduled('contrm_send_reminders');
    ?>
    
    <div class="wrap crm-settings-wrap">
        <div class="crm-header">
            <div class="crm-title-area">
                <h1><?php esc_html_e('Content Refresh Manager from WP Support Specialists', 'content-refresh-manager'); ?></h1>
            </div>
            <div class="crm-header-actions">
                <a href="https://www.wpsupportspecialists.com/content-refresh-manager" 
                target="_blank" 
                class="crm-go-pro"><?php esc_html_e('Unlock Extra Features With Content Refresh Manager Pro', 'content-refresh-manager'); ?></a>
                <a href="https://www.wpsupportspecialists.com/" 
                target="_blank" 
                class="crm-logo-link">
                <img src="<?php echo esc_url(plugins_url('images/wp-support-specialists-logo-74x26.png', __FILE__)); ?>" 
                alt="WP Support Specialists" 
                class="crm-logo" 
                style="height: 26px; width: 74px;" />
            </a>
        </div>
    </div>

    <!-- Stats Banner -->
    <div class="crm-stats-banner">
        <div class="crm-stat-item">
            <span class="dashicons dashicons-admin-post"></span>
            <div class="crm-stat-content">
                <span class="crm-stat-number"><?php echo esc_html($total_monitored); ?></span>
                <span class="crm-stat-label"><?php esc_html_e('Monitored Items', 'content-refresh-manager'); ?></span>
            </div>
        </div>
        <div class="crm-stat-item">
            <span class="dashicons dashicons-warning"></span>
            <div class="crm-stat-content">
                <span class="crm-stat-number"><?php echo esc_html($outdated_count); ?></span>
                <span class="crm-stat-label"><?php esc_html_e('Outdated Items', 'content-refresh-manager'); ?></span>
            </div>
        </div>
        <div class="crm-stat-item">
            <span class="dashicons dashicons-calendar-alt"></span>
            <div class="crm-stat-content">
                <span class="crm-stat-number"><?php echo esc_html(human_time_diff(time(), $next_reminder)); ?></span>
                <span class="crm-stat-label"><?php esc_html_e('Until Next Reminder', 'content-refresh-manager'); ?></span>
            </div>
        </div>
    </div>

    <!-- Manual Reminder Card -->
    <div class="crm-card">
        <div class="crm-card-header">
            <span class="dashicons dashicons-email-alt"></span>
            <h2><?php esc_html_e('Manual Reminder', 'content-refresh-manager'); ?></h2>
        </div>
        <div class="crm-card-content">
            <p><?php esc_html_e('Send a reminder email immediately:', 'content-refresh-manager'); ?></p>
            <button type="button" id="cur-send-manual" class="button button-primary">
                <?php esc_html_e('Send Reminder Now', 'content-refresh-manager'); ?>
            </button>
            <span id="cur-manual-status"></span>
        </div>
    </div>

    <!-- Settings Form -->
    <form method="post" action="options.php">
        <?php
        settings_fields('contrm_settings');
        do_settings_sections('contrm_settings');
        ?>

        <!-- Email Settings Card -->
        <div class="crm-card">
            <div class="crm-card-header">
                <span class="dashicons dashicons-admin-settings"></span>
                <h2><?php esc_html_e('Email Settings', 'content-refresh-manager'); ?></h2>
            </div>
            <div class="crm-card-content">
                <div class="crm-field-row">
                    <label for="contrm_reminder_interval">
                        <?php esc_html_e('Reminder Interval (days)', 'content-refresh-manager'); ?>
                    </label>
                    <input type="number" 
                    id="contrm_reminder_interval"
                    name="contrm_reminder_interval" 
                    value="<?php echo esc_attr(get_option('contrm_reminder_interval', 30)); ?>"
                    min="1"
                    required />
                </div>

                <div class="crm-field-row">
                    <label for="contrm_recipient_email">
                        <?php esc_html_e('Recipient Email', 'content-refresh-manager'); ?>
                    </label>
                    <input type="email"
                    id="contrm_recipient_email"
                    name="contrm_recipient_email"
                    value="<?php echo esc_attr(get_option('contrm_recipient_email', get_option('admin_email'))); ?>"
                    required />
                </div>

                <div class="crm-field-row">
                    <label><?php esc_html_e('Email Format', 'content-refresh-manager'); ?></label>
                    <div class="crm-radio-group">
                        <label class="crm-radio">
                            <input type="radio" 
                            name="contrm_email_format" 
                            value="html" 
                            <?php checked(get_option('contrm_email_format', 'html'), 'html'); ?> />
                            <span><?php esc_html_e('HTML', 'content-refresh-manager'); ?></span>
                        </label>
                        <label class="crm-radio">
                            <input type="radio" 
                            name="contrm_email_format" 
                            value="plain" 
                            <?php checked(get_option('contrm_email_format', 'html'), 'plain'); ?> />
                            <span><?php esc_html_e('Plain Text', 'content-refresh-manager'); ?></span>
                        </label>
                    </div>
                </div>
            </div>
        </div>

        <!-- Content Settings Card -->
        <div class="crm-card">
            <div class="crm-card-header">
                <span class="dashicons dashicons-admin-post"></span>
                <h2><?php esc_html_e('Content Settings', 'content-refresh-manager'); ?></h2>
            </div>
            <div class="crm-card-content">
                <div class="crm-monitoring-grid">
                    <!-- Post Types Section -->
                    <div class="crm-monitoring-section">
                        <h3><?php esc_html_e('Post Types to Monitor', 'content-refresh-manager'); ?></h3>
                        <div class="crm-checkbox-group">
                            <?php
                            $post_types = get_post_types(array('public' => true), 'objects');
                            $selected_post_types = get_option('contrm_post_types', array('post'));
                            foreach ($post_types as $post_type) {
                                ?>
                                <label class="crm-checkbox">
                                    <input type="checkbox" 
                                    name="contrm_post_types[]" 
                                    value="<?php echo esc_attr($post_type->name); ?>" 
                                    <?php checked(in_array($post_type->name, $selected_post_types), true); ?> />
                                    <span><?php echo esc_html($post_type->label); ?></span>
                                </label>
                                <?php
                            }
                            ?>
                        </div>
                    </div>

                    <!-- Categories Section -->
                    <div class="crm-monitoring-section">
                        <h3><?php esc_html_e('Categories to Monitor', 'content-refresh-manager'); ?></h3>
                        <div class="crm-checkbox-group">
                            <?php
                            $categories = get_categories(array('hide_empty' => false));
                            $selected_categories = get_option('contrm_categories', array());
                            foreach ($categories as $category) {
                                ?>
                                <label class="crm-checkbox">
                                    <input type="checkbox" 
                                    name="contrm_categories[]" 
                                    value="<?php echo esc_attr($category->term_id); ?>" 
                                    <?php checked(in_array($category->term_id, $selected_categories), true); ?> />
                                    <span><?php echo esc_html($category->name); ?></span>
                                </label>
                                <?php
                            }
                            ?>
                        </div>
                    </div>

                    <!-- Tags Section -->
                    <div class="crm-monitoring-section">
                        <h3><?php esc_html_e('Tags to Monitor', 'content-refresh-manager'); ?></h3>
                        <div class="crm-checkbox-group">
                            <?php
                            $tags = get_tags(array('hide_empty' => false));
                            $selected_tags = get_option('contrm_tags', array());
                            foreach ($tags as $tag) {
                                ?>
                                <label class="crm-checkbox">
                                    <input type="checkbox" 
                                    name="contrm_tags[]" 
                                    value="<?php echo esc_attr($tag->term_id); ?>" 
                                    <?php checked(in_array($tag->term_id, $selected_tags), true); ?> />
                                    <span><?php echo esc_html($tag->name); ?></span>
                                </label>
                                <?php
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <!-- Display Settings Card -->
        <div class="crm-card">
            <div class="crm-card-header">
                <span class="dashicons dashicons-admin-appearance"></span>
                <h2><?php esc_html_e('Display Settings', 'content-refresh-manager'); ?></h2>
            </div>
            <div class="crm-card-content">
                <div class="crm-field-row">
                    <label class="crm-toggle">
                        <input type="checkbox" 
                        name="contrm_show_powered_by" 
                        value="1" 
                        <?php checked(get_option('contrm_show_powered_by', true)); ?> />
                        <span class="crm-toggle-slider"></span>
                        <span class="crm-toggle-label"><?php esc_html_e('Show "Powered by" badge', 'content-refresh-manager'); ?></span>
                    </label>
                </div>

            </div>               
        </div>

        <div class="crm-field-row" style="display: flex; align-items: center; gap: 10px;">
            <a href="mailto:?subject=<?php echo esc_attr__('Check out this WordPress plugin', 'content-refresh-manager'); ?>&body=<?php echo esc_attr__("I've been using the Content Refresh Manager plugin and it's great. You should check it out: ", 'content-refresh-manager'); ?>https%3A%2F%2Fwww.wpsupportspecialists.com%2Fplugins%2Fcontent-refresh-manager" class="button button-primary">
                <?php esc_html_e('Suggest to a Friend', 'content-refresh-manager'); ?>
            </a>
            <p style="margin: 0;"><?php esc_html_e('Enjoying the Content Refresh Manager plugin? Help your fellow website owners by recommending it!', 'content-refresh-manager'); ?></p>
        </div>

        <?php submit_button(null, 'primary', 'submit', false, array('id' => 'crm-save-settings')); ?>
    </form>      
</div>   
<?php
}

// Add these helper methods to your class
private function get_outdated_content_count() {
    $interval = absint(get_option('contrm_reminder_interval', 30));
    $post_types = $this->sanitize_post_types(get_option('contrm_post_types', array('post')));
	$categories = get_option('contrm_categories', array());
    $tags = get_option('contrm_tags', array());
    
    $args = array(
        'post_type' => $post_types,
        'posts_per_page' => -1,
        'date_query' => array(
            array(
                'column' => 'post_modified_gmt',
                'before' => $interval . ' days ago',
            ),
        ),
        'fields' => 'ids',
    );
	
	if (!empty($categories) || !empty($tags)) {
        $args['tax_query'] = array('relation' => 'OR');

        if (!empty($categories)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'category',
                'field' => 'term_id',
                'terms' => $categories,
            );
        }

        if (!empty($tags)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'post_tag',
                'field' => 'term_id',
                'terms' => $tags,
            );
        }
    }
    
    $query = new WP_Query($args);
    return $query->found_posts;
}

private function get_total_monitored_content() {
    $post_types = $this->sanitize_post_types(get_option('contrm_post_types', array('post')));
	$categories = get_option('contrm_categories', array());
    $tags = get_option('contrm_tags', array());
    
    $args = array(
        'post_type' => $post_types,
        'posts_per_page' => -1,
        'fields' => 'ids',
    );
	
	if (!empty($categories) || !empty($tags)) {
        $args['tax_query'] = array('relation' => 'OR');

        if (!empty($categories)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'category',
                'field' => 'term_id',
                'terms' => $categories,
            );
        }

        if (!empty($tags)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'post_tag',
                'field' => 'term_id',
                'terms' => $tags,
            );
        }
    }
    
    $query = new WP_Query($args);
    return $query->found_posts;
}

public function ajax_send_manual_reminder() {
    check_ajax_referer('contrm_manual_reminder', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error('insufficient_permissions');
    }

    $this->send_reminders();
    wp_send_json_success();
}

public function schedule_reminders() {
    if (!wp_next_scheduled('contrm_send_reminders')) {
        wp_schedule_event(time(), 'daily', 'contrm_send_reminders');
    }
}

public function send_reminders() {
    $interval = absint(get_option('contrm_reminder_interval', 30));
    $recipient = sanitize_email(get_option('contrm_recipient_email', get_option('admin_email')));
    $post_types = $this->sanitize_post_types(get_option('contrm_post_types', array('post')));
    $format = get_option('contrm_email_format', 'html');

        // Collect outdated posts by post type
    $outdated_by_type = array();
    foreach ($post_types as $post_type) {
        $args = array(
            'post_type' => $post_type,
            'posts_per_page' => -1,
            'date_query' => array(
                array(
                    'column' => 'post_modified_gmt',
                    'before' => $interval . ' days ago',
                ),
            ),
            'no_found_rows' => true,
        );

            // Add taxonomy queries if set
        $categories = get_option('contrm_categories', array());
        $tags = get_option('contrm_tags', array());

        if (!empty($categories) || !empty($tags)) {
            $args['tax_query'] = array('relation' => 'OR');

            if (!empty($categories)) {
                $args['tax_query'][] = array(
                    'taxonomy' => 'category',
                    'field' => 'term_id',
                    'terms' => $categories,
                );
            }

            if (!empty($tags)) {
                $args['tax_query'][] = array(
                    'taxonomy' => 'post_tag',
                    'field' => 'term_id',
                    'terms' => $tags,
                );
            }
        }

        $outdated_posts = new WP_Query($args);
        if ($outdated_posts->have_posts()) {
            $outdated_by_type[$post_type] = $outdated_posts->posts;
        }
        wp_reset_postdata();
    }

    if (!empty($outdated_by_type)) {
        $message = $format === 'html' ? 
        $this->generate_html_email($outdated_by_type, $interval) : 
        $this->generate_plain_email($outdated_by_type, $interval);

        $headers = $format === 'html' ? 
        array('Content-Type: text/html; charset=UTF-8') : 
        array('Content-Type: text/plain; charset=UTF-8');

        wp_mail($recipient, esc_html__('Your Content Update from Content Refresh Manager', 'content-refresh-manager'), $message, $headers);
    }
}

private function generate_html_email($outdated_by_type, $interval) {
    ob_start();
    ?>
    <!DOCTYPE html>
    <html lang="<?php echo esc_attr(get_locale()); ?>">
    <head>
        <meta charset="<?php echo esc_attr(get_bloginfo('charset')); ?>">
        <title><?php esc_html_e('Content Refresh Manager', 'content-refresh-manager'); ?></title>
    </head>
    <body style="font-family: sans-serif; line-height: 1.6; color: #333;">
        <h1 style="color: #23282d;"><?php esc_html_e('Content Refresh Manager from WP Support Specialists', 'content-refresh-manager'); ?></h1>
        <p>
            <?php
            /* translators: %d: number of days */ printf(
                esc_html__('The following content has not been updated in %d days or more:', 'content-refresh-manager'),
                esc_html(absint($interval))
            );
            ?>
        </p>
        <?php
        foreach ($outdated_by_type as $post_type => $posts) {
            $post_type_obj = get_post_type_object($post_type);
            ?>
            <h2 style="color: #0073aa; margin-top: 20px;"><?php echo esc_html($post_type_obj->labels->name); ?></h2>
            <table style="border-collapse: collapse; width: 100%; margin-bottom: 20px;">
                <tr>
                    <th style="border: 1px solid #ddd; padding: 8px; text-align: left; background-color: #f8f9fa;"><?php esc_html_e('Title', 'content-refresh-manager'); ?></th>
                    <th style="border: 1px solid #ddd; padding: 8px; text-align: left; background-color: #f8f9fa;"><?php esc_html_e('Published', 'content-refresh-manager'); ?></th>
                    <th style="border: 1px solid #ddd; padding: 8px; text-align: left; background-color: #f8f9fa;"><?php esc_html_e('Last Updated', 'content-refresh-manager'); ?></th>
                    <th style="border: 1px solid #ddd; padding: 8px; text-align: left; background-color: #f8f9fa;"><?php esc_html_e('Age', 'content-refresh-manager'); ?></th>
                </tr>
                <?php
                foreach ($posts as $post) {
                    $publish_date = get_the_date('U', $post->ID);
                    $modified_date = get_the_modified_date('U', $post->ID);
                    $current_time = current_time('timestamp');
                    $age = floor(($current_time - $modified_date) / DAY_IN_SECONDS);
                    $age_class = $age < 30 ? 'color: #4CAF50;' : ($age < 90 ? 'color: #FFC107' : 'color: #F44336');
                    ?>
                    <tr>
                        <td style="border: 1px solid #ddd; padding: 8px; text-align: left;"><a href="<?php echo esc_url(get_edit_post_link($post->ID)); ?>"><?php echo esc_html($post->post_title); ?></a></td>
                        <td style="border: 1px solid #ddd; padding: 8px; text-align: left;"><?php echo esc_html(get_the_date('Y-m-d', $post->ID)); ?></td>
                        <td style="border: 1px solid #ddd; padding: 8px; text-align: left;"><?php echo esc_html(get_the_modified_date('Y-m-d', $post->ID)); ?></td>
                        <td style="border: 1px solid #ddd; padding: 8px; text-align: left; <?php echo esc_attr($age_class); ?>" class=""><?php echo esc_html($age . ' ' . __('days', 'content-refresh-manager')); ?></td>
                    </tr>
                    <?php
                }
                ?>
            </table>
            <?php
        }
        ?>
        <p style="margin-top: 30px; border-top: 1px solid #ddd; padding-top: 20px; font-style: italic;">
         <?php esc_html_e('WP Support Specialists are here to help you with all your WordPress Support, Development, Design, Optimisation and Migration Needs - https://www.wpsupportspecialists.com', 'content-refresh-manager'); ?>
     </p>
 </body>
 </html>
 <?php
 return ob_get_clean();
}

private function generate_plain_email($outdated_by_type, $interval) {
    $message = sprintf(
        /* translators: %d: number of days */
        __("Content Refresh Manager\n\nThe following content hasn't been updated in %d days or more:\n\n", 'content-refresh-manager'),
        $interval
    );

    foreach ($outdated_by_type as $post_type => $posts) {
        $post_type_obj = get_post_type_object($post_type);
        $message .= "\n" . strtoupper($post_type_obj->labels->name) . "\n";
        $message .= str_repeat('-', strlen($post_type_obj->labels->name)) . "\n\n";

        foreach ($posts as $post) {
            $message .= sprintf(
                "%s\nPublished: %s\nLast Updated: %s\nEdit Link: %s\n\n",
                $post->post_title,
                get_the_date('Y-m-d', $post->ID),
                get_the_modified_date('Y-m-d', $post->ID),
                get_edit_post_link($post->ID)
            );
        }
    }

    return $message;
}

public function add_powered_by_badge() {
   if (get_option('contrm_show_powered_by', true)) {
       printf(
           '<div style="position: fixed; bottom: 10px; right: 10px; background: #f1f1f1; padding: 5px; border-radius: 5px; font-size: 12px;">%1$s <a href="%2$s" target="_blank">%3$s</a></div>',
           esc_html__('Powered by', 'content-refresh-manager'),
           esc_url('https://www.wpsupportspecialists.com'),
           esc_html__('WP Support Specialists', 'content-refresh-manager')
       );
   }
}

public function add_dashboard_widget() {
    wp_add_dashboard_widget(
        'contrm_dashboard_widget',
        esc_html__('Content Needing Updates', 'content-refresh-manager'),
        array($this, 'dashboard_widget_content')
    );
}

public function dashboard_widget_content() {
    $interval = absint(get_option('contrm_reminder_interval', 30));
    $post_types = $this->sanitize_post_types(get_option('contrm_post_types', array('post')));

    $args = array(
        'post_type' => $post_types,
        'posts_per_page' => 5,
        'date_query' => array(
            array(
                'column' => 'post_modified_gmt',
                'before' => $interval . ' days ago',
            ),
        ),
        'no_found_rows' => true,
    );

    $outdated_posts = new WP_Query($args);

    if ($outdated_posts->have_posts()) {
        echo '<ul>';
        while ($outdated_posts->have_posts()) {
            $outdated_posts->the_post();
            printf(
                '<li><a href="%1$s">%2$s</a> (%3$s, %4$s: %5$s)</li>',
                esc_url(get_edit_post_link()),
                esc_html(get_the_title()),
                esc_html(get_post_type_object(get_post_type())->labels->singular_name),
                esc_html__('Last updated', 'content-refresh-manager'),
                esc_html(get_the_modified_date())
            );
        }
        echo '</ul>';
    } else {
        echo '<p>' . esc_html__('No content currently needs updating. Great job!', 'content-refresh-manager') . '</p>';
    }

    wp_reset_postdata();
}

public function add_age_column($columns) {
    $columns['content_age'] = esc_html__('Content Age', 'content-refresh-manager');
    return $columns;
}

public function show_age_column($column_name, $post_id) {
    if ($column_name === 'content_age') {
        static $shown_for_posts = array();
        if (in_array($post_id, $shown_for_posts)) {
            return;
        }
        $shown_for_posts[] = $post_id;

        $publish_date = get_the_date('U', $post_id);
        $modified_date = get_the_modified_date('U', $post_id);
        $current_time = current_time('timestamp');

        $age_since_publish = floor(($current_time - $publish_date) / DAY_IN_SECONDS);
        $age_since_modified = floor(($current_time - $modified_date) / DAY_IN_SECONDS);

        $publish_class = $age_since_publish < 30 ? 'content-age-new' : ($age_since_publish < 90 ? 'content-age-medium' : 'content-age-old');
        $modified_class = $age_since_modified < 30 ? 'content-age-new' : ($age_since_modified < 90 ? 'content-age-medium' : 'content-age-old');

        printf(
            '<p><strong>%1$s:</strong> <span class="%2$s">%3$s %4$s</span><br><small>%5$s</small></p>',
            esc_html__('Published', 'content-refresh-manager'),
            esc_attr($publish_class),
            esc_html($age_since_publish),
            esc_html__('days ago', 'content-refresh-manager'),
            esc_html(gmdate('Y-m-d', $publish_date))
        );

        printf(
            '<p><strong>%1$s:</strong> <span class="%2$s">%3$s %4$s</span><br><small>%5$s</small></p>',
            esc_html__('Last updated', 'content-refresh-manager'),
            esc_attr($modified_class),
            esc_html($age_since_modified),
            esc_html__('days ago', 'content-refresh-manager'),
            esc_html(gmdate('Y-m-d', $modified_date))
        );
    }
}

public function register_bulk_update_action($bulk_actions) {
    $bulk_actions['mark_as_updated'] = esc_html__('Mark as Updated', 'content-refresh-manager');
    return $bulk_actions;
}

public function handle_bulk_update_action($redirect_to, $doaction, $post_ids) {
    if ($doaction !== 'mark_as_updated') {
        return $redirect_to;
    }

    $updated = 0;
    foreach ($post_ids as $post_id) {
        wp_update_post(array(
            'ID' => $post_id,
            'post_modified' => current_time('mysql'),
            'post_modified_gmt' => current_time('mysql', 1)
        ));
        $updated++;
    }

    return add_query_arg('bulk_updated_posts', $updated, $redirect_to);
}
}

new Content_Refresh_Manager();