<?php
/**
 * Plugin Name: Content Bridge for Commerce
 * Plugin URI: https://wordpress.org/plugins/content-bridge-for-commerce/
 * Description: Expose WordPress content to external commerce platforms via a REST API.
 * Version: 1.0.2
 * Author: Tomohiro Shiotani
 * Author URI: https://vow.design/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: content-bridge-for-commerce
 *
 * @package content-bridge-for-commerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_action('plugins_loaded', function () {
  load_plugin_textdomain(
    'content-bridge-for-commerce',
    false,
    dirname(plugin_basename(__FILE__)) . '/languages'
  );
});


class VOWCBFC_Plugin {

	const OPTION_KEY        = 'vowcbfc_settings';
	const CACHE_KEYS_OPTION = 'vowcbfc_cache_keys';

	// REST.
	const REST_NS    = 'content-bridge/v1';
	const REST_ROUTE = '/posts';

	// Cache.
	const CACHE_TTL = 300; // 5 minutes.

	/**
	 * Bootstrap.
	 */
	public static function init() {
		add_action( 'admin_menu', [ __CLASS__, 'admin_menu' ] );
		add_action( 'admin_init', [ __CLASS__, 'admin_init' ] );
		add_action( 'admin_enqueue_scripts', [ __CLASS__, 'admin_enqueue' ] );

		add_action( 'rest_api_init', [ __CLASS__, 'register_rest' ] );

		// Flush cache when settings change.
		add_action( 'update_option_' . self::OPTION_KEY, [ __CLASS__, 'flush_cache' ], 10, 0 );

		// Flush cache when posts/terms change (minimal set).
		add_action( 'save_post', [ __CLASS__, 'flush_cache_on_post_change' ], 10, 3 );
		add_action( 'deleted_post', [ __CLASS__, 'flush_cache' ], 10, 0 );
		add_action( 'trashed_post', [ __CLASS__, 'flush_cache' ], 10, 0 );
		add_action( 'set_object_terms', [ __CLASS__, 'flush_cache_on_terms_change' ], 10, 6 );
	}

	/**
	 * Defaults.
	 *
	 * @return array<string,mixed>
	 */
	public static function defaults() {
		return [
			'api_key'       => '',
			'default_count' => 4,
			'max_count'     => 12,
			'image_size'    => 'medium',
			'excerpt_words' => 22,
			'unwrap_jetpack'=> 0,
		];
	}

	/**
	 * Get settings (merged with defaults).
	 *
	 * @return array<string,mixed>
	 */
	public static function get_settings() {
		$defaults = self::defaults();
		$stored   = get_option( self::OPTION_KEY, [] );
		if ( ! is_array( $stored ) ) {
			$stored = [];
		}
		return array_merge( $defaults, $stored );
	}

	/**
	 * Admin menu.
	 */
	public static function admin_menu() {
		add_options_page(
			__( 'Content Bridge for Commerce', 'content-bridge-for-commerce' ),
			__( 'Content Bridge', 'content-bridge-for-commerce' ),
			'manage_options',
			'content-bridge-for-commerce',
			[ __CLASS__, 'render_settings_page' ]
		);
	}

	/**
	 * Register settings/fields.
	 */
	public static function admin_init() {
		register_setting(
			'vowcbfc_settings_group',
			self::OPTION_KEY,
			[
				'type'              => 'array',
				'sanitize_callback' => [ __CLASS__, 'sanitize_settings' ],
				'default'           => self::defaults(),
			]
		);

		add_settings_section(
			'vowcbfc_main_section',
			__( 'Settings', 'content-bridge-for-commerce' ),
			function () {
				echo '<p>' . esc_html__( 'Configure the REST API access and output settings.', 'content-bridge-for-commerce' ) . '</p>';
			},
			'content-bridge-for-commerce'
		);

		add_settings_field(
			'vowcbfc_api_key',
			__( 'API Key', 'content-bridge-for-commerce' ),
			[ __CLASS__, 'field_api_key' ],
			'content-bridge-for-commerce',
			'vowcbfc_main_section'
		);

		add_settings_field(
			'vowcbfc_default_count',
			__( 'Default number of posts', 'content-bridge-for-commerce' ),
			[ __CLASS__, 'field_number' ],
			'content-bridge-for-commerce',
			'vowcbfc_main_section',
			[
				'key' => 'default_count',
				'min' => 1,
				'max' => 50,
			]
		);

		add_settings_field(
			'vowcbfc_max_count',
			__( 'Maximum allowed posts per request', 'content-bridge-for-commerce' ),
			[ __CLASS__, 'field_number' ],
			'content-bridge-for-commerce',
			'vowcbfc_main_section',
			[
				'key' => 'max_count',
				'min' => 1,
				'max' => 100,
			]
		);

		add_settings_field(
			'vowcbfc_image_size',
			__( 'Image size', 'content-bridge-for-commerce' ),
			[ __CLASS__, 'field_select' ],
			'content-bridge-for-commerce',
			'vowcbfc_main_section',
			[
				'key'     => 'image_size',
				'options' => [
					'thumbnail' => __( 'Thumbnail', 'content-bridge-for-commerce' ),
					'medium'    => __( 'Medium', 'content-bridge-for-commerce' ),
					'large'     => __( 'Large', 'content-bridge-for-commerce' ),
					'full'      => __( 'Full', 'content-bridge-for-commerce' ),
				],
			]
		);

		add_settings_field(
			'vowcbfc_excerpt_words',
			__( 'Excerpt length (words)', 'content-bridge-for-commerce' ),
			[ __CLASS__, 'field_number' ],
			'content-bridge-for-commerce',
			'vowcbfc_main_section',
			[
				'key' => 'excerpt_words',
				'min' => 0,
				'max' => 200,
			]
		);

		add_settings_field(
			'vowcbfc_unwrap_jetpack',
			__( 'Unwrap Jetpack CDN URLs (i0.wp.com)', 'content-bridge-for-commerce' ),
			[ __CLASS__, 'field_checkbox' ],
			'content-bridge-for-commerce',
			'vowcbfc_main_section',
			[
				'key' => 'unwrap_jetpack',
			]
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param mixed $input Raw input.
	 * @return array<string,mixed>
	 */
	public static function sanitize_settings( $input ) {
		$defaults = self::defaults();
		$out      = [];

		if ( ! is_array( $input ) ) {
			return $defaults;
		}

		$out['api_key'] = isset( $input['api_key'] ) ? sanitize_text_field( (string) $input['api_key'] ) : '';

		$out['default_count'] = isset( $input['default_count'] ) ? absint( $input['default_count'] ) : (int) $defaults['default_count'];
		$out['max_count']     = isset( $input['max_count'] ) ? absint( $input['max_count'] ) : (int) $defaults['max_count'];

		$allowed_sizes = [ 'thumbnail', 'medium', 'large', 'full' ];
		$out['image_size'] = isset( $input['image_size'] ) && in_array( (string) $input['image_size'], $allowed_sizes, true )
			? (string) $input['image_size']
			: (string) $defaults['image_size'];

		$out['excerpt_words'] = isset( $input['excerpt_words'] ) ? absint( $input['excerpt_words'] ) : (int) $defaults['excerpt_words'];
		$out['unwrap_jetpack'] = ! empty( $input['unwrap_jetpack'] ) ? 1 : 0;

		// Clamp.
		if ( $out['default_count'] < 1 ) {
			$out['default_count'] = 1;
		}
		if ( $out['max_count'] < 1 ) {
			$out['max_count'] = 1;
		}
		if ( $out['default_count'] > $out['max_count'] ) {
			$out['default_count'] = $out['max_count'];
		}

		return $out;
	}

	/**
	 * Enqueue admin JS on our settings page.
	 *
	 * @param string $hook_suffix Admin page hook.
	 */
	public static function admin_enqueue( $hook_suffix ) {
		if ( 'settings_page_content-bridge-for-commerce' !== $hook_suffix ) {
			return;
		}

		$handle = 'vowcbfc-admin';
		wp_register_script( $handle, '', [], '0.2.1', true );
		wp_enqueue_script( $handle );

		$confirm = esc_js(
			__( 'Generate a new API key? Existing integrations will stop working until updated.', 'content-bridge-for-commerce' )
		);

		$inline =
			"(function(){\n" .
			"  function generateKey(len){\n" .
			"    var chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';\n" .
			"    var result = '';\n" .
			"    for (var i=0; i<len; i++){\n" .
			"      result += chars.charAt(Math.floor(Math.random() * chars.length));\n" .
			"    }\n" .
			"    return result;\n" .
			"  }\n\n" .
			"  document.addEventListener('click', function(e){\n" .
			"    var btn = e.target.closest('[data-vowcbfc-generate-key]');\n" .
			"    if(!btn) return;\n" .
			"    e.preventDefault();\n\n" .
			"    if(!confirm('" . $confirm . "')) return;\n\n" .
			"    var field = document.getElementById('vowcbfc-api-key-field');\n" .
			"    if(field){\n" .
			"      field.value = generateKey(32);\n" .
			"      field.focus();\n" .
			"    }\n" .
			"  });\n" .
			"})();";

		wp_add_inline_script( $handle, $inline );
	}

	/**
	 * Field: API key with generate button.
	 */
	public static function field_api_key() {
		$settings = self::get_settings();
		$value    = isset( $settings['api_key'] ) ? (string) $settings['api_key'] : '';

		printf(
			'<input type="text" class="regular-text" id="%1$s" name="%2$s" value="%3$s" autocomplete="off" />',
			esc_attr( 'vowcbfc-api-key-field' ),
			esc_attr( self::OPTION_KEY . '[api_key]' ),
			esc_attr( $value )
		);

		echo ' ';
		echo '<button type="button" class="button" data-vowcbfc-generate-key>';
		echo esc_html__( 'Generate API Key', 'content-bridge-for-commerce' );
		echo '</button>';

		echo '<p class="description">' . esc_html__( 'Used to authenticate requests via the X-Content-Bridge-Key header.', 'content-bridge-for-commerce' ) . '</p>';
	}

	/**
	 * Field: number.
	 *
	 * @param array $args Args.
	 */
	public static function field_number( $args ) {
		$key      = isset( $args['key'] ) ? (string) $args['key'] : '';
		$min      = isset( $args['min'] ) ? (int) $args['min'] : 0;
		$max      = isset( $args['max'] ) ? (int) $args['max'] : 0;
		$settings = self::get_settings();
		$value    = isset( $settings[ $key ] ) ? (int) $settings[ $key ] : 0;

		printf(
			'<input type="number" class="small-text" name="%1$s" value="%2$d" min="%3$d" max="%4$d" />',
			esc_attr( self::OPTION_KEY . '[' . $key . ']' ),
			(int) $value,
			(int) $min,
			(int) $max
		);
	}

	/**
	 * Field: checkbox.
	 *
	 * @param array $args Args.
	 */
	public static function field_checkbox( $args ) {
		$key      = isset( $args['key'] ) ? (string) $args['key'] : '';
		$settings = self::get_settings();
		$checked  = ! empty( $settings[ $key ] );

		printf(
			'<label><input type="checkbox" name="%1$s" value="1" %2$s /> %3$s</label>',
			esc_attr( self::OPTION_KEY . '[' . $key . ']' ),
			checked( $checked, true, false ),
			esc_html__( 'Enable', 'content-bridge-for-commerce' )
		);
	}

	/**
	 * Field: select.
	 *
	 * @param array $args Args.
	 */
	public static function field_select( $args ) {
		$key      = isset( $args['key'] ) ? (string) $args['key'] : '';
		$options  = isset( $args['options'] ) && is_array( $args['options'] ) ? $args['options'] : [];
		$settings = self::get_settings();
		$value    = isset( $settings[ $key ] ) ? (string) $settings[ $key ] : '';

		echo '<select name="' . esc_attr( self::OPTION_KEY . '[' . $key . ']' ) . '">';
		foreach ( $options as $k => $label ) {
			echo '<option value="' . esc_attr( $k ) . '" ' . selected( $value, (string) $k, false ) . '>' . esc_html( $label ) . '</option>';
		}
		echo '</select>';
	}

	/**
	 * Render settings page.
	 */
	public static function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$endpoint = esc_url( rest_url( self::REST_NS . self::REST_ROUTE ) );

		echo '<div class="wrap">';
		echo '<h1>' . esc_html__( 'Content Bridge for Commerce', 'content-bridge-for-commerce' ) . '</h1>';

		echo '<p>' . esc_html__( 'REST API Endpoint:', 'content-bridge-for-commerce' ) . ' ';
		echo '<code>' . esc_html( $endpoint ) . '</code></p>';

		echo '<p>' . esc_html__( 'Requests must include the header:', 'content-bridge-for-commerce' ) . ' ';
		echo '<code>' . esc_html( 'X-Content-Bridge-Key: YOUR_API_KEY' ) . '</code></p>';

		echo '<form action="options.php" method="post">';
		settings_fields( 'vowcbfc_settings_group' );
		do_settings_sections( 'content-bridge-for-commerce' );
		submit_button();
		echo '</form>';
		echo '</div>';
	}

	/**
	 * REST: Register routes.
	 */
	public static function register_rest() {
		register_rest_route(
			self::REST_NS,
			self::REST_ROUTE,
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ __CLASS__, 'rest_posts' ],
				'permission_callback' => [ __CLASS__, 'rest_permission' ],
				'args'                => [
					'per_page' => [
						'type'              => 'integer',
						'required'          => false,
						'sanitize_callback' => 'absint',
					],
					'tax' => [
						'type'              => 'string',
						'required'          => false,
						'sanitize_callback' => 'sanitize_key',
					],
					'term' => [
						'type'              => 'string',
						'required'          => false,
						'sanitize_callback' => 'sanitize_title',
					],
				],
			]
		);
	}

	/**
	 * REST permission: API key header check.
	 *
	 * @param WP_REST_Request $request Request.
	 * @return true|WP_Error
	 */
	public static function rest_permission( $request ) {
		$settings = self::get_settings();
		$api_key  = isset( $settings['api_key'] ) ? (string) $settings['api_key'] : '';

		// If no key is set, deny access (secure by default).
		if ( '' === $api_key ) {
			return new WP_Error(
				'vowcbfc_no_api_key',
				__( 'API key is not configured.', 'content-bridge-for-commerce' ),
				[ 'status' => 401 ]
			);
		}

		$provided = $request->get_header( 'x-content-bridge-key' );
		$provided = is_string( $provided ) ? $provided : '';

		if ( '' === $provided || ! hash_equals( $api_key, $provided ) ) {
			return new WP_Error(
				'vowcbfc_unauthorized',
				__( 'Unauthorized.', 'content-bridge-for-commerce' ),
				[ 'status' => 401 ]
			);
		}

		return true;
	}

	/**
	 * REST callback: posts list.
	 *
	 * @param WP_REST_Request $request Request.
	 * @return WP_REST_Response
	 */
	public static function rest_posts( $request ) {
		$settings = self::get_settings();

		$per_page = (int) $request->get_param( 'per_page' );
		if ( $per_page <= 0 ) {
			$per_page = (int) $settings['default_count'];
		}
		$per_page = min( $per_page, (int) $settings['max_count'] );

		$tax  = (string) $request->get_param( 'tax' );
		$term = (string) $request->get_param( 'term' );

		// Allow friendly "tag" alias.
		if ( 'tag' === $tax ) {
			$tax = 'post_tag';
		}

		// Only allow the MVP taxonomies.
		if ( $tax && ! in_array( $tax, [ 'category', 'post_tag' ], true ) ) {
			$tax = '';
			$term = '';
		}
		if ( ! $tax || ! $term ) {
			$tax  = '';
			$term = '';
		}

		$cache_key = self::build_cache_key(
			[
				'per_page'      => $per_page,
				'tax'           => $tax,
				'term'          => $term,
				'image_size'    => (string) $settings['image_size'],
				'excerpt_words' => (int) $settings['excerpt_words'],
				'unwrap_jetpack'=> (int) $settings['unwrap_jetpack'],
			]
		);

		$cached = get_transient( $cache_key );
		if ( is_array( $cached ) ) {
			return rest_ensure_response( $cached );
		}

		$args = [
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $per_page,
			'no_found_rows'       => true,
			'ignore_sticky_posts' => true,
		];

		if ( $tax && $term ) {
			$args['tax_query'] = [
				[
					'taxonomy' => $tax,
					'field'    => 'slug',
					'terms'    => $term,
				],
			];
		}

		$q = new WP_Query( $args );

		$items = [];
		if ( $q->have_posts() ) {
			foreach ( $q->posts as $post ) {
				$items[] = self::format_post_item( $post, $settings );
			}
		}

		$data = [
			'items' => $items,
		];

		set_transient( $cache_key, $data, self::CACHE_TTL );
		self::track_cache_key( $cache_key );

		return rest_ensure_response( $data );
	}

	/**
	 * Build cache key.
	 *
	 * @param array<string,mixed> $parts Parts.
	 * @return string
	 */
	public static function build_cache_key( $parts ) {
		return 'vowcbfc_posts_' . md5( wp_json_encode( $parts ) );
	}

	/**
	 * Track cache keys for cleanup (uninstall / flush).
	 *
	 * @param string $key Transient key.
	 */
	public static function track_cache_key( $key ) {
		$keys = get_option( self::CACHE_KEYS_OPTION, [] );
		if ( ! is_array( $keys ) ) {
			$keys = [];
		}
		$keys[ $key ] = time();
		update_option( self::CACHE_KEYS_OPTION, $keys, false );
	}

	/**
	 * Flush tracked transients.
	 */
	public static function flush_cache() {
		$keys = get_option( self::CACHE_KEYS_OPTION, [] );
		if ( is_array( $keys ) ) {
			foreach ( $keys as $k => $v ) {
				$name = is_string( $k ) ? $k : ( is_string( $v ) ? $v : '' );
				if ( $name ) {
					delete_transient( $name );
				}
			}
		}
		delete_option( self::CACHE_KEYS_OPTION );
	}

	/**
	 * Flush cache when a post is created/updated.
	 */
	public static function flush_cache_on_post_change( $post_id, $post, $update ) {
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		if ( wp_is_post_revision( $post_id ) ) {
			return;
		}
		if ( ! $post || 'post' !== $post->post_type ) {
			return;
		}
		self::flush_cache();
	}

	/**
	 * Flush cache when terms are changed for a post.
	 */
	public static function flush_cache_on_terms_change( $object_id, $terms, $tt_ids, $taxonomy, $append, $old_tt_ids ) {
		if ( ! in_array( $taxonomy, [ 'category', 'post_tag' ], true ) ) {
			return;
		}
		if ( 'post' !== get_post_type( $object_id ) ) {
			return;
		}
		self::flush_cache();
	}

	/**
	 * Format one post item.
	 *
	 * @param WP_Post $post Post.
	 * @param array<string,mixed> $settings Settings.
	 * @return array<string,mixed>
	 */
	public static function format_post_item( $post, $settings ) {
		$title = get_the_title( $post );
		$url   = get_permalink( $post );
		$date  = get_the_date( 'c', $post );

		$excerpt_words = isset( $settings['excerpt_words'] ) ? (int) $settings['excerpt_words'] : 0;
		$excerpt_raw   = has_excerpt( $post ) ? (string) $post->post_excerpt : (string) wp_strip_all_tags( $post->post_content );
		$excerpt_raw   = trim( preg_replace( '/\s+/', ' ', $excerpt_raw ) );
		$excerpt       = ( $excerpt_words > 0 ) ? wp_trim_words( $excerpt_raw, $excerpt_words, '…' ) : '';

		$image = '';
		$size  = isset( $settings['image_size'] ) ? (string) $settings['image_size'] : 'medium';
		if ( has_post_thumbnail( $post ) ) {
			$src = wp_get_attachment_image_src( get_post_thumbnail_id( $post ), $size );
			if ( is_array( $src ) && ! empty( $src[0] ) ) {
				$image = (string) $src[0];
			}
		}

		if ( $image && ! empty( $settings['unwrap_jetpack'] ) ) {
			$image = self::unwrap_wpcom_cdn_url( $image );
		}

		return [
			'title'   => $title,
			'url'     => $url,
			'date'    => $date,
			'excerpt' => $excerpt,
			'image'   => $image,
		];
	}

	/**
	 * Convert i0.wp.com URL back to original URL when possible.
	 *
	 * @param string $url URL.
	 * @return string
	 */
	public static function unwrap_wpcom_cdn_url( $url ) {
		$parts = wp_parse_url( $url );
		if ( empty( $parts['host'] ) || 'i0.wp.com' !== $parts['host'] ) {
			return $url;
		}

		$path = isset( $parts['path'] ) ? ltrim( (string) $parts['path'], '/' ) : '';
		if ( 0 === strpos( $path, 'http/' ) ) {
			$path = substr( $path, 5 );
			return 'http://' . $path;
		}
		if ( 0 === strpos( $path, 'https/' ) ) {
			$path = substr( $path, 6 );
			return 'https://' . $path;
		}

		return $url;
	}
	
}

VOWCBFC_Plugin::init();
