<?php
namespace Contact360;

class API {

    const VERSION = '1.3.0';

    const PANEL_URL = 'https://panel.stelsoft.pl';
    const PANEL_URL_DEV = 'https://dev.contact360.pl';

    const STATUS_OK = 'ok';

    const CART_TYPE_IAI_SHOP = 0;
    const CART_TYPE_SHOPER = 1;
    const CART_TYPE_PRESTASHOP = 2;
    const CART_TYPE_OTHER = 99;

    private $_clientID;
    private $_secret;
    private $_debug;
    private $_dev_mode;
    private $_endpoint;
    private $_call_results = array();

    /**
     * Entry class for Contact360 API communication.
     *
     * @param string $clientID - field "Client ID" from Contact360 panel
     * @param string $secret - field "API secret" from Contact360 panel
     * @param bool $debug - debug mode switch
     * @param bool $dev_mode - true switches to DEV endpoint url if $endpoint is not defined
     * @param string $endpoint - defines URL to use
     */
    public function __construct($clientID, $secret, $debug = false, $dev_mode = false, $endpoint = null)
    {
        $this->_clientID = $clientID;
        $this->_secret = $secret;
        $this->_debug = $debug;
        $this->_dev_mode = $dev_mode;
        $this->_endpoint = trim($endpoint!==null?$endpoint:($this->_dev_mode?self::PANEL_URL_DEV:self::PANEL_URL), '/'). '/api/';
    }

    /**
     * Insert contact form data
     *
     * @param String $name - form name
     * @param String $enquiry_content - full text content of enquiry
     * @param array $fields - form fields as associative array, supported keys: "email", "phone", "name" or ("firstname" and "lastname")
     * @return bool
     */
    public function insertContactForm($name, $enquiry_content, $fields)
    {
        $clientId = $this->_clientID;
        $requestTime = time();

        $data = array(
            'id' => $clientId,
            'requestTime' => $requestTime,
            'checksum' => sha1($requestTime . $clientId . $this->_secret),
            'name' => $name,
            'website' => isset($_SERVER['SERVER_NAME'])?$_SERVER['SERVER_NAME']:'',
            'email_contents' => $enquiry_content,
            'fields' => $fields,
            'source' => self::getTrafficSource()
        );

        $json = json_encode($data);

        $result = $this->_doPostRequest('form', $json);
        $r_json = @json_decode($result);
        if($r_json && $r_json->status === self::STATUS_OK){
            return true;
        }
        return false;
    }

    /**
     * Insert cart
     *
     * @param integer $type One of self::CART_TYPE_
     * @param float $sum Total cart value
     * @param string $email Email of the user
     * @param string $phone Phone number of the user
     * @param string $order_id Order id generated by the cart
     * @param array $additional_fields Additional fields
     * @param string $currency_name Cart currency
     * @param string|null $ip_address IP address of the user
     * @return bool
     */
    public function insertCart($type, $sum, $email, $phone, $order_id, $additional_fields = array(), $currency_name = 'PLN', $ip_address = null)
    {
        $clientId = $this->_clientID;
        $requestTime = time();

        $data = array(
            'source' => self::getTrafficSource(),
            'id' => $clientId,
            'requestTime' => $requestTime,
            'checksum' => sha1($requestTime . $clientId . $this->_secret),
            'cart_type' => $type,
            'domain' => isset($_SERVER['SERVER_NAME'])?$_SERVER['SERVER_NAME']:'',
            'fields' => $additional_fields,
            'sum' => $sum,
            'email' => $email,
            'ip_address' => $ip_address,
            'currency_name' => $currency_name,
            'phone' => $phone,
            'order_id' => $order_id,
        );

        $json = json_encode($data);

        $result = $this->_doPostRequest('cart', $json);
        $r_json = @json_decode($result);
        if($r_json && $r_json->status === self::STATUS_OK){
            return true;
        }
        return false;
    }

    private function _doPostRequest($method, $data)
    {
        $url = $this->_endpoint . $method;

        //@TODO CURL alternatives?
        $ch = curl_init($url);

        if($this->_debug){
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_VERBOSE, true);
            $verbose = fopen('php://temp', 'w+');
            curl_setopt($ch, CURLOPT_STDERR, $verbose);
        }

        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER,
            array(
                'Content-Type: application/json',
                'Content-Length: ' . strlen($data)
            )
        );
        $result = curl_exec($ch);

        if ($this->_debug && $result === FALSE) {
            printf("cUrl error (#%d): %s<br>\n", curl_errno($ch), htmlspecialchars(curl_error($ch)));
            rewind($verbose);
            $verboseLog = stream_get_contents($verbose);
            echo "Verbose information:\n<pre>", htmlspecialchars($verboseLog), "</pre>\n";
        }
        if($this->_debug){
            $this->_call_results[] = $result;
        }
        return $result;
    }

    public function getCallResults()
    {
        return $this->_call_results;
    }

    public static function getTrafficSource()
    {
        $traffic_source = !empty($_COOKIE['traffic_source'])?$_COOKIE['traffic_source']:'';
        $exp = explode('>>', $traffic_source);
        return count($exp)&&$exp[0]?$exp[0]:'none or direct';
    }

}