<?php
/**
 * Utility functions for ChimpMatic.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class Cmatic_Utils
 *
 * Enterprise-level utility functions for ChimpMatic plugins.
 * Inspired by WPSEO_Utils from Yoast SEO.
 *
 * @since 0.9.69
 */
final class Cmatic_Utils {

	const CMATIC_FB_A = 'chimpmatic';

	/**
	 * Validate a value as boolean.
	 *
	 * Handles multiple input types and normalizes to bool:
	 * - bool: returned as-is
	 * - int/float: 0 = false, non-zero = true
	 * - string: 'true', 'on', 'yes', 'y', '1' = true
	 *           'false', 'off', 'no', 'n', '0', '' = false
	 *
	 * @since 0.9.69
	 *
	 * @param mixed $value The value to validate.
	 * @return bool The validated boolean value.
	 */
	public static function validate_bool( $value ): bool {
		if ( is_bool( $value ) ) {
			return $value;
		}

		if ( is_int( $value ) || is_float( $value ) ) {
			return (bool) $value;
		}

		if ( is_string( $value ) ) {
			$v = strtolower( trim( $value ) );

			if ( '' === $v ) {
				return false;
			}

			if ( is_numeric( $v ) ) {
				return 0.0 !== (float) $v;
			}

			static $true  = array( 'true', 'on', 'yes', 'y', '1' );
			static $false = array( 'false', 'off', 'no', 'n', '0' );

			if ( in_array( $v, $true, true ) ) {
				return true;
			}
			if ( in_array( $v, $false, true ) ) {
				return false;
			}
		}

		return false;
	}

	/**
	 * Safely get the first element of an array.
	 *
	 * Prevents "Undefined offset: 0" errors when accessing $array[0]
	 * on potentially empty arrays.
	 *
	 * @since 0.9.70
	 *
	 * @param mixed $array   The array to read from.
	 * @param mixed $default Default value if array is empty or invalid.
	 * @return mixed First element or default.
	 */
	public static function get_first( $array, $default = '' ) {
		if ( is_array( $array ) && ! empty( $array ) ) {
			return reset( $array );
		}
		return $default;
	}

	/**
	 * Prevent instantiation.
	 */
	private function __construct() {}

	/**
	 * Prevent cloning.
	 */
	private function __clone() {}

	/**
	 * Prevent unserialization.
	 *
	 * @throws \Exception Always throws exception.
	 */
	public function __wakeup() {
		throw new \Exception( 'Cannot unserialize singleton' );
	}
}
