<?php
/**
 * Metrics data collector.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

namespace Cmatic\Metrics\Core;

defined( 'ABSPATH' ) || exit;

class Collector {

	public static function collect( $event = 'heartbeat' ) {
		return array(
			'install_id'  => Storage::get_install_id(),
			'timestamp'   => time(),
			'event'       => $event,
			'install'     => self::collect_install(),
			'metadata'    => self::collect_metadata(),
			'lifecycle'   => self::collect_lifecycle(),
			'environment' => self::collect_environment(),
			'api'         => self::collect_api(),
			'submissions' => self::collect_submissions(),
			'features'    => self::collect_features(),
			'forms'       => self::collect_forms(),
			'performance' => self::collect_performance(),
			'plugins'     => self::collect_plugins(),
			'competitors' => self::collect_competitors(),
			'server'      => self::collect_server(),
			'wordpress'   => self::collect_wordpress(),
		);
	}

	private static function collect_install() {
		return array(
			'plugin_slug' => mce_get_cmatic( 'install.plugin_slug', 'contact-form-7-mailchimp-extension' ),
			'quest'       => (int) mce_get_cmatic( 'install.quest', 0 ),
			'pro'         => array(
				'installed'       => (bool) mce_get_cmatic( 'install.pro.installed', false ),
				'activated'       => (bool) mce_get_cmatic( 'install.pro.activated', false ),
				'version'         => mce_get_cmatic( 'install.pro.version', null ),
				'licensed'        => (bool) mce_get_cmatic( 'install.pro.licensed', false ),
				'license_expires' => mce_get_cmatic( 'install.pro.license_expires', null ),
			),
		);
	}

	private static function collect_metadata() {
		$first_installed = mce_get_cmatic( 'install.quest', 0 );
		$total_uptime    = $first_installed > 0 ? time() - $first_installed : 0;
		$failed_count    = (int) mce_get_cmatic( 'telemetry.failed_count', 0 );

		return array(
			'schedule'             => mce_get_cmatic( 'telemetry.schedule', 'frequent' ),
			'frequent_started_at'  => (int) mce_get_cmatic( 'telemetry.frequent_started_at', 0 ),
			'is_reactivation'      => Storage::is_reactivation(),
			'disabled_count'       => (int) mce_get_cmatic( 'telemetry.disabled_count', 0 ),
			'opt_in_date'          => (int) mce_get_cmatic( 'telemetry.opt_in_date', 0 ),
			'last_heartbeat'       => (int) mce_get_cmatic( 'telemetry.last_heartbeat', 0 ),
			'failed_heartbeats'    => $failed_count,
			'total_uptime_seconds' => $total_uptime,
			'telemetry_version'    => SPARTAN_MCE_VERSION,
		);
	}

	private static function collect_lifecycle() {
		$activations   = mce_get_cmatic( 'lifecycle.activations', array() );
		$deactivations = mce_get_cmatic( 'lifecycle.deactivations', array() );
		$upgrades      = mce_get_cmatic( 'lifecycle.upgrades', array() );

		$first_activated  = mce_get_cmatic( 'install.quest', 0 );
		$last_activated   = ! empty( $activations ) ? max( $activations ) : 0;
		$last_deactivated = ! empty( $deactivations ) ? max( $deactivations ) : 0;
		$last_upgrade     = ! empty( $upgrades ) ? max( $upgrades ) : 0;

		$days_since_first = 0;
		if ( $first_activated > 0 ) {
			$days_since_first = floor( ( time() - $first_activated ) / DAY_IN_SECONDS );
		}

		$avg_session_length = 0;
		if ( count( $activations ) > 0 && count( $deactivations ) > 0 ) {
			$total_session_time = 0;
			$session_count      = 0;
			foreach ( $activations as $index => $activation_time ) {
				if ( isset( $deactivations[ $index ] ) ) {
					$total_session_time += $deactivations[ $index ] - $activation_time;
					++$session_count;
				}
			}
			if ( $session_count > 0 ) {
				$avg_session_length = floor( $total_session_time / $session_count );
			}
		}

		$version_history         = mce_get_cmatic( 'lifecycle.version_history', array() );
		$previous_version        = mce_get_cmatic( 'lifecycle.previous_version', '' );
		$days_since_last_upgrade = $last_upgrade > 0 ? floor( ( time() - $last_upgrade ) / DAY_IN_SECONDS ) : 0;

		// Build data array, filter out zero/empty/false values (keep active_session boolean).
		$active_session = empty( $deactivations ) || $last_activated > $last_deactivated;

		$data = array(
			'activation_count'            => count( $activations ),
			'deactivation_count'          => count( $deactivations ),
			'upgrade_count'               => count( $upgrades ),
			'first_activated'             => $first_activated,
			'last_activated'              => $last_activated,
			'last_deactivated'            => $last_deactivated,
			'last_upgrade'                => $last_upgrade,
			'days_since_first_activation' => (int) $days_since_first,
			'days_since_last_upgrade'     => (int) $days_since_last_upgrade,
			'avg_session_length_seconds'  => $avg_session_length,
			'total_sessions'              => count( $activations ),
			'previous_version'            => $previous_version,
			'version_history_count'       => count( $version_history ),
			'install_method'              => mce_get_cmatic( 'lifecycle.install_method', 'unknown' ),
			'days_on_current_version'     => $last_upgrade > 0 ? floor( ( time() - $last_upgrade ) / DAY_IN_SECONDS ) : $days_since_first,
			'activation_timestamps'       => $activations,
			'deactivation_timestamps'     => $deactivations,
			'upgrade_timestamps'          => $upgrades,
		);

		// Filter out zero/empty values to reduce payload size.
		$data = array_filter( $data, fn( $v ) => $v !== 0 && $v !== '' && $v !== 'unknown' );

		// Always include active_session (boolean flag).
		$data['active_session'] = $active_session;

		return $data;
	}

	private static function collect_environment() {
		global $wp_version, $wpdb;

		$php_extensions      = get_loaded_extensions();
		$critical_extensions = array( 'curl', 'json', 'mbstring', 'openssl', 'zip', 'gd', 'xml', 'dom', 'SimpleXML' );
		$loaded_critical     = array_intersect( $critical_extensions, $php_extensions );

		$theme        = wp_get_theme();
		$parent_theme = $theme->parent() ? $theme->parent()->get( 'Name' ) : '';

		$data = array(
			'php_version'                    => phpversion(),
			'php_sapi'                       => php_sapi_name(),
			'php_os'                         => PHP_OS,
			'php_architecture'               => PHP_INT_SIZE === 8 ? '64-bit' : '32-bit',
			'php_memory_limit'               => ini_get( 'memory_limit' ),
			'php_max_execution_time'         => (int) ini_get( 'max_execution_time' ),
			'php_max_input_time'             => (int) ini_get( 'max_input_time' ),
			'php_max_input_vars'             => (int) ini_get( 'max_input_vars' ),
			'php_post_max_size'              => ini_get( 'post_max_size' ),
			'php_upload_max_filesize'        => ini_get( 'upload_max_filesize' ),
			'php_default_timezone'           => ini_get( 'date.timezone' ),
			// Removed: php_display_errors, php_error_reporting (low value - we have wp_debug flags).
			'php_log_errors'                 => ini_get( 'log_errors' ),
			'php_extensions_count'           => count( $php_extensions ),
			'php_critical_extensions'        => implode( ',', $loaded_critical ),
			'php_curl_version'               => function_exists( 'curl_version' ) ? curl_version()['version'] : '',
			'php_openssl_version'            => OPENSSL_VERSION_TEXT,
			'wp_version'                     => $wp_version,
			'wp_db_version'                  => get_option( 'db_version' ),
			'wp_memory_limit'                => WP_MEMORY_LIMIT,
			'wp_max_memory_limit'            => WP_MAX_MEMORY_LIMIT,
			'wp_debug'                       => defined( 'WP_DEBUG' ) && WP_DEBUG,
			'wp_debug_log'                   => defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG,
			'wp_debug_display'               => defined( 'WP_DEBUG_DISPLAY' ) && WP_DEBUG_DISPLAY,
			'script_debug'                   => defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG,
			'wp_cache'                       => defined( 'WP_CACHE' ) && WP_CACHE,
			'wp_cron_disabled'               => defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON,
			'wp_auto_update_core'            => get_option( 'auto_update_core', 'enabled' ),
			'mysql_version'                  => $wpdb->db_version(),
			'mysql_client_version'           => $wpdb->get_var( 'SELECT VERSION()' ),
			'db_charset'                     => $wpdb->charset,
			'db_collate'                     => $wpdb->collate,
			'db_prefix'                      => strlen( $wpdb->prefix ),
			'server_software'                => isset( $_SERVER['SERVER_SOFTWARE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) : '',
			'server_protocol'                => isset( $_SERVER['SERVER_PROTOCOL'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_PROTOCOL'] ) ) : '',
			'server_port'                    => isset( $_SERVER['SERVER_PORT'] ) ? (int) $_SERVER['SERVER_PORT'] : 0,
			'https'                          => is_ssl(),
			'http_host'                      => hash( 'sha256', isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '' ),
			'locale'                         => get_locale(),
			'timezone'                       => wp_timezone_string(),
			'site_language'                  => get_bloginfo( 'language' ),
			'site_charset'                   => get_bloginfo( 'charset' ),
			'permalink_structure'            => get_option( 'permalink_structure' ),
			'home_url'                       => hash( 'sha256', home_url() ),
			'site_url'                       => hash( 'sha256', site_url() ),
			'admin_email'                    => hash( 'sha256', get_option( 'admin_email' ) ),
			'theme'                          => $theme->get( 'Name' ),
			'theme_version'                  => $theme->get( 'Version' ),
			'theme_author'                   => $theme->get( 'Author' ),
			'parent_theme'                   => $parent_theme,
			'is_child_theme'                 => ! empty( $parent_theme ),
			'theme_supports_html5'           => current_theme_supports( 'html5' ),
			'theme_supports_post_thumbnails' => current_theme_supports( 'post-thumbnails' ),
			'active_plugins_count'           => count( get_option( 'active_plugins', array() ) ),
			'total_plugins_count'            => count( get_plugins() ),
			'must_use_plugins_count'         => count( wp_get_mu_plugins() ),
			'is_multisite'                   => is_multisite(),
			'is_subdomain_install'           => is_multisite() ? ( defined( 'SUBDOMAIN_INSTALL' ) && SUBDOMAIN_INSTALL ) : false,
			'network_count'                  => is_multisite() ? get_blog_count() : 1,
			'is_main_site'                   => is_multisite() ? is_main_site() : true,
			'cf7_version'                    => defined( 'WPCF7_VERSION' ) ? WPCF7_VERSION : '',
			'cf7_installed'                  => class_exists( 'WPCF7_ContactForm' ),
			'plugin_version'                 => defined( 'SPARTAN_MCE_VERSION' ) ? SPARTAN_MCE_VERSION : '',
			'user_agent'                     => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
		);

		// Filter out zero/false/empty values to reduce payload size.
		return array_filter( $data, fn( $v ) => $v !== 0 && $v !== false && $v !== '' && $v !== 'none' );
	}

	private static function collect_api() {
		$has_api_key     = false;
		$api_data_center = '';
		$api_key_length  = 0;
		$forms_with_api  = 0;
		$cf7_forms       = self::get_cf7_forms();

		foreach ( $cf7_forms as $form ) {
			$form_id = $form->id();
			$cf7_mch = get_option( 'cf7_mch_' . $form_id, array() );

			if ( ! empty( $cf7_mch['api'] ) ) {
				$has_api_key = true;
				++$forms_with_api;

				if ( empty( $api_data_center ) && preg_match( '/-([a-z0-9]+)$/i', $cf7_mch['api'], $matches ) ) {
					$api_data_center = $matches[1];
				}

				if ( 0 === $api_key_length ) {
					$api_key_length = strlen( $cf7_mch['api'] );
				}
			}
		}

		$total_sent        = (int) mce_get_cmatic( 'stats.sent', 0 );
		$total_attempts    = (int) mce_get_cmatic( 'api.total_attempts', $total_sent );
		$total_successes   = (int) mce_get_cmatic( 'api.total_successes', $total_sent );
		$total_failures    = (int) mce_get_cmatic( 'api.total_failures', 0 );
		$success_rate      = $total_attempts > 0 ? round( ( $total_successes / $total_attempts ) * 100, 2 ) : 0;
		$first_connected   = (int) mce_get_cmatic( 'api.first_connected', 0 );
		$last_success      = (int) mce_get_cmatic( 'api.last_success', 0 );
		$last_failure      = (int) mce_get_cmatic( 'api.last_failure', 0 );
		$avg_response_time = (int) mce_get_cmatic( 'api.avg_response_time', 0 );

		$error_codes   = mce_get_cmatic( 'api.error_codes', array() );
		$error_summary = array();
		foreach ( $error_codes as $code => $count ) {
			if ( $count > 0 ) {
				$error_summary[ $code ] = (int) $count;
			}
		}

		$rate_limit_hits      = (int) mce_get_cmatic( 'api.rate_limit_hits', 0 );
		$last_rate_limit      = (int) mce_get_cmatic( 'api.last_rate_limit', 0 );
		$consecutive_failures = (int) mce_get_cmatic( 'api.consecutive_failures', 0 );
		$uptime_percentage    = $total_attempts > 0 ? round( ( $total_successes / $total_attempts ) * 100, 2 ) : 100;

		$days_since_first_connection = $first_connected > 0 ? floor( ( time() - $first_connected ) / DAY_IN_SECONDS ) : 0;
		$days_since_last_success     = $last_success > 0 ? floor( ( time() - $last_success ) / DAY_IN_SECONDS ) : 0;
		$days_since_last_failure     = $last_failure > 0 ? floor( ( time() - $last_failure ) / DAY_IN_SECONDS ) : 0;

		// Build data, excluding rarely-useful fields (days_since_first_connection, unique_error_count, etc).
		$data = array(
			'is_connected'              => $has_api_key,
			'forms_with_api'            => $forms_with_api,
			'api_data_center'           => $api_data_center,
			'api_key_length'            => $api_key_length,
			'first_connected'           => $first_connected,
			'total_attempts'            => $total_attempts,
			'total_successes'           => $total_successes,
			'total_failures'            => $total_failures,
			'success_rate'              => $success_rate,
			'uptime_percentage'         => $uptime_percentage,
			'last_success'              => $last_success,
			'last_failure'              => $last_failure,
			'days_since_last_success'   => $days_since_last_success,
			'days_since_last_failure'   => $days_since_last_failure,
			'avg_response_time_ms'      => $avg_response_time,
			'error_codes'               => $error_summary,
			'api_health_score'          => min( 100, max( 0, $uptime_percentage - ( $consecutive_failures * 5 ) ) ),
			'setup_sync_attempted'      => mce_get_cmatic( 'api.sync_attempted', false ),
			'setup_sync_attempts_count' => (int) mce_get_cmatic( 'api.sync_attempts_count', 0 ),
			'setup_first_success'       => mce_get_cmatic( 'api.setup_first_success', false ),
			'setup_first_failure'       => mce_get_cmatic( 'api.setup_first_failure', false ),
			'setup_failure_count'       => (int) mce_get_cmatic( 'api.setup_failure_count', 0 ),
			'setup_audience_selected'   => mce_get_cmatic( 'api.audience_selected', false ),
		);

		// Filter out zero/false/empty values.
		return array_filter( $data, fn( $v ) => $v !== 0 && $v !== false && $v !== '' && $v !== array() );
	}

	private static function collect_submissions() {
		$total_sent       = (int) mce_get_cmatic( 'stats.sent', 0 );
		$total_failed     = (int) mce_get_cmatic( 'submissions.failed', 0 );
		$first_submission = (int) mce_get_cmatic( 'submissions.first', 0 );
		$last_submission  = (int) mce_get_cmatic( 'submissions.last', 0 );
		$last_success     = (int) mce_get_cmatic( 'submissions.last_success', 0 );
		$last_failure     = (int) mce_get_cmatic( 'submissions.last_failure', 0 );

		$first_activated = mce_get_cmatic( 'install.quest', 0 );

		$days_active = 1;
		if ( $first_activated > 0 ) {
			$days_active = max( 1, floor( ( time() - $first_activated ) / DAY_IN_SECONDS ) );
		}

		$total_submissions = $total_sent + $total_failed;
		$avg_per_day       = $days_active > 0 ? round( $total_submissions / $days_active, 2 ) : 0;
		$success_rate      = $total_submissions > 0 ? round( ( $total_sent / $total_submissions ) * 100, 2 ) : 100;

		$days_since_first = $first_submission > 0 ? floor( ( time() - $first_submission ) / DAY_IN_SECONDS ) : 0;
		$days_since_last  = $last_submission > 0 ? floor( ( time() - $last_submission ) / DAY_IN_SECONDS ) : 0;
		$hours_since_last = $last_submission > 0 ? floor( ( time() - $last_submission ) / HOUR_IN_SECONDS ) : 0;

		$hourly_distribution = mce_get_cmatic( 'submissions.hourly', array() );
		$busiest_hour        = 0;
		$max_submissions     = 0;
		foreach ( $hourly_distribution as $hour => $count ) {
			if ( $count > $max_submissions ) {
				$max_submissions = $count;
				$busiest_hour    = (int) $hour;
			}
		}

		$daily_distribution  = mce_get_cmatic( 'submissions.daily', array() );
		$busiest_day         = 0;
		$max_day_submissions = 0;
		foreach ( $daily_distribution as $day => $count ) {
			if ( $count > $max_day_submissions ) {
				$max_day_submissions = $count;
				$busiest_day         = (int) $day;
			}
		}

		$this_month = (int) mce_get_cmatic( 'submissions.this_month', 0 );
		$last_month = (int) mce_get_cmatic( 'submissions.last_month', 0 );
		$peak_month = (int) mce_get_cmatic( 'submissions.peak_month', 0 );

		$consecutive_successes = (int) mce_get_cmatic( 'submissions.consecutive_successes', 0 );
		$consecutive_failures  = (int) mce_get_cmatic( 'submissions.consecutive_failures', 0 );

		// Build data array, filter out zero/empty values (keep success_rate even if 100).
		$data = array(
			'total_sent'                   => $total_sent,
			'total_failed'                 => $total_failed,
			'total_submissions'            => $total_submissions,
			'successful_submissions_count' => $total_sent,
			'failed_count'                 => $total_failed,
			'success_rate'                 => $success_rate,
			'first_submission'             => $first_submission,
			'last_submission'              => $last_submission,
			'last_success'                 => $last_success,
			'last_failure'                 => $last_failure,
			'days_since_first'             => $days_since_first,
			'days_since_last'              => $days_since_last,
			'hours_since_last'             => $hours_since_last,
			'avg_per_day'                  => $avg_per_day,
			'avg_per_week'                 => round( $avg_per_day * 7, 2 ),
			'avg_per_month'                => round( $avg_per_day * 30, 2 ),
			'busiest_hour'                 => $busiest_hour,
			'busiest_day'                  => $busiest_day,
			'submissions_busiest_hour'     => $max_submissions,
			'submissions_busiest_day'      => $max_day_submissions,
			'this_month'                   => $this_month,
			'last_month'                   => $last_month,
			'peak_month'                   => $peak_month,
			'month_over_month_change'      => $last_month > 0 ? round( ( ( $this_month - $last_month ) / $last_month ) * 100, 2 ) : 0,
			'consecutive_successes'        => $consecutive_successes,
			'consecutive_failures'         => $consecutive_failures,
			'longest_success_streak'       => (int) mce_get_cmatic( 'submissions.longest_success_streak', 0 ),
			'active_forms_count'           => (int) mce_get_cmatic( 'submissions.active_forms', 0 ),
			'forms_with_submissions'       => count( mce_get_cmatic( 'submissions.forms_used', array() ) ),
		);

		// Filter out zero values to reduce payload size.
		return array_filter( $data, fn( $v ) => $v !== 0 && $v !== 0.0 );
	}

	private static function collect_features() {
		$features = array(
			'double_optin'         => 0,
			'required_consent'     => 0,
			'debug_logger'         => 0,
			'custom_merge_fields'  => 0,
			'interest_groups'      => 0,  // Forms with at least one group mapped.
			'groups_total_mapped'  => 0,  // Total group mappings across all forms.
			'tags_enabled'         => 0,
			'tags_total_selected'  => 0,  // Total tag chips selected across all forms.
			'arbitrary_tags'       => 0,  // Forms using arbitrary tags text input.
			'conditional_logic'    => 0,
			'auto_update'          => (bool) mce_get_cmatic( 'auto_update', true ),
			'telemetry_enabled'    => true,
		);

		$cf7_forms = self::get_cf7_forms();

		foreach ( $cf7_forms as $form ) {
			$form_id = $form->id();
			$cf7_mch = get_option( 'cf7_mch_' . $form_id, array() );

			if ( isset( $cf7_mch['confsubs'] ) && '1' === $cf7_mch['confsubs'] ) {
				++$features['double_optin'];
			}

			// Required Consent stored in 'consent_required'; value is field name or ' ' when not configured.
			if ( ! empty( $cf7_mch['consent_required'] ) && ' ' !== $cf7_mch['consent_required'] ) {
				++$features['required_consent'];
			}

			if ( isset( $cf7_mch['logfileEnabled'] ) && '1' === $cf7_mch['logfileEnabled'] ) {
				++$features['debug_logger'];
			}

			// Custom merge fields = tags NOT in defaults (EMAIL, FNAME, LNAME, ADDRESS, PHONE).
			$merge_fields_raw = array();
			if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
				$merge_fields_raw = $cf7_mch['merge_fields'];
			} elseif ( ! empty( $cf7_mch['merge-vars'] ) && is_array( $cf7_mch['merge-vars'] ) ) {
				$merge_fields_raw = $cf7_mch['merge-vars'];
			}
			if ( ! empty( $merge_fields_raw ) ) {
				$default_tags = array( 'EMAIL', 'FNAME', 'LNAME', 'ADDRESS', 'PHONE' );
				foreach ( $merge_fields_raw as $field ) {
					if ( isset( $field['tag'] ) && ! in_array( $field['tag'], $default_tags, true ) ) {
						++$features['custom_merge_fields'];
					}
				}
			}

			// Interest groups stored as ggCustomKey1/ggCustomValue1; active when both exist and non-empty.
			$group_count = 0;
			for ( $i = 1; $i <= 20; $i++ ) {
				$key   = $cf7_mch[ "ggCustomKey{$i}" ] ?? '';
				$value = $cf7_mch[ "ggCustomValue{$i}" ] ?? '';
				if ( ! empty( $key ) && ! empty( trim( $value ) ) && ' ' !== $value ) {
					++$group_count;
				}
			}
			if ( $group_count > 0 ) {
				++$features['interest_groups'];
				$features['groups_total_mapped'] += $group_count;
			}

			// Tags stored in 'labeltags' array; key=field name, value='1' if enabled.
			if ( ! empty( $cf7_mch['labeltags'] ) && is_array( $cf7_mch['labeltags'] ) ) {
				$enabled_tags = array_filter( $cf7_mch['labeltags'], fn( $v ) => '1' === $v );
				if ( count( $enabled_tags ) > 0 ) {
					++$features['tags_enabled'];
					$features['tags_total_selected'] += count( $enabled_tags );
				}
			}

			// Arbitrary tags stored in 'labeltags_cm-tag' text field.
			if ( ! empty( $cf7_mch['labeltags_cm-tag'] ) && trim( $cf7_mch['labeltags_cm-tag'] ) !== '' ) {
				++$features['arbitrary_tags'];
			}

			if ( ! empty( $cf7_mch['conditional_logic'] ) ) {
				++$features['conditional_logic'];
			}
		}

		// Build data array (removed rarely-used first_use fields).
		$data = array(
			'double_optin_count'        => $features['double_optin'],
			'required_consent_count'    => $features['required_consent'],
			'debug_logger_count'        => $features['debug_logger'],
			'custom_merge_fields_count' => $features['custom_merge_fields'],
			'interest_groups_count'     => $features['interest_groups'],
			'groups_total_mapped'       => $features['groups_total_mapped'],
			'tags_enabled_count'        => $features['tags_enabled'],
			'tags_total_selected'       => $features['tags_total_selected'],
			'arbitrary_tags_count'      => $features['arbitrary_tags'],
			'conditional_logic_count'   => $features['conditional_logic'],
			'double_optin'              => $features['double_optin'] > 0,
			'required_consent'          => $features['required_consent'] > 0,
			'debug_logger'              => $features['debug_logger'] > 0,
			'custom_merge_fields'       => $features['custom_merge_fields'] > 0,
			'interest_groups'           => $features['interest_groups'] > 0,
			'tags_enabled'              => $features['tags_enabled'] > 0,
			'arbitrary_tags'            => $features['arbitrary_tags'] > 0,
			'conditional_logic'         => $features['conditional_logic'] > 0,
			'auto_update'               => $features['auto_update'],
			'telemetry_enabled'         => $features['telemetry_enabled'],
			'total_features_enabled'    => count( array_filter( $features ) ),
			'features_usage_percentage' => round( ( count( array_filter( $features ) ) / count( $features ) ) * 100, 2 ),
			'webhook_enabled'           => (bool) mce_get_cmatic( 'features.webhook_enabled', false ),
			'custom_api_endpoint'       => (bool) mce_get_cmatic( 'features.custom_api_endpoint', false ),
			'email_notifications'       => (bool) mce_get_cmatic( 'features.email_notifications', false ),
		);

		// Filter out zero/false values.
		return array_filter( $data, fn( $v ) => $v !== 0 && $v !== false && $v !== '' );
	}

	private static function collect_forms() {
		$cf7_forms        = self::get_cf7_forms();
		$total_forms      = count( $cf7_forms );
		$active_forms     = 0;
		$forms_with_api   = 0;
		$forms_with_lists = 0;
		$total_lists      = 0;
		$total_fields     = 0;
		$unique_audiences = array();
		$forms_data       = array();
		$paired_lists     = array(); // Track which lists are paired to forms.

		// NEW: Granular form details for per-form field analysis.
		$forms_detail      = array();
		$field_type_counts = array();
		$total_mappings    = 0;
		$total_mc_fields   = 0;

		// Build audiences from global lisdata (single source of truth, always freshest).
		$global_lisdata = mce_get_cmatic( 'lisdata', array() );
		if ( ! empty( $global_lisdata['lists'] ) && is_array( $global_lisdata['lists'] ) ) {
			foreach ( $global_lisdata['lists'] as $list ) {
				$list_id = $list['id'] ?? '';
				if ( empty( $list_id ) ) {
					continue;
				}

				$unique_audiences[ $list_id ] = array(
					'audience_id'           => hash( 'sha256', $list_id ),
					'member_count'          => (int) ( $list['stats']['member_count'] ?? 0 ),
					'merge_field_count'     => (int) ( $list['stats']['merge_field_count'] ?? 0 ),
					'double_optin'          => ! empty( $list['double_optin'] ),
					'marketing_permissions' => ! empty( $list['marketing_permissions'] ), // GDPR.
					'campaign_count'        => (int) ( $list['stats']['campaign_count'] ?? 0 ),
					'is_paired'             => false, // Will be set true if any form uses this list.
				);
			}
		}

		// Collect form data and mark which audiences are paired.
		foreach ( $cf7_forms as $form ) {
			$form_id = $form->id();
			$cf7_mch = get_option( 'cf7_mch_' . $form_id, array() );

			if ( ! empty( $cf7_mch['api'] ) ) {
				++$forms_with_api;
				++$active_forms;
			}

			$selected_list = $cf7_mch['list'] ?? '';
			// Handle both string and array formats (legacy vs new storage).
			if ( is_array( $selected_list ) ) {
				$selected_list = $selected_list[0] ?? '';
			}

			// Mark audience as paired if this form uses it.
			if ( ! empty( $selected_list ) && isset( $unique_audiences[ $selected_list ] ) ) {
				$unique_audiences[ $selected_list ]['is_paired'] = true;
				$paired_lists[ $selected_list ]                  = true;
			}

			// Count forms with lists (from per-form lisdata for backwards compat).
			$audience_count = 0;
			$has_list       = false;
			if ( isset( $cf7_mch['lisdata']['lists'] ) && is_array( $cf7_mch['lisdata']['lists'] ) ) {
				$audience_count = count( $cf7_mch['lisdata']['lists'] );
				$has_list       = $audience_count > 0;
				$total_lists   += $audience_count;
				if ( $has_list ) {
					++$forms_with_lists;
				}
			}

			$form_fields   = $form->scan_form_tags();
			$total_fields += count( $form_fields );

			// NEW: Extract field details (limit to 30 fields per form).
			$form_field_details = array();
			$form_field_limit   = 30;
			foreach ( $form_fields as $ff_index => $tag ) {
				if ( $ff_index >= $form_field_limit ) {
					break;
				}
				$basetype = '';
				if ( is_object( $tag ) && isset( $tag->basetype ) ) {
					$basetype = $tag->basetype;
				} elseif ( is_array( $tag ) && isset( $tag['basetype'] ) ) {
					$basetype = $tag['basetype'];
				}
				$field_name = '';
				if ( is_object( $tag ) && isset( $tag->name ) ) {
					$field_name = $tag->name;
				} elseif ( is_array( $tag ) && isset( $tag['name'] ) ) {
					$field_name = $tag['name'];
				}

				// Only include fields with names (skip submit, response output).
				if ( ! empty( $field_name ) && ! empty( $basetype ) ) {
					$form_field_details[] = array(
						'name' => $field_name,
						'type' => $basetype,
					);

					// Aggregate field type counts.
					if ( ! isset( $field_type_counts[ $basetype ] ) ) {
						$field_type_counts[ $basetype ] = 0;
					}
					++$field_type_counts[ $basetype ];
				}
			}

			// NEW: Extract field mappings (CustomKey/CustomValue pairs).
			$form_mappings     = array();
			$unmapped_cf7      = 0;
			$unmapped_mc       = 0;
			$mapped_cf7_fields = array();

			for ( $i = 1; $i <= 20; $i++ ) {
				$mc_tag    = $cf7_mch[ 'CustomKey' . $i ] ?? '';
				$mc_type   = $cf7_mch[ 'CustomKeyType' . $i ] ?? '';
				$cf7_field = trim( $cf7_mch[ 'CustomValue' . $i ] ?? '' );

				if ( ! empty( $mc_tag ) ) {
					++$total_mc_fields;
					if ( '' !== $cf7_field ) {
						$form_mappings[] = array(
							'cf7_field' => $cf7_field,
							'mc_tag'    => $mc_tag,
							'mc_type'   => $mc_type,
						);
						$mapped_cf7_fields[] = $cf7_field;
						++$total_mappings;
					} else {
						++$unmapped_mc;
					}
				}
			}

			// Count CF7 fields not mapped to any MC field.
			foreach ( $form_field_details as $field ) {
				if ( ! in_array( $field['name'], $mapped_cf7_fields, true ) ) {
					++$unmapped_cf7;
				}
			}

			// Per-form feature detection for granular UI display.
			$form_features = array();

			// Double Opt-in.
			if ( isset( $cf7_mch['confsubs'] ) && '1' === $cf7_mch['confsubs'] ) {
				$form_features['double_optin'] = true;
			}

			// Required Consent (stored in 'consent_required', not 'accept').
			if ( ! empty( $cf7_mch['consent_required'] ) && ' ' !== $cf7_mch['consent_required'] ) {
				$form_features['required_consent'] = true;
			}

			// Debug Logger.
			if ( isset( $cf7_mch['logfileEnabled'] ) && '1' === $cf7_mch['logfileEnabled'] ) {
				$form_features['debug_logger'] = true;
			}

			// Tags (stored in 'labeltags' array).
			if ( ! empty( $cf7_mch['labeltags'] ) && is_array( $cf7_mch['labeltags'] ) ) {
				$enabled_tags = array_filter( $cf7_mch['labeltags'], fn( $v ) => '1' === $v );
				if ( count( $enabled_tags ) > 0 ) {
					$form_features['tags_enabled'] = true;
				}
			}

			// Interest Groups (stored as ggCustomKey{n}/ggCustomValue{n}).
			$form_group_count = 0;
			for ( $gi = 1; $gi <= 20; $gi++ ) {
				$gkey   = $cf7_mch[ "ggCustomKey{$gi}" ] ?? '';
				$gvalue = $cf7_mch[ "ggCustomValue{$gi}" ] ?? '';
				if ( ! empty( $gkey ) && ! empty( trim( $gvalue ) ) && ' ' !== $gvalue ) {
					++$form_group_count;
				}
			}
			if ( $form_group_count > 0 ) {
				$form_features['interest_groups'] = true;
			}

			// Custom Merge Fields (non-default tags).
			$form_merge_fields_raw = array();
			if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
				$form_merge_fields_raw = $cf7_mch['merge_fields'];
			} elseif ( ! empty( $cf7_mch['merge-vars'] ) && is_array( $cf7_mch['merge-vars'] ) ) {
				$form_merge_fields_raw = $cf7_mch['merge-vars'];
			}
			if ( ! empty( $form_merge_fields_raw ) ) {
				$default_tags       = array( 'EMAIL', 'FNAME', 'LNAME', 'ADDRESS', 'PHONE' );
				$custom_field_count = 0;
				foreach ( $form_merge_fields_raw as $mfield ) {
					if ( isset( $mfield['tag'] ) && ! in_array( $mfield['tag'], $default_tags, true ) ) {
						++$custom_field_count;
					}
				}
				if ( $custom_field_count > 0 ) {
					$form_features['custom_merge_fields'] = true;
				}
			}

			// Conditional Logic.
			if ( ! empty( $cf7_mch['conditional_logic'] ) ) {
				$form_features['conditional_logic'] = true;
			}

			// Build forms_detail entry (limit to 50 forms total).
			if ( count( $forms_detail ) < 50 ) {
				$forms_detail[] = array(
					'form_id'             => hash( 'sha256', (string) $form_id ),
					'field_count'         => count( $form_field_details ),
					'fields'              => $form_field_details,
					'paired_audience_id'  => ! empty( $selected_list ) ? hash( 'sha256', $selected_list ) : null,
					'mappings'            => $form_mappings,
					'unmapped_cf7_fields' => $unmapped_cf7,
					'unmapped_mc_fields'  => $unmapped_mc,
					'features'            => $form_features,
				);
			}

			$forms_data[] = array(
				'form_id'         => hash( 'sha256', (string) $form_id ),
				'has_api'         => ! empty( $cf7_mch['api'] ),
				'has_list'        => $has_list,
				'audience_count'  => $audience_count,
				'field_count'     => count( $form_fields ),
				'has_double_opt'  => isset( $cf7_mch['confsubs'] ) && '1' === $cf7_mch['confsubs'],
				'has_consent'     => ! empty( $cf7_mch['accept'] ) && ' ' !== $cf7_mch['accept'],
				'submissions'     => (int) get_option( 'cf7_mch_submissions_' . $form_id, 0 ),
				'last_submission' => (int) get_option( 'cf7_mch_last_submission_' . $form_id, 0 ),
			);
		}

		$avg_fields_per_form = $total_forms > 0 ? round( $total_fields / $total_forms, 2 ) : 0;
		$avg_lists_per_form  = $forms_with_lists > 0 ? round( $total_lists / $forms_with_lists, 2 ) : 0;

		$oldest_form = 0;
		$newest_form = 0;
		foreach ( $cf7_forms as $form ) {
			$created   = get_post_field( 'post_date', $form->id(), 'raw' );
			$timestamp = strtotime( $created );

			if ( 0 === $oldest_form || $timestamp < $oldest_form ) {
				$oldest_form = $timestamp;
			}
			if ( 0 === $newest_form || $timestamp > $newest_form ) {
				$newest_form = $timestamp;
			}
		}

		$audience_data          = array_values( $unique_audiences );
		$total_unique_audiences = count( $audience_data );
		$total_contacts         = array_sum( array_column( $audience_data, 'member_count' ) );

		return array(
			'total_forms'                 => $total_forms,
			'active_forms'                => $active_forms,
			'forms_with_api'              => $forms_with_api,
			'forms_with_lists'            => $forms_with_lists,
			'inactive_forms'              => $total_forms - $active_forms,
			'total_audiences'             => $total_unique_audiences,
			'audiences'                   => $audience_data,
			'total_contacts'              => $total_contacts,
			'avg_lists_per_form'          => $avg_lists_per_form,
			'max_lists_per_form'          => $total_lists > 0 ? max( array_column( $forms_data, 'audience_count' ) ) : 0,
			'total_fields_all_forms'      => $total_fields,
			'avg_fields_per_form'         => $avg_fields_per_form,
			'min_fields_per_form'         => $total_forms > 0 ? min( array_column( $forms_data, 'field_count' ) ) : 0,
			'max_fields_per_form'         => $total_forms > 0 ? max( array_column( $forms_data, 'field_count' ) ) : 0,
			'oldest_form_created'         => $oldest_form,
			'newest_form_created'         => $newest_form,
			'days_since_oldest_form'      => $oldest_form > 0 ? floor( ( time() - $oldest_form ) / DAY_IN_SECONDS ) : 0,
			'days_since_newest_form'      => $newest_form > 0 ? floor( ( time() - $newest_form ) / DAY_IN_SECONDS ) : 0,
			'forms_with_submissions'      => count( array_filter( $forms_data, fn( $f ) => $f['submissions'] > 0 ) ),
			'forms_never_submitted'       => count( array_filter( $forms_data, fn( $f ) => 0 === $f['submissions'] ) ),
			'forms_with_double_opt'       => count( array_filter( $forms_data, fn( $f ) => $f['has_double_opt'] ) ),
			'forms_with_consent'          => count( array_filter( $forms_data, fn( $f ) => $f['has_consent'] ) ),
			'total_submissions_all_forms' => array_sum( array_column( $forms_data, 'submissions' ) ),
			'form_utilization_rate'       => $total_forms > 0 ? round( ( $active_forms / $total_forms ) * 100, 2 ) : 0,

			// NEW: Granular form and field data.
			'forms_detail'          => $forms_detail,
			'forms_truncated'       => count( $cf7_forms ) > 50,
			'field_types_aggregate' => $field_type_counts,
			'mapping_stats'         => array(
				'total_cf7_fields' => $total_fields,
				'total_mc_fields'  => $total_mc_fields,
				'mapped_fields'    => $total_mappings,
				'mapping_rate'     => $total_fields > 0 ? round( ( $total_mappings / $total_fields ) * 100, 2 ) : 0,
			),
		);
	}

	private static function collect_performance() {
		$memory_current       = memory_get_usage( true );
		$memory_peak          = memory_get_peak_usage( true );
		$memory_limit         = ini_get( 'memory_limit' );
		$memory_limit_bytes   = self::convert_to_bytes( $memory_limit );
		$memory_usage_percent = $memory_limit_bytes > 0 ? round( ( $memory_peak / $memory_limit_bytes ) * 100, 2 ) : 0;

		$db_queries = get_num_queries();
		$db_time    = timer_stop( 0, 3 );

		$page_load_time = isset( $_SERVER['REQUEST_TIME_FLOAT'] ) ? ( microtime( true ) - floatval( $_SERVER['REQUEST_TIME_FLOAT'] ) ) * 1000 : 0;

		$object_cache_hits   = 0;
		$object_cache_misses = 0;
		if ( function_exists( 'wp_cache_get_stats' ) ) {
			$cache_stats         = wp_cache_get_stats();
			$object_cache_hits   = isset( $cache_stats['hits'] ) ? $cache_stats['hits'] : 0;
			$object_cache_misses = isset( $cache_stats['misses'] ) ? $cache_stats['misses'] : 0;
		}

		$plugin_load_time = (float) mce_get_cmatic( 'performance.plugin_load_time', 0 );
		$api_avg_response = (float) mce_get_cmatic( 'performance.api_avg_response', 0 );

		// Build data array, then filter out zero/empty values.
		$data = array(
			'memory_current'          => $memory_current,
			'memory_peak'             => $memory_peak,
			'memory_limit'            => $memory_limit,
			'memory_limit_bytes'      => $memory_limit_bytes,
			'memory_usage_percent'    => $memory_usage_percent,
			'memory_available'        => max( 0, $memory_limit_bytes - $memory_peak ),
			'php_max_execution_time'  => (int) ini_get( 'max_execution_time' ),
			'page_load_time_ms'       => round( $page_load_time, 2 ),
			'plugin_load_time_ms'     => round( $plugin_load_time, 2 ),
			'db_queries_count'        => $db_queries,
			'db_query_time_seconds'   => (float) $db_time,
			'db_size_mb'              => self::get_database_size(),
			'api_avg_response_ms'     => round( $api_avg_response, 2 ),
			'api_slowest_response_ms' => (int) mce_get_cmatic( 'performance.api_slowest', 0 ),
			'api_fastest_response_ms' => (int) mce_get_cmatic( 'performance.api_fastest', 0 ),
			'object_cache_enabled'    => wp_using_ext_object_cache(),
			'object_cache_hits'       => $object_cache_hits,
			'object_cache_misses'     => $object_cache_misses,
			'object_cache_hit_rate'   => ( $object_cache_hits + $object_cache_misses ) > 0 ? round( ( $object_cache_hits / ( $object_cache_hits + $object_cache_misses ) ) * 100, 2 ) : 0,
			'opcache_enabled'         => self::is_opcache_enabled(),
			'opcache_hit_rate'        => self::get_opcache_hit_rate(),
		);

		// Filter out zero/false/null values to reduce payload size.
		return array_filter( $data, fn( $v ) => $v !== 0 && $v !== 0.0 && $v !== false && $v !== null && $v !== '' );
	}

	private static function is_opcache_enabled() {
		if ( ! function_exists( 'opcache_get_status' ) ) {
			return false;
		}
		$status = @opcache_get_status();
		return false !== $status && is_array( $status );
	}

	private static function convert_to_bytes( $value ) {
		$value = trim( $value );
		if ( empty( $value ) ) {
			return 0;
		}
		$last  = strtolower( $value[ strlen( $value ) - 1 ] );
		$value = (int) $value;

		switch ( $last ) {
			case 'g':
				$value *= 1024;
				// Fall through.
			case 'm':
				$value *= 1024;
				// Fall through.
			case 'k':
				$value *= 1024;
		}

		return $value;
	}

	private static function get_database_size() {
		global $wpdb;

		$size = $wpdb->get_var(
			$wpdb->prepare(
				'SELECT SUM(data_length + index_length) / 1024 / 1024
				FROM information_schema.TABLES
				WHERE table_schema = %s',
				DB_NAME
			)
		);

		return round( (float) $size, 2 );
	}

	private static function get_opcache_hit_rate() {
		if ( ! function_exists( 'opcache_get_status' ) ) {
			return 0;
		}

		$status = @opcache_get_status();
		if ( false === $status || ! is_array( $status ) || ! isset( $status['opcache_statistics'] ) ) {
			return 0;
		}

		$stats  = $status['opcache_statistics'];
		$hits   = isset( $stats['hits'] ) ? (int) $stats['hits'] : 0;
		$misses = isset( $stats['misses'] ) ? (int) $stats['misses'] : 0;

		if ( ( $hits + $misses ) === 0 ) {
			return 0;
		}

		return round( ( $hits / ( $hits + $misses ) ) * 100, 2 );
	}

	private static function collect_plugins() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins    = get_plugins();
		$active_plugins = get_option( 'active_plugins', array() );
		$mu_plugins     = get_mu_plugins();

		$plugin_list    = array();
		$network_active = is_multisite() ? get_site_option( 'active_sitewide_plugins', array() ) : array();

		foreach ( $all_plugins as $plugin_path => $plugin_data ) {
			$is_active  = in_array( $plugin_path, $active_plugins, true );
			$is_network = isset( $network_active[ $plugin_path ] );

			$status = 'inactive';
			if ( $is_network ) {
				$status = 'network-active';
			} elseif ( $is_active ) {
				$status = 'active';
			}

			$plugin_list[] = array(
				'name'    => $plugin_data['Name'],
				'version' => $plugin_data['Version'],
				'author'  => wp_strip_all_tags( $plugin_data['Author'] ),
				'status'  => $status,
			);
		}

		foreach ( $mu_plugins as $mu_plugin_path => $mu_plugin_data ) {
			$plugin_list[] = array(
				'name'    => $mu_plugin_data['Name'],
				'version' => $mu_plugin_data['Version'],
				'author'  => wp_strip_all_tags( $mu_plugin_data['Author'] ),
				'status'  => 'mu-plugin',
			);
		}

		$premium_plugins   = 0;
		$cf7_addons        = 0;
		$mailchimp_plugins = 0;
		$security_plugins  = 0;
		$cache_plugins     = 0;
		$seo_plugins       = 0;

		foreach ( $all_plugins as $plugin_path => $plugin_data ) {
			$name = strtolower( $plugin_data['Name'] );

			if ( strpos( $name, 'pro' ) !== false || strpos( $name, 'premium' ) !== false ) {
				++$premium_plugins;
			}
			if ( strpos( $name, 'contact form 7' ) !== false ) {
				++$cf7_addons;
			}
			if ( strpos( $name, 'mailchimp' ) !== false ) {
				++$mailchimp_plugins;
			}
			if ( strpos( $name, 'security' ) !== false || strpos( $name, 'wordfence' ) !== false || strpos( $name, 'sucuri' ) !== false ) {
				++$security_plugins;
			}
			if ( strpos( $name, 'cache' ) !== false || strpos( $name, 'wp rocket' ) !== false || strpos( $name, 'w3 total cache' ) !== false ) {
				++$cache_plugins;
			}
			if ( strpos( $name, 'seo' ) !== false || strpos( $name, 'yoast' ) !== false ) {
				++$seo_plugins;
			}
		}

		// Removed: akismet (not relevant for our use case).
		$known_plugins = array(
			'woocommerce' => is_plugin_active( 'woocommerce/woocommerce.php' ),
			'elementor'   => is_plugin_active( 'elementor/elementor.php' ),
			'jetpack'     => is_plugin_active( 'jetpack/jetpack.php' ),
			'wordfence'   => is_plugin_active( 'wordfence/wordfence.php' ),
			'yoast_seo'   => is_plugin_active( 'wordpress-seo/wp-seo.php' ),
		);

		$data = array(
			'total_plugins'     => count( $all_plugins ),
			'active_plugins'    => count( $active_plugins ),
			'inactive_plugins'  => count( $all_plugins ) - count( $active_plugins ),
			'mu_plugins'        => count( $mu_plugins ),
			'premium_plugins'   => $premium_plugins,
			'cf7_addons'        => $cf7_addons,
			'mailchimp_plugins' => $mailchimp_plugins,
			'security_plugins'  => $security_plugins,
			'cache_plugins'     => $cache_plugins,
			'seo_plugins'       => $seo_plugins,
			'has_woocommerce'   => $known_plugins['woocommerce'],
			'has_elementor'     => $known_plugins['elementor'],
			'has_jetpack'       => $known_plugins['jetpack'],
			'has_wordfence'     => $known_plugins['wordfence'],
			'has_yoast_seo'     => $known_plugins['yoast_seo'],
			'plugin_list'       => $plugin_list,
		);

		// Filter out zero/false values.
		return array_filter( $data, fn( $v ) => $v !== 0 && $v !== false && $v !== '' && $v !== array() );
	}

	/**
	 * Collect competitor plugin information.
	 *
	 * Detects known CF7-Mailchimp competitors to understand:
	 * - Churn risk (competitor installed alongside us)
	 * - Competitive landscape (which competitors are most common)
	 * - Migration opportunities (competitor active, we're inactive)
	 */
	private static function collect_competitors() {
		if ( ! function_exists( 'is_plugin_active' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		// Known competitors: CF7 + Mailchimp integration plugins.
		$competitors = array(
			// MC4WP: Mailchimp for WordPress (most popular competitor).
			'mc4wp'              => array(
				'slug'      => 'mailchimp-for-wp/mailchimp-for-wp.php',
				'name'      => 'MC4WP: Mailchimp for WordPress',
				'installed' => false,
				'active'    => false,
			),
			// MC4WP Premium (paid version).
			'mc4wp_premium'      => array(
				'slug'      => 'mc4wp-premium/mc4wp-premium.php',
				'name'      => 'MC4WP Premium',
				'installed' => false,
				'active'    => false,
			),
			// Mailchimp for WooCommerce (official plugin).
			'mailchimp_woo'      => array(
				'slug'      => 'mailchimp-for-woocommerce/mailchimp-woocommerce.php',
				'name'      => 'Mailchimp for WooCommerce',
				'installed' => false,
				'active'    => false,
			),
			// CRM Perks: Contact Form 7 Mailchimp Add-on.
			'crm_perks'          => array(
				'slug'      => 'cf7-mailchimp/cf7-mailchimp.php',
				'name'      => 'CRM Perks CF7 Mailchimp',
				'installed' => false,
				'active'    => false,
			),
			// Easy Forms for Mailchimp.
			'easy_forms'         => array(
				'slug'      => 'jetwp-easy-mailchimp/jetwp-easy-mailchimp.php',
				'name'      => 'Easy Forms for Mailchimp',
				'installed' => false,
				'active'    => false,
			),
			// Jetrail CF7 to Mailchimp.
			'jetrail'            => array(
				'slug'      => 'jetrail-cf7-mailchimp/jetrail-cf7-mailchimp.php',
				'name'      => 'Jetrail CF7 Mailchimp',
				'installed' => false,
				'active'    => false,
			),
			// Contact Form 7 Mailchimp Extension by Jetrail (another variant).
			'cf7_mailchimp_ext'  => array(
				'slug'      => 'contact-form-7-mailchimp-extension-jetrail/cf7-mailchimp-ext.php',
				'name'      => 'CF7 Mailchimp Extension Jetrail',
				'installed' => false,
				'active'    => false,
			),
			// Newsletter plugin (generic competitor).
			'newsletter'         => array(
				'slug'      => 'newsletter/plugin.php',
				'name'      => 'Newsletter',
				'installed' => false,
				'active'    => false,
			),
			// MailPoet (email marketing competitor).
			'mailpoet'           => array(
				'slug'      => 'mailpoet/mailpoet.php',
				'name'      => 'MailPoet',
				'installed' => false,
				'active'    => false,
			),
			// Fluent Forms (form builder with Mailchimp integration).
			'fluent_forms'       => array(
				'slug'      => 'fluentform/fluentform.php',
				'name'      => 'Fluent Forms',
				'installed' => false,
				'active'    => false,
			),
			// WPForms (form builder with Mailchimp addon).
			'wpforms'            => array(
				'slug'      => 'wpforms-lite/wpforms.php',
				'name'      => 'WPForms',
				'installed' => false,
				'active'    => false,
			),
			// Gravity Forms (premium form builder).
			'gravity_forms'      => array(
				'slug'      => 'gravityforms/gravityforms.php',
				'name'      => 'Gravity Forms',
				'installed' => false,
				'active'    => false,
			),
			// Ninja Forms.
			'ninja_forms'        => array(
				'slug'      => 'ninja-forms/ninja-forms.php',
				'name'      => 'Ninja Forms',
				'installed' => false,
				'active'    => false,
			),
			// Formidable Forms.
			'formidable'         => array(
				'slug'      => 'formidable/formidable.php',
				'name'      => 'Formidable Forms',
				'installed' => false,
				'active'    => false,
			),
			// HubSpot (CRM with forms).
			'hubspot'            => array(
				'slug'      => 'leadin/leadin.php',
				'name'      => 'HubSpot',
				'installed' => false,
				'active'    => false,
			),
			// Elementor Forms (Pro feature).
			'elementor_pro'      => array(
				'slug'      => 'elementor-pro/elementor-pro.php',
				'name'      => 'Elementor Pro',
				'installed' => false,
				'active'    => false,
			),
		);

		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$all_plugins = get_plugins();

		// Check each competitor.
		foreach ( $competitors as $key => &$competitor ) {
			$competitor['installed'] = isset( $all_plugins[ $competitor['slug'] ] );
			$competitor['active']    = is_plugin_active( $competitor['slug'] );
		}
		unset( $competitor );

		// Build summary.
		$installed_count = 0;
		$active_count    = 0;
		$installed_list  = array();
		$active_list     = array();

		foreach ( $competitors as $key => $competitor ) {
			if ( $competitor['installed'] ) {
				++$installed_count;
				$installed_list[] = $key;
			}
			if ( $competitor['active'] ) {
				++$active_count;
				$active_list[] = $key;
			}
		}

		// Determine risk level.
		$risk_level = 'none';
		if ( $active_count > 0 ) {
			$risk_level = 'high'; // Competitor is active alongside us.
		} elseif ( $installed_count > 0 ) {
			$risk_level = 'medium'; // Competitor installed but not active.
		}

		return array(
			// Summary flags (quick lookups).
			'has_competitors'        => $installed_count > 0,
			'competitors_installed'  => $installed_count,
			'competitors_active'     => $active_count,
			'churn_risk'             => $risk_level,

			// Lists for analysis.
			'installed_list'         => $installed_list,
			'active_list'            => $active_list,

			// Individual competitor status (flat structure for easy querying).
			'mc4wp_installed'        => $competitors['mc4wp']['installed'],
			'mc4wp_active'           => $competitors['mc4wp']['active'],
			'mc4wp_premium_installed' => $competitors['mc4wp_premium']['installed'],
			'mc4wp_premium_active'   => $competitors['mc4wp_premium']['active'],
			'mailchimp_woo_installed' => $competitors['mailchimp_woo']['installed'],
			'mailchimp_woo_active'   => $competitors['mailchimp_woo']['active'],
			'crm_perks_installed'    => $competitors['crm_perks']['installed'],
			'crm_perks_active'       => $competitors['crm_perks']['active'],
			'mailpoet_installed'     => $competitors['mailpoet']['installed'],
			'mailpoet_active'        => $competitors['mailpoet']['active'],
			'wpforms_installed'      => $competitors['wpforms']['installed'],
			'wpforms_active'         => $competitors['wpforms']['active'],
			'fluent_forms_installed' => $competitors['fluent_forms']['installed'],
			'fluent_forms_active'    => $competitors['fluent_forms']['active'],
			'gravity_forms_installed' => $competitors['gravity_forms']['installed'],
			'gravity_forms_active'   => $competitors['gravity_forms']['active'],
			'ninja_forms_installed'  => $competitors['ninja_forms']['installed'],
			'ninja_forms_active'     => $competitors['ninja_forms']['active'],
			'elementor_pro_installed' => $competitors['elementor_pro']['installed'],
			'elementor_pro_active'   => $competitors['elementor_pro']['active'],
			'hubspot_installed'      => $competitors['hubspot']['installed'],
			'hubspot_active'         => $competitors['hubspot']['active'],
		);
	}

	private static function collect_server() {
		$server_load = array( 0, 0, 0 );
		if ( function_exists( 'sys_getloadavg' ) ) {
			$load = @sys_getloadavg();
			if ( false !== $load && is_array( $load ) ) {
				$server_load = $load;
			}
		}

		$disk_free  = 0;
		$disk_total = 0;
		if ( function_exists( 'disk_free_space' ) ) {
			$free = @disk_free_space( ABSPATH );
			if ( false !== $free ) {
				$disk_free = $free;
			}
		}
		if ( function_exists( 'disk_total_space' ) ) {
			$total = @disk_total_space( ABSPATH );
			if ( false !== $total ) {
				$disk_total = $total;
			}
		}
		$disk_used          = $disk_total - $disk_free;
		$disk_usage_percent = $disk_total > 0 ? round( ( $disk_used / $disk_total ) * 100, 2 ) : 0;

		$hostname = '';
		if ( function_exists( 'gethostname' ) ) {
			$name = @gethostname();
			if ( false !== $name ) {
				$hostname = $name;
			}
		}

		$architecture = '';
		if ( function_exists( 'php_uname' ) ) {
			$arch = @php_uname( 'm' );
			if ( false !== $arch ) {
				$architecture = $arch;
			}
		}

		// Removed: disk_free_bytes, disk_total_bytes, disk_used_bytes, disk_free_gb (redundant - keep percent and total_gb only).
		return array(
			'load_average_1min'   => isset( $server_load[0] ) ? round( (float) $server_load[0], 2 ) : 0,
			'load_average_5min'   => isset( $server_load[1] ) ? round( (float) $server_load[1], 2 ) : 0,
			'load_average_15min'  => isset( $server_load[2] ) ? round( (float) $server_load[2], 2 ) : 0,
			'disk_usage_percent'  => $disk_usage_percent,
			'disk_total_gb'       => $disk_total ? round( $disk_total / 1024 / 1024 / 1024, 2 ) : 0,
			'server_ip'           => hash( 'sha256', isset( $_SERVER['SERVER_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_ADDR'] ) ) : '' ),
			'server_hostname'     => hash( 'sha256', $hostname ),
			'server_os'           => PHP_OS,
			'server_architecture' => $architecture,
		);
	}

	private static function collect_wordpress() {
		global $wpdb;

		$post_counts    = wp_count_posts( 'post' );
		$page_counts    = wp_count_posts( 'page' );
		$comment_counts = wp_count_comments();
		$user_count     = count_users();
		$media_counts   = wp_count_posts( 'attachment' );
		$category_count = wp_count_terms( 'category' );
		$tag_count      = wp_count_terms( 'post_tag' );
		$revision_count = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'revision'" );

		// Build data array, filter out zero values (removed pages_draft, comments_total, etc).
		$data = array(
			'posts_published'      => isset( $post_counts->publish ) ? (int) $post_counts->publish : 0,
			'posts_draft'          => isset( $post_counts->draft ) ? (int) $post_counts->draft : 0,
			'pages_published'      => isset( $page_counts->publish ) ? (int) $page_counts->publish : 0,
			'media_items'          => isset( $media_counts->inherit ) ? (int) $media_counts->inherit : 0,
			'comments_pending'     => isset( $comment_counts->moderated ) ? (int) $comment_counts->moderated : 0,
			'comments_spam'        => isset( $comment_counts->spam ) ? (int) $comment_counts->spam : 0,
			'users_total'          => isset( $user_count['total_users'] ) ? (int) $user_count['total_users'] : 0,
			'users_administrators' => isset( $user_count['avail_roles']['administrator'] ) ? (int) $user_count['avail_roles']['administrator'] : 0,
			'users_editors'        => isset( $user_count['avail_roles']['editor'] ) ? (int) $user_count['avail_roles']['editor'] : 0,
			'users_authors'        => isset( $user_count['avail_roles']['author'] ) ? (int) $user_count['avail_roles']['author'] : 0,
			'users_subscribers'    => isset( $user_count['avail_roles']['subscriber'] ) ? (int) $user_count['avail_roles']['subscriber'] : 0,
			'categories_count'     => is_wp_error( $category_count ) ? 0 : (int) $category_count,
			'tags_count'           => is_wp_error( $tag_count ) ? 0 : (int) $tag_count,
			'revisions_count'      => (int) $revision_count,
		);

		// Filter out zero values to reduce payload size.
		$data = array_filter( $data, fn( $v ) => $v !== 0 );

		// Always include auto_updates_enabled (boolean flag, even if false).
		$data['auto_updates_enabled'] = (bool) get_option( 'auto_update_plugins', false );

		return $data;
	}

	private static function get_cf7_forms() {
		if ( ! class_exists( 'WPCF7_ContactForm' ) ) {
			return array();
		}

		$posts = get_posts(
			array(
				'post_type'      => 'wpcf7_contact_form',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			)
		);

		$forms = array();
		foreach ( $posts as $post ) {
			$form = \WPCF7_ContactForm::get_instance( $post->ID );
			if ( $form ) {
				$forms[] = $form;
			}
		}

		return $forms;
	}
}
