<?php
/**
 * REST API endpoints.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;


function cmatic_register_rest_routes() {
	register_rest_route(
		'chimpmatic-lite/v1',
		'/lists',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_get_lists',
			'permission_callback' => 'cmatic_rest_api_key_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'validate_callback' => function ( $param ) {
						return is_numeric( $param ) && $param > 0;
					},
				),
				'api_key' => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'validate_callback' => function ( $param ) {
						return preg_match( '/^[a-f0-9]{32}-[a-z]{2,3}\d+$/', $param );
					},
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/settings/toggle',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_toggle_setting',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'field'   => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
				'enabled' => array(
					'required'          => true,
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/notices/dismiss',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_dismiss_notice',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'notice_id' => array(
					'required'          => false,
					'type'              => 'string',
					'default'           => 'news',
					'sanitize_callback' => 'sanitize_text_field',
					'enum'              => array( 'news', 'upgrade' ),
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/logs/browser',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_log_browser_console',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'level'   => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'validate_callback' => function ( $param ) {
						return in_array( $param, array( 'log', 'info', 'warn', 'error', 'debug' ), true );
					},
				),
				'message' => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
				'data'    => array(
					'required'          => false,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_textarea_field',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/settings/reset',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_reset_settings',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'type'    => array(
					'required'          => false,
					'type'              => 'string',
					'default'           => 'form',
					'enum'              => array( 'form', 'nuclear' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				'form_id' => array(
					'required'          => false,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/merge-fields',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_get_merge_fields',
			'permission_callback' => 'cmatic_rest_api_key_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
				),
				'list_id' => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/tags/toggle',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_toggle_tag',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
				),
				'tag'     => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
				'enabled' => array(
					'required'          => true,
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/form/field',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_save_form_field',
			'permission_callback' => 'cmatic_rest_api_key_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
				),
				'field'   => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
				'value'   => array(
					'required'          => false,
					'default'           => null,
				),
			),
		)
	);

	register_rest_route(
		'cmatic',
		'/form/setting',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_save_form_setting',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'validate_callback' => function ( $param ) {
						return is_numeric( $param ) && $param > 0;
					},
				),
				'field'   => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_key',
				),
				'value'   => array(
					'required'          => true,
					'type'              => 'boolean',
					'sanitize_callback' => function ( $value ) {
						return Cmatic_Utils::validate_bool( $value );
					},
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/api-key/(?P<form_id>\d+)',
		array(
			'methods'             => 'GET',
			'callback'            => 'cmatic_rest_get_api_key',
			'permission_callback' => 'cmatic_rest_api_key_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'validate_callback' => function ( $param ) {
						return is_numeric( $param ) && $param > 0;
					},
				),
			),
		)
	);
}
add_action( 'rest_api_init', 'cmatic_register_rest_routes' );

function cmatic_rest_permission_check( $request ) {
	if ( ! current_user_can( 'manage_options' ) ) {
		return new WP_Error(
			'rest_forbidden',
			esc_html__( 'You do not have permission to access this endpoint.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	$nonce = $request->get_header( 'X-WP-Nonce' );
	if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
		return new WP_Error(
			'rest_cookie_invalid_nonce',
			esc_html__( 'Cookie nonce is invalid.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	return true;
}

function cmatic_rest_api_key_permission_check( $request ) {
	$form_id = $request->get_param( 'form_id' );

	if ( ! current_user_can( 'wpcf7_edit_contact_form', $form_id ) ) {
		return new WP_Error(
			'rest_forbidden',
			esc_html__( 'You do not have permission to access the API key.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	$nonce = $request->get_header( 'X-WP-Nonce' );
	if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
		return new WP_Error(
			'rest_cookie_invalid_nonce',
			esc_html__( 'Cookie nonce is invalid.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	return true;
}


function cmatic_rest_get_lists( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$api_key = $request->get_param( 'api_key' );

	if ( ! mce_get_cmatic( 'api.sync_attempted' ) ) {
		mce_update_cmatic( 'api.sync_attempted', time() );
	}
	$current_count = (int) mce_get_cmatic( 'api.sync_attempts_count', 0 );
	mce_update_cmatic( 'api.sync_attempts_count', $current_count + 1 );

	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	if ( ! is_array( $cf7_mch ) ) {
		$cf7_mch = array();
	}

	$logfile_enabled = (bool) get_option( CMATIC_LOG_OPTION, false );

	try {
		$validation_result = wpcf7_mce_validate_api_key( $api_key, $logfile_enabled, $option_name );
		$api_valid         = $validation_result['api-validation'] ?? 0;

		$lists_result      = wpcf7_mce_listasasociadas( $api_key, $logfile_enabled, $option_name, $api_valid );
		$lists_data        = $lists_result['lisdata'] ?? array();
		$merge_fields_data = $lists_result['merge_fields'] ?? array();

		$lists = array();
		if ( $api_valid === 1 && isset( $lists_data['lists'] ) && is_array( $lists_data['lists'] ) ) {
			foreach ( $lists_data['lists'] as $list ) {
				if ( is_array( $list ) && isset( $list['id'], $list['name'] ) ) {
					$member_count = isset( $list['stats']['member_count'] ) ? intval( $list['stats']['member_count'] ) : 0;
					$field_count  = isset( $list['stats']['merge_field_count'] ) ? intval( $list['stats']['merge_field_count'] ) : 0;

					$lists[] = array(
						'id'           => $list['id'],
						'name'         => $list['name'],
						'member_count' => $member_count,
						'field_count'  => $field_count,
					);
				}
			}
		}

		$excluded_types = array( 'address', 'birthday', 'imageurl', 'zip' );
		$merge_fields   = array();

		$merge_fields[] = array(
			'tag'  => 'EMAIL',
			'name' => 'Subscriber Email',
			'type' => 'email',
		);

		if ( isset( $merge_fields_data['merge_fields'] ) && is_array( $merge_fields_data['merge_fields'] ) ) {
			$fields_to_process = $merge_fields_data['merge_fields'];
			usort(
				$fields_to_process,
				function ( $a, $b ) {
					return ( $a['display_order'] ?? 0 ) - ( $b['display_order'] ?? 0 );
				}
			);

			$count = 1;
			foreach ( $fields_to_process as $field ) {
				$field_type = strtolower( $field['type'] ?? '' );
				$field_tag  = $field['tag'] ?? '';

				if ( $field_tag === 'EMAIL' ) {
					continue;
				}

				if ( in_array( $field_type, $excluded_types, true ) ) {
					continue;
				}

				if ( $count >= CMATIC_LITE_FIELDS ) {
					break;
				}

				$merge_fields[] = array(
					'tag'  => $field_tag,
					'name' => $field['name'] ?? '',
					'type' => $field_type,
				);
				++$count;
			}
		}

		$settings_to_save = array_merge(
			$cf7_mch,
			$validation_result,
			$lists_result,
			array(
				'api'          => $api_key,
				'merge_fields' => $merge_fields,
			)
		);
		update_option( $option_name, $settings_to_save );

		if ( ! empty( $lists_result['lisdata'] ) ) {
			mce_update_cmatic( 'lisdata', $lists_result['lisdata'] );
			mce_update_cmatic( 'lisdata_updated', time() );
		}

		return rest_ensure_response(
			array(
				'success'      => true,
				'api_valid'    => $api_valid === 1,
				'lists'        => $lists,
				'total'        => count( $lists ),
				'merge_fields' => $merge_fields,
			)
		);

	} catch ( Exception $e ) {
		$logger = new Cmatic_Logger( true, 'REST-API-Error', 0 );
		$logger->log( 'ERROR', 'REST API list loading failed.', $e->getMessage() );

		return new WP_Error(
			'api_request_failed',
			esc_html__( 'Failed to load Mailchimp lists. Check debug log for details.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}
}

function cmatic_rest_toggle_setting( $request ) {
	$field   = $request->get_param( 'field' );
	$enabled = $request->get_param( 'enabled' );

	$allowed_fields = array(
		'double_optin' => array(
			'type' => 'cmatic',
			'path' => 'double_optin',
		),
		'unsubscribed' => array(
			'type' => 'cmatic',
			'path' => 'unsubscribed',
		),
		'debug'        => array(
			'type' => 'cmatic',
			'path' => 'debug',
		),
		'backlink'     => array(
			'type' => 'cmatic',
			'path' => 'backlink',
		),
		'auto_update'  => array(
			'type' => 'cmatic',
			'path' => 'auto_update',
		),
		'telemetry'    => array(
			'type' => 'cmatic',
			'path' => 'telemetry.enabled',
		),
	);

	if ( ! array_key_exists( $field, $allowed_fields ) ) {
		return new WP_Error(
			'invalid_field',
			__( 'Invalid settings field.', 'chimpmatic-lite' ),
			array( 'status' => 400 )
		);
	}

	$field_config = $allowed_fields[ $field ];

	if ( 'telemetry' === $field ) {
		if ( class_exists( 'Cmatic\\Metrics\\Core\\Storage' ) && class_exists( 'Cmatic\\Metrics\\Core\\Tracker' ) ) {
			$storage_enabled = \Cmatic\Metrics\Core\Storage::is_enabled();
			if ( ! $enabled && $storage_enabled ) {
				\Cmatic\Metrics\Core\Tracker::on_opt_out();
			}
			if ( $enabled && ! $storage_enabled ) {
				\Cmatic\Metrics\Core\Tracker::on_re_enable();
			}
		}
	}

	mce_update_cmatic( $field_config['path'], $enabled ? 1 : 0 );

	$field_labels = array(
		'double_optin' => __( 'Double Opt-in', 'chimpmatic-lite' ),
		'unsubscribed' => __( 'Unsubscribed', 'chimpmatic-lite' ),
		'debug'        => __( 'Debug Logger', 'chimpmatic-lite' ),
		'backlink'     => __( 'Developer Backlink', 'chimpmatic-lite' ),
		'auto_update'  => __( 'Auto Update', 'chimpmatic-lite' ),
		'telemetry'    => __( 'Usage Statistics', 'chimpmatic-lite' ),
	);

	$label = $field_labels[ $field ] ?? ucfirst( str_replace( '_', ' ', $field ) );

	return rest_ensure_response(
		array(
			'success' => true,
			'field'   => $field,
			'enabled' => $enabled,
			'message' => $enabled
				? sprintf( __( '%s enabled.', 'chimpmatic-lite' ), $label )
				: sprintf( __( '%s disabled.', 'chimpmatic-lite' ), $label ),
		)
	);
}

function cmatic_rest_toggle_tag( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$tag     = $request->get_param( 'tag' );
	$enabled = $request->get_param( 'enabled' );

	if ( ! $form_id || ! $tag ) {
		return new WP_Error(
			'missing_params',
			__( 'Missing required parameters.', 'chimpmatic-lite' ),
			array( 'status' => 400 )
		);
	}

	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	if ( ! isset( $cf7_mch['labeltags'] ) || ! is_array( $cf7_mch['labeltags'] ) ) {
		$cf7_mch['labeltags'] = array();
	}

	if ( $enabled ) {
		$cf7_mch['labeltags'][ $tag ] = '1';
	} else {
		unset( $cf7_mch['labeltags'][ $tag ] );
	}

	update_option( $option_name, $cf7_mch );

	return rest_ensure_response(
		array(
			'success' => true,
			'tag'     => $tag,
			'enabled' => $enabled,
			'message' => $enabled
				? sprintf( __( 'Tag [%s] enabled.', 'chimpmatic-lite' ), $tag )
				: sprintf( __( 'Tag [%s] disabled.', 'chimpmatic-lite' ), $tag ),
		)
	);
}

function cmatic_rest_save_form_field( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$field   = $request->get_param( 'field' );
	$value   = $request->get_param( 'value' );

	$allowed_patterns = array(
		'labeltags\\.(.+)'      => array(
			'type'     => 'boolean',
			'pro_only' => false,
			'nested'   => 'labeltags',
		),
		'field(\\d+)'           => array(
			'type'     => 'string',
			'pro_only' => false,
			'direct'   => true,
		),
		'customKey(\\d+)'       => array(
			'type'     => 'string',
			'pro_only' => false,
			'direct'   => true,
		),
		'customValue(\\d+)'     => array(
			'type'     => 'string',
			'pro_only' => false,
			'direct'   => true,
		),
		'GDPRCheck(\\d+)'       => array(
			'type'     => 'boolean',
			'pro_only' => true,
			'direct'   => true,
		),
		'GDPRCustomValue(\\d+)' => array(
			'type'     => 'string',
			'pro_only' => true,
			'direct'   => true,
		),
		'ggCheck(\\d+)'         => array(
			'type'     => 'boolean',
			'pro_only' => true,
			'direct'   => true,
		),
		'ggCustomValue(\\d+)'   => array(
			'type'     => 'string',
			'pro_only' => true,
			'direct'   => true,
		),
	);

	$matched_config = null;
	$matched_key    = null;

	foreach ( $allowed_patterns as $pattern => $config ) {
		if ( preg_match( '/^' . $pattern . '$/', $field, $matches ) ) {
			$matched_config = $config;
			$matched_key    = isset( $matches[1] ) ? $matches[1] : null;
			break;
		}
	}

	if ( null === $matched_config ) {
		return new WP_Error(
			'invalid_field',
			sprintf( __( 'Field "%s" is not allowed.', 'chimpmatic-lite' ), $field ),
			array( 'status' => 400 )
		);
	}

	if ( $matched_config['pro_only'] && ! defined( 'CMATIC_VERSION' ) ) {
		return new WP_Error(
			'pro_required',
			__( 'This field requires ChimpMatic PRO.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	if ( 'boolean' === $matched_config['type'] ) {
		$value = rest_sanitize_boolean( $value );
	} elseif ( 'string' === $matched_config['type'] ) {
		$value = trim( sanitize_text_field( $value ) );
	}

	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	if ( isset( $matched_config['nested'] ) ) {
		$nested_key = $matched_config['nested'];
		if ( ! isset( $cf7_mch[ $nested_key ] ) ) {
			$cf7_mch[ $nested_key ] = array();
		}

		if ( $value ) {
			$cf7_mch[ $nested_key ][ $matched_key ] = true;
		} else {
			unset( $cf7_mch[ $nested_key ][ $matched_key ] );
		}
	} elseif ( null === $value || '' === $value ) {
		unset( $cf7_mch[ $field ] );
	} else {
		$cf7_mch[ $field ] = $value;
	}

	update_option( $option_name, $cf7_mch );

	return rest_ensure_response(
		array(
			'success' => true,
			'field'   => $field,
			'value'   => $value,
			'message' => __( 'Field saved successfully.', 'chimpmatic-lite' ),
		)
	);
}

/**
 * Save a per-form boolean setting.
 *
 * Endpoint: POST /wp-json/cmatic/form/setting
 *
 * @since 0.9.69
 *
 * @param WP_REST_Request $request The REST request.
 * @return WP_REST_Response|WP_Error Response or error.
 */
function cmatic_rest_save_form_setting( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$field   = $request->get_param( 'field' );
	$value   = $request->get_param( 'value' );

	$allowed_fields = array(
		'sync_tags'    => array(
			'label'    => __( 'Sync Tags', 'chimpmatic-lite' ),
			'pro_only' => false,
		),
		'double_optin' => array(
			'label'    => __( 'Double Opt-in', 'chimpmatic-lite' ),
			'pro_only' => false,
		),
	);

	/**
	 * Filter the allowed per-form setting fields.
	 *
	 * Pro can extend this to add GDPR, Groups/Interests, etc.
	 *
	 * @since 0.9.69
	 *
	 * @param array $allowed_fields Associative array of field_name => config.
	 * @param int   $form_id        The CF7 form ID.
	 */
	$allowed_fields = apply_filters( 'cmatic_form_setting_fields', $allowed_fields, $form_id );

	if ( ! array_key_exists( $field, $allowed_fields ) ) {
		return new WP_Error(
			'invalid_field',
			sprintf(
				/* translators: %s: field name */
				__( 'Field "%s" is not a valid setting.', 'chimpmatic-lite' ),
				$field
			),
			array( 'status' => 400 )
		);
	}

	$field_config = $allowed_fields[ $field ];

	if ( ! empty( $field_config['pro_only'] ) && ! defined( 'CMATIC_VERSION' ) ) {
		return new WP_Error(
			'pro_required',
			sprintf(
				/* translators: %s: field label */
				__( '%s requires ChimpMatic Pro.', 'chimpmatic-lite' ),
				$field_config['label']
			),
			array( 'status' => 403 )
		);
	}

	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	if ( ! is_array( $cf7_mch ) ) {
		$cf7_mch = array();
	}

	$cf7_mch[ $field ] = $value ? 1 : 0;

	update_option( $option_name, $cf7_mch );

	return rest_ensure_response(
		array(
			'success' => true,
			'form_id' => $form_id,
			'field'   => $field,
			'value'   => (bool) $value,
			'message' => $value
				? sprintf(
					/* translators: %s: field label */
					__( '%s enabled.', 'chimpmatic-lite' ),
					$field_config['label']
				)
				: sprintf(
					/* translators: %s: field label */
					__( '%s disabled.', 'chimpmatic-lite' ),
					$field_config['label']
				),
		)
	);
}

function cmatic_rest_get_debug_log( $request ) {
	$log_file_path = WP_CONTENT_DIR . '/debug.log';

	if ( ! file_exists( $log_file_path ) || 0 === filesize( $log_file_path ) ) {
		return rest_ensure_response(
			array(
				'success' => true,
				'content' => esc_html__( 'The debug.log file was not found or is empty.', 'chimpmatic-lite' ),
				'entries' => array(),
			)
		);
	}

	$lines        = file( $log_file_path );
	$recent_lines = array_slice( $lines, -500 );
	$line_count   = count( $recent_lines );
	$filtered_log = '';

	for ( $i = 0; $i < $line_count; $i++ ) {
		if ( strpos( $recent_lines[ $i ], '[ChimpMatic Lite]' ) !== false ) {
			$current_entry = '';
			for ( $j = $i; $j < $line_count; $j++ ) {
				$line_to_check = $recent_lines[ $j ];
				if ( $j > $i && preg_match( '/^\[\d{2}-\w{3}-\d{4} \d{2}:\d{2}:\d{2} UTC\]/', $line_to_check ) ) {
					break;
				}
				$current_entry .= $line_to_check;
			}
			$filtered_log .= $current_entry;
			$i             = $j - 1;
		}
	}

	if ( empty( $filtered_log ) ) {
		$filtered_log = esc_html__( 'No ChimpMatic Lite entries found in the recent log data. Note: This viewer only shows the last 500 lines of the log file.', 'chimpmatic-lite' );
	}

	return rest_ensure_response(
		array(
			'success' => true,
			'content' => trim( $filtered_log ),
		)
	);
}

function cmatic_rest_log_browser_console( $request ) {
	$level   = $request->get_param( 'level' );
	$message = $request->get_param( 'message' );
	$data    = $request->get_param( 'data' );

	$level_map = array(
		'log'   => 'INFO',
		'info'  => 'INFO',
		'warn'  => 'WARNING',
		'error' => 'ERROR',
		'debug' => 'DEBUG',
	);

	$wp_level    = $level_map[ $level ] ?? 'INFO';
	$log_message = sprintf(
		'[%s] [ChimpMatic Lite] [Browser Console - %s] %s',
		gmdate( 'd-M-Y H:i:s' ) . ' UTC',
		strtoupper( $level ),
		$message
	);

	if ( ! empty( $data ) ) {
		$log_message .= ' | Data: ' . $data;
	}

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		error_log( $log_message );
	}

	$cf7_mch        = get_option( 'cf7_mch_' . ( $_POST['form_id'] ?? 0 ), array() );
	$logfile_enabled = (bool) get_option( CMATIC_LOG_OPTION, false );

	if ( class_exists( 'Cmatic_Logger' ) ) {
		$logger = new Cmatic_Logger( $logfile_enabled, 'Browser-Console', 0 );
		$logger->log( $wp_level, 'Browser: ' . $message, $data ? json_decode( $data, true ) : null );
	}

	return rest_ensure_response(
		array(
			'success' => true,
			'logged'  => true,
		)
	);
}

function cmatic_rest_clear_debug_log( $request ) {
	$log_file_path = WP_CONTENT_DIR . '/debug.log';

	if ( ! file_exists( $log_file_path ) ) {
		return rest_ensure_response(
			array(
				'success' => true,
				'message' => esc_html__( 'Debug log file does not exist.', 'chimpmatic-lite' ),
				'cleared' => false,
			)
		);
	}

	$file_handle = @fopen( $log_file_path, 'w' );

	if ( false === $file_handle ) {
		return new WP_Error(
			'file_clear_failed',
			esc_html__( 'Failed to clear debug log file.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}

	fclose( $file_handle );

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		error_log(
			sprintf(
				'[%s] [ChimpMatic Lite] Debug log cleared by user: %s',
				gmdate( 'd-M-Y H:i:s' ) . ' UTC',
				wp_get_current_user()->user_login
			)
		);
	}

	return rest_ensure_response(
		array(
			'success' => true,
			'message' => esc_html__( 'Debug log cleared successfully.', 'chimpmatic-lite' ),
			'cleared' => true,
		)
	);
}

function cmatic_rest_dismiss_notice( $request ) {
	$notice_id = $request->get_param( 'notice_id' );

	switch ( $notice_id ) {
		case 'upgrade':
			mce_update_cmatic( 'ui.upgrade_clicked', true );
			$message = __( 'Upgrade notice dismissed.', 'chimpmatic-lite' );
			break;

		case 'news':
		default:
			mce_update_cmatic( 'ui.news', false );
			$message = __( 'Notice dismissed successfully.', 'chimpmatic-lite' );
			break;
	}

	return rest_ensure_response(
		array(
			'success'   => true,
			'message'   => esc_html( $message ),
			'dismissed' => $notice_id,
		)
	);
}


function cmatic_rest_toggle_telemetry( $request ) {
	$enabled = $request->get_param( 'enabled' );

	if ( class_exists( 'Cmatic\\Metrics\\Core\\Storage' ) && class_exists( 'Cmatic\\Metrics\\Core\\Tracker' ) ) {
		$storage_enabled = \Cmatic\Metrics\Core\Storage::is_enabled();

		if ( ! $enabled && $storage_enabled ) {
			\Cmatic\Metrics\Core\Tracker::on_opt_out();
		}

		if ( $enabled && ! $storage_enabled ) {
			\Cmatic\Metrics\Core\Tracker::on_re_enable();
		}
	}

	mce_update_cmatic( 'telemetry.enabled', $enabled );

	return rest_ensure_response(
		array(
			'success' => true,
			'enabled' => $enabled,
			'message' => $enabled
				? esc_html__( 'Telemetry enabled. Thank you for helping improve the plugin!', 'chimpmatic-lite' )
				: esc_html__( 'Telemetry disabled.', 'chimpmatic-lite' ),
		)
	);
}


function cmatic_rest_get_merge_fields( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$list_id = $request->get_param( 'list_id' );

	$option_name    = 'cf7_mch_' . $form_id;
	$cf7_mch        = get_option( $option_name, array() );
	$api_key        = $cf7_mch['api'] ?? '';
	$logfile_enabled = (bool) get_option( CMATIC_LOG_OPTION, false );

	if ( empty( $api_key ) ) {
		return new WP_Error(
			'missing_api_key',
			esc_html__( 'API key not found. Please connect to Mailchimp first.', 'chimpmatic-lite' ),
			array( 'status' => 400 )
		);
	}

	try {
		$merge_fields_result = wpcf7_mce_get_merge_fields( $api_key, $list_id, $logfile_enabled );
		$merge_fields_data   = $merge_fields_result['merge_fields'] ?? array();

		$excluded_types = array( 'address', 'birthday', 'imageurl', 'zip' );
		$merge_fields   = array();

		$merge_fields[] = array(
			'tag'  => 'EMAIL',
			'name' => 'Subscriber Email',
			'type' => 'email',
		);

		if ( isset( $merge_fields_data['merge_fields'] ) && is_array( $merge_fields_data['merge_fields'] ) ) {
			$fields_to_process = $merge_fields_data['merge_fields'];
			usort(
				$fields_to_process,
				function ( $a, $b ) {
					return ( $a['display_order'] ?? 0 ) - ( $b['display_order'] ?? 0 );
				}
			);

			$count = 1;
			foreach ( $fields_to_process as $field ) {
				$field_type = strtolower( $field['type'] ?? '' );
				$field_tag  = $field['tag'] ?? '';

				if ( $field_tag === 'EMAIL' ) {
					continue;
				}

				if ( in_array( $field_type, $excluded_types, true ) ) {
					continue;
				}

				if ( $count >= CMATIC_LITE_FIELDS ) {
					break;
				}

				$merge_fields[] = array(
					'tag'  => $field_tag,
					'name' => $field['name'] ?? '',
					'type' => $field_type,
				);
				++$count;
			}
		}

		$cf7_mch['merge_fields'] = $merge_fields;
		$cf7_mch['list']         = $list_id;
		update_option( $option_name, $cf7_mch );

		if ( ! mce_get_cmatic( 'api.audience_selected' ) ) {
			mce_update_cmatic( 'api.audience_selected', time() );
		}

		if ( class_exists( 'Cmatic\\Metrics\\Core\\Sync' ) && class_exists( 'Cmatic\\Metrics\\Core\\Collector' ) ) {
			$payload = \Cmatic\Metrics\Core\Collector::collect( 'list_selected' );
			\Cmatic\Metrics\Core\Sync::send_async( $payload );
		}

		return rest_ensure_response(
			array(
				'success'      => true,
				'merge_fields' => $merge_fields,
			)
		);

	} catch ( Exception $e ) {
		return new WP_Error(
			'api_request_failed',
			esc_html__( 'Failed to load merge fields. Check debug log for details.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}
}

function cmatic_rest_reset_settings( $request ) {
	$type = $request->get_param( 'type' );

	if ( 'nuclear' === $type ) {
		return cmatic_rest_nuclear_license_reset( $request );
	}

	$form_id = $request->get_param( 'form_id' );
	if ( ! $form_id ) {
		return new WP_Error(
			'missing_form_id',
			__( 'Form ID is required for form reset.', 'chimpmatic-lite' ),
			array( 'status' => 400 )
		);
	}

	$option_name = 'cf7_mch_' . $form_id;
	delete_option( $option_name );

	return rest_ensure_response(
		array(
			'success' => true,
			'message' => __( 'Form settings cleared successfully.', 'chimpmatic-lite' ),
		)
	);
}

function cmatic_rest_nuclear_license_reset( $request ) {
	global $wpdb;

	$current_user   = wp_get_current_user();
	$username       = $current_user->user_login ?? 'unknown';
	$deleted_counts = array();

	$license_options = array(
		'chimpmatic_license_activation',
		'chimpmatic_license_status',
		'chimpmatic_license_state',
		'chimpmatic_license_last_check',
		'chimpmatic_license_error_state',
		'cmatic_license_activated',
		'cmatic_license_api_key',
		'cmatic_product_id',
		'wc_am_client_chimpmatic',
		'wc_am_product_id_chimpmatic',
		'wc_am_client_chimpmatic_activated',
		'wc_am_client_chimpmatic_instance',
		'wc_am_client_chimpmatic_deactivate_checkbox',
		'chimpmatic_product_id',
	);

	$options_deleted = 0;
	foreach ( $license_options as $option ) {
		if ( delete_option( $option ) ) {
			++$options_deleted;
		}
	}
	$deleted_counts['options'] = $options_deleted;

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
	$transients_deleted           = $wpdb->query(
		"DELETE FROM {$wpdb->options}
        WHERE option_name LIKE '_transient_%chimpmatic%'
           OR option_name LIKE '_transient_timeout_%chimpmatic%'
           OR option_name LIKE '_site_transient_%chimpmatic%'
           OR option_name LIKE '_site_transient_timeout_%chimpmatic%'
           OR option_name LIKE 'site_transient_%chimpmatic%'
           OR option_name LIKE '_transient_%cmatic%'
           OR option_name LIKE '_transient_timeout_%cmatic%'
           OR option_name LIKE '_site_transient_%cmatic%'
           OR option_name LIKE '_site_transient_timeout_%cmatic%'
           OR option_name LIKE 'site_transient_%cmatic%'"
	);
	$deleted_counts['transients'] = (int) $transients_deleted;

	$cache_flushed = false;
	if ( function_exists( 'wp_cache_flush' ) ) {
		$cache_flushed = wp_cache_flush();
	}
	$deleted_counts['cache_flushed'] = $cache_flushed;

	delete_site_transient( 'update_plugins' );
	update_option( 'chimpmatic_license_status', 'deactivated' );

	return rest_ensure_response(
		array(
			'success'        => true,
			'message'        => 'License data completely wiped (nuclear reset)',
			'deleted_counts' => $deleted_counts,
			'performed_by'   => $username,
			'timestamp'      => current_time( 'mysql' ),
			'actions_taken'  => array(
				'Deleted ' . $options_deleted . ' license options',
				'Deleted ' . $transients_deleted . ' cached transients',
				'Flushed object cache: ' . ( $cache_flushed ? 'yes' : 'no' ),
				'Cleared plugin update cache',
				'Set license status to deactivated',
			),
		)
	);
}

function cmatic_rest_get_api_key( $request ) {
	$form_id     = $request->get_param( 'form_id' );
	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	if ( ! is_array( $cf7_mch ) ) {
		$cf7_mch = array();
	}

	$api_key = isset( $cf7_mch['api'] ) ? $cf7_mch['api'] : '';

	return rest_ensure_response(
		array(
			'success' => true,
			'api_key' => $api_key,
		)
	);
}
