<?php
/**
 * Database migration utilities.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;

/** Run migration on plugins_loaded (copies old scattered options to consolidated cmatic array). */
function cmatic_run_migration() {
	if ( ! function_exists( 'mce_get_cmatic' ) || ! function_exists( 'mce_update_cmatic' ) ) {
		return;
	}

	$db_version = mce_get_cmatic( 'db_version' );
	if ( $db_version && $db_version >= 2 ) {
		return;
	}

	$migrated = array();

	$old_auto_update = get_option( 'chimpmatic-update' );
	if ( false !== $old_auto_update ) {
		$new_value = ( '1' === $old_auto_update ) ? 1 : 0;
		mce_update_cmatic( 'auto_update', $new_value );

		if ( (int) mce_get_cmatic( 'auto_update' ) === $new_value ) {
			delete_option( 'chimpmatic-update' );
			$migrated[] = 'auto_update=' . $new_value;
		}
	}

	$old_debug = get_option( 'cmatic_log_on' );
	if ( false !== $old_debug ) {
		$new_value = ( 'on' === $old_debug || '1' === $old_debug ) ? 1 : 0;
		mce_update_cmatic( 'debug', $new_value );

		if ( (int) mce_get_cmatic( 'debug' ) === $new_value ) {
			delete_option( 'cmatic_log_on' );
			$migrated[] = 'debug=' . $new_value;
		}
	}

	$old_redirect = get_option( 'cmatic_do_activation_redirect' );
	if ( false !== $old_redirect ) {
		$new_value = (bool) $old_redirect;
		mce_update_cmatic( 'activation_redirect', $new_value );

		if ( (bool) mce_get_cmatic( 'activation_redirect' ) === $new_value ) {
			delete_option( 'cmatic_do_activation_redirect' );
			$migrated[] = 'activation_redirect=' . ( $new_value ? 'true' : 'false' );
		}
	}

	$old_news_count = get_option( 'cmatic_news_retry_count' );
	if ( false !== $old_news_count ) {
		$new_value = (int) $old_news_count;
		mce_update_cmatic( 'news.retry_count', $new_value );

		if ( (int) mce_get_cmatic( 'news.retry_count' ) === $new_value ) {
			delete_option( 'cmatic_news_retry_count' );
			$migrated[] = 'news.retry_count=' . $new_value;
		}
	}

	$old_last_run = get_option( 'csyncr_last_weekly_run' );
	if ( false !== $old_last_run ) {
		$new_value = (int) $old_last_run;
		mce_update_cmatic( 'telemetry.last_run', $new_value );

		if ( (int) mce_get_cmatic( 'telemetry.last_run' ) === $new_value ) {
			delete_option( 'csyncr_last_weekly_run' );
			$migrated[] = 'telemetry.last_run=' . $new_value;
		}
	}

	mce_update_cmatic( 'db_version', 2 );

	if ( defined( 'WP_DEBUG' ) && WP_DEBUG && ! empty( $migrated ) ) {
		error_log( '[ChimpMatic Lite] Migration complete. Consolidated: ' . implode( ', ', $migrated ) );
	}
}

add_action( 'plugins_loaded', 'cmatic_run_migration', 1 );

/** Clean up old csyncr_ cron events and transients. */
function cmatic_migrate_csyncr_to_cmatic() {
	if ( ! function_exists( 'mce_get_cmatic' ) || ! function_exists( 'mce_update_cmatic' ) ) {
		return;
	}

	$db_version = mce_get_cmatic( 'db_version' );
	if ( $db_version && $db_version >= 3 ) {
		return;
	}

	$migrated = array();

	$weekly_timestamp = wp_next_scheduled( 'csyncr_weekly_telemetry' );
	if ( $weekly_timestamp ) {
		wp_unschedule_event( $weekly_timestamp, 'csyncr_weekly_telemetry' );
		$migrated[] = 'csyncr_weekly_telemetry';
	}

	$heartbeat_timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
	if ( $heartbeat_timestamp ) {
		wp_unschedule_event( $heartbeat_timestamp, 'csyncr_metrics_heartbeat' );
		$migrated[] = 'csyncr_metrics_heartbeat';
	}

	delete_transient( 'csyncr_admin_checked' );

	$crons = _get_cron_array();
	if ( is_array( $crons ) ) {
		foreach ( $crons as $timestamp => $cron_hooks ) {
			foreach ( $cron_hooks as $hook => $events ) {
				if ( strpos( $hook, 'csyncr_' ) === 0 ) {
					foreach ( $events as $key => $event ) {
						wp_unschedule_event( $timestamp, $hook );
						if ( ! in_array( $hook, $migrated, true ) ) {
							$migrated[] = $hook;
						}
					}
				}
			}
		}
	}

	mce_update_cmatic( 'db_version', 3 );

	if ( defined( 'WP_DEBUG' ) && WP_DEBUG && ! empty( $migrated ) ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only.
		error_log( '[ChimpMatic Lite] Cron migration complete. Cleaned: ' . implode( ', ', $migrated ) );
	}
}

add_action( 'plugins_loaded', 'cmatic_migrate_csyncr_to_cmatic', 0 );

/** Add plugin_slug to install section. */
function cmatic_migrate_add_plugin_slug() {
	if ( ! function_exists( 'mce_get_cmatic' ) || ! function_exists( 'mce_update_cmatic' ) ) {
		return;
	}

	$db_version = mce_get_cmatic( 'db_version' );
	if ( $db_version && $db_version >= 4 ) {
		return;
	}

	mce_update_cmatic( 'install.plugin_slug', 'contact-form-7-mailchimp-extension' );
	mce_update_cmatic( 'db_version', 4 );

	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only.
		error_log( '[ChimpMatic Lite] Added plugin_slug to install section.' );
	}
}

add_action( 'plugins_loaded', 'cmatic_migrate_add_plugin_slug', 2 );
