<?php
/**
 * CF7 editor panel and form handler.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;

add_filter( 'wpcf7_editor_panels', 'cmatic_add_editor_panel' );
add_action( 'wpcf7_after_save', 'cmatic_save_form_settings' );


if ( ! defined( 'CMATIC_VERSION' ) ) {
	add_action( 'wpcf7_before_send_mail', 'cmatic_subscribe_submission' );
}

add_action( 'wpcf7_admin_misc_pub_section', 'cmatic_add_version_to_submitdiv' );
add_action( 'wpcf7_admin_footer', 'cmatic_render_footer_banner', 10, 1 );

function cmatic_add_editor_panel( $panels ) {
	if ( defined( 'CMATIC_VERSION' ) ) {
		return $panels;
	}

	$post_id = isset( $_GET['post'] ) ? absint( $_GET['post'] ) : 0;
	if ( ! current_user_can( 'wpcf7_edit_contact_form', $post_id ) ) {
		return $panels;
	}

	$panels['Chimpmatic'] = array(
		'title'    => __( 'Chimpmatic', 'chimpmatic-lite' ),
		'callback' => 'cmatic_editor_panel_content',
	);
	return $panels;
}


function cmatic_editor_panel_content( $contact_form ) {
	$form_id              = $contact_form->id();
	$defaults             = array();
	$cf7_mch              = get_option( 'cf7_mch_' . $form_id, $defaults );
	if ( ! is_array( $cf7_mch ) ) {
		$cf7_mch = array();
	}
	$mce_txcomodin        = $form_id;
	$form_tags_with_types = mce_get_form_tags_with_types( $contact_form );
	$checkbox_fields      = mce_get_checkbox_fields( $contact_form );

	$logfile_enabled = isset( $cf7_mch['logfileEnabled'] ) && '1' === $cf7_mch['logfileEnabled'];
	$mceapi          = $cf7_mch['api'] ?? null;
	$apivalid        = $cf7_mch['api-validation'] ?? null;
	$listdata        = isset( $cf7_mch['lisdata'] ) && is_array( $cf7_mch['lisdata'] ) ? $cf7_mch['lisdata'] : null;

	$chm_valid   = '<span class="chmm valid">API Connected</span>';
	$chm_invalid = '<span class="chmm invalid">API Inactive</span>';

	include SPARTAN_MCE_PLUGIN_DIR . '/lib/view.php';
}


function cmatic_save_form_settings( $contact_form ) {
	if ( ! isset( $_POST['wpcf7-mailchimp'] ) ) {
		return;
	}

	if ( class_exists( 'Cmatic\\Metrics\\Core\\Sync' ) && class_exists( 'Cmatic\\Metrics\\Core\\Collector' ) ) {
		$payload = \Cmatic\Metrics\Core\Collector::collect( 'form_saved' );
		\Cmatic\Metrics\Core\Sync::send( $payload );
	}

	$form_id            = $contact_form->id();
	$option_name        = 'cf7_mch_' . $form_id;
	$old_settings       = get_option( $option_name, array() );
	$posted_data        = $_POST['wpcf7-mailchimp'];
	$sanitized_settings = array();
	$text_fields        = array( 'api', 'list', 'accept' );
	$max_field_index    = CMATIC_LITE_FIELDS + 2;
	for ( $i = 3; $i <= $max_field_index; $i++ ) {
		$text_fields[] = 'field' . $i;
	}

	for ( $i = 1; $i <= 10; $i++ ) {
		$text_fields[] = 'CustomValue' . $i;
		$text_fields[] = 'CustomKey' . $i;
	}

	foreach ( $text_fields as $field ) {
		if ( isset( $posted_data[ $field ] ) ) {
			$sanitized_value = trim( sanitize_text_field( $posted_data[ $field ] ) );
			if ( '' !== $sanitized_value ) {
				$sanitized_settings[ $field ] = $sanitized_value;
			}
		}
	}

	if ( isset( $sanitized_settings['api'] ) && strpos( $sanitized_settings['api'], '•' ) !== false ) {
		if ( ! empty( $old_settings['api'] ) && strpos( $old_settings['api'], '•' ) === false ) {
			$sanitized_settings['api'] = $old_settings['api'];
		}
	}

	$checkbox_fields = array(
		'cfactive',
		'confsubs',
		'addunsubscr',
		'logfileEnabled',
		'cf-supp',
	);

	foreach ( $checkbox_fields as $field ) {
		$sanitized_settings[ $field ] = isset( $posted_data[ $field ] ) ? '1' : '0';
	}

	if ( empty( $sanitized_settings['api'] ) ) {
		delete_option( $option_name );
		return;
	}

	$updated_settings = array_merge( $old_settings, $sanitized_settings );
	$max_field_index  = CMATIC_LITE_FIELDS + 2;
	for ( $i = $max_field_index + 1; $i <= 20; $i++ ) {
		$field_key = 'field' . $i;
		if ( isset( $updated_settings[ $field_key ] ) ) {
			unset( $updated_settings[ $field_key ] );
		}
	}

	update_option( $option_name, $updated_settings );
}


function cmatic_subscribe_submission( $contact_form ) {
	$submission = WPCF7_Submission::get_instance();
	if ( ! $submission ) {
		return;
	}

	$form_id = $contact_form->id();
	$cf7_mch = get_option( 'cf7_mch_' . $form_id );

	if ( empty( $cf7_mch ) || empty( $cf7_mch['api-validation'] ) || 1 != $cf7_mch['api-validation'] || empty( $cf7_mch['api'] ) ) {
		return;
	}

	$logfile_enabled = (bool) mce_get_cmatic( 'debug', false );
	$logger          = new Cmatic_File_Logger( 'api-events', $logfile_enabled );
	$posted_data     = $submission->get_posted_data();
	$regex           = '/\[\s*([a-zA-Z_][0-9a-zA-Z:._-]*)\s*\]/';

	$email = '';
	if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
		foreach ( $cf7_mch['merge_fields'] as $idx => $merge_field ) {
			if ( ( $merge_field['tag'] ?? '' ) === 'EMAIL' ) {
				$field_key = 'field' . ( $idx + 3 );
				if ( ! empty( $cf7_mch[ $field_key ] ) ) {
					$email = cmatic_tag_replace( $regex, $cf7_mch[ $field_key ], $posted_data );
				}
				break;
			}
		}
	}

	if ( ! is_email( $email ) ) {
		$logger->log( 'WARNING', 'Subscription attempt with invalid email address.', $email );
		Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::failure( 'invalid_email', '', $email ) );
		return;
	}

	$list_id = cmatic_tag_replace( $regex, $cf7_mch['list'] ?? '', $posted_data );
	$status  = 'subscribed';

	if ( ! empty( $cf7_mch['confsubs'] ) ) {
		$status = 'pending';
	} elseif ( ! empty( $cf7_mch['accept'] ) ) {
		$acceptance_value = cmatic_tag_replace( $regex, $cf7_mch['accept'], $posted_data );
		if ( empty( $acceptance_value ) ) {
			if ( ! empty( $cf7_mch['addunsubscr'] ) ) {
				$status = 'unsubscribed';
			} else {
				$logger->log( 'INFO', 'Subscription skipped: acceptance checkbox was not checked.' );
				Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::skipped( 'acceptance_not_checked' ) );
				return;
			}
		}
	}

	$merge_vars = array();
	if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
		$field_index     = 3;
		$max_field_index = CMATIC_LITE_FIELDS + 2;

		foreach ( $cf7_mch['merge_fields'] as $merge_field ) {
			$field_key = 'field' . $field_index;
			$merge_tag = $merge_field['tag'] ?? '';

			if ( ! empty( $cf7_mch[ $field_key ] ) && ! empty( $merge_tag ) ) {
				$value = cmatic_tag_replace( $regex, $cf7_mch[ $field_key ], $posted_data );
				if ( ! empty( $value ) ) {
					$merge_vars[ $merge_tag ] = $value;
				}
			}

			++$field_index;
			if ( $field_index > $max_field_index ) {
				break;
			}
		}
	}

	$merge_vars = array_filter(
		$merge_vars,
		function ( $value ) {
			return ! empty( $value ) || 0 === $value || '0' === $value;
		}
	);

	try {
		$logger->log(
			'INFO',
			'Starting subscription process.',
			array(
				'email'   => $email,
				'list_id' => $list_id,
			)
		);

		$payload = array(
			'email_address' => $email,
			'status'        => $status,
		);

		if ( ! empty( $merge_vars ) ) {
			$payload['merge_fields'] = (object) $merge_vars;
		}

		$api_key        = $cf7_mch['api'];
		list($key, $dc) = explode( '-', $api_key );
		$url            = "https://{$dc}.api.mailchimp.com/3.0/lists/{$list_id}/members/" . md5( strtolower( $email ) );
		$info           = wp_json_encode( $payload );

		$logger->log(
			'INFO',
			'Sending data to Mailchimp.',
			array(
				'url'     => $url,
				'payload' => $payload,
			)
		);
		$response_data = cmatic_call_api_put( $api_key, $url, $info );

		$api_response = isset( $response_data[0] ) ? $response_data[0] : array();

		$logger->log( 'INFO', 'Mailchimp API Response.', $api_response );

		if ( false === $response_data[0] ) {
			$logger->log( 'ERROR', 'Network request failed.', array( 'response' => $response_data[1] ) );
			Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::failure( 'network_error', '', $email ) );
		} elseif ( empty( $api_response ) ) {
			$logger->log( 'ERROR', 'Empty API response received.' );
			Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::failure( 'api_error', 'Empty response from Mailchimp API.', $email ) );
		} elseif ( ! empty( $response_data[0]['errors'] ) ) {
			$php_logger = new Cmatic_File_Logger( 'php-errors', $logfile_enabled );
			foreach ( $response_data[0]['errors'] as $error ) {
				$php_logger->log( 'ERROR', 'Mailchimp API Error received.', $error );
			}
			Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::parse_api_error( $api_response, $email ) );
		} elseif ( isset( $api_response['status'] ) && is_int( $api_response['status'] ) && $api_response['status'] >= 400 ) {
			Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::parse_api_error( $api_response, $email ) );
		} elseif ( isset( $api_response['title'] ) && stripos( $api_response['title'], 'error' ) !== false ) {
			Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::parse_api_error( $api_response, $email ) );
		} else {
			mce_save_contador();

			Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::success( $email, $status, $merge_vars, $api_response ) );

			do_action( 'cmatic_subscription_success', $form_id, $email );
		}
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'Subscription process failed with exception.', $e->getMessage() );
		Cmatic_Submission_Feedback::set_result( Cmatic_Submission_Feedback::failure( 'network_error', $e->getMessage(), $email ) );
	}
}

function cmatic_tag_replace( $pattern, $subject, $posted_data ) {
	if ( preg_match( $pattern, $subject, $matches ) > 0 ) {
		if ( isset( $posted_data[ $matches[1] ] ) ) {
			$submitted = $posted_data[ $matches[1] ];
			return is_array( $submitted ) ? implode( ', ', $submitted ) : $submitted;
		}
		return $matches[0];
	}
	return $subject;
}

function mce_save_contador() {
	$count = (int) mce_get_cmatic( 'stats.sent', 0 );
	mce_update_cmatic( 'stats.sent', $count + 1 );
}



function mce_html_panel_listmail( $apivalid, $listdata, $cf7_mch ) {
	$vlist = isset( $cf7_mch['list'] ) ? sanitize_text_field( $cf7_mch['list'] ) : '';
	$count = isset( $listdata['lists'] ) && is_array( $listdata['lists'] ) ? count( $listdata['lists'] ) : 0;
	?>
	<?php if ( 1 === $apivalid && $count > 0 ) : ?>
		<label for="wpcf7-mailchimp-list">
			<?php
			printf( esc_html__( 'Total Mailchimp Audiences: %d', 'chimpmatic-lite' ), esc_html( $count ) );
			?>
		</label><br/>
		<select id="wpcf7-mailchimp-list" name="wpcf7-mailchimp[list]">
			<?php foreach ( $listdata['lists'] as $i => $list ) : ?>
				<?php
				if ( ! is_array( $list ) || ! isset( $list['id'], $list['name'] ) ) {
					continue;
				}
				$member_count = isset( $list['stats']['member_count'] ) ? intval( $list['stats']['member_count'] ) : 0;
				$field_count  = isset( $list['stats']['merge_field_count'] ) ? intval( $list['stats']['merge_field_count'] ) : 0;
				$option_text  = ( $i + 1 ) . ':' . $member_count . ' ' . $list['name'] . '  ' . $field_count . ' fields #' . $list['id'];
				?>
				<option value="<?php echo esc_attr( $list['id'] ); ?>" <?php selected( $vlist, $list['id'] ); ?>>
					<?php echo esc_html( $option_text ); ?>
				</option>
			<?php endforeach; ?>
		</select>

		<button type="button" id="mce_fetch_fields" class="button" <?php echo empty( $vlist ) ? 'disabled' : ''; ?>>
			<?php esc_html_e( 'Sync Fields', 'chimpmatic-lite' ); ?>
		</button>

	<?php elseif ( 1 === $apivalid ) : ?>
		<p><?php esc_html_e( 'No lists found in your Mailchimp account.', 'chimpmatic-lite' ); ?></p>
		<?php
	endif;
}

function mce_render_field_dropdown( $field_name, $form_tags, $cf7_mch, $filter = '', $merge_tag = '' ) {
	$field_name  = sanitize_key( $field_name );
	$saved_value = isset( $cf7_mch[ $field_name ] ) ? trim( sanitize_text_field( $cf7_mch[ $field_name ] ) ) : '';

	if ( '' === $saved_value && ! empty( $merge_tag ) ) {
		$merge_tag_lower = strtolower( $merge_tag );
		foreach ( $form_tags as $tag ) {
			if ( 'email' === $filter && ( 'email' === $tag['basetype'] || false !== strpos( strtolower( $tag['name'] ), 'email' ) ) ) {
				$saved_value = '[' . $tag['name'] . ']';
				break;
			} elseif ( 'email' !== $filter && false !== strpos( strtolower( $tag['name'] ), $merge_tag_lower ) ) {
				$saved_value = '[' . $tag['name'] . ']';
				break;
			}
		}
	}

	?>
	<select class="chm-select" id="wpcf7-mailchimp-<?php echo esc_attr( $field_name ); ?>" name="wpcf7-mailchimp[<?php echo esc_attr( $field_name ); ?>]">
		<?php if ( 'email' !== $filter ) : ?>
			<option value="" <?php selected( $saved_value, '' ); ?>>
				<?php esc_html_e( 'Choose..', 'chimpmatic-lite' ); ?>
			</option>
		<?php endif; ?>

		<?php foreach ( $form_tags as $tag ) : ?>
			<?php
			if ( 'opt-in' === $tag['name'] ) {
				continue;
			}
			if ( 'email' === $filter ) {
				$is_email_field = ( 'email' === $tag['basetype'] || false !== strpos( strtolower( $tag['name'] ), 'email' ) );
				if ( ! $is_email_field ) {
					continue;
				}
			}
			$tag_value = '[' . $tag['name'] . ']';
			?>
			<option value="<?php echo esc_attr( $tag_value ); ?>" <?php selected( $saved_value, $tag_value ); ?>>
				<?php echo esc_html( $tag_value ); ?> - type: <?php echo esc_html( $tag['basetype'] ); ?>
			</option>
		<?php endforeach; ?>
	</select>
	<?php
}


function mce_get_form_tags_with_types( $contact_form ) {
	if ( ! $contact_form ) {
		return array();
	}

	$mail_tags       = $contact_form->collect_mail_tags();
	$all_tags        = $contact_form->scan_form_tags();
	$tags_with_types = array();
	foreach ( $all_tags as $tag ) {
		if ( ! empty( $tag->name ) && in_array( $tag->name, $mail_tags ) ) {
			$tags_with_types[] = array(
				'name'     => $tag->name,
				'basetype' => $tag->basetype,
			);
		}
	}

	return $tags_with_types;
}

function mce_get_checkbox_fields( $contact_form ) {
	if ( ! $contact_form ) {
		return array();
	}

	$tags            = $contact_form->scan_form_tags( array( 'type' => array( 'checkbox', 'acceptance' ) ) );
	$checkbox_fields = array();
	foreach ( $tags as $tag ) {
		if ( ! empty( $tag->name ) ) {
			$checkbox_fields[] = array(
				'name'     => $tag->name,
				'type'     => $tag->type,
				'basetype' => $tag->basetype,
			);
		}
	}

	return $checkbox_fields;
}

function wpcf7_mce_validate_api_key( $input, $logfile_enabled, $idform = '' ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
	$logger = new Cmatic_File_Logger( 'API-Validation', $logfile_enabled );
	try {
		if ( empty( $input ) || ! preg_match( '/^[a-f0-9]{32}-[a-z]{2,3}\d+$/', $input ) ) {
			$logger->log( 'ERROR', 'Invalid API Key format provided.', $input );
			if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
				mce_update_cmatic( 'api.setup_first_failure', time() );
			}
			mce_update_cmatic( 'api.setup_last_failure', time() );
			$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
			mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

			return array( 'api-validation' => 0 );
		}

		list($key, $dc) = explode( '-', $input );
		if ( empty( $key ) || empty( $dc ) ) {
			if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
				mce_update_cmatic( 'api.setup_first_failure', time() );
			}
			mce_update_cmatic( 'api.setup_last_failure', time() );
			$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
			mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

			return array( 'api-validation' => 0 );
		}

		$urltoken = "https://{$dc}.api.mailchimp.com/3.0/ping";
		$response = cmatic_call_api_get( $key, $urltoken );

		if ( is_wp_error( $response[2] ) || 200 !== wp_remote_retrieve_response_code( $response[2] ) ) {
			$error = is_wp_error( $response[2] ) ? $response[2]->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code( $response[2] );
			$logger->log( 'ERROR', 'API Key validation ping failed.', $error );
			if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
				mce_update_cmatic( 'api.setup_first_failure', time() );
			}
			mce_update_cmatic( 'api.setup_last_failure', time() );
			$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
			mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

			return array( 'api-validation' => 0 );
		}

		$logger->log( 'INFO', 'API Key validated successfully.' );
		if ( ! mce_get_cmatic( 'api.setup_first_success' ) ) {
			mce_update_cmatic( 'api.setup_first_success', time() );
		}

		return array( 'api-validation' => 1 );
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'API validation threw an exception.', $e->getMessage() );
		if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
			mce_update_cmatic( 'api.setup_first_failure', time() );
		}
		mce_update_cmatic( 'api.setup_last_failure', time() );
		$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
		mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

		return array( 'api-validation' => 0 );
	}
}

function wpcf7_mce_listasasociadas( $apikey, $logfile_enabled, $idform, $apivalid ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
	$logger = new Cmatic_File_Logger( 'List-Retrieval', $logfile_enabled );
	if ( 1 !== $apivalid ) {
		return array( 'lisdata' => array() );
	}

	try {
		list($key, $dc) = explode( '-', $apikey );
		if ( empty( $key ) || empty( $dc ) ) {
			return array( 'lisdata' => array() );
		}

		$urltoken = "https://{$dc}.api.mailchimp.com/3.0/lists?count=999";
		$response = cmatic_call_api_get( $key, $urltoken );

		if ( is_wp_error( $response[2] ) || 200 !== wp_remote_retrieve_response_code( $response[2] ) ) {
			$error = is_wp_error( $response[2] ) ? $response[2]->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code( $response[2] );
			$logger->log( 'ERROR', 'Failed to retrieve lists from Mailchimp.', $error );
			return array( 'lisdata' => array() );
		}

		$lists_data = $response[0];
		$logger->log( 'INFO', 'Successfully retrieved lists from Mailchimp.', $lists_data );
		return array(
			'lisdata'      => $lists_data,
			'merge_fields' => array(),
		);
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'List retrieval threw an exception.', $e->getMessage() );
		return array( 'lisdata' => array() );
	}
}

function wpcf7_mce_get_merge_fields( $apikey, $list_id, $logfile_enabled ) {
	$logger = new Cmatic_File_Logger( 'Merge-Fields-Retrieval', $logfile_enabled );

	if ( empty( $apikey ) || empty( $list_id ) ) {
		return array( 'merge_fields' => array() );
	}

	try {
		list($key, $dc) = explode( '-', $apikey );
		if ( empty( $key ) || empty( $dc ) ) {
			return array( 'merge_fields' => array() );
		}

		$urltoken = "https://{$dc}.api.mailchimp.com/3.0/lists/{$list_id}/merge-fields?count=50";
		$response = cmatic_call_api_get( $key, $urltoken );

		if ( is_wp_error( $response[2] ) || 200 !== wp_remote_retrieve_response_code( $response[2] ) ) {
			$error = is_wp_error( $response[2] ) ? $response[2]->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code( $response[2] );
			$logger->log( 'ERROR', 'Failed to retrieve merge fields from Mailchimp.', $error );
			return array( 'merge_fields' => array() );
		}

		$merge_fields_data = $response[0];
		$logger->log( 'INFO', 'Successfully retrieved merge fields from Mailchimp.', $merge_fields_data );
		return array( 'merge_fields' => $merge_fields_data );
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'Merge fields retrieval threw an exception.', $e->getMessage() );
		return array( 'merge_fields' => array() );
	}
}

function chmp_new_usr() {
	$help_url = Cmatic_Pursuit::docs( 'how-to-find-your-mailchimp-api-key', 'new_user_help' );
	?>
	<h2>
		<a href="<?php echo esc_url( $help_url ); ?>" class="helping-field" target="_blank" rel="noopener noreferrer">
			<?php esc_html_e( 'How to get your Mailchimp API key.', 'chimpmatic-lite' ); ?>
		</a>
	</h2>
	<?php
}

function cmatic_add_version_to_submitdiv( $post_id ) {
	$cf7_mch   = get_option( 'cf7_mch_' . $post_id, array() );
	$apivalid  = $cf7_mch['api-validation'] ?? null;
	$mce_sents = mce_get_cmatic( 'stats.sent', 0 );

	if ( 1 === (int) $apivalid ) {
		$status_text = '<span class="chmm valid">API Connected</span>';
	} else {
		$status_text = '<span class="chmm invalid">API Inactive</span>';
	}

	?>
	<div class="misc-pub-section chimpmatic-info" id="chimpmatic-version-info">
		<div style="margin-bottom: 3px;">
			<strong><?php echo esc_html__( 'ChimpMatic Lite', 'chimpmatic-lite' ) . ' ' . esc_html( SPARTAN_MCE_VERSION ); ?></strong>
		</div>
		<div style="margin-top: 5px;">
			<div class="mc-stats" style="color: #646970; font-size: 12px; margin-bottom: 3px;">
				<?php
				echo esc_html( $mce_sents ) . ' synced contacts in ' .
					esc_html( mce_diferdays_dateact_date() ) . ' days';
				?>
			</div>
			<div style="margin-bottom: 3px;">
				<?php echo wp_kses_post( $status_text ); ?>
			</div>
		</div>
	</div>
	<?php
}


function cmatic_render_footer_banner( $post ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
	$pricing_fetcher = new CMatic_Remote_Fetcher(
		array(
			'url'            => 'https://api.chimpmatic.com/promo',
			'cache_key'      => 'cmatic_pricing_data',
			'cache_duration' => DAY_IN_SECONDS,
			'fallback_data'  => array(
				'regular_price'    => 39,
				'sale_price'       => 29.25,
				'discount_percent' => 40,
				'coupon_code'      => 'NOW40',
				'formatted'        => '$39 → $29.25 • Save 40%',
			),
		)
	);

	$pricing_data     = $pricing_fetcher->get_data();
	$pricing_text     = isset( $pricing_data['formatted'] ) ? $pricing_data['formatted'] : '$39 → $29.25 • Save 40%';
	$discount_percent = isset( $pricing_data['discount_percent'] ) ? (int) $pricing_data['discount_percent'] : 40;
	$install_id       = mce_get_cmatic( 'install.id' );

	if ( ! $install_id ) {
		mce_migrate_to_cmatic_option();
		$install_id = mce_get_cmatic( 'install.id', 'unknown' );
	}

	$pursuit_addy = add_query_arg(
		array(
			'source' => $install_id,
			'promo'  => 'pro' . $discount_percent,
		),
		Cmatic_Pursuit::promo( 'footer_banner', $discount_percent )
	);

	?>
	<div id="informationdiv_aux" class="postbox mce-move mc-lateral" style="display:none">
		<div class="inside bg-f2">
			<h3>Upgrade to PRO</h3>
			<p>Get the best Contact Form 7 and Mailchimp integration tool available. Now with these new features:</p>
			<ul>
				<li>Tag Existing Subscribers</li>
				<li>Group Existing Subscribers</li>
				<li>Email Verification</li>
				<li>AWESOME Support And more!</li>
			</ul>
		</div>
		<div class="promo-2022">
			<h1><?php echo (int) $discount_percent; ?><span>%</span> Off!</h1>
			<p class="interesting">Unlock advanced tagging, subscriber groups, email verification, and priority support for your Mailchimp campaigns.</p>
			<div class="cm-form">
				<a href="<?php echo esc_url( $pursuit_addy ); ?>" target="_blank" class="button cm-submit">Get PRO Now</a>
				<span class="cm-pricing"><?php echo esc_html( $pricing_text ); ?></span>
			</div>
		</div>
	</div>
	<?php
}
