<?php
/**
 * Asset enqueueing and cache busting.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;


class Cmatic_Buster {


	private $plugin_version;


	private $is_debug;


	public function __construct( $plugin_version = SPARTAN_MCE_VERSION, $is_debug = null ) {
		$this->plugin_version = $plugin_version;
		$this->is_debug       = $is_debug ?? ( defined( 'WP_DEBUG' ) && WP_DEBUG );
	}


	public function get_version( $file_path ) {
		$version_parts = array( $this->plugin_version );

		if ( file_exists( $file_path ) ) {
			$version_parts[] = filemtime( $file_path );
			$version_parts[] = substr( md5_file( $file_path ), 0, 8 );
		}

		if ( $this->is_debug ) {
			$version_parts[] = time();
		}

		return implode( '-', $version_parts );
	}

	/** Get singleton instance. */
	public static function instance() {
		static $instance = null;

		if ( null === $instance ) {
			$instance = new self();
		}

		return $instance;
	}
}


function cmatic_enqueue_admin_assets( $hook_suffix ) {
	if ( false === strpos( $hook_suffix, 'wpcf7' ) ) {
		return;
	}

	$css_file_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/css/chimpmatic-lite.css';
	wp_enqueue_style(
		'chimpmatic-lite-css',
		SPARTAN_MCE_PLUGIN_URL . 'assets/css/chimpmatic-lite.css',
		array(),
		Cmatic_Buster::instance()->get_version( $css_file_path )
	);

	$modal_css_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/css/chimpmatic-lite-deactivate.css';
	wp_enqueue_style(
		'cmatic-modal-css',
		SPARTAN_MCE_PLUGIN_URL . 'assets/css/chimpmatic-lite-deactivate.css',
		array(),
		Cmatic_Buster::instance()->get_version( $modal_css_path )
	);

	wp_enqueue_style( 'site-health' );

	cmatic_enqueue_lite_js( $hook_suffix );

	$is_pro_installed = defined( 'CMATIC_VERSION' );
	$is_pro_blessed   = function_exists( 'cmatic_is_blessed' ) && cmatic_is_blessed();

	if ( $is_pro_installed ) {
		cmatic_enqueue_pro_js( $hook_suffix, $is_pro_blessed );
	}
}

add_action( 'admin_enqueue_scripts', 'cmatic_enqueue_admin_assets' );

/** Add body class to CF7 admin pages. */
function cmatic_admin_body_class( $classes ) {
	$screen = get_current_screen();

	if ( $screen && strpos( $screen->id, 'wpcf7' ) !== false ) {
		$classes .= ' chimpmatic-lite';

		if ( function_exists( 'cmatic_is_blessed' ) && cmatic_is_blessed() ) {
			$classes .= ' chimpmatic';
		}
	}

	return $classes;
}
add_filter( 'admin_body_class', 'cmatic_admin_body_class' );


function cmatic_enqueue_lite_js( $hook_suffix ) {
	$js_file_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic-lite.js';
	wp_enqueue_script(
		'chimpmatic-lite-js',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic-lite.js',
		array(),
		Cmatic_Buster::instance()->get_version( $js_file_path ),
		true
	);

	$form_id         = isset( $_GET['post'] ) ? absint( $_GET['post'] ) : 0;
	$merge_fields    = array();
	$logging_enabled = false;

	if ( $form_id > 0 ) {
		$option_name = 'cf7_mch_' . $form_id;
		$cf7_mch     = get_option( $option_name, array() );

		if ( isset( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
			$merge_fields = $cf7_mch['merge_fields'];
		}

		$logging_enabled = ! empty( $cf7_mch['logfileEnabled'] );
	}

	wp_localize_script(
		'chimpmatic-lite-js',
		'chimpmaticLite',
		array(
			'restUrl'         => esc_url_raw( rest_url( 'chimpmatic-lite/v1/' ) ),
			'restNonce'       => wp_create_nonce( 'wp_rest' ),
			'licenseResetUrl' => esc_url_raw( rest_url( 'chimpmatic-lite/v1/settings/reset' ) ),
			'nonce'           => wp_create_nonce( 'wp_rest' ),
			'pluginUrl'       => SPARTAN_MCE_PLUGIN_URL,
			'formId'          => $form_id,
			'mergeFields'     => $merge_fields,
			'loggingEnabled'  => $logging_enabled,
			'i18n'            => array(
				'loading'       => __( 'Loading...', 'chimpmatic-lite' ),
				'error'         => __( 'An error occurred. Check the browser console for details.', 'chimpmatic-lite' ),
				'apiKeyValid'   => __( 'API Connected', 'chimpmatic-lite' ),
				'apiKeyInvalid' => __( 'API Inactive', 'chimpmatic-lite' ),
			),
		)
	);
}


function cmatic_enqueue_pro_js( $hook_suffix, $is_pro_blessed ) {
	$pro_js_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic.js';

	wp_enqueue_script(
		'chimpmatic-pro',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic.js',
		array(),
		Cmatic_Buster::instance()->get_version( $pro_js_path ),
		true
	);

	wp_localize_script(
		'chimpmatic-pro',
		'chmConfig',
		array(
			'restUrl'   => rest_url( 'chimpmatic/v1/' ),
			'nonce'     => wp_create_nonce( 'wp_rest' ),
			'isBlessed' => $is_pro_blessed,
		)
	);

	wp_localize_script(
		'chimpmatic-pro',
		'wpApiSettings',
		array(
			'root'  => esc_url_raw( rest_url() ),
			'nonce' => wp_create_nonce( 'wp_rest' ),
		)
	);
}


function cmatic_enqueue_notices_script( $hook_suffix ) {
	if ( false === strpos( $hook_suffix, 'wpcf7' ) ) {
		return;
	}

	$notices_js_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic-lite-notices.js';
	wp_enqueue_script(
		'chimpmatic-lite-notices',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic-lite-notices.js',
		array(),
		Cmatic_Buster::instance()->get_version( $notices_js_path ),
		true
	);

	wp_localize_script(
		'chimpmatic-lite-notices',
		'chimpmaticNotices',
		array(
			'restUrl'   => esc_url_raw( rest_url( 'chimpmatic-lite/v1' ) ),
			'restNonce' => wp_create_nonce( 'wp_rest' ),
		)
	);

}
add_action( 'admin_enqueue_scripts', 'cmatic_enqueue_notices_script' );


/** Enqueue CF7 frontend styles in admin for Test Submission modal. */
function cmatic_enqueue_cf7_frontend_for_test( $hook_suffix ) {
	if ( 'toplevel_page_wpcf7' !== $hook_suffix ) {
		return;
	}

	$form_id = isset( $_GET['post'] ) ? absint( $_GET['post'] ) : 0;
	if ( ! $form_id ) {
		return;
	}

	$cf7_path = WP_PLUGIN_DIR . '/contact-form-7/';
	$cf7_url  = plugins_url( '/', $cf7_path . 'wp-contact-form-7.php' );

	if ( ! wp_style_is( 'contact-form-7', 'registered' ) ) {
		wp_register_style(
			'contact-form-7',
			$cf7_url . 'includes/css/styles.css',
			array(),
			defined( 'WPCF7_VERSION' ) ? WPCF7_VERSION : '5.0',
			'all'
		);
	}

	wp_enqueue_style( 'contact-form-7' );
}
add_action( 'admin_enqueue_scripts', 'cmatic_enqueue_cf7_frontend_for_test' );
