<?php
/**
 * Debug log viewer component.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;

/** Log Viewer class - handles REST API, rendering, and log retrieval. */
class Cmatic_Log_Viewer {

	/** @var string REST namespace. */
	protected static $namespace = 'chimpmatic-lite/v1';

	/** @var string Log prefix to filter for. */
	protected static $log_prefix = '[ChimpMatic Lite]';

	/** @var string Text domain for translations. */
	protected static $text_domain = 'chimpmatic-lite';

	/** @var int Number of lines to read from log file. */
	protected static $max_lines = 500;

	/** @var bool Whether initialized. */
	protected static $initialized = false;

	/** Initialize the log viewer. */
	public static function init( $namespace = null, $log_prefix = null, $text_domain = null ) {
		if ( self::$initialized ) {
			return;
		}

		if ( $namespace ) {
			self::$namespace = $namespace . '/v1';
		}
		if ( $log_prefix ) {
			self::$log_prefix = $log_prefix;
		}
		if ( $text_domain ) {
			self::$text_domain = $text_domain;
		}

		add_action( 'rest_api_init', array( static::class, 'register_routes' ) );
		add_action( 'admin_enqueue_scripts', array( static::class, 'enqueue_assets' ) );

		self::$initialized = true;
	}

	/** Register REST API routes. */
	public static function register_routes() {
		register_rest_route(
			self::$namespace,
			'/logs',
			array(
				'methods'             => 'GET',
				'callback'            => array( static::class, 'get_logs' ),
				'permission_callback' => array( static::class, 'check_permission' ),
			)
		);

		register_rest_route(
			self::$namespace,
			'/logs/clear',
			array(
				'methods'             => 'POST',
				'callback'            => array( static::class, 'clear_logs' ),
				'permission_callback' => array( static::class, 'check_permission' ),
			)
		);
	}

	/** Check if user has permission to access logs. */
	public static function check_permission() {
		return current_user_can( 'manage_options' );
	}

	/** Get the log prefix to filter for. */
	public static function get_log_prefix() {
		return static::$log_prefix;
	}

	/** Get the debug log file path. */
	public static function get_log_path() {
		if ( defined( 'WP_DEBUG_LOG' ) && is_string( WP_DEBUG_LOG ) ) {
			return WP_DEBUG_LOG;
		}
		return WP_CONTENT_DIR . '/debug.log';
	}

	/** REST API callback: Get filtered logs. */
	public static function get_logs( $request ) {
		$log_path = static::get_log_path();
		$prefix   = static::get_log_prefix();

		if ( ! file_exists( $log_path ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Debug log file not found. Ensure WP_DEBUG_LOG is enabled.', self::$text_domain ),
					'logs'    => '',
				),
				200
			);
		}

		$lines = static::read_last_lines( $log_path, self::$max_lines );

		// Filter lines containing our prefix.
		$filtered = array();
		foreach ( $lines as $line ) {
			if ( strpos( $line, $prefix ) !== false ) {
				$filtered[] = $line;
			}
		}

		if ( empty( $filtered ) ) {
			$message = sprintf(
				/* translators: %d: number of lines checked */
				__( 'No %1$s entries found in the recent log data. Note: This viewer only shows the last %2$d lines of the log file.', self::$text_domain ),
				$prefix,
				self::$max_lines
			);
			return new WP_REST_Response(
				array(
					'success' => true,
					'message' => $message,
					'logs'    => '',
					'count'   => 0,
				),
				200
			);
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => '',
				'logs'    => implode( "\n", $filtered ),
				'count'   => count( $filtered ),
			),
			200
		);
	}

	/** REST API callback: Clear the entire debug.log file. */
	public static function clear_logs( $request ) {
		$log_path = static::get_log_path();

		if ( ! file_exists( $log_path ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'cleared' => false,
					'message' => __( 'Debug log file does not exist.', self::$text_domain ),
				),
				200
			);
		}

		$file_handle = @fopen( $log_path, 'w' );

		if ( false === $file_handle ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'cleared' => false,
					'message' => __( 'Failed to clear debug log file.', self::$text_domain ),
				),
				500
			);
		}

		fclose( $file_handle );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log(
				sprintf(
					'[%s] [ChimpMatic Lite] Debug log cleared by user: %s',
					gmdate( 'd-M-Y H:i:s' ) . ' UTC',
					wp_get_current_user()->user_login
				)
			);
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'cleared' => true,
				'message' => __( 'Debug log cleared successfully.', self::$text_domain ),
			),
			200
		);
	}

	/** Read the last N lines from a file efficiently. */
	protected static function read_last_lines( $filepath, $lines = 500 ) {
		$handle = fopen( $filepath, 'r' );
		if ( ! $handle ) {
			return array();
		}

		$result    = array();
		$chunk     = 4096;
		$file_size = filesize( $filepath );

		if ( 0 === $file_size ) {
			fclose( $handle );
			return array();
		}

		$pos    = $file_size;
		$buffer = '';

		while ( $pos > 0 && count( $result ) < $lines ) {
			$read_size = min( $chunk, $pos );
			$pos      -= $read_size;

			fseek( $handle, $pos );
			$buffer       = fread( $handle, $read_size ) . $buffer;
			$buffer_lines = explode( "\n", $buffer );
			$buffer       = array_shift( $buffer_lines );
			$result       = array_merge( $buffer_lines, $result );
		}

		if ( 0 === $pos && ! empty( $buffer ) ) {
			array_unshift( $result, $buffer );
		}

		fclose( $handle );
		return array_slice( $result, -$lines );
	}

	/** Enqueue JavaScript for the log viewer. */
	public static function enqueue_assets( $hook ) {
	}

	/** Get the inline JavaScript for the log viewer. */
	protected static function get_inline_js() {
		$namespace = self::$namespace;

		return <<<JS
(function($) {
	'use strict';

	var CmaticLogViewer = {
		namespace: '{$namespace}',

		// Get REST API root URL (supports both LITE and PRO configurations).
		getRestRoot: function() {
			if (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) {
				return wpApiSettings.root;
			}
			if (typeof chimpmaticLite !== 'undefined' && chimpmaticLite.restUrl) {
				// chimpmaticLite.restUrl includes 'chimpmatic-lite/v1/' already.
				return chimpmaticLite.restUrl.replace(/chimpmatic-lite\/v1\/$/, '');
			}
			// Fallback: construct from current URL.
			return window.location.origin + '/wp-json/';
		},

		// Get REST API nonce.
		getNonce: function() {
			if (typeof wpApiSettings !== 'undefined' && wpApiSettings.nonce) {
				return wpApiSettings.nonce;
			}
			if (typeof chimpmaticLite !== 'undefined' && chimpmaticLite.restNonce) {
				return chimpmaticLite.restNonce;
			}
			return '';
		},

		init: function() {
			$(document).on('click', '.cme-trigger-log', this.toggleLogs.bind(this));
			$(document).on('click', '.vc-clear-logs', this.clearLogs.bind(this));
		},

		toggleLogs: function(e) {
			e.preventDefault();
			var \$container = $('#eventlog-sys');
			var \$trigger = $(e.currentTarget);

			if (\$container.is(':visible')) {
				\$container.slideUp(200);
				\$trigger.text('View Debug Logs');
			} else {
				\$container.slideDown(200);
				\$trigger.text('Hide Debug Logs');
				this.fetchLogs();
			}
		},

		fetchLogs: function() {
			var self = this;
			var \$panel = $('#log_panel');

			\$panel.text('Loading logs...');

			$.ajax({
				url: this.getRestRoot() + this.namespace + '/logs',
				method: 'GET',
				beforeSend: function(xhr) {
					var nonce = self.getNonce();
					if (nonce) {
						xhr.setRequestHeader('X-WP-Nonce', nonce);
					}
				},
				success: function(response) {
					if (response.logs) {
						\$panel.text(response.logs);
					} else {
						\$panel.text(response.message || 'No logs found.');
					}
				},
				error: function(xhr) {
					\$panel.text('Error loading logs: ' + xhr.statusText);
				}
			});
		},

		clearLogs: function(e) {
			e.preventDefault();
			$('#log_panel').text('Logs cleared.');
		},

		refresh: function() {
			if ($('#eventlog-sys').is(':visible')) {
				this.fetchLogs();
			}
		}
	};

	$(document).ready(function() {
		CmaticLogViewer.init();
	});

	// Expose for external use (e.g., after test submission).
	window.CmaticLogViewer = CmaticLogViewer;
})(jQuery);
JS;
	}

	/** Render the log viewer HTML. */
	public static function render( $args = array() ) {
		$defaults = array(
			'title'       => __( 'Submission Logs', self::$text_domain ),
			'clear_text'  => __( 'Clear Logs', self::$text_domain ),
			'placeholder' => __( 'Click "View Debug Logs" to fetch the log content.', self::$text_domain ),
			'class'       => '',
		);

		$args = wp_parse_args( $args, $defaults );
		?>
		<div id="eventlog-sys" class="vc-logs <?php echo esc_attr( $args['class'] ); ?>" style="margin-top: 1em; margin-bottom: 1em; display: none;">
			<div class="mce-custom-fields">
				<div class="vc-logs-header">
					<span class="vc-logs-title"><?php echo esc_html( $args['title'] ); ?></span>
					<a href="#" class="vc-clear-logs"><?php echo esc_html( $args['clear_text'] ); ?></a>
				</div>
				<pre><code id="log_panel"><?php echo esc_html( $args['placeholder'] ); ?></code></pre>
			</div>
		</div>
		<?php
	}

	/** Get inline CSS for the log viewer. */
	public static function get_css() {
		return <<<CSS
/* Log Viewer - Dark Terminal Theme */
.vc-logs .mce-custom-fields {
	background: #0d1b2a;
	font-family: 'JetBrains Mono', 'SF Mono', 'Monaco', 'Inconsolata', 'Fira Mono', 'Droid Sans Mono', 'Source Code Pro', monospace;
	font-size: 12px;
	color: #e0e1dd;
	border-radius: 6px;
	padding: 0;
	overflow: hidden;
}

.vc-logs .vc-logs-header {
	display: flex;
	justify-content: space-between;
	align-items: center;
	padding: 10px 16px;
	border-bottom: 1px solid #1b2838;
	background: #0a1628;
}

.vc-logs .vc-logs-title {
	color: #778da9;
	font-weight: 500;
	font-size: 12px;
}

.vc-logs .vc-clear-logs {
	color: #e63946;
	text-decoration: none;
	font-size: 12px;
}

.vc-logs .vc-clear-logs:hover {
	color: #ff6b6b;
	text-decoration: underline;
}

.vc-logs pre {
	margin: 0;
	padding: 16px;
	background: transparent;
	overflow-x: auto;
	max-height: 400px;
	overflow-y: auto;
}

.vc-logs code {
	background: transparent;
	color: #e0e1dd;
	font-family: inherit;
	font-size: inherit;
	white-space: pre-wrap;
	word-wrap: break-word;
}
CSS;
	}
}
