<?php
/**
 * Plugin activation handler.
 *
 * @package   contact-form-7-mailchimp-extension
 * @author    renzo.johnson@gmail.com
 * @copyright 2014-2026 https://renzojohnson.com
 * @license   GPL-3.0+
 */

defined( 'ABSPATH' ) || exit;


function mce_error() {

	if ( ! file_exists( WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php' ) ) {

		wp_cache_flush();

	} elseif ( ! class_exists( 'WPCF7' ) ) {

		plugin_activation( 'contact-form-7/wp-contact-form-7.php' );

	}
}
add_action( 'admin_notices', 'mce_error' );



function mce_save_date_activation() {
	// Run migration to modern 'cmatic' option structure
	mce_migrate_to_cmatic_option();
}

/**
 * Migrate old mce_* options to consolidated 'cmatic' structure.
 */
function mce_migrate_to_cmatic_option() {
	$cmatic = get_option( 'cmatic' );
	if ( $cmatic && isset( $cmatic['version'] ) ) {
		return;
	}

	$old_loyalty       = get_option( 'mce_loyalty' );
	$old_install_id    = get_option( 'mce_install_id' );
	$old_sent          = get_option( 'mce_sent', 0 );
	$old_news          = get_option( 'mce_show_update_news', 1 );
	$old_notice        = get_option( 'mce_show_notice', 1 );
	$old_panel_content = get_option( 'mce_conten_panel_master' );
	$old_panel_title   = get_option( 'mce_conten_tittle_master' );

	$quest_timestamp = time();
	if ( is_array( $old_loyalty ) && isset( $old_loyalty[0] ) ) {
		$quest_timestamp = $old_loyalty[0];
	}

	if ( ! $old_install_id ) {
		$old_install_id = substr( md5( home_url() . time() . wp_generate_password( 20, false ) ), 0, 12 );
	}

	$cmatic_data = array(
		'version'  => SPARTAN_MCE_VERSION,
		'install'  => array(
			'id'           => $old_install_id,
			'plugin_slug'  => 'contact-form-7-mailchimp-extension',
			'quest'        => $quest_timestamp,
			'activated_at' => gmdate( 'Y-m-d H:i:s', $quest_timestamp ),
		),
		'stats'    => array(
			'sent' => (int) $old_sent,
		),
		'ui'       => array(
			'news'          => (bool) $old_news,
			'notice_banner' => (bool) $old_notice,
			'welcome_panel' => array(
				'title'   => $old_panel_title ? $old_panel_title : 'Chimpmatic Lite is now ' . SPARTAN_MCE_VERSION . '!',
				'content' => $old_panel_content ? $old_panel_content : '',
			),
		),
		'migrated' => true,
	);

	delete_option( 'cmatic' );
	add_option( 'cmatic', $cmatic_data, '', 'no' );

	delete_option( 'mce_loyalty' );
	delete_option( 'mce_install_id' );
	delete_option( 'mce_sent' );
	delete_option( 'mce_show_update_news' );
	delete_option( 'mce_show_notice' );
	delete_option( 'mce_conten_panel_master' );
	delete_option( 'mce_conten_tittle_master' );
}

/**
 * Get value from cmatic option using dot notation.
 *
 * @param string $key     Dot-notation key (e.g., 'install.id').
 * @param mixed  $default Default value if not found.
 * @return mixed
 */
function mce_get_cmatic( $key, $default = null ) {
	$cmatic = get_option( 'cmatic' );
	if ( ! $cmatic ) {
		return $default;
	}

	$keys  = explode( '.', $key );
	$value = $cmatic;

	foreach ( $keys as $k ) {
		if ( ! isset( $value[ $k ] ) ) {
			return $default;
		}
		$value = $value[ $k ];
	}

	return $value;
}

/**
 * Update value in cmatic option using dot notation.
 *
 * @param string $key   Dot-notation key (e.g., 'stats.sent').
 * @param mixed  $value Value to set.
 * @return bool
 */
function mce_update_cmatic( $key, $value ) {
	$cmatic = get_option( 'cmatic', array() );
	$keys   = explode( '.', $key );
	$ref    = &$cmatic;

	foreach ( $keys as $i => $k ) {
		if ( $i === count( $keys ) - 1 ) {
			$ref[ $k ] = $value;
		} else {
			if ( ! isset( $ref[ $k ] ) || ! is_array( $ref[ $k ] ) ) {
				$ref[ $k ] = array();
			}
			$ref = &$ref[ $k ];
		}
	}

	return update_option( 'cmatic', $cmatic );
}



function mce_difer_dateact_date() {
	mce_save_date_activation();

	$quest = mce_get_cmatic( 'install.quest', time() );

	$datetime_ini = new DateTime( 'now' );
	$datetime_fin = new DateTime( '@' . $quest ); // Unix timestamp

	$fechaF = date_diff( $datetime_ini, $datetime_fin );

	if ( $fechaF->y > 0 ) {

		if ( $fechaF->m > 0 ) {

			$differenceFormat = '%y Years %m Months %d Days ';

		} else {

			$differenceFormat = '%y Years %d Days ';

		}
	} elseif ( $fechaF->m > 0 ) {

			$differenceFormat = '%m Months %d Days ';

	} else {

		$differenceFormat = '%d Days ';
	}

	$resultf = $fechaF->format( $differenceFormat );

	return $resultf;
}



function mce_diferdays_dateact_date() {
	mce_save_date_activation();

	$quest = mce_get_cmatic( 'install.quest', time() );

	$datetime_ini = new DateTime( 'now' );
	$datetime_fin = new DateTime( '@' . $quest ); // Unix timestamp

	$fechaF = date_diff( $datetime_ini, $datetime_fin );

	$resultf = $fechaF->format( '%a' );
	return $resultf;
}

if ( ! function_exists( 'mc_meta_links' ) ) {

	function mc_meta_links( $links, $file ) {

		if ( $file === 'contact-form-7-mailchimp-extension/chimpmatic-lite.php' ) {

			$links[] = '<a href="' . esc_url( Cmatic_Pursuit::docs( '', 'plugin_row_meta' ) ) . '" target="_blank" title="Chimpmatic Lite Documentation">Chimpmatic Documentation</a>';
		}
		return $links;
	}

}
add_filter( 'plugin_row_meta', 'mc_meta_links', 10, 2 );



/** Register plugin action links using shared Cmatic_Plugin_Links class. */
if ( ! function_exists( 'mc_after_setup_theme' ) ) {

	function mc_after_setup_theme() {
		if ( class_exists( 'Cmatic_Plugin_Links' ) ) {
			add_filter(
				'plugin_action_links_' . SPARTAN_MCE_PLUGIN_BASENAME,
				array( 'Cmatic_Plugin_Links', 'filter_action_links' )
			);
		}
	}
	add_action( 'after_setup_theme', 'mc_after_setup_theme' );

}

/**
 * Detect existing Mailchimp API keys from other plugins.
 *
 * @return string|false API key if found, false otherwise.
 */
if ( ! function_exists( 'cmatic_detect_existing_api_key' ) ) {
	function cmatic_detect_existing_api_key() {
		$mc4wp = get_option( 'mc4wp' );
		if ( ! empty( $mc4wp['api_key'] ) ) {
			return $mc4wp['api_key'];
		}

		$yikes = get_option( 'yikes-mc-api-key' );
		if ( ! empty( $yikes ) ) {
			return $yikes;
		}

		$easy_forms = get_option( 'yikes-easy-mailchimp-extender-api-key' );
		if ( ! empty( $easy_forms ) ) {
			return $easy_forms;
		}

		$woo_mc = get_option( 'mailchimp-woocommerce' );
		if ( ! empty( $woo_mc['mailchimp_api_key'] ) ) {
			return $woo_mc['mailchimp_api_key'];
		}

		$mc4wp_top_bar = get_option( 'mc4wp_top_bar' );
		if ( ! empty( $mc4wp_top_bar['api_key'] ) ) {
			return $mc4wp_top_bar['api_key'];
		}

		return false;
	}
}

/**
 * Import API key from other plugins to newest CF7 form.
 *
 * @return bool True if imported, false otherwise.
 */
if ( ! function_exists( 'cmatic_import_api_key_on_activation' ) ) {
	function cmatic_import_api_key_on_activation() {
		$existing_key = cmatic_detect_existing_api_key();
		if ( ! $existing_key ) {
			return false;
		}

		$form_id = cmatic_newest();
		if ( ! $form_id ) {
			return false;
		}

		$option_name = 'cf7_mch_' . $form_id;
		$cf7_mch     = get_option( $option_name, array() );

		if ( ! empty( $cf7_mch['api'] ) ) {
			return false;
		}

		$cf7_mch['api'] = $existing_key;
		update_option( $option_name, $cf7_mch );

		return true;
	}
}

/** Plugin activation handler. */
if ( ! function_exists( 'cmatic_on_activation' ) ) {
	function cmatic_on_activation() {
		mce_save_date_activation();
		\Cmatic\Metrics\Core\Sync::send_lifecycle_signal( 'activation' );
		do_action( 'cmatic_metrics_on_activation' );

		if ( cmatic_can_redirect_on_activation() ) {
			mce_update_cmatic( 'activation_redirect', true );
		}
	}
}

/** Plugin deactivation handler. */
if ( ! function_exists( 'cmatic_on_deactivation' ) ) {
	function cmatic_on_deactivation() {
		\Cmatic\Metrics\Core\Sync::send_lifecycle_signal( 'deactivation' );
		do_action( 'cmatic_metrics_on_deactivation' );
	}
}

/** Check if plugin should redirect on activation. */
if ( ! function_exists( 'cmatic_can_redirect_on_activation' ) ) {
	function cmatic_can_redirect_on_activation() {
		if ( isset( $_GET['activate-multi'] ) ) {
			return false;
		}
		if ( is_network_admin() ) {
			return false;
		}
		if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
			return false;
		}
		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			return false;
		}
		return true;
	}
}

/** Redirect to plugin settings on activation. */
if ( ! function_exists( 'cmatic_activation_redirect' ) ) {
	function cmatic_activation_redirect() {
		if ( mce_get_cmatic( 'activation_redirect' ) ) {
			mce_update_cmatic( 'activation_redirect', false );

			if ( isset( $_GET['activate-multi'] ) ) {
				return;
			}

			$latest_form_id = cmatic_newest();
			$redirect_url   = admin_url( 'admin.php?page=wpcf7&post=' . $latest_form_id . '&active-tab=chimpmatic-lite-panel' );

			wp_safe_redirect( $redirect_url );
			exit;
		}
	}

	add_action( 'admin_init', 'cmatic_activation_redirect' );
}

/** Update Pro plugin status in cmatic option. */
if ( ! function_exists( 'cmatic_update_pro_status' ) ) {
	function cmatic_update_pro_status() {
		if ( ! function_exists( 'mce_get_cmatic' ) || ! function_exists( 'mce_update_cmatic' ) ) {
			return;
		}

		$pro_plugin_file = WP_PLUGIN_DIR . '/chimpmatic/chimpmatic.php';
		$pro_installed   = file_exists( $pro_plugin_file );
		$pro_activated   = is_plugin_active( 'chimpmatic/chimpmatic.php' );

		// Check license from function if Pro active, otherwise read from database.
		if ( function_exists( 'cmatic_is_blessed' ) ) {
			$pro_licensed = cmatic_is_blessed();
		} else {
			$license_data = get_option( 'chimpmatic_license_activation', array() );
			$pro_licensed = ! empty( $license_data['activated'] );
		}

		// Get version from constant if active, otherwise read from plugin header.
		$pro_version = null;
		if ( $pro_activated && defined( 'CMATIC_VERSION' ) ) {
			$pro_version = CMATIC_VERSION;
		} elseif ( $pro_installed ) {
			$plugin_data = get_file_data( $pro_plugin_file, array( 'Version' => 'Version' ) );
			$pro_version = $plugin_data['Version'] ?? null;
		}

		// Get license expiration from Pro's stored license data (Pro handles server calls).
		$license_expires = null;
		$license_data    = get_option( 'chimpmatic_license_activation', array() );
		if ( ! empty( $license_data['expires_at'] ) ) {
			// If stored as date string, convert to Unix timestamp.
			$license_expires = is_numeric( $license_data['expires_at'] )
				? (int) $license_data['expires_at']
				: strtotime( $license_data['expires_at'] );
		}

		$new_pro = array(
			'installed'       => $pro_installed,
			'activated'       => $pro_activated,
			'version'         => $pro_version,
			'licensed'        => $pro_licensed,
			'license_expires' => $license_expires,
		);

		$current_pro = mce_get_cmatic( 'install.pro', array() );

		if ( $current_pro !== $new_pro ) {
			mce_update_cmatic( 'install.pro', $new_pro );
		}
	}

	add_action( 'plugins_loaded', 'cmatic_update_pro_status', 99 );
}


/** Sync PRO license instance ID for update compatibility. */
if ( ! function_exists( 'cmatic_sync_pro_license_instance' ) ) {
	function cmatic_sync_pro_license_instance() {
		$activation = get_option( 'chimpmatic_license_activation' );
		if ( ! $activation ) {
			return false;
		}

		if ( is_string( $activation ) ) {
			$activation = maybe_unserialize( $activation );
		}

		$activation_instance = $activation['instance_id'] ?? null;
		if ( ! $activation_instance ) {
			return false;
		}

		$current_instance = get_option( 'cmatic_license_instance' );

		if ( $current_instance !== $activation_instance ) {
			update_option( 'cmatic_license_instance', $activation_instance );
			delete_option( '_site_transient_update_plugins' );
			delete_site_transient( 'update_plugins' );
			return true;
		}

		return false;
	}
}

/**
 * Query Chimpmatic update API directly.
 *
 * @param string $current_version Current installed version.
 * @return object|false Update info object or false on failure.
 */
if ( ! function_exists( 'cmatic_query_pro_update_api' ) ) {
	function cmatic_query_pro_update_api( $current_version ) {
		$activation = get_option( 'chimpmatic_license_activation' );
		if ( ! $activation ) {
			return false;
		}
		if ( is_string( $activation ) ) {
			$activation = maybe_unserialize( $activation );
		}

		$api_key     = $activation['license_key'] ?? '';
		$instance_id = $activation['instance_id'] ?? '';
		if ( empty( $api_key ) || empty( $instance_id ) ) {
			return false;
		}

		$domain  = str_ireplace( array( 'http://', 'https://' ), '', home_url() );
		$api_url = 'https://chimpmatic.com/';
		$args    = array(
			'wc_am_action' => 'update',
			'slug'         => 'chimpmatic',
			'plugin_name'  => 'chimpmatic/chimpmatic.php',
			'version'      => $current_version,
			'product_id'   => 436,
			'api_key'      => $api_key,
			'instance'     => $instance_id,
			'object'       => $domain,
		);

		$target_url = add_query_arg( 'wc-api', 'wc-am-api', $api_url ) . '&' . http_build_query( $args );
		$response   = wp_safe_remote_get( esc_url_raw( $target_url ), array( 'timeout' => 15 ) );

		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		if ( empty( $body ) ) {
			return false;
		}

		$data = json_decode( $body, true );
		if ( ! isset( $data['success'] ) || ! $data['success'] ) {
			return false;
		}

		if ( empty( $data['data']['package']['package'] ) ) {
			return false;
		}

		return (object) array(
			'new_version' => $data['data']['package']['new_version'] ?? '',
			'package'     => $data['data']['package']['package'] ?? '',
			'slug'        => 'chimpmatic',
		);
	}
}

/** Silently update PRO plugin to latest available version. */
if ( ! function_exists( 'cmatic_force_pro_update' ) ) {
	function cmatic_force_pro_update() {
		if ( ! is_admin() ) {
			return;
		}

		if ( wp_doing_ajax() || wp_doing_cron() || defined( 'REST_REQUEST' ) ) {
			return;
		}

		// Check every 12 hours.
		if ( get_transient( 'cmatic_pro_update_check' ) ) {
			return;
		}

		$pro_plugin_file = WP_PLUGIN_DIR . '/chimpmatic/chimpmatic.php';
		if ( ! file_exists( $pro_plugin_file ) ) {
			return;
		}

		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data     = get_plugin_data( $pro_plugin_file, false, false );
		$current_version = $plugin_data['Version'] ?? '0';

		set_transient( 'cmatic_pro_update_check', 1, 12 * HOUR_IN_SECONDS );
		cmatic_sync_pro_license_instance();

		$update_info = cmatic_query_pro_update_api( $current_version );
		if ( ! $update_info || empty( $update_info->package ) ) {
			return;
		}

		// Only update if API returns a newer version.
		if ( version_compare( $update_info->new_version, $current_version, '<=' ) ) {
			return;
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';

		$updates = get_site_transient( 'update_plugins' );
		if ( ! is_object( $updates ) ) {
			$updates = new stdClass();
		}
		if ( ! isset( $updates->response ) ) {
			$updates->response = array();
		}
		$updates->response['chimpmatic/chimpmatic.php'] = $update_info;
		set_site_transient( 'update_plugins', $updates );

		$skin     = new Automatic_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader( $skin );
		$result   = $upgrader->upgrade( 'chimpmatic/chimpmatic.php' );

		if ( true === $result || ( is_array( $result ) && ! empty( $result ) ) ) {
			mce_update_cmatic( 'pro_last_auto_update', $update_info->new_version );
			mce_update_cmatic( 'pro_last_auto_update_at', gmdate( 'Y-m-d H:i:s' ) );
		}
	}

	add_action( 'admin_init', 'cmatic_force_pro_update', 999 );
}
