<?php
/*
* Plugin Name: Consumer Terminal Checkout
* Plugin URI: https://consumerterminalcheckout.com
* Description: CT Checkout is a custom WooCommerce checkout plugin designed to work seamlessly with the CTC app.
* Version: 1.0
* License: GPL2
* Author: Consumer Terminal Checkout
* Text Domain: consumer-terminal-checkout
*/

if (!defined('ABSPATH')) {
    exit;
}

// Ensure WooCommerce is active
function ctcp_check_woocommerce_active()
{
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function () {
            echo '<div class="error"><p><strong>' . esc_html__('CTC Payment Gateway', 'consumer-terminal-checkout') . '</strong> ' . esc_html__('requires WooCommerce to be installed and active.', 'consumer-terminal-checkout') . '</p></div>';
        });
        return;
    }
}
add_action('plugins_loaded', 'ctcp_check_woocommerce_active');

// Register the custom payment gateway
function ctcp_register_payment_gateway($gateways)
{
    $gateways[] = 'CTCP_WC_PaymentGateway';
    return $gateways;
}
add_filter('woocommerce_payment_gateways', 'ctcp_register_payment_gateway');

// Initialize the payment gateway
function ctcp_init_payment_gateway()
{
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class CTCP_WC_PaymentGateway extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'ctc_payment';
            $this->icon = apply_filters('ctcp_icon', plugins_url('assets/icon.png', __FILE__));
            $this->has_fields = true;
            $this->method_title = esc_html__('CTC Payment', 'consumer-terminal-checkout');
            $this->method_description = esc_html__('Pay using our custom method by entering an email.', 'consumer-terminal-checkout');

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_checkout_process', array($this, 'ctcp_validate_payment_fields'));
            add_action('woocommerce_checkout_update_order_meta', array($this, 'ctcp_save_payment_fields'));
            add_action('woocommerce_admin_order_data_after_billing_address', array($this, 'ctcp_display_admin_fields'), 10, 1);
        }

        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => esc_html__('Enable/Disable', 'consumer-terminal-checkout'),
                    'type' => 'checkbox',
                    'label' => esc_html__('Enable / Disable CTC Payment', 'consumer-terminal-checkout'),
                    'default' => 'no',
                ),
                'title' => array(
                    'title' => esc_html__('Title', 'consumer-terminal-checkout'),
                    'type' => 'text',
                    'description' => esc_html__('CTC Payment', 'consumer-terminal-checkout'),
                    'default' => esc_html__('CTC Payment', 'consumer-terminal-checkout'),
                    'desc_tip' => true,
                ),
                'description' => array(
                    'title' => esc_html__('Description', 'consumer-terminal-checkout'),
                    'type' => 'textarea',
                    'description' => esc_html__('This payment method allows use of encrypted card tokens for a payment method.', 'consumer-terminal-checkout'),
                    'default' => esc_html__('Enter your email to complete payment through our custom method.', 'consumer-terminal-checkout'),
                ),
            );
        }

        public function is_available()
        {
            return ('yes' === $this->get_option('enabled'));
        }

        public function payment_fields()
        {
            if ($this->description) {
                echo wp_kses_post(wpautop(wp_kses_post($this->description)));
            }

            // Add a nonce field for checkout
            wp_nonce_field('ctcp_payment_process', 'ctcp_payment_nonce');
?>
            <p>
                <label for="ctcp_payment_email"><?php esc_html_e('Email for Payment:', 'consumer-terminal-checkout'); ?></label>
                <input type="email" name="ctcp_payment_email" id="ctcp_payment_email" required />
            </p>
            <p>
                <label for="ctcp_card_token"><?php esc_html_e('CTC Card Token Number:', 'consumer-terminal-checkout'); ?></label>
                <input type="text" name="ctcp_card_token" id="ctcp_card_token" required />
            </p>
            <p>
                <label for="ctcp_card_exp_date"><?php esc_html_e('Card Expiration Date (MM/YY):', 'consumer-terminal-checkout'); ?></label>
                <input type="text" name="ctcp_card_exp_date" id="ctcp_card_exp_date" placeholder="<?php esc_attr_e('MM/YY', 'consumer-terminal-checkout'); ?>" required />
            </p>
    <?php
        }

        public function ctcp_validate_payment_fields()
        {
            if (
                !isset($_POST['ctcp_payment_nonce']) ||
                !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ctcp_payment_nonce'])), 'ctcp_payment_process')
            ) {
                wc_add_notice(__('Security check failed. Please refresh and try again.', 'consumer-terminal-checkout'), 'error');
                return;
            }

            $email = isset($_POST['ctcp_payment_email']) ? sanitize_email(wp_unslash($_POST['ctcp_payment_email'])) : '';
            $token = isset($_POST['ctcp_card_token']) ? sanitize_text_field(wp_unslash($_POST['ctcp_card_token'])) : '';
            $exp   = isset($_POST['ctcp_card_exp_date']) ? sanitize_text_field(wp_unslash($_POST['ctcp_card_exp_date'])) : '';

            if ('ctc_payment' === WC()->session->get('chosen_payment_method')) {
                if (empty($email) || !is_email($email)) {
                    wc_add_notice(__('Please enter a valid email address for payment.', 'consumer-terminal-checkout'), 'error');
                }
                if (empty($token)) {
                    wc_add_notice(__('Please enter your CTC card token number.', 'consumer-terminal-checkout'), 'error');
                }
                if (empty($exp) || !preg_match('/^(0[1-9]|1[0-2])\/\d{2}$/', $exp)) {
                    wc_add_notice(__('Please enter a valid card expiration date in MM/YY format.', 'consumer-terminal-checkout'), 'error');
                }
            }
        }

        public function ctcp_save_payment_fields($order_id)
        {
            if (
                !isset($_POST['ctcp_payment_nonce']) ||
                !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ctcp_payment_nonce'])), 'ctcp_payment_process')
            ) {
                return; // Nonce check failed
            }

            if (!empty($_POST['ctcp_payment_email'])) {
                update_post_meta($order_id, '_ctcp_payment_email', sanitize_email(wp_unslash($_POST['ctcp_payment_email'])));
            }
            if (!empty($_POST['ctcp_card_token'])) {
                update_post_meta($order_id, '_ctcp_card_token', sanitize_text_field(wp_unslash($_POST['ctcp_card_token'])));
            }
            if (!empty($_POST['ctcp_card_exp_date'])) {
                update_post_meta($order_id, '_ctcp_card_exp_date', sanitize_text_field(wp_unslash($_POST['ctcp_card_exp_date'])));
            }
        }

        public function ctcp_display_admin_fields($order)
        {
            $email = get_post_meta($order->get_id(), '_ctcp_payment_email', true);
            $token = get_post_meta($order->get_id(), '_ctcp_card_token', true);
            $exp_date = get_post_meta($order->get_id(), '_ctcp_card_exp_date', true);

            if ($email) {
                echo '<p><strong>' . esc_html__('Payment Email:', 'consumer-terminal-checkout') . '</strong> ' . esc_html($email) . '</p>';
            }
            if ($token) {
                echo '<p><strong>' . esc_html__('CTC Card Token:', 'consumer-terminal-checkout') . '</strong> ' . esc_html($token) . '</p>';
            }
            if ($exp_date) {
                echo '<p><strong>' . esc_html__('Card Expiration Date:', 'consumer-terminal-checkout') . '</strong> ' . esc_html($exp_date) . '</p>';
            }
        }

        public function process_payment($order_id)
        {
            // Nonce verification
            if (
                !isset($_POST['ctcp_payment_nonce']) ||
                !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ctcp_payment_nonce'])), 'ctcp_payment_process')
            ) {
                wc_add_notice(__('Security check failed. Please refresh and try again.', 'consumer-terminal-checkout'), 'error');
                return;
            }

            $order = wc_get_order($order_id);
            $email = isset($_POST['ctcp_payment_email']) ? sanitize_email(wp_unslash($_POST['ctcp_payment_email'])) : '';
            $card_token = isset($_POST['ctcp_card_token']) ? sanitize_text_field(wp_unslash($_POST['ctcp_card_token'])) : '';
            $card_exp_date = isset($_POST['ctcp_card_exp_date']) ? sanitize_text_field(wp_unslash($_POST['ctcp_card_exp_date'])) : '';

            if (empty($email) || empty($card_token) || empty($card_exp_date)) {
                wc_add_notice(__('Missing required payment fields.', 'consumer-terminal-checkout'), 'error');
                return;
            }

            $items = array();
            foreach ($order->get_items() as $item) {
                $items[] = array(
                    'product_name' => $item->get_name(),
                    'quantity' => $item->get_quantity(),
                );
            }

            $billing_details = array(
                'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                'email' => $order->get_billing_email(),
                'phone' => $order->get_billing_phone(),
                'address' => $order->get_billing_address_1(),
                'address2' => $order->get_billing_address_2(),
                'city' => $order->get_billing_city(),
                'region' => $order->get_billing_state(),
                'postal' => $order->get_billing_postcode(),
                'country' => $order->get_billing_country(),
            );

            $merchant_id = get_option('ctcp_merchant_id', get_option('ctc_merchant_id', ''));
            if (empty($merchant_id)) {
                wc_add_notice(__('Merchant ID is not configured.', 'consumer-terminal-checkout'), 'error');
                return;
            }

            $api_url = 'https://ctc-app-api-1cc2713168ac.herokuapp.com/plugin/';
            $response = wp_remote_post(esc_url_raw($api_url), array(
                'method'    => 'POST',
                'headers'   => array(
                    'Content-Type' => 'application/json',
                    'Accept'       => 'application/json',
                ),
                'body'      => wp_json_encode(array(
                    'merchid'         => $merchant_id,
                    'email'           => $email,
                    'account'         => $card_token,
                    'expiry'          => $card_exp_date,
                    'order_id'        => $order_id,
                    'amount'          => $order->get_total(),
                    'items'           => $items,
                    'billing_details' => $billing_details,
                )),
                'data_format' => 'body',
                'timeout'     => 15,
            ));

            if (is_wp_error($response)) {
                wc_add_notice(__('Payment failed: ', 'consumer-terminal-checkout') . esc_html($response->get_error_message()), 'error');
                return;
            }

            $http_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $result = json_decode($body, true);

            if ($http_code !== 200 || empty($result) || !empty($result['error'])) {
                wc_add_notice(__('Payment API error: ', 'consumer-terminal-checkout') . esc_html($result['error'] ?? 'Unknown error'), 'error');
                return;
            }

            $order->update_status('on-hold', __('Awaiting custom payment confirmation.', 'consumer-terminal-checkout'));
            wc_reduce_stock_levels($order_id);

            return array(
                'result'   => 'success',
                'redirect' => $this->get_return_url($order),
            );
        }
    }

    add_filter('woocommerce_default_gateway', function () {
        return 'ctc_payment';
    });
}
add_action('plugins_loaded', 'ctcp_init_payment_gateway', 11);

// Admin Menu
add_action('admin_menu', 'ctcp_admin_menu');
function ctcp_admin_menu()
{
    add_menu_page('CTC Payment Settings', 'CTC Settings', 'manage_options', 'ctc-settings', 'ctcp_settings_page', 'dashicons-admin-generic', 80);
}

// Settings Page
function ctcp_settings_page()
{
    if (isset($_POST['ctcp_settings_submit'])) {
        if (!isset($_POST['ctcp_settings_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ctcp_settings_nonce'])), 'ctcp_settings_save')) {
            echo '<div class="error"><p>Invalid nonce. Settings not saved.</p></div>';
            return;
        }

        if (isset($_POST['ctcp_merchant_id'])) {
            update_option('ctcp_merchant_id', sanitize_text_field(wp_unslash($_POST['ctcp_merchant_id'])));
            echo '<div class="updated"><p>Settings saved successfully.</p></div>';
        }
    }

    $merchant_id = get_option('ctcp_merchant_id', get_option('ctc_merchant_id', ''));
    ?>
    <div class="wrap">
        <h1>CTC Payment Settings</h1>
        <form method="post" action="">
            <?php wp_nonce_field('ctcp_settings_save', 'ctcp_settings_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th><label for="ctcp_merchant_id">Merchant ID</label></th>
                    <td>
                        <input type="text" name="ctcp_merchant_id" id="ctcp_merchant_id" value="<?php echo esc_attr($merchant_id); ?>" class="regular-text" />
                        <p class="description">Enter your Merchant ID for processing transactions.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button('Save Settings', 'primary', 'ctcp_settings_submit'); ?>
        </form>
    </div>
<?php
}

// Add icon styling
function ctcp_enqueue_styles()
{
    if (is_checkout()) {
        wp_enqueue_style(
            'ctcp-style',
            plugin_dir_url(__FILE__) . 'assets/css/style.css',
            array(),
            '1.0'
        );
    }
}
add_action('wp_enqueue_scripts', 'ctcp_enqueue_styles');
