<?php
/**
 * Submisttion class for CF7
 *
 * @package Cf7zh
 */

namespace Procoders\Cf7zh\Includes;

if (!defined('ABSPATH')) {
	exit('restricted access');
}

use com\zoho\crm\api\HeaderMap;
use com\zoho\crm\api\record\Field;
use com\zoho\crm\api\record\Leads as Leads;
use com\zoho\crm\api\tags\Tag;
use Procoders\Cf7zh\Admin\Logs as Logs;
use Procoders\Cf7zh\Functions as Functions;
use Procoders\Cf7zh\Admin\Init as Init;

use com\zoho\crm\api\record\Record;
use com\zoho\crm\api\record\RecordOperations;
use com\zoho\crm\api\record\APIException;
use com\zoho\crm\api\record\ActionWrapper;
use com\zoho\crm\api\record\SuccessResponse;
use com\zoho\crm\api\record\BodyWrapper;

/**
 * Create or update a person in Zoho.
 *
 * @param Zoho $Zoho The Zoho instance.
 * @param string $name The name of the person.
 * @param string $email The email address of the person.
 * @param string $phone The phone number of the person.
 * @param int|null $org_id The ID of the associated organization (optional).
 *
 * @return int The ID of the created or updated person.
 */
class Submission
{

	/**
	 * Initialize the submission process.
	 *
	 * @param mixed $form The form being submitted.
	 * @param boolean $abort Flag to indicate whether the submission should be aborted.
	 * @param mixed $object The object being submitted.
	 *
	 * @return void
	 */
	public function init($form, &$abort, $object): void
	{
		$access_token = get_option('Cf7zh_access_token');

		// PCF7_Submission i`ts from contact form 7.
		$submission = \WPCF7_Submission::get_instance();

		if (!$submission) {
			Functions::return_error('Contact Form 7 plugin is required.');
		}
		//$post_id = $submission->get_meta( 'container_post_id' );
		$request = $submission->get_posted_data();
		$form_id = $submission->get_contact_form()->id();

		if ($form_id && '0' == !get_post_meta($form_id, 'Cf7zh_active', true)) {
			$Cf7zh_fields = get_post_meta($form_id, 'Cf7zh_fields', true);
			if (null !== $Cf7zh_fields) {
				$data = $this->prepare_data($request, $Cf7zh_fields);
				$data = $this->process_data($data);

				$Cf7zh_update_person = get_post_meta($form_id, 'Cf7zh_update_person', true);
				$Cf7zh_update_org = get_post_meta($form_id, 'Cf7zh_update_org', true);

				$init = new Init();
				$init->zohoInit();

				try {

					$res = $this->createZohoLead($data);

					if (!$res['success']) {
						Logs::handleErrorResponse($res['message'], $form_id);
						$submission->set_status('validation_failed');
						$abort = true;
						$submission->set_response('API submission errors: ' . $this->error_special_cases($res['message']));
					}

				} catch (ZohoException $e) {
					// echo 'Error: ' . $e->getMessage();
					Logs::handleErrorResponse($e->getMessage(), $form_id);
					$submission->set_status('validation_failed');
					$abort = true;
					$submission->set_response('API submission errors: ' . $this->error_special_cases($e->getMessage()));
				}
			}
		}
	}


	/**
	 * Створення нового Lead в Zoho CRM з наявних даних
	 *
	 * @param array $data Масив з даними для створення Lead
	 * @return array Результат створення Lead
	 */
	public function createZohoLead($data)
	{
		try {
			$records = [];
			$tagList = [];
			$record = new Record();

			// Populate $data
			foreach ($data as $type => $field) {
				foreach ($field as $valKey => $val) {
					$valKey = str_replace('_', '', $valKey);
					$type = ucfirst($type);
					if ($type == 'Leads' && $val) {
						$record->addFieldValue(Leads::$valKey(), $val);
					}
				}
			}

			$tag = new Tag();
			$tag->setName("Lead from " . get_site_url());
			$tagList[] = $tag;
			$record->setTag($tagList);
			$records[] = $record;

			$bodyWrapper = new BodyWrapper();
			$bodyWrapper->setData($records);

			$recordOperations = new RecordOperations('leads');
			$headerInstance = new HeaderMap();
			// Відправляємо запит на створення Lead
			$response = $recordOperations->createRecords($bodyWrapper, $headerInstance);

			// Обробка відповіді
			if ($response != null) {
				// Перевіряємо статус-код відповіді
				$statusCode = $response->getStatusCode();

				if ($statusCode == 201) {
					$actionHandler = $response->getObject();

					if ($actionHandler instanceof ActionWrapper) {
						$actionResponses = $actionHandler->getData();

						if ($actionResponses != null && count($actionResponses) > 0) {
							$actionResponse = $actionResponses[0];

							if ($actionResponse instanceof SuccessResponse) {
								$successResponse = $actionResponse;

								return [
									'success' => true,
									'message' => 'Success lead creation',
									'code' => $successResponse->getCode()->getValue(),
									'status' => $successResponse->getStatus()->getValue(),
									'details' => $successResponse->getDetails(),
									'leadId' => $successResponse->getDetails()['id'] ?? null
								];
							} else if ($actionResponse instanceof APIException) {
								$exception = $actionResponse;

								return [
									'success' => false,
									'message' => $exception->getMessage()->getValue(),
									'code' => $exception->getCode()->getValue(),
									'status' => $exception->getStatus()->getValue(),
									'details' => $exception->getDetails()
								];
							}
						}
					} else if ($actionHandler instanceof APIException) {
						$exception = $actionHandler;

						return [
							'success' => false,
							'message' => $exception->getMessage()->getValue(),
							'code' => $exception->getCode()->getValue(),
							'status' => $exception->getStatus()->getValue(),
							'details' => $exception->getDetails()
						];
					}
				} else {
					return [
						'success' => false,
						'message' => 'Lead creation error. Code: ' . $statusCode
					];
				}
			}
		} catch (Exception $e) {
			return [
				'success' => false,
				'message' => 'Error: ' . $e->getMessage()
			];
		}

		return [
			'success' => false,
			'message' => 'Unknown error'
		];
	}


	/**
	 * Create a file in Zoho using the submitted files and attachment fields.
	 *
	 * @param \WPCF7_Submission $submission The submitted form data.
	 * @param Zoho $Zoho The Zoho API client.
	 * @param array $attachment_fields The mapping configuration for attachment fields.
	 * @param int $personId The ID of the person associated with the file.
	 * @param int $organizationId The ID of the organization associated with the file.
	 * @param string $leadId The ID of the lead associated with the file.
	 *
	 * @return int The ID of the created file in Zoho.
	 */
	private function createFile(
		\WPCF7_Submission $submission,
		Zoho              $Zoho,
		array             $attachment_fields,
		int               $personId,
		?int              $organizationId,
		string            $leadId
	): int
	{
		$files = $submission->uploaded_files();
		if (!$files) {
			return false;
		}
		$res = new \stdClass;

		foreach ($attachment_fields as $attachment_field_key => $attachment_field) {
			if (!isset($files[$attachment_field_key]) || !$files[$attachment_field_key]) {
				continue;
			}

			$file = is_array($files[$attachment_field_key])
				? $files[$attachment_field_key][0]
				: $files[$attachment_field_key];

			$file_res = new \SplFileInfo($file);
			$res = $Zoho->files->add([
				'file' => $file_res,
				'person_id' => $personId,
				'lead_id' => $leadId,
				'org_id' => $organizationId,
			])->getData();
		}

		return $res->id;
	}

	/**
	 * Create a note in Zoho.
	 *
	 * @param Zoho $Zoho The Zoho API client.
	 * @param string $content The content of the note.
	 * @param string $lead_id The ID of the lead associated with the note.
	 * @param int $person_id The ID of the person associated with the note.
	 * @param int|null $org_id (Optional) The ID of the organization associated with the note. Default is null.
	 *
	 * @return int The ID of the created note.
	 */
	private function createNotes(
		Zoho   $Zoho,
		string $content,
		string $lead_id,
		int    $person_id,
		?int   $organizationId
	): int
	{
		$notesData = [
			"content" => $content,
			"lead_id" => $lead_id,
			"person_id" => $person_id,
			"add_time" => current_time('mysql'),
		];
		if ($organizationId) {
			$notesData['org_id'] = $organizationId;
		}
		$note = $Zoho->notes->add($notesData)->getData();

		return $note->id;
	}

	/**
	 * Create a lead in Zoho.
	 *
	 * @param Zoho $Zoho The Zoho instance.
	 * @param string $title The title of the lead.
	 * @param int $person_id The ID of the associated person.
	 * @param int|null $org_id The ID of the associated organization (optional).
	 *
	 * @return string The ID of the created lead.
	 */
	private function createLead(
		Zoho   $Zoho,
		string $title,
		int    $person_id,
		string $label,
		?int   $organizationId
	): string
	{
		$leadData = [
			'title' => $title,
			'person_id' => $person_id,
		];
		if (!empty($label)) {
			$leadData['label_ids'] = [$label];
		}
		if ($organizationId) {
			$leadData['organization_id'] = $organizationId;
		}
		$lead = $Zoho->leads->add($leadData)->getData();

		return $lead->id;
	}

	/**
	 * Create or update an organization in Zoho.
	 *
	 * @param Zoho $Zoho The Zoho instance.
	 * @param string $person_name The name of the person associated with the organization.
	 * @param string $address The address of the organization.
	 * @param string $org_name The name of the organization.
	 *
	 * @return int The ID of the created or updated organization.
	 */
	private function createOrUpdateOrganization(
		Zoho   $Zoho,
		string $person_name,
		string $address,
		string $org_name,
		string $label,
		string $people,
		string $update
	): int
	{
		$name = !empty($org_name) ? $org_name : $person_name;
		$organizations = $Zoho->organizations->search($name, ['name'])->getData();

		if (count($organizations->items) > 0 && $update === '1') {
			$organization = $Zoho->organizations->update($organizations->items[0]->item->id, [
				'address' => $address,
				'label' => $label,
				'people_count' => $people
			])->getData();
		} else {
			$organization = $Zoho->organizations->add([
				'name' => $name,
				'address' => $address,
				'label' => $label,
				'people_count' => $people
			])->getData();
		}

		return $organization->id;
	}

	/**
	 * Create or update a person in Zoho.
	 *
	 * @param Zoho $Zoho Instance of Zoho API client.
	 * @param string $name The name of the person.
	 * @param string $email The email of the person.
	 * @param string $phone The phone number of the person.
	 * @param int|null $org_id Optional. The organization ID to associate the person with.
	 *
	 * @return int The ID of the created or updated person.
	 */
	function createOrUpdatePerson(
		Zoho   $Zoho,
		string $name,
		string $firstName,
		string $lastName,
		string $email,
		string $phone,
		string $label,
		string $marketing_status,
		string $update,
		int    $org_id = null
	): int
	{

		$personData = [
			'name' => $name,
			'first_name' => $firstName,
			'last_name' => $lastName,
			'label' => (int)$label,
			'marketing_status' => $marketing_status,
			'email' => [
				'value' => $email,
				'primary' => true,
				'label' => 'Work'
			],
		];
		if (!empty($phone)) {
			$personData['phone'] = [
				'value' => $phone,
				'primary' => true,
				'label' => 'Mobile',
			];
		}
		if ($org_id) {
			$personData['org_id'] = $org_id;
		}

		$response = $Zoho->persons->search($email, ['email']);
		$persons = $response->getData();

		if (count($persons->items) > 0 && $update === '1') {
			$person = $Zoho->persons->update($persons->items[0]->item->id, $personData)->getData();
		} else {
			$person = $Zoho->persons->add($personData)->getData();
		}

		return $person->id;
	}

	/**
	 * Prepare data for submission.
	 *
	 * @param array $request The form submission data.
	 * @param array $Cf7zh_fields Fields mapping configuration.
	 *
	 * @return array Prepared data for submission.
	 */
	private function prepare_data(array $request, array $Cf7zh_fields): array
	{
		$data = [];
		foreach ($Cf7zh_fields as $Cf7zh_field_key => $Cf7zh_field) {
			if (isset($Cf7zh_field['key']) && $Cf7zh_field['key']) {

				$value = $request[$Cf7zh_field_key] ?? null;
				$value = $this->format_value($value, $Cf7zh_field);

				if (null !== $value) {
					$data[$Cf7zh_field['key']] = wp_strip_all_tags($value);
				}
			}
		}

		return $data;
	}

	/**
	 * Format the value based on its type.
	 *
	 * @param mixed $value The value to be formatted.
	 * @param array $Cf7zh_field Field configuration.
	 *
	 * @return mixed The formatted value.
	 */
	private function format_value($value, array $Cf7zh_field)
	{
		if (is_array($value)) {
			$value = implode(';', $value);
		}

		if (('datetime' === $Cf7zh_field['type'] || 'date' === $Cf7zh_field['type']) && $value) {
			$value = strtotime($value) . '000';
		}

		return $value;
	}

	/**
	 * Processes the given data and organizes it into a nested array.
	 *
	 * @param array $data The data to be processed.
	 *
	 * @return array The processed data in the form of a nested array.
	 */
	private function process_data(array $data): array
	{

		$request = [];

		foreach ($data as $key => $value) {
			$parts = explode('_', $key);
			$group = $parts[0];
			array_shift($parts);
			$field_name = implode('_', $parts);
			$request[$group][$field_name] = $value;
		}

		return $request;
	}

	/**
	 * Handle special error cases.
	 *
	 * @param string $error The error message.
	 *
	 * @return string The modified error message if it matches a special case, otherwise the original error message.
	 */
	private function error_special_cases(string $error): string
	{
		switch (trim($error)) {
			case "provided dataset is not valid" :
			{
				return $error . __(', the data sent may not be in the required structure or could be of the wrong data type. Check yours fields mapping.', 'connect-cf7-to-zoho');
			}
			default:
			{
				return $error;
			}
		}
	}

}
